/********************************************************************
 *
 * NAME:     
 *   FXExThread - multiplatform threadding objects
 *
 * AUTHORS:
 *   Daniel Gehriger (gehriger@linkcad.com)
 *
 *   adapted to FOX from the omniThread library.
 *
 *   Copyright (c) 2000 by Daniel Gehriger.
 *   Copyright (c) 1994-1997 Olivetti & Oracle Research Laboratory
 *   Copyright (c) 1994-1999 AT&T Laboratories Cambridge
 *
 * PUPROSE:
 *   Thread abstraction class
 *
 * NOTE
 *
 * This library is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU Library General Public   
 * License as published by the Free Software Foundation; either  
 * version 2 of the License, or (at your option) any later version.
 *                                                                 
 * This library is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.                 
 *                                                                  
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free       
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Id: FXExThread.h,v 1.4 2000/11/27 10:11:48 dgehrige Exp $
 *
 ********************************************************************/
#ifndef FXEX_THREAD_H
#define FXEX_THREAD_H

#ifndef WIN32

# include <pthread.h>
# define FXEX_THREAD_WRAPPER    void* fxexThreadWrapper(void* ptr)

# if defined(__alpha__) && defined(__osf1__) || defined(__hpux__) 
#   ifndef EXC_HANDLING
#     define EXC_HANDLING // stop unnecessary definitions of TRY, etc on OSF
#   endif
# endif

#else /* WIN32 */

# include <windows.h>
# ifndef __BCPLUSPLUS__
#   define FXEX_THREAD_WRAPPER  unsigned __stdcall fxexThreadWrapper(LPVOID ptr)
# else
#   define FXEX_THREAD_WRAPPER  void _USERENTRY fxexThreadWrapper(void *ptr)
# endif

#endif

class FXExThreadEvent;
class FXExMutex;
class FXExCondition;
class FXExSemaphore;
class FXExThread;

////////////////////////////////////////////////////////////////////
//
// FXExThreadFatal : thrown in the event of a fatal error
//
////////////////////////////////////////////////////////////////////
class FXAPI FXExThreadFatal 
{
public:
  int error;
  FXExThreadFatal(int e = 0) : error(e) {}
};

////////////////////////////////////////////////////////////////////
//
// FXExThreadInvalid : thrown when an operation is invoked with 
//                     invalid arguments
//
////////////////////////////////////////////////////////////////////
class FXAPI FXExThreadInvalid {};


////////////////////////////////////////////////////////////////////
//
// FXExMutex class
//
////////////////////////////////////////////////////////////////////
class FXAPI FXExMutex 
{
public:
  FXExMutex(void);
  ~FXExMutex(void);

  // the names lock and unlock are preferred over acquire and release
  // since we are attempting to be as POSIX-like as possible.
  void lock(void);
  void unlock(void);
  void acquire(void) { lock(); }
  void release(void) { unlock(); }

  friend class FXExCondition;

private:
  // dummy copy constructor and operator= to prevent copying
  FXExMutex(const FXExMutex&);
  FXExMutex& operator=(const FXExMutex&);

private:

#ifndef WIN32
  pthread_mutex_t   posix_mutex;
#else
  CRITICAL_SECTION  crit;
#endif
};

////////////////////////////////////////////////////////////////////
//
// FXExMutexLock class
//
// As an alternative to:
// {
//   mutex.lock();
//   .....
//   mutex.unlock();
// }
//
// you can use a single instance of the FXExMutexLock class:
//
// {
//   FXExMutexLock l(mutex);
//   ....
// }
//
// This has the advantage that mutex.unlock() will be called 
// automatically when an exception is thrown.
//
////////////////////////////////////////////////////////////////////
class FXAPI FXExMutexLock 
{
  FXExMutex&    mutex;
  
public:
  FXExMutexLock(FXExMutex& m) : mutex(m) { mutex.lock(); }
  ~FXExMutexLock(void) { mutex.unlock(); }

private:
  // dummy copy constructor and operator= to prevent copying
  FXExMutexLock(const FXExMutexLock&);
  FXExMutexLock& operator=(const FXExMutexLock&);
};

////////////////////////////////////////////////////////////////////
//
// FXExCondition : condition variable
//
////////////////////////////////////////////////////////////////////
class FXAPI FXExCondition 
{
  FXExMutex*    mutex;
  
public:
  // constructor must be given a pointer to an existing mutex. The
  // condition variable is then linked to the mutex, so that there is an
  // implicit unlock and lock around wait() and timed_wait().
  FXExCondition(FXExMutex* m);
  
  ~FXExCondition(void);
  
  // wait for the condition variable to be signalled.  The mutex is
  // implicitly released before waiting and locked again after waking up.
  // If wait() is called by multiple threads, a signal may wake up more
  // than one thread.  See POSIX threads documentation for details.
  void wait(void);
  
  // timedwait() is given an absolute time to wait until.  To wait for a
  // relative time from now, use FXExThread::get_time. See POSIX threads
  // documentation for why absolute times are better than relative.
  // Returns 1 (true) if successfully signalled, 0 (false) if time
  // expired.
  int timedwait(unsigned long secs, unsigned long nanosecs = 0);
  
  // if one or more threads have called wait(), signal wakes up at least
  // one of them, possibly more.  See POSIX threads documentation for
  // details.
  void signal(void);
  
  // broadcast is like signal but wakes all threads which have called
  // wait().
  void broadcast(void);
  
private:
  // dummy copy constructor and operator= to prevent copying
  FXExCondition(const FXExCondition&);
  FXExCondition& operator=(const FXExCondition&);
  
private:

#ifndef WIN32
  pthread_cond_t        posix_cond;
#else
  CRITICAL_SECTION      crit;
  FXExThread*           waiting_head;
  FXExThread*           waiting_tail;
#endif
};


////////////////////////////////////////////////////////////////////
//
// FXExCondition : condition variable
//
////////////////////////////////////////////////////////////////////
class FXAPI FXExSemaphore 
{
public:
  FXExSemaphore(unsigned int initial = 1);
  ~FXExSemaphore(void);
  
  // if semaphore value is > 0 then decrement it and carry on. If it's
  // already 0 then block.
  void wait(void);
  
  // if semaphore value is > 0 then decrement it and return 1 (true).
  // If it's already 0 then return 0 (false).
  int trywait(void);
  
  // if any threads are blocked in wait(), wake one of them up. Otherwise
  // increment the value of the semaphore.
  void post(void);
  
private:
  // dummy copy constructor and operator= to prevent copying
  FXExSemaphore(const FXExSemaphore&);
  FXExSemaphore& operator=(const FXExSemaphore&);
  
private:

#ifndef WIN32
  FXExMutex              m;
  FXExCondition          c;
  int                     value;
#else
  HANDLE                  nt_sem;  
#endif
};

////////////////////////////////////////////////////////////////////
//
// FXExSemaphoreLock : helper class for FXExSemaphore
//
////////////////////////////////////////////////////////////////////
class FXAPI FXExSemaphoreLock 
{
  FXExSemaphore& sem;
public:
  FXExSemaphoreLock(FXExSemaphore& s) : sem(s) { sem.wait(); }
  ~FXExSemaphoreLock(void) { sem.post(); }
private:
  // dummy copy constructor and operator= to prevent copying
  FXExSemaphoreLock(const FXExSemaphoreLock&);
  FXExSemaphoreLock& operator=(const FXExSemaphoreLock&);
};


////////////////////////////////////////////////////////////////////
//                                                                                            
// FXExThread class
//
////////////////////////////////////////////////////////////////////
extern "C" FXEX_THREAD_WRAPPER;

class FXAPI FXExThread  
{
public:

  enum priority_t 
  {
    PRIORITY_LOW,
    PRIORITY_NORMAL,
    PRIORITY_HIGH
  };

  enum state_t 
  {
    STATE_NEW,        // thread object exists but thread hasn't started yet.
    STATE_RUNNING,    // thread is running.
    STATE_TERMINATED  // thread has terminated but storage has not
                      // been reclaimed (i.e. waiting to be joined).
  };

  // Constructors set up the thread object but the thread won't start until
  // start() is called. The create method can be used to construct and start
  // a thread in a single call.

  // construct detached thread  (thread function has (void) return type)
  FXExThread(void (*fn)(void*), 
             void* arg = NULL,
             priority_t pri = PRIORITY_NORMAL);

  // construct undetached (joinable) thread (thread function has (void*) return type)
  FXExThread(void* (*fn)(void*), 
             void* arg = NULL,
             priority_t pri = PRIORITY_NORMAL);

  // start() causes a thread created with one of the constructors to
  // start executing the appropriate function.
  void start(void);

protected:

  // this constructor is used in a derived class.  The thread will
  // execute the run() or run_undetached() member functions depending on
  // whether start() or start_undetached() is called respectively.
  FXExThread(void* arg = NULL, priority_t pri = PRIORITY_NORMAL);

  // can be used with the above constructor in a derived class to cause
  // the thread to be undetached.  In this case the thread executes the
  // run_undetached member function.
  void start_undetached(void);

  // destructor cannot be called by user (except via a derived class).
  // Use exit() or cancel() instead. This also means a thread object must
  // be allocated with new - it cannot be statically or automatically
  // allocated. The destructor of a class that inherits from FXExThread
  // shouldn't be public either (otherwise the thread object can be
  // destroyed while the underlying thread is still running).
  virtual ~FXExThread(void);

public:

  // join causes the calling thread to wait for another's completion,
  // putting the return value in the variable of type void* whose address
  // is given (unless passed a null pointer). Only undetached threads
  // may be joined. Storage for the thread will be reclaimed.
  void join(void**);

  // set the priority of the thread.
  void set_priority(priority_t);

  // create spawns a new thread executing the given function with the
  // given argument at the given priority. Returns a pointer to the
  // thread object. It simply constructs a new thread object then calls
  // start.
  static FXExThread* create(void (*fn)(void*), 
                            void* arg = NULL,
                            priority_t pri = PRIORITY_NORMAL);

  static FXExThread* create(void* (*fn)(void*), 
                            void* arg = NULL,
                            priority_t pri = PRIORITY_NORMAL);

  // terminates the thread that calls this member
  static void exit(void* return_value = NULL);

  // returns the calling thread's FXExThread object.
  // If the calling thread is not the main thread and
  // is not created using this library, returns 0.
  static FXExThread* self(void);

  // allows another thread to run.
  static void yield(void);

  // sleeps for the given time.
  static void sleep(unsigned long secs, unsigned long nanosecs = 0);

  // calculates an absolute time in seconds and nanoseconds, suitable for
  // use in timed_waits on condition variables, which is the current time
  // plus the given relative offset.
  static void get_time(unsigned long* abs_sec, 
                       unsigned long* abs_nsec,
                       unsigned long rel_sec = 0, 
                       unsigned long rel_nsec=0);


  // Use this value as the stack size when spawning a new thread.
  // The default value (0) means that the thread library default is
  // to be used.
  static void stacksize(unsigned long sz);
  static unsigned long stacksize();

private:

  // can be overridden in a derived class.  When constructed using the
  // the constructor FXExThread(void*, priority_t), these functions are
  // called by start() and start_undetached() respectively.
  virtual void  run(void* arg) {}
  virtual void* run_undetached(void* arg) { return NULL; }

  // implements the common parts of the constructors.
  void common_constructor(void* arg, priority_t pri, int det);

  // used to protect any members which can change after construction,
  // i.e. the following 2 members:
  FXExMutex             mutex;

  state_t               _state;
  priority_t            _priority;

  static FXExMutex*     next_id_mutex;
  static int            next_id;
  int                   _id;

  void                  (*fn_void)(void*);
  void*                 (*fn_ret)(void*);
  void*                 thread_arg;
  int                   detached;

public:

  // return this thread's priority.
  priority_t priority(void) { FXExMutexLock l(mutex); return _priority; }

  // return thread state (invalid, new, running or terminated).
  state_t state(void) { FXExMutexLock l(mutex); return _state; }

  // return unique thread id within the current process.
  int id(void) { return _id; }

  // This class plus the instance of it declared below allows us to execute
  // some initialisation code before main() is called.
  class FXAPI init_t 
  { 
    static int count;   
  public:
    init_t(void);
  };

  friend class init_t;

private:
#ifndef WIN32
  pthread_t             posix_thread;
  static int            posix_priority(priority_t);
#else
  HANDLE                handle;
  DWORD                 nt_id;
  void*                 return_val;
  HANDLE                cond_semaphore;
  FXExThread*           cond_next;
  FXExThread*           cond_prev;
  BOOL                  cond_waiting;
  static int            nt_priority(priority_t);
  friend class  FXExCondition;
#endif

  friend FXEX_THREAD_WRAPPER;
};

static FXExThread::init_t omni_thread_init;

#endif /* FXEX_THREAD */