/********************************************************************************
*                                                                               *
*                  Binary log file object                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXBINARYLOGGER_H
#define FXBINARYLOGGER_H

namespace FXEX {

// A simple set of defines used to simplify the binary logging syntax
// Note that the idea of using a binary logger precludes the 'disabling' of the logger as
// a compile time option.  The reasoning is that, this form of logging is often used where
// you have a long 
#define BINDBG(value)        FXBinaryLogger::instance().debug  (value)
#define BINLOG(value)        FXBinaryLogger::instance().log    (value)
#define BINWRN(value)        FXBinaryLogger::instance().warning(value)
#define BINERR(value)        FXBinaryLogger::instance().error  (value)
#define BINSTATUS(arg1,arg2) FXBinaryLogger::instance().status (arg1,arg2)
#define BINUNKNOWN()         FXBinaryLogger::instance().unknown()

/// Binary logger options
enum {
  BINARYLOGGER_SUBSECOND=0x01   /// log entries contain subsecond resolution
  };

/**
 * A binary log file allows applications to log entries to a file, faster and more compact
 * than a normal log file.
 *
 * How this differs from a normal log file is that, up to 1000% of the speed of a log file
 * is consumed in the printf style parsing of the input string.  By using numbers only,
 * no parsing is necessary.  As an added bonus, you dont log many bytes per log entry so
 * the disk I/O time is reduced.
 *
 * The upshot is that a binary logger can run many times quicker than a text file logger,
 * and that the storage of that information is more compact.  The downside is that you need
 * a custom program to read the file (see FXBinaryLogReader).
 *
 * File format:
 * - file header (2 bytes)
 *   - 1 byte version code
 *   - 1 byte options value
 * - written per log entry (8-12 bytes)
 *   - 32bit date stamp
 *     - 1 second resolution
 *   - 32bit subsecond date stamp
 *     - microsecond resolution
 *     - optional
 *   - 32bit log entry
 *     - upper 16bits are used for indicating error severity
 *     - lower 16bits are used for indicating enumerated value
 *
 * Notes:
 * 1. The log file stores the endian-ness of the CPU architecture so that the log file
 *    can later be read back on any type of CPU.
 * 2. Log entries are automatically written on file open/close (as debug entries)
 *
 * It is envisaged that you could use this logger in an embedded application.  You would
 * log the values you need, when you identify a condition that needs to be logged.  You 
 * should create a specific enumerated value, for all possible log cases.
 */

class FXAPI FXBinaryLogger {
private:
  static FXBinaryLogger* thisLog;   // self reference indicator
  FXFileIO              *fileio;    // file IO object
  FXlong                 filesize;  // max size of file
  FXuchar                options;

protected:
  /// opens the log file for writing
  FXbool open();

  /// closes the log file
  void close();

  /**
   * Ensures that the log file is no bigger than size
   * Returns state whether the log file is open
   */
  FXbool trim();

public:
  enum {
    CODE_NORMAL =0x0000,  // normal log code
    CODE_WARNING=0x0001,  // warning log code
    CODE_ERROR  =0x0002,  // error log code
    CODE_DEBUG  =0x0003   // debug log code
    };

  enum {
    LOG_STARTED =0x7FFF,  // log started
    LOG_ENDED   =0x7FFE,  // log ended
    LOG_UNKNOWN =0x7FFD   // log entry unknown
    };

public:
  /// Give me a log file
  FXBinaryLogger(const FXString& file="",FXlong size=100000,FXuint opts=0);

  /// Use the current instance
  static FXBinaryLogger& instance();

  /// Set the size of the log file - checked on next log entry
  void size(const FXlong size);

  /// Get the size of the log file
  FXlong size() { return filesize; }

  /// Change the location of the log file - change is immediate
  FXbool name(const FXString& file);

  /// get the current filename
  FXString name();

  /// indicates whether the log file can/will be written to
  FXbool opened();

  /// log a numeric entry
  void log(FXshort code,FXshort val);

  /// log numeric entries based on generic code
  void log(FXshort val);
  void warning(FXshort val);
  void error(FXshort val);
  void debug(FXshort val);

  /// log an error, using the 'unknown' value
  void unknown();

  /// Save to stream
  friend FXAPI FXStream& operator<<(FXStream& store,const FXBinaryLogger& b);

  /// load from stream
  friend FXAPI FXStream& operator>>(FXStream& store,FXBinaryLogger& b);

  /// done
  virtual ~FXBinaryLogger();
  };

} // namespace FXEX
#endif // FXBINARYLOGGER_H
