/********************************************************************************
*                                                                               *
*               Condition Variable                                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXCONDITION_H
#define FXCONDITION_H

#ifndef FXWAITABLE_H
#include "FXWaitable.h"
#endif
namespace FXEX {
class FXMutex;

/**
 * A condition variable is a mechanism by which a programmer can synchronise
 * multiple threads (those waiting in wait())
 */
class FXAPI FXCondition : public FXWaitable {
  FXDECLARE(FXCondition)

private:
  FXThreadCondition  condition;      // sinalling object
  FXMutex           *conditionMutex; // points to real application mutex

private:
  // dummy copy constructor and operator= to prevent copying
  FXCondition(const FXCondition&);
  FXCondition& operator=(const FXCondition&);
  
public:
  /**
   * Constructor must be given a pointer to an existing mutex. The
   * condition variable is then linked to the mutex, so that there is an
   * implicit unlock and lock around wait().
   * 
   * If you specify a target (such as the main window), a broadcast will
   * also generate an event to that target, on broadcast.
   */
  FXCondition(FXMutex* m=NULL,FXObject *tgt=NULL,FXSelector sel=0);

  /// wait for the condition variable to be signalled.
  void wait();
  
  /**
   * Wait for the condition variable to be signalled.  The mutex is
   * implicitly released before waiting and locked again after waking up.
   * If wait() is called by multiple threads, a signal may wake up more
   * than one thread.  wait() can be given a time to wait until.
   * a time of zero, causes it to wait indefinately, thus always returns
   * true once it has been signaled.
   */
  FXbool trywait(FXuint ms);
  
  /**
   * If one or more threads have called wait(), activate() wakes up at least
   * one of them, possibly more.
   */
  void activate();
  
  /// activateAll() is like activate() but wakes all threads which have called wait().
  void activateAll();
  
  /**
   * A condition variable requires a pointer to a mutex, we allow the CV
   * to be set to other mutex's.  Implicit activateAll so as to ensure that there
   * are no mutex's waiting on us.
   */
  void setMutex(FXMutex *m);

  /// dtor - implicit activateAll() on destruction
  virtual ~FXCondition();
  };

} // namespace FXEX
#endif // FXCONDITION_H
