/********************************************************************************
*                                                                               *
*                  File I/O object                                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXFILEIO_H
#define FXFILEIO_H

#ifndef FXIOHANDLE_H
#include "FXIOHandle.h"
#endif
namespace FXEX {
class FXFileMonitor;

/**
 * The FXFileIO implements file read/write access, including ability to seek
 * around the file, and support for read and write locking all or parts of a file.
 * Detects when new data is available from the file (if the file pointer is at the end
 * of the file), then reads in the new data.
 *
 * You can also memory-map a section of the file, from the current seek point, then
 * treat the pointer returned, as a section of raw memory.
 *
 * Note, file permissions are set to 'secure IO' mode, by default, for all new objects.
 */

enum {
  FILEIO_EXCLUSIVE       = 0x00000001,  // exclusive access to file
  FILEIO_TRUNCATE        = 0x00000002,  // truncate file on opening
  FILEIO_SYNC            = 0x00000004,  // make writes sychronous if possible
  FILEIO_REMOVE_ON_CLOSE = 0x00000008   // remove the file when closed
  };


class FXAPI FXFileIO : public FXIOHandle {
  FXDECLARE(FXFileIO)

  public:
    enum{
      LockUnknown=-1,   // couldn't determine lock status
      Unlocked=0,       // not locked
      LockRead=1,       // this process is currently holding an read lock
      LockWrite=2,      // this process is currently holding an write lock
      LockedRead=3,     // read locked by another process
      LockedWrite=4,    // write locked by another process
      };

  private:
    static FXuint filemodeDefault; // value used for all new FileIO objects

  protected:
    FXString       filename;        // file name
    FXbool         temp;            // indicates if file is a temp file
    FXuint         filemode;        // file permissions (mode flags)
    FXuint         retryInterval;   // retry interval for 'lock' polling
    FXint          locked;          // locked status
    void          *region;          // mapped region
    FXuint         mappedLength;    // length of mapped region
    FXFileMonitor *monitor;         // monitors file activity - used to simulate 'data ready'

  protected:
    FXFileIO(){}

    /// instantiates this class
    virtual FXFileIO* newInstance(FXInputHandle h);

  public:
    enum {
      ID_FILEIO=FXIOHandle::ID_LAST,
      ID_LAST
      };

  public:
    long onConnect(FXObject*,FXSelector,void*);
    long onFileCreated(FXObject*,FXSelector,void*);
    long onFileChanged(FXObject*,FXSelector,void*);
    long onFileClosed(FXObject*,FXSelector,void*);

  public:
    /// get access to a file
    FXFileIO(FXApp *a,const FXString& file="",FXObject *tgt=NULL,FXSelector sel=0,FXuint opts=0);

    /// use an already open file :-)
    FXFileIO(FXInputHandle f,FXApp *a,FXObject *tgt=NULL,FXSelector sel=0);

    /// create a temporary file, deletes on close
    FXFileIO(FXApp *a,FXObject *tgt=NULL,FXSelector=0);

    /// create resource
    virtual void create();

    /// get the filename
    FXString name() const;

    /// set to new file, but only if the file is not open (fails silently)
    void name(const FXString& file);

    /**
     * Set file permissions (mode flags).
     * The permissions that can be set, are taken from the 'unix way' - I have looked at the
     * Win32 security permissions/ACL's for files; get this, there is a permission to stop
     * you deleting a file, but you can happily truncate to file to zero bytes - what the?
     * If you have write access, then you might as well have delete access.
     * In any case, if a better idea comes along we can always change it...
     *
     * If the file exists, the permission is applied to a file immediately.  If the file is
     * yet to be opened, it is applied to the file on opening time.
     */
    void mode(FXuint m);

    /**
     * Returns the permission (mode flags) on the file; if the file exists, the permission
     * is the value from the file; if the file doesn't exist, the value returned is the
     * 'default' value if no previous call to modeDefault() has been called, otherwise it
     * is the value passed in the modeDefault() call
     */
    FXuint mode();

    /// open the file
    virtual FXbool open();

    /// close the file
    virtual void close();

    /// get the current size
    FXlong size();

    /// is the file a temp file
    FXbool isTemp() const { return temp; }

    /// mark the file, as a temp file - temp files are removed on close
    void setTemp(FXbool t=TRUE);

    /**
     * Enable FXFileIO to detect when new data is available from the file.
     * This moves the file pointer to the end of the file, reading in the file as it moves.
     * It then tries to maintain the file pointer at the end of the file.  The idea is that
     * you can treat a file, like a socket, ie continually incoming data.
     *
     * It is also used to enable 'auto-open' feature of FXFileIO which will detect when
     * the filename exists, and if enabled will open the file for you.
     *
     * If you want to have the 'auto-open' but not the 'new-data-available' functionality,
     * handle SEL_IO_CONNECT in your app, then disable the FXFileIO object, finally return 0.
     * eg
     *
     * FXMAPFUNC(SEL_IO_CONNECT,App::ID_FILE,App::onAutoOpen),
     *
     * long App::onAutoOpen(FXObject *sender,FXSelector,void*){
     *   sender->handle(this,FXSEL(SEL_COMMAND,ID_DISABLE),NULL);
     *   return 0;
     *   }
     */
    virtual void enable();

    /// Disable the ability to detect when mroe data is available from the file
    virtual void disable();

    /**
     * Flush/sync data to disk - returns when sync is complete.
     * Used to sunchronise memory mapped files, or when FILEIO_SYNC is not specified.
     */
    void sync();

    /// what is current file pointer position
    FXlong current();

    /// move to the start of file plus some number of pos bytes
    FXbool seekbegin(FXlong pos=0);

    /// move to some point in the file, from where it is currently
    FXbool seek(FXlong pos);

    /// move to the end of the file minus pos
    /// Note: zero is the first byte beyond the end of the file,
    ///       -1 is one past the end of the file (differs from lseek)
    FXbool seekend(FXlong pos=0);

    /// truncate the file to some length
    /// Note: you should really check the size of the file, after you call truncate
    FXbool truncate(FXlong len);

    /// nice and simple lock interface
    FXbool lock() { return writeLock(); }

    /// nice and simple unlock interface
    FXbool unlock(FXlong len);

    /// is the file already locked
    /// - a length of zero implies the whole file
    /// - for the simple interface, a non-zero return code implies some form of lock
    /// - otherwise, returns 'Unlocked' (0) if not locked, or the type of lock
    FXint isLocked(FXlong len=0);

    /// lock a file - read lock
    /// - multiple read locks can be in-force (by multiple processes) at any given time
    /// - never unlocks any other lock
    /// - locking with a length of zero means lock the whole file
    /// - otherwise lock the number of bytes, begining from the current position
    /// - timeout is in milliseconds
    FXbool readLock(FXlong len=0,FXint timeout=0);

    /// lock a file - write lock
    /// - only one lock can be in force at any given time, ie the write lock
    /// - cannot unlock any other lock
    /// - a write lock still allows you to read the file (ie it is a more powerful read lock)
    /// - locking with a length of zero means lock the whole file
    /// - otherwise lock the number of bytes, begining from the current position
    /// - timeout is in milliseconds
    FXbool writeLock(FXlong len=0,FXint timeout=0);
    
    /// this allows us to adjust or retry interval when trying to set a lock
    /// - retry in nanoseconds
    void setLockRetry(FXuint retry) { retryInterval=retry; }

    /// this class supports memory mapping of the file, but only one region at a time...
    /// load part of the file into memory, from the current seek position
    FXbool map(FXlong len,FXbool writeable=TRUE);

    /// unload part of a file from memory
    void unmap();

    /// is the file mapped into memory
    FXbool isMapped() { return region?TRUE:FALSE; }

    /// return pointer to mapped memory
    void* getMapping() { return region; }

    /// duplicate this file handle
    FXFileIO* duplicate(FXInputHandle newHandle=INVALID_HANDLE);

    /// set the default mode for all new FileIO objects
    static void modeDefault(FXuint mode) { filemodeDefault=mode; }

    /// get the default mode of all new FileIO objects
    static FXuint modeDefault() { return filemodeDefault; }

    /// save object to stream
    virtual void save(FXStream& store) const;

    /// load object from stream
    virtual void load(FXStream& store);

    /**
     * Destructor
     * If the object contains a mapped region, the region is unmapped before destruction.
     */
    virtual ~FXFileIO();
  };

} // namespace FXEX
#endif // FXFileIO
