/********************************************************************************
*                                                                               *
*                  Generic history list traversal                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXHISTORYITEMS_H
#define FXHISTORYITEMS_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {

/**
 * Implements generic history list traversal
 */
class FXAPI FXHistoryItems : public FXBaseObject {
  FXDECLARE(FXHistoryItems)

protected:
  struct ItemDict{
    FXString  name;
    void*     data;
    FXint     key;
    ItemDict *previous;
    ItemDict *next;
    };

protected:
  FXString    group;          // MRU Item group
  FXint       maxitems;       // Maximum number of items to track
  FXint       key;            // position of the most recently tracked item (ie *not* the most recent item)
  FXint       amount;         // The current amount of entries being tracked
  FXbool      destroySave;    // Indicates the save state on object destroy
  FXbool      stringData;     // Indicates that the 'data' is an FXString type
  ItemDict   *listStart;      // The first item in the list
  ItemDict   *currentItem;    // The current item in the list

private:
  FXHistoryItems(const FXHistoryItems&);
  FXHistoryItems &operator=(const FXHistoryItems&);

protected:
  /// serialisation
  FXHistoryItems(){};

  /// Reset the keys for each item
  void renumber();

  /// Remove any extra items
  void removeExtras();

public:
  long onCmdClear(FXObject*,FXSelector,void*);
  long onCmdPrevious(FXObject*,FXSelector,void*);
  long onCmdNext(FXObject*,FXSelector,void*);
  long onUpdPrevious(FXObject*,FXSelector,void*);
  long onUpdNext(FXObject*,FXSelector,void*);
  long onCmdItem(FXObject*,FXSelector,void*);
  long onUpdItem(FXObject*,FXSelector,void*);
  long onUpdAnyItems(FXObject*,FXSelector,void*);
  long onUpdNextItems(FXObject*,FXSelector,void*);
  long onUpdPreviousItems(FXObject*,FXSelector,void*);

public:
  // FIXME hmm...
  // since we are derived from FXBaseObject, this enum is kinda broken, but in this
  // case everything still happens to work...
  enum{
    ID_ITEM_NOT_FOUND=-1,
    ID_ITEM_0,
    ID_ITEM_1,
    ID_ITEM_2,
    ID_ITEM_3,
    ID_ITEM_4,
    ID_ITEM_5,
    ID_ITEM_6,
    ID_ITEM_7,
    ID_ITEM_8,
    ID_ITEM_9,
    ID_ITEM_10,
    ID_ITEM_NEXT,
    ID_ITEM_PREVIOUS,
    ID_CLEAR,
    ID_ANY_ITEMS,
    ID_NEXT_ITEMS,
    ID_PREVIOUS_ITEMS,
    ID_LAST
    };

public:
  /**
   * Make new History Items Group with groupname gp,
   * capable of tracking an unlimited (-1) amount of items
   */
  FXHistoryItems(FXApp *a,const FXString& gp="History Items",FXObject* tgt=NULL,FXSelector sel=0,FXint size=-1);

  /// Change number of items we're tracking
  FXint size(const FXint mx);

  /// Return the maximum number of items capable being tracked
  FXint size() const { return maxitems; }

  /// Get the amount of entries in the list
  FXint no() { return amount; }

  /// Clear the list of items
  void clear();

  /// Get current index number
  FXint index() const { return key; }

  /// Set current index number - returns the current item pointed to
  FXint index(const FXint item);

  /// Get the previous/older item index, if it exists - stays on the same index
  FXint previous();

  /// Get the next/newer item index, if it exists - stays on the same index
  FXint next();

  /// Get the index of particular item name
  FXint find(const FXString& name);

  /// Get name of current item
  FXString name() { return currentItem->name; }

  /// get data of current item
  virtual void* data() { return currentItem->data; }

  /// Remove current item
  FXint remove();

  /// Remove an item by name
  FXint remove(const FXString& name);

  /// Remove an item by index
  FXint remove(const FXint item);

  /**
   * Add an item to the start of the list
   * --> made virtual so that decendants can change the 'add' behaviour
   */
  virtual FXint add(const FXString& name,void* data=NULL);

  /// Remove items newer than specified item - returns the current item pointed to
  FXint removeNewer(const FXint item);

  /// Remove items older than specified item - returns the current item pointed to
  FXint removeOlder(const FXint item);

  /**
   * Moves the 'named item' to the top of the list
   * Sets the current item reference to the top of the list
   */
  FXint moveFirst(const FXString& name);

  /**
   * Moves the 'item' to the top of the list
   * Sets the current item reference to the top of the list
   */
  FXint moveFirst(const FXint item);

  /// Saves the list to the registry on object destroy
  void saveOnDestroy(FXbool state);

  /// Saves the 'data' to the registry if it declared to an FXString type
  void dataIsString(FXbool state);

  /// Removes this list from the registry
  void clearReg();

  /// Save the current history to the registory - returns the current item
  FXint writeReg();

  /// Load the history from the registory - returns the most recent item
  FXint readReg();

  /// Save to a stream
  virtual void save(FXStream& store) const;

  /// Load from a stream
  virtual void load(FXStream& store);

  /// Destructor
  virtual ~FXHistoryItems();
  };

} // namespace FXEX
#endif // FXHISTORYITEMS_H
