/*******************************************************************************
* Copyright (C) 2001,2002 by H. J. Daniel III. All Rights Reserved.            *
********************************************************************************
* This library is free software; you can redistribute it and/or                *
* modify it under the terms of the GNU Lesser General Public                   *
* License as published by the Free Software Foundation; either                 *
* version 2.1 of the License, or (at your option) any later version.           *
*                                                                              *
* This library is distributed in the hope that it will be useful,              *
* but WITHOUT ANY WARRANTY; without even the implied warranty of               *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU            *
* Lesser General Public License for more details.                              *
*                                                                              *
* You should have received a copy of the GNU Lesser General Public             *
* License along with this library; if not, write to the Free Software          *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.   *
*******************************************************************************/
#ifndef FXWIZARDDIALOG_H
#define FXWIZARDDIALOG_H

#ifndef FXDIALOGBOX_H
#include <fox/FXDialogBox.h>
using namespace FX;
#endif
namespace FXEX {
class FXWizardPage;

/**
 * When used in describing the default FXWizardDialog image, it sets the default 
 * image as having NO image. When used in describing a FXWizardPage, it sets
 * the page as having NO image.  NULL means use default image.
 */
#define FXWizardEmptyImage ((FXImage*) -1)

///  The following are some predefined placement types, for FXWizardDialog
#define FXPlacementDefault    FXPoint(-1,-1)   /// Place it at the default size and location
#define FXPlacementVisible    FXPoint(-2,-2)   /// Place window to be fully visible
#define FXPlacementCursor     FXPoint(-3,-3)   /// Place it under the cursor position
#define FXPlacementOwner      FXPoint(-4,-4)   /// Place it centered on its owner
#define FXPlacementScreen     FXPoint(-5,-5)   /// Place it centered on the screen
#define FXPlacementMaximized  FXPoint(-6,-6)   /// Place it maximized to the screen size

/**
 *  FXWizardDialog : Adds support for Windows like wizard dialogs.
 *
 *  FXWizardDialog is the central class for implementing 'wizard-like' dialogs. These
 *  dialogs are mostly familiar to Windows users and are nothing else but a 
 *  sequence of 'pages' each of them displayed inside a dialog which has the 
 *  buttons to pass to the next (and previous) pages.
 *  
 *  The wizards are typically used to decompose a complex dialog into several 
 *  simple steps and are mainly useful to the novice users, hence it is 
 *  important to keep them as simple as possible.
 *  
 *  To show a wizard dialog, you must first create an object of FXWizardDialog class.
 *  Then you should add all pages you want the wizard to show and then call 
 *  FXWizardDialog::run.
 */
class FXAPI FXWizardDialog : public FXDialogBox {
  FXDECLARE(FXWizardDialog);

private:
  // the global image to display with the wizard pages. This is the image that will
  // be displayed if the current pages' image is set to FXNullImage. If the page
  // does not want any image displayed it should use FXEmptyImage.
  FXImage *wizImage;

  // the current page or NULL
  FXWizardPage *page;       

  // wizard dimensions
  FXuint width;       // the size of the wizard itself
  FXuint height;      // (total width is width + m_x)

  // the page size requested by user
  FXSize sizePage;

  // the dialog position from the ctor
  FXPoint posWizard;

  // wizard controls and contents
  FXButton *btnPrev;     // the "<Back" button
  FXButton *btnNext;     // the "Next>" or "Finish" button
  FXButton *btnCancel;   // the "Cancel" button
  FXHorizontalSeparator *separator;  //the separator
  FXImageFrame *wizImageFrame; // frame around image

private:
  // Moves the upper right hand corner of all of the FXWizardPage's to a
  // specified location in the FXWizard's coordinate space.
  void movePages(FXuint x, FXuint y);

  // Displays the wizard with the currently active FXWizardPage.
  FXbool showPage(FXWizardPage *pg, FXbool goingForward);
  
  // Construct the default controls used in the wizard
  void constructControls();
  
  // Was the dialog really created?
  bool wasCreated() const { return btnPrev != NULL; }

  // Routine to find the minimum size required to fully display all
  // the pages.
  FXSize getPageSizes();

  // Routine to set the minimum size required to fully display all
  // the pages.
  void setPageSizes(FXuint w, FXuint h);
  
  // Finds the maximum size required by all of the page images.
  FXSize getImageSizes();

protected:
  FXWizardDialog(){}  //no default constructor allowed.

public:
  enum {
    /// This message is sent to the wizard when the <PREV> button is selected
    ID_WIZARD_PREV=FXDialogBox::ID_LAST,
    /// This message is sent to the wizard when the <NEXT> button is selected
    ID_WIZARD_NEXT,
    /*! This message is sent to the wizard when the <CANCEL> button is selected,
     * and, if the wizard contains a title bar, when the user selects [X].
    */
    ID_WIZARD_CANCEL,
    /// This message is sent to the wizard when the <HELP> button is selected
    ID_WIZARD_HELP,
    ID_LAST
    };

public:
  // these should not be overridden.

  /// Called when the <NEXT>, <PREV>, and <FINISH>(last page only) is pressed.
  long onPrevOrNext(FXObject*,FXSelector,void* ptr);

  /// Called when the <CANCEL> or [X](if toolbar is active) is pressed.
  long onCancel(FXObject*,FXSelector,void* ptr);

  /// Called when the <HELP> button is pressed.
  long onHelp(FXObject*,FXSelector,void* ptr);

public:
  /*! Holds an array of FXWizard pages. This is the only thing that effects 
   * portablility(it uses template classes). Therefore future versions may replace 
   * this with a more portable method. But for now it works :)
   * This should also be made private and accessed through member functions but
   * i'll save that for later.
  */
  FXArray<FXWizardPage*> pageArray;

public:
  /*! Construct a free-floating FXWizardDialog dialog. This is the construtor to 
   * use if creating a dialog-based FXWizardDialog application.
   * \param app The application object.
   * \param name The name of the wizard which will be display on the title bar
   * \param image The default image to display on the left of the pages. 
   *              This should be a user image or one the @link 
   *              DefaultImageGroup default image types@endlink.
   * \param position A FXPoint location on the display where the wizard will
   *                 will be displayed. The value can be an explicit FXPoint
   *                 or one of the @link DefaultPlacementGroup default 
   *                 placement types@endlink.
   * \param padding The padding between the different components in the FXWizardDialog.
  */
  FXWizardDialog(FXApp *app,const FXString& name,FXImage* image=NULL,FXPoint position=FXPlacementScreen,FXint padding=DEFAULT_PAD);

  /*! Construct a model FXWizardDialog dialog which will always float over the owner window.
   * \param owner The parent window to which the wizard is owned by.
   * \param name The name of the wizard which will be display on the title bar
   * \param image The default image to display on the left of the pages. 
   *              This should be a user image or one the @link 
   *              DefaultImageGroup default image types@endlink.
   * \param position A FXPoint location on the display where the wizard will
   *                 will be displayed. The value can be an explicit FXPoint
   *                 or one of the @link DefaultPlacementGroup default 
   *                 placement types@endlink.
   * \param padding The padding between the different components in the FXWizardDialog.
  */
  FXWizardDialog(FXWindow *owner,const FXString& name,FXImage* image=NULL,FXPoint position=FXPlacementScreen,FXint padding=DEFAULT_PAD);

  /// Create the server side resources for the wizard.
  void create();
  
  /** Executes the wizard starting from the given page, returns TRUE if it 
   *  was successfully finished or FALSE if user cancelled it. The first Page
   *  can not be NULL.
  */
  int run(FXWizardPage* startPage);

  /** Moves the wizard window to a specific location. This can be called anytime 
   *  before FXWizardDialog::run()
  */
  void move(FXuint x, FXuint y);

  /** Moves the wizard window to a specific location. This can be called anytime 
   *  before FXWizardDialog::run()
  */
  void move(FXPoint pos);

  /** A convenience function to make the pages follow each other. A NULL 
   *  value should terminate the list.
   */
  static void chain(FXWizardPage *first, ...);

  /// dtor
  virtual ~FXWizardDialog();
  };

} // namespace FXEX
#endif // FXWIZARDDIALOG_H
