/********************************************************************************
*                                                                               *
*                  Binary log file object                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXFile.h>
#include <fox/FXFont.h>
#include <fox/FXComposite.h>
using namespace FX;
#include "fxexdefs.h"
#include "fxexutils.h"
#include "FXDateTime.h"
#include "FXFileIO.h"
#include "FXBinaryLogger.h"
using namespace FXEX;
namespace FXEX {

/*
 * Notes:
 */

#define ENDIAN_MASK  (0x80)

FXBinaryLogger* FXBinaryLogger::thisLog = NULL;

// ctor
FXBinaryLogger::FXBinaryLogger(const FXString& file,FXlong size,FXuint opts) {
  if (size<4) fxerror("FXBinaryLogger: file size too small\n");
  filesize=size;
  options=opts;
  if (FXStream::isLittleEndian()) options|=ENDIAN_MASK;
  fileio=new FXFileIO(FXApp::instance(),file);
  if (file.length() && size) open();
  thisLog=this;
  }

// dtor
FXBinaryLogger::~FXBinaryLogger(){
  thisLog=(FXBinaryLogger*)-1;
  close();
  delete fileio;
  }

// instance
FXBinaryLogger& FXBinaryLogger::instance(){
  if (!thisLog) {
    FXString name=FXApp::instance()->getAppName();
    name.append(".binlog");
    new FXBinaryLogger(name);
    }
  return *thisLog;
  }

// set the size of the file
void FXBinaryLogger::size(const FXlong size){
  if (size<4) fxerror("FXBinaryLogger: file size too small\n");
  filesize=size;
  }

// set the logfile filename
FXbool FXBinaryLogger::name(const FXString& file){
  close();
  fileio->name(file);
  fileio->mode(FXUtils::fxfilemode(FILEPERM_DEFAULT_IO));
  return open();
  }

// get the logfile filename
FXString FXBinaryLogger::name(){
  return fileio->name();
  }

// return indication if file is open
FXbool FXBinaryLogger::opened(){
  return fileio->opened();
  }

// open the logfile - if we can
FXbool FXBinaryLogger::open(){
  trim();
  if (!fileio->opened()){
    fileio->create();
    if (!fileio->opened()) return FALSE;
    FXuchar data[2];
    data[0] = 1;  // binary logger version number
    data[1] = options;
    fileio->write(data,2);
    log(CODE_DEBUG,LOG_STARTED);
    }
  return TRUE;
  }

// close the logfile
void FXBinaryLogger::close(){
  if (fileio->opened()){
    log(CODE_DEBUG,LOG_ENDED);
    fileio->destroy();
    }
  }

// checks the current file size against the required size
FXbool FXBinaryLogger::trim(){
  if ( FXFile::size(fileio->name()) > filesize ){
    close();
    FXFile::move(fileio->name(), FXStringFormat("%s.old",fileio->name().text()), TRUE);
    open();
    }
  return fileio->opened();
  }

// save to stream
FXStream& operator<<(FXStream& store,const FXBinaryLogger& b){
  store << b.fileio;
  store << b.filesize;
  FXuchar opts = b.options&~ENDIAN_MASK;
  store << opts;
  return store;
  }

// load from stream
FXStream& operator>>(FXStream& store,FXBinaryLogger& b){
  store >> b.fileio;
  store >> b.filesize;
  store >> b.options;
  if(FXStream::isLittleEndian()) b.options&=ENDIAN_MASK;
  return store;
  }

// write the data to the log file
void FXBinaryLogger::log(FXshort code,FXshort val){
  if (fileio->opened() && trim()){
    FXint time = (FXint) FXDateTime::now();
    if (options & BINARYLOGGER_SUBSECOND) {
      FXuchar data[12];
      FXint sub = FXDateTime::microseconds();
      memcpy(&data[0],&time,4);
      memcpy(&data[4],&sub,4);
      memcpy(&data[8],&code,2);
      memcpy(&data[10],&val,2);
      fileio->write(data,12);
      }
    else {
      FXuchar data[8];
      memcpy(&data[0],&time,4);
      memcpy(&data[4],&code,2);
      memcpy(&data[6],&val,2);
      fileio->write(data,8);
      }
    }
  }

// log a normal entry
void FXBinaryLogger::log(FXshort val){
  log(CODE_NORMAL,val);
  }

// log a warning entry
void FXBinaryLogger::warning(FXshort val){
  log(CODE_WARNING,val);
  }

// log an error entry
void FXBinaryLogger::error(FXshort val){
  log(CODE_ERROR,val);
  }

// log a debug entry
void FXBinaryLogger::debug(FXshort val){
  log(CODE_DEBUG,val);
  }

}

