/********************************************************************************
*                                                                               *
*                  BZip compression on arbitrary streams                        *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#ifdef HAVE_LIBBZ2
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXObject.h>
using namespace FX;
#include "fxexdefs.h"
#include "FXBzipStream.h"
#include <bzlib.h>
using namespace FXEX;
namespace FXEX {

// setup the BZIP library
FXBzipStream::FXBzipStream(FXStream *s,FXint compFactor,const FXObject* cont) : FXBufferedStream(s,cont),blockSize(compFactor) {
  code=FXStreamOK;
  verbosity=0;
  workFactor=0;
  small=0;
  }

// Take data in 'ptr' and compress it, placing result back in 'ptr', then allowing base
// class to push data into target stream
//
// Note: we also store the size of the data chunk, to the resultant buffer - this is so that we can
//       retrieve its value during the load operation, so we know how much memory to allocate
FXbool FXBzipStream::bufferSave(){
  unsigned long tmppos=pos;
  unsigned long possize=sizeof(pos);
  FXuchar *tmpptr = ptr;
  FXuchar *dest;
  unsigned long size;
  unsigned long total;

  // we must allocate enough memory to ensure that the compressed data will fit
  // => 1% larger than the source buffer + 600 bytes (as recommended by the bzip docs)
  size = lround(pos * 1.01 + 600 + 0.5);

  // Note: the extra bytes in the sizeof() allows us to save the source buffer size to the stream
  total = possize + size;
  if (! FXMALLOC(&dest,FXuchar,total)) return FALSE;
  memcpy(dest,&pos,possize);

  // compress the buffer - if compresses is ok:
  // - resize the temp buffer to be the same size as the compressed data
  // - make the temp buffer the current buffer
  if (BZ_OK != BZ2_bzBuffToBuffCompress((char*)&dest[possize],(unsigned int*)&size,(char*)ptr,pos,blockSize,verbosity,workFactor) ){
    FXFREE(&dest);
    return FALSE;
    }
  total = possize + size;
  if (! FXRESIZE(&dest,FXuchar,total)) { FXFREE(&dest); return FALSE; }
  ptr=dest;
  pos=total;

  // allow base class to forward the data to target stream
  FXbool result= FXBufferedStream::bufferSave();
  FXFREE(&dest);
  ptr=tmpptr;
  pos=tmppos;
  return result;
  }

// allow the base class to load in the data from the underlying stream, then
// take the data in 'ptr' and decompress it, placing the result back in 'ptr' :-)
// FIXME: this appears to make ptr grow in size
FXbool FXBzipStream::bufferLoad(){
  if ( FXBufferedStream::bufferLoad() ){

    // figure out how much memory to allocate for the incoming uncompressed data
    unsigned long size=0;
    unsigned long ssize=sizeof(size);
    memcpy(&size,ptr,ssize);
    FXuchar *dest;
    if (! FXMALLOC(&dest,FXuchar,size)) return FALSE;

    // decompress the data
    if (BZ_OK == BZ2_bzBuffToBuffDecompress((char*)dest,(unsigned int*)&size,(char*)&ptr[ssize],pos,small,verbosity)) {
      FXFREE(&ptr);
      ptr=dest;
      return position(0);
      }
    FXFREE(&dest);
    }
  return FALSE;
  }

}
#endif
