/********************************************************************************
*                                                                               *
*                  File I/O object                                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#ifndef WIN32
#  ifndef _POSIX_MAPPED_FILES
#    define _POSIX_MAPPED_FILES
#  endif
#  ifndef _POSIX_SYNCHRONIZED_IO
#    define _POSIX_SYNCHRONIZED_IO
#  endif
#endif
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXFile.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXFileMonitor.h"
#include "FXFileIO.h"
#include "FXDateTime.h"
using namespace FXEX;
namespace FXEX {

/* 
 * Notes:
 * - we prefer to use fcntl rather than flock/lockf since each vender may implement
 *   those function calls slightly differently
 */

#define RETRY_INTERVAL 1000000       // 1 second

// map
FXDEFMAP (FXFileIO) FXFileIOMap[]={
  FXMAPFUNC(SEL_IO_WRITE,FXFileIO::ID_FILEIO,FXFileIO::onConnect),
  FXMAPFUNC(SEL_CREATE,FXFileIO::ID_FILEIO,FXFileIO::onFileCreated),
  FXMAPFUNC(SEL_CHANGED,FXFileIO::ID_FILEIO,FXFileIO::onFileChanged),
  FXMAPFUNC(SEL_DELETE,FXFileIO::ID_FILEIO,FXFileIO::onFileClosed),
  FXMAPFUNC(SEL_CLOSED,FXFileIO::ID_FILEIO,FXFileIO::onFileClosed),
  };
FXIMPLEMENT(FXFileIO,FXIOHandle,NULL,0)

// set default permission for all FXFileIO objects
FXuint FXFileIO::filemodeDefault = FILEPERM_SECURE_IO;


// open file using specific filename
FXFileIO::FXFileIO(FXApp *a,const FXString& file,FXObject *tgt,FXSelector sel,FXuint opts) : FXIOHandle(a,tgt,sel) {
  temp=FALSE;
  filename=file;
  filemode=filemodeDefault;
  retryInterval=RETRY_INTERVAL;
  options&=opts;
  region=NULL;
  if(file.length()) code=FXIOStatusOk;
  monitor=new FXFileMonitor(getApp(),file,2000,0,this,ID_FILEIO);
  }

// use pre-exisitng open'ed file
FXFileIO::FXFileIO(FXInputHandle f,FXApp *a,FXObject *tgt,FXSelector sel) : FXIOHandle(f,a,tgt,sel){
  temp=FALSE;
  filemode=filemodeDefault;
  retryInterval=RETRY_INTERVAL;
  region=NULL;
  state=FXIOStateOk;
  code=FXIOStatusOk;
  monitor=NULL;
  }

// open a temporary file
FXFileIO::FXFileIO(FXApp *a,FXObject *tgt,FXSelector sel) : FXIOHandle(a,tgt,sel){
  temp=TRUE;
  filemode=filemodeDefault;
  retryInterval=RETRY_INTERVAL;
  region=NULL;
  code=FXIOStatusOk;
  monitor=NULL;
  }

// dtor
FXFileIO::~FXFileIO(){
  if (region) unmap();
  if (monitor) delete monitor;
  }

// create resources
void FXFileIO::create(){
  if (state != FXIOStateNone && !filename.length()) return;
  FXTRACE((100,"%s::create %p\n",getClassName(),this));
  FXIOHandle::create();
  if (code!=FXIOStatusOk) fxwarning("%s: create() error\n",getClassName());
  if (monitor) monitor->create();
  if(!opened()) open();
  }

// save to stream
void FXFileIO::save(FXStream &store) const {
  FXIOHandle::save(store);
  store << filename;
  store << filemode;
  store << retryInterval;
  }

// load from stream
void FXFileIO::load(FXStream &store) {
  FXIOHandle::load(store);
  store >> filename;
  store >> filemode;
  store >> retryInterval;
  }

// return the filename
FXString FXFileIO::name() const {
  return temp ? FXString::null : filename;
  }

// set to new filename, if file is closed
void FXFileIO::name(const FXString& file){
  if (closed()) {
    filename=file;
    code=FXIOStatusOk;
    }
  }

// Set permission on the already existing file, or the one about to be created.
void FXFileIO::mode(FXuint m) {
  filemode=m;
  if (FXFile::exists(name())) FXFile::mode(name(),m);
  }

// get the permission on an already existing file, or the one about to be created.
FXuint FXFileIO::mode() {
  if (FXFile::exists(name())) return FXFile::mode(name());
  return filemode;
  }

// open the file
FXbool FXFileIO::open() {
  // make sure we haven't already opened a file,
  // see what the parents says about our handle...
  if (state!=FXIOStateNone) return FALSE;
  if (!FXIOHandle::open()) return FALSE;

  // open the file
  if (temp) {
    FXString tmpfile=FXFile::getTempDirectory() + PATHSEP + "XXXXXX";
    FXchar t[MAXPATHLEN];
    strcpy(t,tmpfile.text());
#ifndef WIN32
    iohandle=::mkstemp(t);
    if (iohandle==INVALID_HANDLE) { code=errno; return FALSE; }
#else
     FIXME implement something for Win32
#endif
    filename=t;
    filemode=FXFile::mode(t);
    }
  else {
    if (!FXFile::exists(name())) return FALSE;
#ifndef WIN32
    FXint flags= O_RDWR|O_CREAT;
    flags |= options&FILEIO_EXCLUSIVE ? O_EXCL : 0;
    flags |= options&FILEIO_TRUNCATE ? O_TRUNC : 0;
    flags |= options&FILEIO_SYNC ? O_SYNC : 0;
    iohandle=::open(filename.text(),flags,filemode);
    if (iohandle==INVALID_HANDLE) { code=errno; return FALSE; }
#else
    DWORD flags = GENERIC_READ|GENERIC_WRITE;
    DWORD share = options&FILEIO_EXCLUSIVE ? 0 : FILE_SHARE_READ|FILE_SHARE_WRITE;
    DWORD trunc = options&FILEIO_TRUNCATE ? CREATE_ALWAYS : OPEN_ALWAYS;
    DOWRD sync  = options&FILEIO_SYNC ? SYNCHRONIZE : 0;
    iohandle=CreateFile(filename.text(),sync&flags,share,NULL,trunc,NULL);
    if (iohandle==INVALID_HANDLE) { code=errno; return FALSE; }
    FXFile::mode(filename,filemode);
#endif
    }
  FXTRACE((100,"file opened"));
  state=FXIOStateOpen;
  handle(this,FXSEL(SEL_OPENED,0),(void*)this);
  return TRUE;
  }

// delete the file if open as temp
void FXFileIO::close(){
  FXIOHandle::close();
  if (temp || options&FILEIO_REMOVE_ON_CLOSE) FXFile::remove(name());
  flags&=~FLAG_ENABLED;
  }

// file size
FXlong FXFileIO::size(){
  return (FXlong) FXFile::size(name());
  }

// mark the file as a temp file
void FXFileIO::setTemp(FXbool t){
  if (name().length()) temp=t;
  }

// enable FXFileIO to detect file growth
void FXFileIO::enable(){
  if (name().length()) flags|=FLAG_ENABLED;
  }

// disable FXFileIO from detecting file growth
void FXFileIO::disable(){
  if (name().length()) flags&=~FLAG_ENABLED;
  }

// flush data to disk
void FXFileIO::sync(){
#ifndef WIN32
  if (region) ::msync(region,mappedLength,MS_SYNC|MS_INVALIDATE);
  ::fsync(iohandle);
#else
#endif
  }

// report the current file pointer position
FXlong FXFileIO::current(){
#ifndef WIN32
  return ::lseek(iohandle,0,SEEK_CUR);
#else
  return SetFilePointer(iohandle,0,0,FILE_CURRENT);
#endif
  }

// move around file, wrt the start of the file
FXbool FXFileIO::seekbegin(FXlong pos){
  if (pos<0) fxerror("%s: cannot seek beyond start of file\n",getClassName());
#ifndef WIN32
  return ::lseek(iohandle,pos,SEEK_SET)>=0?TRUE:FALSE;
#else
  return SetFilePointer(iohandle,pos,NULL,FILE_BEGIN)>=0?TRUE:FALSE;
#endif
  }

// move around file, wrt the current position within the file
FXbool FXFileIO::seek(FXlong pos){
#ifndef WIN32
  return ::lseek(iohandle,pos,SEEK_CUR)>=0?TRUE:FALSE;
#else
  return SetFilePointer(iohandle,pos,NULL,FILE_CURRENT)>=0?TRUE:FALSE;
#endif
  }

// move around file, wrt the end of the file
FXbool FXFileIO::seekend(FXlong pos){
#ifndef WIN32
  return ::lseek(iohandle,pos,SEEK_END)>=0?TRUE:FALSE;
#else
  return SetFilePointer(iohandle,pos,NULL,FILE_END)>=0?TRUE:FALSE;
#endif
  }

// get lock status
FXint FXFileIO::isLocked(FXlong len){
#ifndef WIN32
  struct flock lock;
  lock.l_type=F_WRLCK;
  lock.l_start=current();
  lock.l_whence=SEEK_CUR;
  lock.l_len=len;
  if (::fcntl(iohandle,F_GETLK,&lock)<0) return LockUnknown;

  // handle unlocked case
  if (lock.l_type & F_UNLCK) return Unlocked;

  // handle 'this process' case
  // (first test for write, then for read)
  if (lock.l_pid == fxgetpid()) {
    if (lock.l_type & F_WRLCK) return LockWrite;
    else return LockRead;
    }

  // handle 'some other process case'
  // (first test for write, then for read)
  if (lock.l_type & F_WRLCK) return LockedWrite;
  else return LockedRead;

#else

  // FIXME implement test for file locking - need to support Read and Write
  WINBOOL result= LockFileEx(iohandle,LOCKFILE_FAIL_IMMEDIATELY,0,(DWORD)len,0,NULL);
  if (result) UnlockFileEx(iohandle,9,(DWORD)len,0,NULL);
  return result?Unlocked:LockedWrite;
#endif
  }

// set a read lock
FXbool FXFileIO::readLock(FXlong len,FXint timeout){
  // see if we already have a lock
  FXint locking=isLocked(len);
  if (locking == LockRead || locking == LockWrite) return TRUE;
  // there is no lock on the file, or some other process has a lock,
  // just try and continually lock the file until we timeout
 
#ifndef WIN32
  struct flock lock;
  lock.l_type=F_RDLCK;
  lock.l_whence=SEEK_CUR;
  lock.l_start=current();
  lock.l_len=len;
  if (timeout==0) return ::fcntl(iohandle,F_SETLK,&lock)<0?FALSE:TRUE;
  if (timeout<0) {
    while (::fcntl(iohandle,F_SETLK,&lock)<0) {}
    return TRUE;
    }
  register FXint to=timeout*1000000;
  while(to>0){
    if (::fcntl(iohandle,F_SETLK,&lock)>-1) return TRUE;
    fxsleep(retryInterval);
    to-=retryInterval;
    }
  return FALSE;
#else
  if (timeout==0) return LockFileEx(iohandle,LOCKFILE_FAIL_IMMEDIATELY,0,(DWORD)len,0,NULL)?TRUE:FALSE;
  if (timeout<0) return LockFileEx(iohandle,0,0,(DWORD)len,0,NULL)?TRUE:FALSE; // FIXME check
  register FXint to=timeout*1000000;
  while(to>0){
    if (LockFileEx(iohandle,LOCKFILE_FAIL_IMMEDIATELY,0,(DWORD)len,0,NULL)) return TRUE;
    fxsleep(retryInterval);
    to-=retryInterval;
    }
  return FALSE;
#endif
  }

// set a write lock
FXbool FXFileIO::writeLock(FXlong len,FXint timeout){

  // see if we already have a lock
  FXint locking=isLocked(len);
  if (locking == LockWrite) return TRUE;

  // there is no lock on the file, or some other process has a lock,
  // just try and continually lock the file until we timeout

#ifndef WIN32
  struct flock lock;
  lock.l_type=F_WRLCK;
  lock.l_whence=SEEK_CUR;
  lock.l_start=current();
  lock.l_len=len;
  if (timeout==0) return ::fcntl(iohandle,F_SETLKW,&lock)<0?FALSE:TRUE;
  if (timeout<0) {
    while (::fcntl(iohandle,F_SETLKW,&lock)<0) {}
    return TRUE;
    }
  register FXint to=timeout*1000000;
  while(to>0){
    if (::fcntl(iohandle,F_SETLKW,&lock)>-1) return TRUE;
    fxsleep(retryInterval);
    to-=retryInterval;
    }
  return FALSE;
#else

  if (timeout==0) return LockFileEx(iohandle,LOCKFILE_FAIL_IMMEDIATELY,0,(DWORD)len,0,NULL)?TRUE:FALSE;
  if (timeout<0) return LockFileEx(iohandle,0,0,(DWORD)len,0,NULL)?TRUE:FALSE; // FIXME check
  register FXint to=timeout*1000000;
  while(to>0){
    if (LockFileEx(iohandle,LOCKFILE_FAIL_IMMEDIATELY,0,(DWORD)len,0,NULL)) return TRUE;
    fxsleep(retryInterval);
    to-=retryInterval;
    }
  return FALSE;
#endif
  }

// unlock the last lock
FXbool FXFileIO::unlock(FXlong len){
#ifndef WIN32
  struct flock lock;
  lock.l_type=F_UNLCK;
  lock.l_start=current();
  lock.l_len=len;
  return ::fcntl(iohandle,F_SETLKW,&lock)<0?FALSE:TRUE;
#else
  return UnlockFileEx(iohandle,0,(DWORD)len,0)?TRUE:FALSE;
#endif
  }

// truncate the file to a specified length
FXbool FXFileIO::truncate(FXlong len){
  FXlong s=(FXlong)size();
  if (s == len) return TRUE;
  if (!seek(len)) return FALSE;  // set file pointer to new EOF position
  if (s < len) return TRUE;  // if the origonal size is less than the length, nothing left to do
#ifndef WIN32
  return ::ftruncate(iohandle,len)>=0?TRUE:FALSE;
#else
  return SetEndOfFile(iohandle)?TRUE:FALSE;
#endif
  }

// memory map a file
FXbool FXFileIO::map(FXlong len,FXbool writable){
  if (!len) { unmap(); return TRUE; }
#ifndef WIN32
  FXint flags= writable ? MAP_SHARED : MAP_PRIVATE;
  region=::mmap(0,(size_t)len,PROT_EXEC|PROT_READ|PROT_WRITE,flags,iohandle,current());
  if (region == MAP_FAILED) { region=NULL; code=errno; return FALSE; }
#else
  region=MapViewOfFile(iohandle,FILE_MAP_ALL_ACCESS,0,(DWORD)current(),(DWORD)len);
  if (!region) { code=GetLastError(); return FALSE; }
#endif
  mappedLength=len;
  return TRUE;
  }

// un-memory map a file
void FXFileIO::unmap(){
  if (!region) fxerror("%s: no region mapped\n",getClassName());
#ifndef WIN32
  ::munmap(region,mappedLength);
#else
  UnmapViewOfFile(region);
#endif
  mappedLength=0;
  region=NULL;
  }

/*
 * Create a file handle which is effectively a duplicate of this one
 */
FXFileIO* FXFileIO::duplicate(FXInputHandle newHandle){
#ifndef WIN32
  return dynamic_cast<FXFileIO*>(FXIOHandle::duplicate(newHandle));
#else
#error FIXME
#endif
  }

// implement the instance creation
FXFileIO* FXFileIO::newInstance(FXInputHandle h){
  return new FXFileIO(h,getApp(),target,message);
  }

// open file, now that it exists
long FXFileIO::onConnect(FXObject*,FXSelector,void *ptr){
  if (!handle(this,FXSEL(SEL_IO_CONNECT,0),ptr)) open();
  return 1;
  }

// simulate 'connect/open' event
long FXFileIO::onFileCreated(FXObject*,FXSelector,void *ptr){
  if (!opened() && flags&FLAG_ENABLED) handle(this,FXSEL(SEL_IO_WRITE,ID_FILEIO),ptr);
  return 1;
  }

// simulate 'data ready' event
long FXFileIO::onFileChanged(FXObject*,FXSelector,void *ptr){
  if (flags&FLAG_ENABLED) handle(this,FXSEL(SEL_IO_READ,FXIOHandle::ID_IOHANDLE),ptr);
  return 1;
  }

// we shouldn't receive a 'close' event unless the file doesn't exist,
// in which case we need to errorClose().
long FXFileIO::onFileClosed(FXObject*,FXSelector,void*){
  if (flags&FLAG_ENABLED) {
    code=ENOENT;
    errorClose();
    }
  return 1;
  }

}
