/********************************************************************************
*                                                                               *
*                  Monitor the status of a file                                 *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXFile.h>
using namespace FX;
#include "FXFileMonitor.h"
using namespace FXEX;
namespace FXEX {

FXDEFMAP(FXFileMonitor) FXFileMonitorMap[]={
  FXMAPFUNC(SEL_TIMEOUT,FXFileMonitor::ID_FILE_CREATE,FXFileMonitor::onFileCreate),
  FXMAPFUNC(SEL_TIMEOUT,FXFileMonitor::ID_FILE_CHANGED,FXFileMonitor::onFileChange),
  FXMAPFUNC(SEL_TIMEOUT,FXFileMonitor::ID_FILE_END,FXFileMonitor::onFileEnd),
  FXMAPFUNC(SEL_CREATE,0,FXFileMonitor::onFileCreated),
  FXMAPFUNC(SEL_CHANGED,0,FXFileMonitor::onFileChanged),
  FXMAPFUNC(SEL_DELETE,0,FXFileMonitor::onFileDeleted),
  FXMAPFUNC(SEL_CLOSED,0,FXFileMonitor::onFileClosed),
  };
FXIMPLEMENT(FXFileMonitor,FXBaseObject,FXFileMonitorMap,ARRAYNUMBER(FXFileMonitorMap))

// ctor
FXFileMonitor::FXFileMonitor(FXApp *a,const FXString& file,FXint ms,FXint to,FXObject *tgt,FXSelector sel) : FXBaseObject(a,tgt,sel) {
  this->file=FXFile::absolute(file);
  if (to > 0 && (to*1000) < ms)
    fxerror("%s: end timeout must be bigger than scan time\n",getClassName());
  growtime=ms;
  endtimeout=to;
  size=0;
  }

// free up all resources
FXFileMonitor::~FXFileMonitor(){
  getApp()->removeTimeout(this,ID_FILE_CREATE);
  getApp()->removeTimeout(this,ID_FILE_CHANGED);
  getApp()->removeTimeout(this,ID_FILE_END);
  }

// create resources
void FXFileMonitor::create(){
  FXBaseObject::create();
  reset();
  }

// save object to stream
void FXFileMonitor::save(FXStream& store) const {
  FXBaseObject::save(store);
  store << file;
  store << growtime;
  store << endtimeout;
  }

// load object from stream
void FXFileMonitor::load(FXStream& store){
  FXBaseObject::load(store);
  store >> file;
  store >> growtime;
  store >> endtimeout;
  }

// monitor a new file
void FXFileMonitor::filename(const FXString& file) {
  this->file=file;
  reset();
  }

// file size-scanning time
void FXFileMonitor::setScanTime(FXint ms) {
  if ((endtimeout*1000) < ms)
    fxerror("%s: end timeout must be bigger than scan time\n",getClassName());
  growtime=ms;
  }

// file stop-growing time
void FXFileMonitor::setEndTimeout(FXint to) {
  if ((to*1000) < growtime)
    fxerror("%s: end timeout must be bigger than scan time\n",getClassName());
  endtimeout=to;
  }

// reset the monitor
void FXFileMonitor::reset(){
  size=0;
  getApp()->removeTimeout(this,ID_FILE_CREATE);
  getApp()->removeTimeout(this,ID_FILE_CHANGED);
  getApp()->removeTimeout(this,ID_FILE_END);
  if(file.length()) onFileCreate(NULL,0,NULL);
  }

// check if the file exists
long FXFileMonitor::onFileCreate(FXObject*,FXSelector,void*){
  if (! FXFile::exists(file))
    getApp()->addTimeout(this,ID_FILE_CREATE,growtime);
  else{
    handle(this,FXSEL(SEL_CREATE,0),NULL);
    onFileChanged(NULL,0,NULL);
    }
  return 1;
  }

// generate a SEL_CHANGED event whenever the file grows or its 'mtime' changes
long FXFileMonitor::onFileChange(FXObject*,FXSelector,void*){
  if (FXFile::exists(file)){
    FXTime mtime=FXFile::modified(file);
    unsigned long sz=FXFile::size(file);
    if (modified!=mtime || size!=sz){
      modified=mtime;
      size=sz;
      getApp()->removeTimeout(this,ID_FILE_END);
      handle(this,FXSEL(SEL_CHANGED,0),(void*)&sz);
      }
    else if (!getApp()->hasTimeout(this,ID_FILE_END) && endtimeout>0) {
      getApp()->addTimeout(this,ID_FILE_END,endtimeout*1000-growtime);
      }
    getApp()->addTimeout(this,ID_FILE_CHANGED,growtime);
    }
  else{
    handle(this,FXSEL(SEL_DELETE,0),(void*)&size);
    reset();
    }
  return 1;
  }

// detect if a file hasn't changed size in a while
long FXFileMonitor::onFileEnd(FXObject*,FXSelector,void*){
  getApp()->removeTimeout(this,ID_FILE_CHANGED);
  handle(this,FXSEL(SEL_CLOSED,0),(void*)&size);
  getApp()->addTimeout(this,ID_FILE_CHANGED,growtime);
  return 1;
  }

// forward file creation event, to target
long FXFileMonitor::onFileCreated(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_CREATE,message),ptr);
  }

// forward file changed event, to target
long FXFileMonitor::onFileChanged(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_CHANGED,message),ptr);
  }
  
// forward file deleted event, to target
long FXFileMonitor::onFileDeleted(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_DELETE,message),ptr);
  }
  
// forward file closed event, to target
long FXFileMonitor::onFileClosed(FXObject*,FXSelector,void *ptr){
  return target && target->handle(this,FXSEL(SEL_CLOSED,message),ptr);
  }
  
}

