// fxvtksamp_.cpp 
//

#include "vtkCommon.h"
#include "vtkGraphics.h"
#include "fxex.h"

// Timer setting (in milliseconds)
const FXuint TIMER_INTERVAL = 100;


// Event Handler Object
class FXVTKTestWindow : public FXMainWindow {
  FXDECLARE(FXVTKTestWindow)

private:
  FXVTKWindow     *VTKWindow;                 // Window for vtk
  vtkActor        *actor;                     // A vtk object

protected:
  // serialisation
  FXVTKTestWindow(){}

  // create a VTK object
  void CreateVTKObject();

  // is the object spinning
  FXbool isSpinning();
  
public:
  // We define additional ID's, starting from the last one used by the base class+1.
  // This way, we know the ID's are all unique for this particular target.
  enum{
    ID_CANVAS=FXMainWindow::ID_LAST,
    ID_SPIN,
    ID_SPINFAST,
    ID_STOP,
    ID_TIMEOUT,
    ID_CHORE
    };
    
  // Message handlers
  long onMouseDown(FXObject*,FXSelector,void*);
  long onMouseUp(FXObject*,FXSelector,void*);
  long onMouseMove(FXObject*,FXSelector,void*);
  long onExpose(FXObject*,FXSelector,void*);
  long onCmdSpin(FXObject*,FXSelector,void*);
  long onUpdSpin(FXObject*,FXSelector,void*);
  long onCmdStop(FXObject*,FXSelector,void*);
  long onUpdStop(FXObject*,FXSelector,void*);
  long onTimeout(FXObject*,FXSelector,void*);
  long onChore(FXObject*,FXSelector,void*);
  long onCmdSpinFast(FXObject*,FXSelector,void*);
  long onUpdSpinFast(FXObject*,FXSelector,void*);
  
public:
  // FXVTKTestWindow constructor
  FXVTKTestWindow(FXApp* a);
  
  // Initialize
  void create();
  
  // Draw scene
  void drawScene();

  // FXVTKTestWindow destructor
  virtual ~FXVTKTestWindow();
  };


// Message Map for the FXVTKSampApp class
FXDEFMAP(FXVTKTestWindow) FXVTKTestWindowMap[]={
  FXMAPFUNC(SEL_PAINT,         FXVTKTestWindow::ID_CANVAS,   FXVTKTestWindow::onExpose),
  FXMAPFUNC(SEL_COMMAND,       FXVTKTestWindow::ID_SPIN,     FXVTKTestWindow::onCmdSpin),
  FXMAPFUNC(SEL_UPDATE,        FXVTKTestWindow::ID_SPIN,     FXVTKTestWindow::onUpdSpin),
  FXMAPFUNC(SEL_COMMAND,       FXVTKTestWindow::ID_SPINFAST, FXVTKTestWindow::onCmdSpinFast),
  FXMAPFUNC(SEL_UPDATE,        FXVTKTestWindow::ID_SPINFAST, FXVTKTestWindow::onUpdSpinFast),
  FXMAPFUNC(SEL_COMMAND,       FXVTKTestWindow::ID_STOP,     FXVTKTestWindow::onCmdStop),
  FXMAPFUNC(SEL_UPDATE,        FXVTKTestWindow::ID_STOP,     FXVTKTestWindow::onUpdStop),
  FXMAPFUNC(SEL_TIMEOUT,       FXVTKTestWindow::ID_TIMEOUT,  FXVTKTestWindow::onTimeout),
  FXMAPFUNC(SEL_CHORE,         FXVTKTestWindow::ID_CHORE,    FXVTKTestWindow::onChore),
  };
FXIMPLEMENT(FXVTKTestWindow,FXMainWindow,FXVTKTestWindowMap,ARRAYNUMBER(FXVTKTestWindowMap))

// Construct a FXVTKTestWindow
FXVTKTestWindow::FXVTKTestWindow(FXApp* a):FXMainWindow(a,"VTK Test Application",NULL,NULL,DECOR_ALL,0,0,800,600){
  FXVerticalFrame *vtkcanvasFrame;
  FXVerticalFrame *buttonFrame;
  FXComposite *vtkpanel;
  FXHorizontalFrame *frame;
  frame=new FXHorizontalFrame(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
  
  // LEFT pane to contain the vtkcanvas
  vtkcanvasFrame=new FXVerticalFrame(frame,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_TOP|LAYOUT_LEFT,0,0,0,0,10,10,10,10);
  
  // Label above the vtkcanvas               
  new FXLabel(vtkcanvasFrame,"VTK Canvas Frame",NULL,JUSTIFY_CENTER_X|LAYOUT_FILL_X);
  
  // Horizontal divider line
  new FXHorizontalSeparator(vtkcanvasFrame,SEPARATOR_GROOVE|LAYOUT_FILL_X);
  
  vtkpanel=new FXVerticalFrame(vtkcanvasFrame,FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_TOP|LAYOUT_LEFT,0,0,0,0, 0,0,0,0);
  
  // Drawing vtkWindow
  VTKWindow=new FXVTKWindow(vtkpanel,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_TOP|LAYOUT_LEFT);
  // Initialize vtk objects
  CreateVTKObject();
  
  // RIGHT pane for the buttons
  buttonFrame=new FXVerticalFrame(frame,LAYOUT_FILL_Y|LAYOUT_TOP|LAYOUT_LEFT,0,0,0,0,10,10,10,10);
  
  // Label above the buttons  
  new FXLabel(buttonFrame,"Button Frame",NULL,JUSTIFY_CENTER_X|LAYOUT_FILL_X);
  
  // Horizontal divider line
  new FXHorizontalSeparator(buttonFrame,SEPARATOR_RIDGE|LAYOUT_FILL_X);
  
  // Button to print
  new FXButton(buttonFrame,"Spin &Timer\tSpin using interval timers\nNote the app blocks until the interal has elapsed...",NULL,this,ID_SPIN,FRAME_THICK|FRAME_RAISED|LAYOUT_FILL_X|LAYOUT_TOP|LAYOUT_LEFT,0,0,0,0,10,10,5,5);
  new FXButton(buttonFrame,"Spin &Chore\tSpin as fast as possible using chores\nNote even though the app is very responsive, it never blocks;\nthere is always something to do...",NULL,this,ID_SPINFAST,FRAME_THICK|FRAME_RAISED|LAYOUT_FILL_X|LAYOUT_TOP|LAYOUT_LEFT,0,0,0,0,10,10,5,5);
  
  // Button to print
  new FXButton(buttonFrame,"&Stop Spin\tStop this mad spinning, I'm getting dizzy",NULL,this,ID_STOP,FRAME_THICK|FRAME_RAISED|LAYOUT_FILL_X|LAYOUT_TOP|LAYOUT_LEFT,0,0,0,0,10,10,5,5);
  
  // Exit button
  new FXButton(buttonFrame,"&Exit\tExit the application",NULL,getApp(),FXApp::ID_QUIT,FRAME_THICK|FRAME_RAISED|LAYOUT_FILL_X|LAYOUT_TOP|LAYOUT_LEFT,0,0,0,0,10,10,5,5);
  
  // Make a tooltip
  new FXToolTip(getApp());
  
  }
    

// Destructor
FXVTKTestWindow::~FXVTKTestWindow(){
  getApp()->removeTimeout(this,ID_TIMEOUT);
  }

 

// Create and initialize 
void FXVTKTestWindow::create(){
  FXMainWindow::create();
  show(PLACEMENT_SCREEN);
  }


// Widget needs repainting
long FXVTKTestWindow::onExpose(FXObject*,FXSelector,void*){
  drawScene();
  return 1;
  }



//  Rotate the boxes when a timer message is received
long FXVTKTestWindow::onTimeout(FXObject*,FXSelector,void*){
  drawScene();
  getApp()->addTimeout(this,ID_TIMEOUT,TIMER_INTERVAL);
  return 1;
  }


// Rotate the boxes when a chore message is received
long FXVTKTestWindow::onChore(FXObject*,FXSelector,void*){
  drawScene();
  getApp()->addChore(this,ID_CHORE);
  return 1;
  }


// Start the boxes spinning
long FXVTKTestWindow::onCmdSpin(FXObject*,FXSelector,void*){
  getApp()->addTimeout(this,ID_TIMEOUT,TIMER_INTERVAL);
  return 1;
  }



// Enable or disable the spin button
long FXVTKTestWindow::onUpdSpin(FXObject* sender,FXSelector,void*){
  if (!isSpinning())
    sender->handle(this,FXSEL(SEL_COMMAND,ID_ENABLE),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,ID_DISABLE),NULL);
  return 1;
  }


// Start the boxes spinning
long FXVTKTestWindow::onCmdSpinFast(FXObject*,FXSelector,void*){
  getApp()->addChore(this,ID_CHORE);
  return 1;
  }


// Enable or disable the spin button
long FXVTKTestWindow::onUpdSpinFast(FXObject* sender,FXSelector,void*){
  if (!isSpinning())
    sender->handle(this,FXSEL(SEL_COMMAND,ID_ENABLE),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,ID_DISABLE),NULL);
  return 1;
  }


// Enable or disable the stop button
long FXVTKTestWindow::onUpdStop(FXObject* sender,FXSelector,void*){
  if (isSpinning())
    sender->handle(this,FXSEL(SEL_COMMAND,ID_ENABLE),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,ID_DISABLE),NULL);
  return 1;
  }


// If boxes are spinning, stop them
long FXVTKTestWindow::onCmdStop(FXObject*,FXSelector,void*){
  getApp()->removeTimeout(this,ID_TIMEOUT);
  getApp()->removeChore(this,ID_CHORE);
  return 1;
  }


// Draw the VTK scene
void FXVTKTestWindow::drawScene(){
//  VTKWindow->getInteractor()->Disable();
  if (spinning) actor->RotateZ(2);
  VTKWindow->handle(this,FXSEL(SEL_PAINT,0),NULL);
  }


// is the object spinning
FXbool FXVTKTestWindow::isSpinning(){
  return (getApp()->hasTimeout(this,ID_TIMEOUT) || getApp()->hasChore(this,ID_CHORE));
  }


// create a VTK object
void FXVTKTestWindow::CreateVTKObject() {
  vtkPlaneSource *plane = vtkPlaneSource::New();
    plane->SetXResolution(50);
    plane->SetYResolution(50);

  vtkTransform *transform = vtkTransform::New();
    transform->Scale(10.0, 10.0, 1.0);

  vtkTransformPolyDataFilter *transF = vtkTransformPolyDataFilter::New();
    transF->SetInput(plane->GetOutput());
    transF->SetTransform(transform);

  vtkTriangleFilter *tri = vtkTriangleFilter::New();
    tri->SetInput(transF->GetOutput());

  vtkStripper *strip = vtkStripper::New();
    strip->SetInput(tri->GetOutput());
    strip->Update();

  vtkPolyData * input = strip->GetOutput();
  int numPts = input->GetNumberOfPoints();

  vtkPoints *newPts = vtkPoints::New();
    newPts->SetDataTypeToFloat();
  vtkScalars *derivs = vtkScalars::New();
    derivs->SetDataTypeToFloat();

  vtkPolyData *bessel = vtkPolyData::New();
    bessel->CopyStructure(input);
    bessel->SetPoints(newPts);
    bessel->GetPointData()->SetScalars(derivs);

  for (int i=0; i<numPts; i++) {
    float *x = input->GetPoint(i);
    float x0 = x[0];
    float x1 = x[1];

    float r = sqrt(x0*x0 +x1*x1);
    float x2 = (x0*x0 - x1*x1)/10.0;
    float deriv = (x0*x0 - x1*x1)/25.0;

    newPts->InsertPoint(i,x0,x1,x2);
    derivs->InsertScalar(i, deriv);
    }
 
  newPts->Delete(); //reference counting - it's ok
  derivs->Delete();

  // warp plane
  vtkWarpScalar *warp = vtkWarpScalar::New();
    warp->SetInput(bessel);
    warp->XYPlaneOn();
    warp->SetScaleFactor(0.5);

  // mapper and actor
  vtkDataSetMapper *mapper = vtkDataSetMapper::New();
    mapper->SetInput(warp->GetOutput());
    mapper->SetScalarRange(-1.0, 1.0);

  vtkActor *carpet = vtkActor::New();
    carpet->SetMapper(mapper);
  actor = carpet;

  vtkOutlineFilter *outline = vtkOutlineFilter::New();
  outline->SetInput(warp->GetOutput());
  vtkPolyDataMapper *outlineMapper = vtkPolyDataMapper::New();
    outlineMapper->SetInput(outline->GetOutput());
  vtkActor *outlineActor = vtkActor::New();
  outlineActor->SetMapper(outlineMapper);
    outlineActor->GetProperty()->SetColor(1,1,1);

  vtkRenderer *renderer = VTKWindow->getRenderer(); 
    renderer->SetBackground(0,0,0);
    renderer->AddActor(carpet);
    renderer->AddActor(outlineActor);
    mapper->Delete();
    carpet->Delete();
  }


// Here we get the ball rolling
int main(int argc,char *argv[]){
  FXApp application("VTKTest","FoxTest");
  application.init(argc,argv);
  new FXVTKTestWindow(&application);
  application.create();
  return application.run();
  }

