/********************************************************************************
*                                                                               *
*            New  M u l t i - L i n e   T e x t   W i d g e t                   *
*                                                                               *
*                              External Process                                 *
*********************************************************************************
*   $Id:$
*
*    Copyright (C) 2001, Stephen J. Hardy
*
*    tekno@users.sourceforge.net
*    920 Cranbrook Ct #32, Davis, CA, 95616
*
*    This program is free software; you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation; either version 2 of the License, or
*    (at your option) any later version.
*
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public License
*    along with this program; if not, write to the Free Software
*    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* $Log:$
*
*
********************************************************************************/
#include <fox/fx.h>
#include <fox/xincs.h>
#include <fox/fxver.h>
#include <fox/fxdefs.h>
#include <fox/fxkeys.h>
#include "FXProcess.h"
#include "exincs.h"

#ifdef WIN32
#  error "No Win32 support"
#endif

/*-------------------- Base Class -------------------------*/

// statics
FXProcess *FXProcess::first=NULL;
FXint FXProcess::childcount=0;

// maps
FXDEFMAP(FXProcess) FXProcessMap[]={
  FXMAPFUNCS(SEL_IO_READ,FXProcess::ID_IO_0,FXProcess::ID_IO_9999,FXProcess::onIORead),
  FXMAPFUNCS(SEL_IO_WRITE,FXProcess::ID_IO_0,FXProcess::ID_IO_9999,FXProcess::onIOWrite),
  FXMAPFUNCS(SEL_IO_EXCEPT,FXProcess::ID_IO_0,FXProcess::ID_IO_9999,FXProcess::onIOExcept),
  FXMAPFUNC(SEL_SIGNAL,FXProcess::ID_SIGCHLD,FXProcess::onSigSIGCHLD),
  };
FXIMPLEMENT(FXProcess,FXBaseObject,FXProcessMap,ARRAYNUMBER(FXProcessMap))

// serialisation
FXProcess::FXProcess() { ok=FALSE; }

// ctor
FXProcess::FXProcess(FXApp *a,FXObject *tgt,FXSelector sel,FXuint opts) : FXBaseObject(a,tgt,sel) {
  ok = TRUE;
  next = first;
  first = this;
  childpid = -1;
  options = opts;
  if (options & PROCESS_STDERR) nstreams = 2;
  else nstreams = 1;
  FXCALLOC(&streams, pstream, nstreams);
  // We always add to head of list, and change the app signal handler to this.
  app->addSignal(SIGCHLD, this, ID_SIGCHLD, FALSE, 0);
  }

FXProcess::pstype FXProcess::getPstype(FXSelector sel) {
  switch(SELTYPE(sel)) {
    case SEL_SIGNAL: return PS_SIG;
    case SEL_CLOSE: return PS_SIG;
    case SEL_IO_READ: return PS_READ;
    case SEL_IO_WRITE: return PS_WRITE;
    case SEL_IO_EXCEPT: return PS_EXC;
    }
  return PS_OTHER;
  }

void FXProcess::setTargetFor(FXint streamno, FXObject * tgt, FXSelector sel) {
  FXASSERT(streamno >= 0 && streamno < nstreams);
  if (streamno >= 0 && streamno < nstreams) {
    pstype pt = getPstype(sel);
    streams[streamno].targets[pt].target=tgt;
    streams[streamno].targets[pt].message=sel;
    }
  }

void FXProcess::hold(FXint streamno, FXuint mode) {
  FXASSERT(streamno >= 0 && streamno < nstreams);
  streams[streamno].hold |= mode;
  }
  
void FXProcess::resume(FXint streamno, FXuint mode) {
  FXASSERT(streamno >= 0 && streamno < nstreams);
  pstream * p = streams + streamno;
  p->hold &= ~mode;
  app->addInput(p->stream, p->active = ~p->hold, this, ID_IO_0+streamno);
  }
  
static int get_pty(int * pty, char * ttydev) {
  int  i;
  char c;
  
  strcpy(ttydev, "/dev/pty??");

  for (c='p'; c<'t'; c++) {
    for (i=0; i<16; i++) {
      ttydev[8] = c;
      ttydev[9] = "0123456789abcdef"[i];
      if ((*pty = open(ttydev, O_RDWR)) >= 0) {
        //printf("get_pty: %s, fd=%d\n", ttydev, *pty);
        return 0;
        }
      }
    }
  return 1;
  }

static int open_slave(char * ttydev, char * mstrdev) {
  int fd;
  strcpy(ttydev, "/dev/tty??");
  ttydev[8] = mstrdev[8];
  ttydev[9] = mstrdev[9];
  fd = open(ttydev, O_RDWR);
  //printf("open_slave: fd=%d\n", fd);
  return fd;
  }

FXint FXProcess::connect(const FXString& resource, FXint argc, FXchar ** argv) {
  FXInputHandle slave;
  FXchar master_dev[40], slave_dev[40];
  FXInputHandle pipes[2];
  FXInputHandle morepipes[2];
  FXbool use_pipe = !!(options & PROCESS_PIPE);
  FXbool sep_stderr = !!(options & PROCESS_STDERR);
  pstream * master = streams;
  pstream * master2 = streams + 1;
   
  if (!use_pipe) {
    if (get_pty(&master->stream, master_dev)) return errno;
    }
  else {
    if (socketpair(PF_UNIX,SOCK_STREAM,0,pipes)) return errno;
    master->stream = pipes[0];
    }

  if (sep_stderr) {
    // Secondary channel for stderr
    if (socketpair(PF_UNIX,SOCK_STREAM,0,morepipes)) return errno;
    master2->stream = morepipes[0];
    }
  
  childpid = fork();
  if (childpid < 0)
    return errno;
  else if (childpid) {
    // parent
    //printf("child pid = %d\n", childpid);
    childcount++;
    if (use_pipe)
      close(pipes[1]);
    fcntl(master->stream, F_SETFL, O_NONBLOCK);
    app->addInput(master->stream, master->active = ~master->hold, this, ID_IO_0);
    if (sep_stderr) {
      close(morepipes[1]);
      fcntl(master2->stream, F_SETFL, O_NONBLOCK);
      app->addInput(master2->stream, master2->active = ~master2->hold, this, ID_IO_1);
      }
    }
  else {
    // child
    if (!use_pipe) {
      setsid();
      slave = open_slave(slave_dev, master_dev);
      if (slave < 0) { perror("open_slave"); exit(98); }
      close(master->stream);
      dup2(slave, 0);
      dup2(slave, 1);
      if (sep_stderr) {
        dup2(morepipes[1], 2);
        if (morepipes[1] > 2)
          close(morepipes[1]);
        }
      else
        dup2(slave, 2);
      if (slave > 2)
        close(slave);
      }
    else {
      dup2(pipes[1], 0);
      dup2(pipes[1], 1);
      if (sep_stderr) {
        dup2(morepipes[1], 2);
        if (morepipes[1] > 2)
          close(morepipes[1]);
        }
      else
        dup2(pipes[1], 2);
      close(pipes[0]);
      if (pipes[1] > 2)
        close(pipes[1]);
      }
    fcntl(1, F_SETFL, O_APPEND);
    setbuf(stdout, NULL);
    setbuf(stderr, NULL);
    execvp(resource.text(), argv);
    perror("execvp");
    exit(99);
    
    }
  return 0;
  }
  
FXint FXProcess::disconnect() {
  if (childpid <= 0)
    return 0;
  if (kill(childpid, SIGHUP))
    return errno;
  
  return 0;
  }

FXint FXProcess::disconnect(FXint streamno, FXuint mode) {
  FXint sh = 0;
  if (mode & INPUT_READ) sh |= 1;
  if (mode & INPUT_WRITE) sh |= 2;
  if (!sh) return 0;
  pstream * p = streams + streamno;
  app->removeInput(p->stream, mode);
  p->active &= ~mode;
  FXint rc = shutdown(p->stream, sh-1);
  if (sh == 3)
    p->stream = -1;
  return rc < 0 ? errno : 0;
  }

FXint FXProcess::disconnectHandle(FXInputHandle fd, FXuint mode) {
  for (FXint s = 0; s < nstreams; s++)
    if (streams[s].stream == fd)
      return disconnect(s, mode);
  return -1;
  }

void FXProcess::removeTarget(const FXObject * target) {
  for (FXProcess * p = first; p; p = p->next) {
    if (p->tgt == target) p->tgt = NULL;
    for (FXint i = 0; i < p->nstreams; i++)
      for (FXint pt = PS_SIG; pt < PS_LAST; pt++)
        if (p->streams[i].targets[pt].target == target)
          p->streams[i].targets[pt].target = NULL;
    }
  }

FXInputHandle FXProcess::getStream(FXint streamno) const {
  return 0;
  }

long FXProcess::onSigSIGCHLD(FXObject*,FXSelector,void*) {
  // Note that we have to handle the possibility of the signal being raised
  // for any FXProcess in existence.  The signal is then routed to the
  // appropriate target.
  int status;	// Child status added to message data.
  FXProcess * p;
  
  for (p = first; p; p = p->next)
    if (p->childpid > 0 && waitpid(p->childpid, &status, WNOHANG) > 0) {
      // Route this message to stream handlers as a SEL_CLOSE
      FXObject * t;
      childcount--;
      for (FXint i = 0; i < p->nstreams; i++)
        if ((t = p->streams[i].targets[PS_SIG].target))
          t->handle(p, MKUINT(p->streams[i].targets[PS_SIG].message, SEL_CLOSE), (void *)status);
      // Default target gets this is a SEL_SIGNAL
      if (p->tgt)
        p->tgt->handle(p, MKUINT(p->message, SEL_SIGNAL), (void *)status);
      for (FXint i = 0; i < p->nstreams; i++)
        p->closeStream(i);
      p->childpid = -1;
      return 1;
      }
  // If no FXProcess handled a SIGCHLD, then one was probably destructed without
  // waiting to reap the child.  This must be the grim reaper, so we wait for any
  // child to prevent a buildup of zombies.  Sounds ghoulish...
  FXASSERT(childcount > 0);
  if (childcount) childcount--;
  wait(NULL);
  return 0;
  }

long FXProcess::onIORead(FXObject*,FXSelector sel, void*ptr) {
  FXint sn = SELID(sel) - ID_IO_0;
  FXInputHandle s = (FXInputHandle)ptr;
  FXASSERT(sn >= 0 && sn < nstreams);
  if (sn >= 0 && sn < nstreams) {
    pstream * p = streams + sn;
    FXASSERT(s == p->stream);
    if (p->hold & INPUT_READ ||
        !(p->targets[PS_READ].target && 
          p->targets[PS_READ].target->handle(this, p->targets[PS_READ].message, ptr)) &&
        !(tgt && tgt->handle(this, MKUINT(message, SEL_IO_READ), ptr)))
      // No target or on hold: remove input to avoid busy wait loop
      app->removeInput(p->stream, INPUT_READ), p->active &= ~INPUT_READ;
    }
  return 1;
  }

long FXProcess::onIOWrite(FXObject*,FXSelector sel, void*ptr) {
  FXint sn = SELID(sel) - ID_IO_0;
  FXInputHandle s = (FXInputHandle)ptr;
  FXASSERT(sn >= 0 && sn < nstreams);
  if (sn >= 0 && sn < nstreams) {
    pstream * p = streams + sn;
    FXASSERT(s == p->stream);
    if (p->hold & INPUT_WRITE ||
        !(p->targets[PS_WRITE].target && 
          p->targets[PS_WRITE].target->handle(this, p->targets[PS_WRITE].message, ptr)) &&
        !(tgt && tgt->handle(this, MKUINT(message, SEL_IO_WRITE), ptr)))
      // No target or on hold: remove input to avoid busy wait loop
      app->removeInput(p->stream, INPUT_WRITE), p->active &= ~INPUT_WRITE;
    }
  return 1;
  }

long FXProcess::onIOExcept(FXObject*,FXSelector sel, void*ptr) {
  FXint sn = SELID(sel) - ID_IO_0;
  FXInputHandle s = (FXInputHandle)ptr;
  FXASSERT(sn >= 0 && sn < nstreams);
  if (sn >= 0 && sn < nstreams) {
    pstream * p = streams + sn;
    FXASSERT(s == p->stream);
    if (p->hold & INPUT_EXCEPT ||
        !(p->targets[PS_EXC].target && 
          p->targets[PS_EXC].target->handle(this, p->targets[PS_EXC].message, ptr)) &&
        !(tgt && tgt->handle(this, MKUINT(message, SEL_IO_EXCEPT), ptr)))
      // No target or on hold: remove input to avoid busy wait loop
      app->removeInput(p->stream, INPUT_EXCEPT), p->active &= ~INPUT_EXCEPT;
    }
  return 1;
  }

long FXProcess::onDefault(FXObject*,FXSelector sel,void*) {
  return 0;
  }

FXint FXProcess::closeStream(FXint idx) {
  if (isOpen(idx)) {
    app->removeInput(streams[idx].stream, streams[idx].active);
    streams[idx].active = 0;
    close(streams[idx].stream);
    streams[idx].stream = -1;
    }
  return 0;
  }

/*---------------------------------------------------------*/

void FXProcess::save(FXStream& /*store*/) const {
  }

void FXProcess::load(FXStream& /*store*/) {
  }

FXProcess::~FXProcess() {
  FXProcess ** p = &first;
  if (this == first) {
    // I am the signal handler.  Pass the baton on to next.
    if (next)
      app->addSignal(SIGCHLD, next, ID_SIGCHLD, FALSE, 0);
    else {
      // We are the last to be removed.  If there is at least one outstanding child,
      // a new (unowned) FXProcess is created in order to reap them.  See onSigSIGCHLD().
      if (childcount > 0) {
        first = NULL;
        new FXProcess(app, NULL, 0, 0); // create grim reaper
        goto cleanup;
        }
      else
        app->removeSignal(SIGCHLD);
      }
    }
  while (*p && *p != this) p = &(*p)->next;
  FXASSERT(*p);
  if (*p)
    *p = next;
cleanup:
  for (FXint i = 0; i < nstreams; i++)
    closeStream(i);
  FXFREE(&streams);
  tgt = (FXObject *)-1;
  app = (FXApp *)-1;
  }

 
