/********************************************************************************
*                                                                               *
*            New  M u l t i - L i n e   T e x t   W i d g e t                   *
*                                                                               *
*                              External Process                                 *
*********************************************************************************
*   $Id:$
*
*    Copyright (C) 2001, Stephen J. Hardy
*
*    tekno@users.sourceforge.net
*    920 Cranbrook Ct #32, Davis, CA, 95616
*
*    This program is free software; you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation; either version 2 of the License, or
*    (at your option) any later version.
*
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public License
*    along with this program; if not, write to the Free Software
*    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* $Log:$
*
*
********************************************************************************/
#ifndef FXPROCESS_H
#define FXPROCESS_H

#include "FXBaseObject.h"

/** Process options.
*/
enum {
  PROCESS_PTY    = 0x00000000,	/// Use PTY for terminal emulation
  PROCESS_PIPE   = 0x00000001,	/// Use socket/pipe rather than PTY (non-interactive)
  PROCESS_STDERR = 0x00000002,	/// Use separate stderr channel, else munged with stdout
  };

/** Base class for external processes.  This object handles creation, communication,
* and termination of child processes.  Currently, it is only defined for POSIX OSs
* and has only been tested on Linux 2.2.12 kernel.  Communication with the child
* process is via sockets or pipes, collectively referred to as streams.
*/

class FXAPI FXProcess : public FXBaseObject {
  FXDECLARE(FXProcess)

protected:
  enum pstype {
    PS_SIG,
    PS_READ,
    PS_WRITE,
    PS_EXC,
    PS_OTHER,
    PS_LAST
    };
    
  struct pstream {
    FXInputHandle stream;
    FXuint hold;	// combination of INPUT_READ,_WRITE,_EXCEPT.  If bit set, want hold.
    FXuint active;	// indicates which modes have outstanding app->addInput()s.
    struct {
      FXObject *target;
      FXSelector message;
      } targets[PS_LAST];
    };

  FXuint options;	// Connection options
  FXint childpid;	// Process ID of child process (-1 if not available, 0 if not a
  			// child process e.g. TCP socket).
  FXbool ok;		// Flag to indicate construction completed successfully
  FXProcess * next;	// Linked list of processes.  Each has to be able to access all
  			//  others since there is only one signal handler per signal.
  static FXProcess * first;
  static FXint childcount; // Number of outstanding processes.
  FXint nstreams;	// Number of streams
  pstream * streams;	// Stream records

public:
  enum {
    ID_SIGCHLD,		/// SIGCHLD signal received
    ID_IO_0,		/// I/O from various streams (usually only 1 or 2).
    ID_IO_1,
    ID_IO_9999 = ID_IO_0 + 9999,
    ID_LAST
    };

  long onSigSIGCHLD(FXObject*,FXSelector,void*);
  long onIORead(FXObject*,FXSelector,void*);
  long onIOWrite(FXObject*,FXSelector,void*);
  long onIOExcept(FXObject*,FXSelector,void*);
  virtual long onDefault(FXObject*,FXSelector,void*);

protected:
  FXProcess();
#ifndef WIN32
  FXbool isOpen(FXint idx) const { return streams[idx].stream > 0; }
#else
  FXbool isOpen(FXint idx) const { return streams[idx].stream != NULL; }
#endif
  FXint closeStream(FXint idx);
  pstype getPstype(FXSelector sel);
    
private:
  FXProcess(const FXProcess&);
  FXProcess& operator=(const FXProcess&);
  
public:
  /// Construct.  owner/message specify a default message target.  opts specifies
  /// the required process type (see PROCESS_* tags above).
  FXProcess(FXApp *a,FXObject *tgt,FXSelector sel, FXuint opts);

  /// Test whether constructor completed successfully.  Since ctors may allocate memory etc.,
  /// it is necessary for the caller to check whether construction was sucessful.  If not,
  /// then the dtor should be invoked as soon as possible and an error flagged to the user.
  /// Note that the ctor is exclusively responsible for setting this flag.  The base class
  /// sets it TRUE by default.
  ///
  /// Note by Mathew:
  ///     this is actually a pretty lame statement, for a couple of reasons:
  ///     1. If you are worried about the constructor failing to allocate memory, etc
  ///        you should have an init() method which is called after the ctor,
  ///     or
  ///     2. Exceptions were specifically designed to overcome this problem - they
  ///        most importantly unwind the call stack - I am surprised to find that
  ///        more programmers dont use exceptions; they are a very good way of
  ///        programming - I particulariy like using exceptions to control program flow
  ///        but there is a catch - most compilers arn't very good at generating
  ///        exception based code; give it time...
  FXbool isOk() const { return ok; }
  
  /// Set target object and message for activity on a given stream.  Different targets
  /// can be selected for SEL_SIGNAL, SEL_IO_READ, SEL_IO_WRITE, SEL_IO_EXCEPT.  Any
  /// other SEL_* gets lumped in "other".  Setting target to NULL puts a temporary
  /// hold on stream activity in the given direction.
  void setTargetFor(FXint streamno,FXObject *tgt,FXSelector sel);
  
  /// Hold activity for specified stream/directions.  This removes the stream from the
  /// list of candidates for application event processing, until resume() called.
  /// By default, streams are set up with all modes active.  If data is ready in the
  /// stream, but no handler is waiting for it, then the mode becomes inactive
  /// automatically.  This prevents a busy wait loop in the event dispatcher.
  /// Mode is bitwise combination of INPUT_READ, _WRITE or _EXCEPT.
  void hold(FXint streamno, FXuint mode);
  
  /// Resume stream processing for specified mode(s) (see hold()).
  void resume(FXint streamno, FXuint mode);
  
  /// Create the child process and/or establish communication.  resource is the name
  /// of an executable.  argc and argv are arguments to be passed to the process.
  /// argv[argc] must be set to NULL.  argv[0] is conventionally set up as a copy of
  /// resource.  Returns 0 on success, or errno value otherwise.  Child process may
  /// print diagnostics on its stderr stream.
  virtual FXint connect(const FXString& resource, FXint argc, FXchar ** argv);
  
  /// Terminate the child process.  The caller should wait for SEL_SIGNAL (SIGCHLD).
  virtual FXint disconnect();
  
  /// Disconnect only a specific stream and direction.  The direction is specified
  /// in mode as a bitwise combination of INPUT_READ or INPUT_WRITE.
  virtual FXint disconnect(FXint streamno, FXuint mode);
  
  /// Similar to disconnect(streamno) except the FXInputHandle is specified instead.
  virtual FXint disconnectHandle(FXInputHandle fd, FXuint mode);
  
  /// Specified target is not to receive any messages from any FXProcess.  Typically
  /// called from target's destructor, just to be sure there are no dangling pointers.
  static void removeTarget(const FXObject * target);
  
  /// Return stream handle given index.  streamno is 0 for stdin/stdout, 1 for stderr
  /// (unless PROCESS_STDERR was not specified, in which case stderr also comes through
  /// stream 0).  This will return 0/NULL for invalid stream index.
  FXInputHandle getStream(FXint streamno) const;
  
  /// Return number of streams.
  FXint getStreams() const { return nstreams; }
  
  /// Save to a stream
  virtual void save(FXStream& store) const;

  /// Load from a stream
  virtual void load(FXStream& store);

  /// Destructor.  Normally, this is called from the SIGCHLD signal handler.  However,
  /// if called early then resources will be freed without waiting for child.  Under
  /// Unix, this may cause the child to enter zombie state.
  virtual ~FXProcess();
  };
  

#endif
