#!/bin/bash

# Coseismic and viscoelastic deformation due to the 1992 Mw7.3 
# Landers, CA earthquake. The Earth model is uniform elastic 
# with a Newtonian viscosity. The lower crust (H = 25-30 km)
# and the asthenosphere (H = 60+ km) are weak and separated by
# a strong (elastic) upper mantle.
#
# The crust and mantle are assigned the same viscosity. The 
# simulation computes time series of displacement in units
# of Maxwell times. 
#
# This example computes the Coulomb stress change on the 1999
# Hector Mine fault.
#
# plot the coseismic surface deformation with
#
#   grdmap.sh -b -80/40/-80/40 -e erpatch.sh coulomb/000
#
# plot the viscoelastic deformation at t=tm/10 with
#
#   grdmap.sh -b -100/60/-100/60 -s 7 -v 0.01 -e erpatch.sh coulomb/001-relax
#
# use Paraview (www.paraview.org) to visualize the Coulomb stress 
# on the fault segments that ruptured during the 1999 Hector Mine 
# earthquake. In Paraview, open 
#
#   coulomb/rfaults-sigma-..vtp
#
# to plot the time series of total stress (co- and postseismic). 
# open
#
#   coulomb/rfaults-dsigma-..vtp
#
# to visualize in 3D the time evolution of the viscoelastic
# stress perturbation without the effect of the coseismic
# rupture.

WDIR=./coulomb

if [ ! -e $WDIR ]; then
	echo adding directory $WDIR
	mkdir $WDIR
fi

time relax <<EOF --no-vtk-output --no-proj-output \
	--no-stress-output $* | tee $WDIR/in.param
# grid size (sx1,sx2,sx3)
512 512 512
# dx1  dx2  dx3 beta nyquist
  0.6  0.6  0.6 0.30     1.0
# origin position and rotation
0 0 0
# geographic origin zone unit
#-116.27 34.595 11 1e3
# observation depth
0 5
# output directory
$WDIR
# elastic parameters and gamma = (1-nu) rho g / mu = 8.33e-7 /m = 8.33e-4 /km
1 1 8.33e-4
# integration time (in time unit of Maxwell time) and time steps
2 -1 1
# number of observation planes
0
# number of observation points (gps points)
33
# index name x1  x2 x3
01 gold 91.8223 -56.8886 0.0
02 lae1 -2.4369 -26.3146 0.0
03 lae2 -0.8133 -23.1109 0.0
04 lae3 2.4215 -19.8964 0.0
05 lae4 15.3918 -5.5286 0.0
06 law1 -6.0025 -29.1520 0.0
07 law2 -7.7525 -32.4061 0.0
08 law3 -10.5600 -36.5557 0.0
09 law4 -15.8454 -36.1895 0.0
10 meek -37.5535 -31.7186 0.0
11 oldd -22.8763 -39.1759 0.0
12 oldw -23.0937 -44.1328 0.0
13 rich -36.8044 -18.0455 0.0
14 sanh -37.7077 -0.5455 0.0
15 agmt -0.1738 -14.6151 0.0
16 bbry -36.9073 -56.2854 0.0
17 bmhl -37.9308 20.2579 0.0
18 bsry 35.6429 -68.0380 0.0
19 ctms -52.2772 -8.8856 0.0
20 gmrc 21.5330 55.6410 0.0
21 hcmn 17.6226 -14.7797 0.0
22 ldes -36.4285 -14.7263 0.0
23 nbps -9.4908 11.2529 0.0
24 oaes -50.1893 19.0109 0.0
25 opbl -24.6737 32.5419 0.0
26 opcl -18.5726 -3.1242 0.0
27 opcp -25.1287 17.3435 0.0
28 opcx -18.2010 11.2056 0.0
29 oprd -6.8861 -1.9950 0.0
30 rdmt 5.2487 -32.5459 0.0
31 widc -73.2830 -10.7303 0.0
32 womt 7.9672 -60.6817 0.0
33 ldsw 11.6220 5.4900 0.0
# number of Coulomb stress patches (using the fault geometry of Simons et al (2002))
263
# no   x1     x2   x3 length width strike dip friction
1  15.337 -9.987 13.685 7.91 6.31 165.1 91.0 0.6
2  15.317 -10.063 9.176 7.91 4.51 165.1 91.0 0.6
3  15.303 -10.117 5.9555 3.96 3.22 165.1 91.0 0.6
4  11.48 -9.1017 5.9555 3.96 3.22 165.1 91.0 0.6
5  15.292 -10.156 3.6551 2.64 2.3 165.1 91.0 0.6
6  12.744 -9.4791 3.6551 2.64 2.3 165.1 91.0 0.6
7  10.196 -8.802 3.6551 2.64 2.3 165.1 91.0 0.6
8  15.285 -10.184 2.012 1.98 1.64 165.1 91.0 0.6
9  13.374 -9.6761 2.012 1.98 1.64 165.1 91.0 0.6
10  11.462 -9.1683 2.012 1.98 1.64 165.1 91.0 0.6
11  9.5513 -8.6604 2.012 1.98 1.64 165.1 91.0 0.6
12  15.28 -10.204 0.83833 1.32 1.17 165.1 91.0 0.6
13  14.005 -9.8651 0.83833 1.32 1.17 165.1 91.0 0.6
14  12.731 -9.5266 0.83833 1.32 1.17 165.1 91.0 0.6
15  11.457 -9.1881 0.83833 1.32 1.17 165.1 91.0 0.6
16  10.183 -8.8495 0.83833 1.32 1.17 165.1 91.0 0.6
17  8.909 -8.511 0.83833 1.32 1.17 165.1 91.0 0.6
18  15.276 -10.218 0 0.879 0.838 165.1 91.0 0.6
19  14.426 -9.9921 0 0.879 0.838 165.1 91.0 0.6
20  13.577 -9.7664 0 0.879 0.838 165.1 91.0 0.6
21  12.728 -9.5407 0 0.879 0.838 165.1 91.0 0.6
22  11.878 -9.315 0 0.879 0.838 165.1 91.0 0.6
23  11.029 -9.0894 0 0.879 0.838 165.1 91.0 0.6
24  10.179 -8.8637 0 0.879 0.838 165.1 91.0 0.6
25  9.3299 -8.638 0 0.879 0.838 165.1 91.0 0.6
26  8.4805 -8.4123 0 0.879 0.838 165.1 91.0 0.6
27  7.8335 -8.3138 13.685 6.97 6.31 142.5 91.0 0.6
28  2.3036 -4.0705 13.685 6.97 6.31 142.5 91.0 0.6
29  7.7856 -8.3762 9.176 4.65 4.51 142.5 91.0 0.6
30  4.099 -5.5474 9.176 4.65 4.51 142.5 91.0 0.6
31  0.41234 -2.7185 9.176 4.65 4.51 142.5 91.0 0.6
32  7.7514 -8.4208 5.9555 3.49 3.22 142.5 91.0 0.6
33  4.9864 -6.2992 5.9555 3.49 3.22 142.5 91.0 0.6
34  2.2214 -4.1775 5.9555 3.49 3.22 142.5 91.0 0.6
35  -0.54354 -2.0559 5.9555 3.49 3.22 142.5 91.0 0.6
36  7.7269 -8.4527 3.6551 2.32 2.3 142.5 91.0 0.6
37  5.8836 -7.0382 3.6551 2.32 2.3 142.5 91.0 0.6
38  4.0403 -5.6238 3.6551 2.32 2.3 142.5 91.0 0.6
39  2.197 -4.2094 3.6551 2.32 2.3 142.5 91.0 0.6
40  0.35367 -2.795 3.6551 2.32 2.3 142.5 91.0 0.6
41  -1.4896 -1.3805 3.6551 2.32 2.3 142.5 91.0 0.6
42  7.7095 -8.4754 2.012 1.74 1.64 142.5 91.0 0.6
43  6.327 -7.4146 2.012 1.74 1.64 142.5 91.0 0.6
44  4.9445 -6.3538 2.012 1.74 1.64 142.5 91.0 0.6
45  3.562 -5.293 2.012 1.74 1.64 142.5 91.0 0.6
46  2.1795 -4.2321 2.012 1.74 1.64 142.5 91.0 0.6
47  0.79704 -3.1713 2.012 1.74 1.64 142.5 91.0 0.6
48  -0.58545 -2.1105 2.012 1.74 1.64 142.5 91.0 0.6
49  -1.9679 -1.0497 2.012 1.74 1.64 142.5 91.0 0.6
50  7.697 -8.4917 0.83833 1.27 1.17 142.5 91.0 0.6
51  6.6916 -7.7202 0.83833 1.27 1.17 142.5 91.0 0.6
52  5.6861 -6.9487 0.83833 1.27 1.17 142.5 91.0 0.6
53  4.6807 -6.1772 0.83833 1.27 1.17 142.5 91.0 0.6
54  3.6752 -5.4056 0.83833 1.27 1.17 142.5 91.0 0.6
55  2.6698 -4.6341 0.83833 1.27 1.17 142.5 91.0 0.6
56  1.6643 -3.8626 0.83833 1.27 1.17 142.5 91.0 0.6
57  0.65889 -3.0911 0.83833 1.27 1.17 142.5 91.0 0.6
58  -0.34656 -2.3196 0.83833 1.27 1.17 142.5 91.0 0.6
59  -1.352 -1.5481 0.83833 1.27 1.17 142.5 91.0 0.6
60  -2.3575 -0.77661 0.83833 1.27 1.17 142.5 91.0 0.6
61  7.6881 -8.5033 0 0.929 0.838 142.5 91.0 0.6
62  6.9508 -7.9375 0 0.929 0.838 142.5 91.0 0.6
63  6.2134 -7.3717 0 0.929 0.838 142.5 91.0 0.6
64  5.4761 -6.806 0 0.929 0.838 142.5 91.0 0.6
65  4.7388 -6.2402 0 0.929 0.838 142.5 91.0 0.6
66  4.0015 -5.6744 0 0.929 0.838 142.5 91.0 0.6
67  3.2641 -5.1087 0 0.929 0.838 142.5 91.0 0.6
68  2.5268 -4.5429 0 0.929 0.838 142.5 91.0 0.6
69  1.7895 -3.9771 0 0.929 0.838 142.5 91.0 0.6
70  1.0522 -3.4113 0 0.929 0.838 142.5 91.0 0.6
71  0.31483 -2.8456 0 0.929 0.838 142.5 91.0 0.6
72  -0.4225 -2.2798 0 0.929 0.838 142.5 91.0 0.6
73  -1.1598 -1.714 0 0.929 0.838 142.5 91.0 0.6
74  -1.8971 -1.1483 0 0.929 0.838 142.5 91.0 0.6
75  -2.6345 -0.58248 0 0.929 0.838 142.5 91.0 0.6
76  15.471 -4.1629 13.685 6.5 6.31 167.0 91.0 0.6
77  9.1377 -2.7008 13.685 6.5 6.31 167.0 91.0 0.6
78  2.8042 -1.2386 13.685 6.5 6.31 167.0 91.0 0.6
79  -3.5292 0.2236 13.685 6.5 6.31 167.0 91.0 0.6
80  15.453 -4.2396 9.176 4.33 4.51 167.0 91.0 0.6
81  11.231 -3.2648 9.176 4.33 4.51 167.0 91.0 0.6
82  7.0088 -2.29 9.176 4.33 4.51 167.0 91.0 0.6
83  2.7865 -1.3153 9.176 4.33 4.51 167.0 91.0 0.6
84  -1.4357 -0.34047 9.176 4.33 4.51 167.0 91.0 0.6
85  -5.658 0.63431 9.176 4.33 4.51 167.0 91.0 0.6
86  15.441 -4.2944 5.9555 3.25 3.22 167.0 91.0 0.6
87  12.274 -3.5633 5.9555 3.25 3.22 167.0 91.0 0.6
88  9.1073 -2.8322 5.9555 3.25 3.22 167.0 91.0 0.6
89  5.9406 -2.1011 5.9555 3.25 3.22 167.0 91.0 0.6
90  2.7739 -1.37 5.9555 3.25 3.22 167.0 91.0 0.6
91  -0.39281 -0.63894 5.9555 3.25 3.22 167.0 91.0 0.6
92  -3.5595 0.092147 5.9555 3.25 3.22 167.0 91.0 0.6
93  -6.7262 0.82324 5.9555 3.25 3.22 167.0 91.0 0.6
94  15.432 -4.3335 3.6551 2.36 2.3 167.0 91.0 0.6
95  13.129 -3.8018 3.6551 2.36 2.3 167.0 91.0 0.6
96  10.826 -3.2701 3.6551 2.36 2.3 167.0 91.0 0.6
97  8.5225 -2.7384 3.6551 2.36 2.3 167.0 91.0 0.6
98  6.2194 -2.2067 3.6551 2.36 2.3 167.0 91.0 0.6
99  3.9164 -1.675 3.6551 2.36 2.3 167.0 91.0 0.6
100  1.6133 -1.1433 3.6551 2.36 2.3 167.0 91.0 0.6
101  -0.68972 -0.61161 3.6551 2.36 2.3 167.0 91.0 0.6
102  -2.9928 -0.079903 3.6551 2.36 2.3 167.0 91.0 0.6
103  -5.2958 0.4518 3.6551 2.36 2.3 167.0 91.0 0.6
104  -7.5989 0.9835 3.6551 2.36 2.3 167.0 91.0 0.6
105  15.425 -4.3615 2.012 1.73 1.64 167.0 91.0 0.6
106  13.736 -3.9716 2.012 1.73 1.64 167.0 91.0 0.6
107  12.047 -3.5816 2.012 1.73 1.64 167.0 91.0 0.6
108  10.358 -3.1917 2.012 1.73 1.64 167.0 91.0 0.6
109  8.6696 -2.8018 2.012 1.73 1.64 167.0 91.0 0.6
110  6.9807 -2.4119 2.012 1.73 1.64 167.0 91.0 0.6
111  5.2918 -2.022 2.012 1.73 1.64 167.0 91.0 0.6
112  3.6029 -1.6321 2.012 1.73 1.64 167.0 91.0 0.6
113  1.914 -1.2421 2.012 1.73 1.64 167.0 91.0 0.6
114  0.22505 -0.85223 2.012 1.73 1.64 167.0 91.0 0.6
115  -1.4639 -0.46232 2.012 1.73 1.64 167.0 91.0 0.6
116  -3.1528 -0.072402 2.012 1.73 1.64 167.0 91.0 0.6
117  -4.8417 0.31751 2.012 1.73 1.64 167.0 91.0 0.6
118  -6.5306 0.70743 2.012 1.73 1.64 167.0 91.0 0.6
119  -8.2195 1.0973 2.012 1.73 1.64 167.0 91.0 0.6
120  15.421 -4.3814 0.83833 1.24 1.17 167.0 91.0 0.6
121  14.214 -4.1029 0.83833 1.24 1.17 167.0 91.0 0.6
122  13.008 -3.8244 0.83833 1.24 1.17 167.0 91.0 0.6
123  11.802 -3.5459 0.83833 1.24 1.17 167.0 91.0 0.6
124  10.595 -3.2674 0.83833 1.24 1.17 167.0 91.0 0.6
125  9.3888 -2.9889 0.83833 1.24 1.17 167.0 91.0 0.6
126  8.1824 -2.7104 0.83833 1.24 1.17 167.0 91.0 0.6
127  6.9761 -2.4319 0.83833 1.24 1.17 167.0 91.0 0.6
128  5.7697 -2.1533 0.83833 1.24 1.17 167.0 91.0 0.6
129  4.5633 -1.8748 0.83833 1.24 1.17 167.0 91.0 0.6
130  3.357 -1.5963 0.83833 1.24 1.17 167.0 91.0 0.6
131  2.1506 -1.3178 0.83833 1.24 1.17 167.0 91.0 0.6
132  0.94426 -1.0393 0.83833 1.24 1.17 167.0 91.0 0.6
133  -0.2621 -0.76079 0.83833 1.24 1.17 167.0 91.0 0.6
134  -1.4685 -0.48228 0.83833 1.24 1.17 167.0 91.0 0.6
135  -2.6748 -0.20377 0.83833 1.24 1.17 167.0 91.0 0.6
136  -3.8812 0.074744 0.83833 1.24 1.17 167.0 91.0 0.6
137  -5.0876 0.35325 0.83833 1.24 1.17 167.0 91.0 0.6
138  -6.2939 0.63177 0.83833 1.24 1.17 167.0 91.0 0.6
139  -7.5003 0.91028 0.83833 1.24 1.17 167.0 91.0 0.6
140  -8.7066 1.1888 0.83833 1.24 1.17 167.0 91.0 0.6
141  15.417 -4.3957 0 0.897 0.838 167.0 91.0 0.6
142  14.544 -4.194 0 0.897 0.838 167.0 91.0 0.6
143  13.67 -3.9923 0 0.897 0.838 167.0 91.0 0.6
144  12.797 -3.7906 0 0.897 0.838 167.0 91.0 0.6
145  11.923 -3.589 0 0.897 0.838 167.0 91.0 0.6
146  11.049 -3.3873 0 0.897 0.838 167.0 91.0 0.6
147  10.176 -3.1856 0 0.897 0.838 167.0 91.0 0.6
148  9.3023 -2.9839 0 0.897 0.838 167.0 91.0 0.6
149  8.4287 -2.7822 0 0.897 0.838 167.0 91.0 0.6
150  7.5552 -2.5806 0 0.897 0.838 167.0 91.0 0.6
151  6.6816 -2.3789 0 0.897 0.838 167.0 91.0 0.6
152  5.808 -2.1772 0 0.897 0.838 167.0 91.0 0.6
153  4.9344 -1.9755 0 0.897 0.838 167.0 91.0 0.6
154  4.0609 -1.7738 0 0.897 0.838 167.0 91.0 0.6
155  3.1873 -1.5722 0 0.897 0.838 167.0 91.0 0.6
156  2.3137 -1.3705 0 0.897 0.838 167.0 91.0 0.6
157  1.4402 -1.1688 0 0.897 0.838 167.0 91.0 0.6
158  0.56658 -0.96712 0 0.897 0.838 167.0 91.0 0.6
159  -0.307 -0.76544 0 0.897 0.838 167.0 91.0 0.6
160  -1.1806 -0.56376 0 0.897 0.838 167.0 91.0 0.6
161  -2.0541 -0.36208 0 0.897 0.838 167.0 91.0 0.6
162  -2.9277 -0.1604 0 0.897 0.838 167.0 91.0 0.6
163  -3.8013 0.041278 0 0.897 0.838 167.0 91.0 0.6
164  -4.6749 0.24296 0 0.897 0.838 167.0 91.0 0.6
165  -5.5484 0.44464 0 0.897 0.838 167.0 91.0 0.6
166  -6.422 0.64632 0 0.897 0.838 167.0 91.0 0.6
167  -7.2956 0.848 0 0.897 0.838 167.0 91.0 0.6
168  -8.1692 1.0497 0 0.897 0.838 167.0 91.0 0.6
169  -9.0427 1.2514 0 0.897 0.838 167.0 91.0 0.6
170  -8.1513 4.6315 13.157 9.42 6.31 157.9 106.0 0.6
171  -8.6189 3.4798 8.8219 4.71 4.51 157.9 106.0 0.6
172  -12.983 5.2518 8.8219 4.71 4.51 157.9 106.0 0.6
173  -8.953 2.6572 5.7256 3.14 3.22 157.9 106.0 0.6
174  -11.862 3.8386 5.7256 3.14 3.22 157.9 106.0 0.6
175  -14.772 5.0199 5.7256 3.14 3.22 157.9 106.0 0.6
176  -9.1915 2.0697 3.5141 2.36 2.3 157.9 106.0 0.6
177  -11.374 2.9557 3.5141 2.36 2.3 157.9 106.0 0.6
178  -13.555 3.8417 3.5141 2.36 2.3 157.9 106.0 0.6
179  -15.737 4.7277 3.5141 2.36 2.3 157.9 106.0 0.6
180  -9.362 1.65 1.9343 1.88 1.64 157.9 106.0 0.6
181  -11.108 2.3588 1.9343 1.88 1.64 157.9 106.0 0.6
182  -12.853 3.0676 1.9343 1.88 1.64 157.9 106.0 0.6
183  -14.599 3.7764 1.9343 1.88 1.64 157.9 106.0 0.6
184  -16.344 4.4852 1.9343 1.88 1.64 157.9 106.0 0.6
185  -9.4837 1.3502 0.80598 1.35 1.17 157.9 106.0 0.6
186  -10.731 1.8565 0.80598 1.35 1.17 157.9 106.0 0.6
187  -11.977 2.3628 0.80598 1.35 1.17 157.9 106.0 0.6
188  -13.224 2.869 0.80598 1.35 1.17 157.9 106.0 0.6
189  -14.471 3.3753 0.80598 1.35 1.17 157.9 106.0 0.6
190  -15.718 3.8816 0.80598 1.35 1.17 157.9 106.0 0.6
191  -16.965 4.3879 0.80598 1.35 1.17 157.9 106.0 0.6
192  -9.5706 1.136 0 0.942 0.838 157.9 106.0 0.6
193  -10.443 1.4905 0 0.942 0.838 157.9 106.0 0.6
194  -11.316 1.8449 0 0.942 0.838 157.9 106.0 0.6
195  -12.189 2.1993 0 0.942 0.838 157.9 106.0 0.6
196  -13.062 2.5537 0 0.942 0.838 157.9 106.0 0.6
197  -13.935 2.9081 0 0.942 0.838 157.9 106.0 0.6
198  -14.807 3.2625 0 0.942 0.838 157.9 106.0 0.6
199  -15.68 3.6169 0 0.942 0.838 157.9 106.0 0.6
200  -16.553 3.9713 0 0.942 0.838 157.9 106.0 0.6
201  -17.426 4.3257 0 0.942 0.838 157.9 106.0 0.6
202  -15.004 5.6461 13.435 5.67 6.31 137.1 101.0 0.6
203  -19.155 9.5035 13.435 5.67 6.31 137.1 101.0 0.6
204  -23.306 13.361 13.435 5.67 6.31 137.1 101.0 0.6
205  -15.589 5.0158 9.0088 4.25 4.51 137.1 101.0 0.6
206  -18.703 7.9088 9.0088 4.25 4.51 137.1 101.0 0.6
207  -21.816 10.802 9.0088 4.25 4.51 137.1 101.0 0.6
208  -24.929 13.695 9.0088 4.25 4.51 137.1 101.0 0.6
209  -16.008 4.5656 5.847 3.4 3.22 137.1 101.0 0.6
210  -18.498 6.88 5.847 3.4 3.22 137.1 101.0 0.6
211  -20.989 9.1945 5.847 3.4 3.22 137.1 101.0 0.6
212  -23.48 11.509 5.847 3.4 3.22 137.1 101.0 0.6
213  -25.97 13.823 5.847 3.4 3.22 137.1 101.0 0.6
214  -16.307 4.244 3.5885 2.43 2.3 137.1 101.0 0.6
215  -18.086 5.8971 3.5885 2.43 2.3 137.1 101.0 0.6
216  -19.865 7.5503 3.5885 2.43 2.3 137.1 101.0 0.6
217  -21.644 9.2035 3.5885 2.43 2.3 137.1 101.0 0.6
218  -23.423 10.857 3.5885 2.43 2.3 137.1 101.0 0.6
219  -25.202 12.51 3.5885 2.43 2.3 137.1 101.0 0.6
220  -26.981 14.163 3.5885 2.43 2.3 137.1 101.0 0.6
221  -16.52 4.0143 1.9753 1.7 1.64 137.1 101.0 0.6
222  -17.765 5.1715 1.9753 1.7 1.64 137.1 101.0 0.6
223  -19.011 6.3287 1.9753 1.7 1.64 137.1 101.0 0.6
224  -20.256 7.4859 1.9753 1.7 1.64 137.1 101.0 0.6
225  -21.501 8.6432 1.9753 1.7 1.64 137.1 101.0 0.6
226  -22.747 9.8004 1.9753 1.7 1.64 137.1 101.0 0.6
227  -23.992 10.958 1.9753 1.7 1.64 137.1 101.0 0.6
228  -25.237 12.115 1.9753 1.7 1.64 137.1 101.0 0.6
229  -26.483 13.272 1.9753 1.7 1.64 137.1 101.0 0.6
230  -27.728 14.429 1.9753 1.7 1.64 137.1 101.0 0.6
231  -16.672 3.8502 0.82305 1.21 1.17 137.1 101.0 0.6
232  -17.562 4.6768 0.82305 1.21 1.17 137.1 101.0 0.6
233  -18.451 5.5034 0.82305 1.21 1.17 137.1 101.0 0.6
234  -19.341 6.33 0.82305 1.21 1.17 137.1 101.0 0.6
235  -20.23 7.1565 0.82305 1.21 1.17 137.1 101.0 0.6
236  -21.12 7.9831 0.82305 1.21 1.17 137.1 101.0 0.6
237  -22.01 8.8097 0.82305 1.21 1.17 137.1 101.0 0.6
238  -22.899 9.6363 0.82305 1.21 1.17 137.1 101.0 0.6
239  -23.789 10.463 0.82305 1.21 1.17 137.1 101.0 0.6
240  -24.678 11.289 0.82305 1.21 1.17 137.1 101.0 0.6
241  -25.568 12.116 0.82305 1.21 1.17 137.1 101.0 0.6
242  -26.457 12.943 0.82305 1.21 1.17 137.1 101.0 0.6
243  -27.347 13.769 0.82305 1.21 1.17 137.1 101.0 0.6
244  -28.236 14.596 0.82305 1.21 1.17 137.1 101.0 0.6
245  -16.781 3.733 0 0.895 0.838 137.1 101.0 0.6
246  -17.437 4.3421 0 0.895 0.838 137.1 101.0 0.6
247  -18.092 4.9511 0 0.895 0.838 137.1 101.0 0.6
248  -18.748 5.5602 0 0.895 0.838 137.1 101.0 0.6
249  -19.403 6.1693 0 0.895 0.838 137.1 101.0 0.6
250  -20.059 6.7783 0 0.895 0.838 137.1 101.0 0.6
251  -20.714 7.3874 0 0.895 0.838 137.1 101.0 0.6
252  -21.369 7.9965 0 0.895 0.838 137.1 101.0 0.6
253  -22.025 8.6055 0 0.895 0.838 137.1 101.0 0.6
254  -22.68 9.2146 0 0.895 0.838 137.1 101.0 0.6
255  -23.336 9.8237 0 0.895 0.838 137.1 101.0 0.6
256  -23.991 10.433 0 0.895 0.838 137.1 101.0 0.6
257  -24.647 11.042 0 0.895 0.838 137.1 101.0 0.6
258  -25.302 11.651 0 0.895 0.838 137.1 101.0 0.6
259  -25.957 12.26 0 0.895 0.838 137.1 101.0 0.6
260  -26.613 12.869 0 0.895 0.838 137.1 101.0 0.6
261  -27.268 13.478 0 0.895 0.838 137.1 101.0 0.6
262  -27.924 14.087 0 0.895 0.838 137.1 101.0 0.6
263  -28.579 14.696 0 0.895 0.838 137.1 101.0 0.6
# number of prestress interfaces with depth
0
# number of linear viscous interfaces
0
# number of powerlaw viscous interfaces
5
# no depth gammadot0 power cohesion
   1    25         1   1.0      0.0
   2    30         1   1.0      0.0
   3    30         0   1.0      0.0
   4    60         0   1.0      0.0
   5    60         1   1.0      0.0
# number of nonlinear ductile zones
0
# number of fault creep interface
0
# number of interseismic dislocation
0
# number of interseismic dykes
0
# number of coseismic events
1
# number of shear dislocations (coseismic slip model of Fialko (2004))
426
# index slip x1 x2 x3 length width strike dip rake
1 1.3475 14.246 -45.439 10.056 5.6 4.94 132.7 91.0 -114.7
2 1.8921 10.446 -41.319 10.056 5.6 4.94 132.7 91.0 -151.8
3 0.46688 14.201 -45.481 6.5269 3.74 3.53 132.7 91.0 -150.6
4 0.38986 11.668 -42.734 6.5269 3.74 3.53 132.7 91.0 -175.6
5 1.3331 9.1346 -39.987 6.5269 3.74 3.53 132.7 91.0 172.8
6 0.20179 14.168 -45.511 4.0058 2.8 2.52 132.7 91.0 157.4
7 0.29966 12.268 -43.45 4.0058 2.8 2.52 132.7 91.0 144.0
8 0.45731 10.369 -41.39 4.0058 2.8 2.52 132.7 91.0 147.0
9 1.4127 8.469 -39.33 4.0058 2.8 2.52 132.7 91.0 172.9
10 0.29987 14.145 -45.532 2.205 1.87 1.8 132.7 91.0 -174.3
11 0.17694 12.879 -44.159 2.205 1.87 1.8 132.7 91.0 158.4
12 0.059721 11.612 -42.785 2.205 1.87 1.8 132.7 91.0 124.6
13 0.12027 10.346 -41.412 2.205 1.87 1.8 132.7 91.0 -172.2
14 0.51562 9.0791 -40.038 2.205 1.87 1.8 132.7 91.0 -163.2
15 1.2953 7.8126 -38.665 2.205 1.87 1.8 132.7 91.0 -170.8
16 0.046779 14.129 -45.547 0.91876 1.4 1.29 132.7 91.0 -169.6
17 0.14997 13.179 -44.517 0.91876 1.4 1.29 132.7 91.0 131.5
18 0.21267 12.229 -43.487 0.91876 1.4 1.29 132.7 91.0 152.4
19 0.13236 11.279 -42.457 0.91876 1.4 1.29 132.7 91.0 171.2
20 0.19214 10.329 -41.427 0.91876 1.4 1.29 132.7 91.0 173.3
21 0.3773 9.3792 -40.397 0.91876 1.4 1.29 132.7 91.0 -163.6
22 0.65384 8.4293 -39.366 0.91876 1.4 1.29 132.7 91.0 -172.6
23 1.3611 7.4795 -38.336 0.91876 1.4 1.29 132.7 91.0 -175.8
24 0.05589 14.117 -45.558 0 1.02 0.919 132.7 91.0 96.2
25 0.038805 13.426 -44.809 0 1.02 0.919 132.7 91.0 -109.7
26 0.087788 12.735 -44.06 0 1.02 0.919 132.7 91.0 163.3
27 0.2068 12.044 -43.311 0 1.02 0.919 132.7 91.0 -156.7
28 0.022622 11.354 -42.561 0 1.02 0.919 132.7 91.0 111.7
29 0.14741 10.663 -41.812 0 1.02 0.919 132.7 91.0 152.5
30 0.21159 9.9719 -41.063 0 1.02 0.919 132.7 91.0 174.3
31 0.04416 9.2811 -40.314 0 1.02 0.919 132.7 91.0 -106.8
32 0.90637 8.5903 -39.565 0 1.02 0.919 132.7 91.0 166.6
33 1.0182 7.8994 -38.815 0 1.02 0.919 132.7 91.0 169.6
34 2.0172 7.2086 -38.066 0 1.02 0.919 132.7 91.0 172.6
35 1.6461 6.4315 -36.997 10.056 5.46 4.94 154.6 91.0 159.8
36 0.8503 6.405 -37.053 6.5269 5.46 3.53 154.6 91.0 128.2
37 0.82255 6.3861 -37.092 4.0058 2.73 2.52 154.6 91.0 159.2
38 0.35483 3.9205 -35.921 4.0058 2.73 2.52 154.6 91.0 120.7
39 1.0389 6.3726 -37.121 2.205 1.82 1.8 154.6 91.0 176.7
40 0.74221 4.7289 -36.34 2.205 1.82 1.8 154.6 91.0 163.8
41 0.65162 3.0851 -35.559 2.205 1.82 1.8 154.6 91.0 122.0
42 1.2486 6.363 -37.141 0.91876 1.36 1.29 154.6 91.0 164.6
43 1.2495 5.1302 -36.555 0.91876 1.36 1.29 154.6 91.0 159.7
44 1.0503 3.8974 -35.97 0.91876 1.36 1.29 154.6 91.0 129.1
45 1.3042 2.6645 -35.384 0.91876 1.36 1.29 154.6 91.0 114.4
46 0.86612 6.3561 -37.156 0 1.09 0.919 154.6 91.0 124.4
47 1.1298 5.3699 -36.687 0 1.09 0.919 154.6 91.0 143.1
48 1.4622 4.3836 -36.218 0 1.09 0.919 154.6 91.0 134.9
49 1.9023 3.3974 -35.75 0 1.09 0.919 154.6 91.0 133.6
50 1.2704 2.4111 -35.281 0 1.09 0.919 154.6 91.0 99.9
51 2.0664 1.972 -34.872 10.056 9.33 4.94 125.5 91.0 152.8
52 3.7394 1.9218 -34.908 6.5269 4.66 3.53 125.5 91.0 173.4
53 4.2129 -0.78635 -31.111 6.5269 4.66 3.53 125.5 91.0 178.7
54 4.4265 1.886 -34.933 4.0058 3.11 2.52 125.5 91.0 177.0
55 3.8343 0.080542 -32.402 4.0058 3.11 2.52 125.5 91.0 -177.6
56 3.0634 -1.7249 -29.87 4.0058 3.11 2.52 125.5 91.0 -175.1
57 3.1116 1.8604 -34.952 2.205 2.33 1.8 125.5 91.0 171.0
58 3.2522 0.50631 -33.053 2.205 2.33 1.8 125.5 91.0 175.4
59 3.2794 -0.84777 -31.154 2.205 2.33 1.8 125.5 91.0 -179.7
60 2.3577 -2.2018 -29.256 2.205 2.33 1.8 125.5 91.0 179.3
61 1.8171 1.8421 -34.965 0.91876 1.55 1.29 125.5 91.0 164.2
62 2.3002 0.93939 -33.699 0.91876 1.55 1.29 125.5 91.0 161.5
63 2.9062 0.036672 -32.433 0.91876 1.55 1.29 125.5 91.0 167.6
64 3.1308 -0.86605 -31.167 0.91876 1.55 1.29 125.5 91.0 171.3
65 3.176 -1.7688 -29.902 0.91876 1.55 1.29 125.5 91.0 179.3
66 3.5948 -2.6715 -28.636 0.91876 1.55 1.29 125.5 91.0 -177.9
67 3.1697 1.8291 -34.974 0 1.04 0.919 125.5 91.0 180.0
68 0.87503 1.2272 -34.13 0 1.04 0.919 125.5 91.0 109.2
69 2.8863 0.62543 -33.286 0 1.04 0.919 125.5 91.0 154.2
70 3.2476 0.023615 -32.443 0 1.04 0.919 125.5 91.0 161.1
71 3.1495 -0.5782 -31.599 0 1.04 0.919 125.5 91.0 163.4
72 0.98179 -1.18 -30.755 0 1.04 0.919 125.5 91.0 125.6
73 2.7862 -1.7818 -29.911 0 1.04 0.919 125.5 91.0 173.8
74 3.0355 -2.3836 -29.067 0 1.04 0.919 125.5 91.0 -175.5
75 2.1897 -2.9854 -28.223 0 1.04 0.919 125.5 91.0 -162.9
76 1.7024 -2.2671 -28.857 10.056 4.78 4.94 142.2 91.0 -165.9
77 1.7742 -6.0464 -25.928 10.056 4.78 4.94 142.2 91.0 -149.0
78 1.513 -9.8257 -22.999 10.056 4.78 4.94 142.2 91.0 -120.6
79 1.4857 -13.605 -20.07 10.056 4.78 4.94 142.2 91.0 -97.3
80 3.0039 -2.3049 -28.905 6.5269 3.83 3.53 142.2 91.0 175.6
81 2.6857 -5.3283 -26.562 6.5269 3.83 3.53 142.2 91.0 -170.2
82 1.0492 -8.3517 -24.219 6.5269 3.83 3.53 142.2 91.0 -116.0
83 0.89433 -11.375 -21.876 6.5269 3.83 3.53 142.2 91.0 -100.0
84 0.51509 -14.399 -19.533 6.5269 3.83 3.53 142.2 91.0 -105.8
85 2.4478 -2.3318 -28.94 4.0058 2.73 2.52 142.2 91.0 153.1
86 4.3577 -4.4914 -27.266 4.0058 2.73 2.52 142.2 91.0 173.5
87 2.9612 -6.651 -25.593 4.0058 2.73 2.52 142.2 91.0 -168.5
88 0.95596 -8.8106 -23.919 4.0058 2.73 2.52 142.2 91.0 -117.9
89 0.76893 -10.97 -22.246 4.0058 2.73 2.52 142.2 91.0 -103.8
90 0.68027 -13.13 -20.572 4.0058 2.73 2.52 142.2 91.0 -99.6
91 0.1762 -15.289 -18.898 4.0058 2.73 2.52 142.2 91.0 -109.5
92 2.0449 -2.3511 -28.965 2.205 1.91 1.8 142.2 91.0 140.1
93 3.4501 -3.8628 -27.793 2.205 1.91 1.8 142.2 91.0 160.2
94 5.2534 -5.3745 -26.622 2.205 1.91 1.8 142.2 91.0 177.0
95 4.3993 -6.8862 -25.45 2.205 1.91 1.8 142.2 91.0 -170.8
96 1.778 -8.3979 -24.279 2.205 1.91 1.8 142.2 91.0 -156.9
97 1.4017 -9.9097 -23.107 2.205 1.91 1.8 142.2 91.0 -158.4
98 1.062 -11.421 -21.936 2.205 1.91 1.8 142.2 91.0 -140.2
99 0.87885 -12.933 -20.764 2.205 1.91 1.8 142.2 91.0 -114.8
100 0.82542 -14.445 -19.593 2.205 1.91 1.8 142.2 91.0 -93.5
101 0.54906 -15.957 -18.421 2.205 1.91 1.8 142.2 91.0 -149.6
102 2.5941 -2.3648 -28.983 0.91876 1.37 1.29 142.2 91.0 164.2
103 3.3956 -3.4446 -28.146 0.91876 1.37 1.29 142.2 91.0 169.9
104 3.6025 -4.5244 -27.309 0.91876 1.37 1.29 142.2 91.0 174.4
105 5.5648 -5.6042 -26.472 0.91876 1.37 1.29 142.2 91.0 180.0
106 5.2227 -6.684 -25.635 0.91876 1.37 1.29 142.2 91.0 -173.5
107 3.4498 -7.7638 -24.799 0.91876 1.37 1.29 142.2 91.0 -161.2
108 2.2608 -8.8436 -23.962 0.91876 1.37 1.29 142.2 91.0 -163.1
109 2.54 -9.9234 -23.125 0.91876 1.37 1.29 142.2 91.0 -175.5
110 2.4954 -11.003 -22.288 0.91876 1.37 1.29 142.2 91.0 -170.9
111 1.3605 -12.083 -21.451 0.91876 1.37 1.29 142.2 91.0 -157.1
112 1.2516 -13.163 -20.615 0.91876 1.37 1.29 142.2 91.0 -155.2
113 1.0291 -14.243 -19.778 0.91876 1.37 1.29 142.2 91.0 -140.5
114 0.60677 -15.322 -18.941 0.91876 1.37 1.29 142.2 91.0 -117.2
115 0.10625 -16.402 -18.104 0.91876 1.37 1.29 142.2 91.0 -112.7
116 2.1554 -2.3747 -28.995 0 1.01 0.919 142.2 91.0 173.0
117 3.0751 -3.1703 -28.379 0 1.01 0.919 142.2 91.0 179.2
118 3.6345 -3.9659 -27.762 0 1.01 0.919 142.2 91.0 -176.9
119 3.0977 -4.7616 -27.146 0 1.01 0.919 142.2 91.0 -178.9
120 5.4142 -5.5572 -26.529 0 1.01 0.919 142.2 91.0 -178.4
121 4.8272 -6.3529 -25.912 0 1.01 0.919 142.2 91.0 -172.1
122 4.0291 -7.1485 -25.296 0 1.01 0.919 142.2 91.0 -163.6
123 1.8484 -7.9441 -24.679 0 1.01 0.919 142.2 91.0 -92.4
124 4.6495 -8.7398 -24.063 0 1.01 0.919 142.2 91.0 -161.8
125 2.0184 -9.5354 -23.446 0 1.01 0.919 142.2 91.0 -168.1
126 1.7757 -10.331 -22.829 0 1.01 0.919 142.2 91.0 -170.4
127 1.3703 -11.127 -22.213 0 1.01 0.919 142.2 91.0 -158.1
128 1.0098 -11.922 -21.596 0 1.01 0.919 142.2 91.0 -150.4
129 0.87225 -12.718 -20.98 0 1.01 0.919 142.2 91.0 -131.4
130 0.67731 -13.514 -20.363 0 1.01 0.919 142.2 91.0 -112.3
131 0.58768 -14.309 -19.746 0 1.01 0.919 142.2 91.0 -98.9
132 0.46444 -15.105 -19.13 0 1.01 0.919 142.2 91.0 -110.1
133 0.986 -15.901 -18.513 0 1.01 0.919 142.2 91.0 -149.4
134 0.34545 -16.696 -17.897 0 1.01 0.919 142.2 91.0 -100.8
135 2.7194 -7.8921 -24.78 10.056 5.94 4.94 155.1 91.0 167.0
136 5.6099 -13.282 -22.284 10.056 5.94 4.94 155.1 91.0 175.0
137 4.9632 -18.671 -19.787 10.056 5.94 4.94 155.1 91.0 178.5
138 6.2634 -24.061 -17.29 10.056 5.94 4.94 155.1 91.0 174.2
139 5.7918 -7.918 -24.836 6.5269 3.96 3.53 155.1 91.0 176.6
140 6.6641 -11.511 -23.172 6.5269 3.96 3.53 155.1 91.0 174.2
141 7.1261 -15.104 -21.507 6.5269 3.96 3.53 155.1 91.0 176.8
142 5.2541 -18.697 -19.843 6.5269 3.96 3.53 155.1 91.0 -177.8
143 1.6796 -22.29 -18.178 6.5269 3.96 3.53 155.1 91.0 -176.2
144 0.98159 -25.883 -16.514 6.5269 3.96 3.53 155.1 91.0 123.8
145 8.0728 -7.9365 -24.876 4.0058 2.64 2.52 155.1 91.0 177.7
146 7.616 -10.332 -23.767 4.0058 2.64 2.52 155.1 91.0 173.5
147 7.5237 -12.727 -22.657 4.0058 2.64 2.52 155.1 91.0 171.8
148 7.2733 -15.122 -21.547 4.0058 2.64 2.52 155.1 91.0 173.2
149 6.3327 -17.518 -20.438 4.0058 2.64 2.52 155.1 91.0 177.3
150 4.6509 -19.913 -19.328 4.0058 2.64 2.52 155.1 91.0 -178.8
151 1.5383 -22.308 -18.218 4.0058 2.64 2.52 155.1 91.0 -172.8
152 0.53757 -24.704 -17.109 4.0058 2.64 2.52 155.1 91.0 92.1
153 1.4039 -27.099 -15.999 4.0058 2.64 2.52 155.1 91.0 90.3
154 7.8605 -7.9497 -24.905 2.205 1.98 1.8 155.1 91.0 176.3
155 7.785 -9.7462 -24.073 2.205 1.98 1.8 155.1 91.0 171.6
156 7.6874 -11.543 -23.24 2.205 1.98 1.8 155.1 91.0 169.4
157 6.5953 -13.339 -22.408 2.205 1.98 1.8 155.1 91.0 168.7
158 5.5559 -15.136 -21.576 2.205 1.98 1.8 155.1 91.0 169.5
159 5.9601 -16.932 -20.744 2.205 1.98 1.8 155.1 91.0 172.4
160 5.353 -18.729 -19.911 2.205 1.98 1.8 155.1 91.0 176.7
161 5.1229 -20.525 -19.079 2.205 1.98 1.8 155.1 91.0 178.0
162 4.1133 -22.322 -18.247 2.205 1.98 1.8 155.1 91.0 178.5
163 0.78389 -24.118 -17.415 2.205 1.98 1.8 155.1 91.0 151.1
164 1.1179 -25.915 -16.582 2.205 1.98 1.8 155.1 91.0 90.3
165 1.7541 -27.711 -15.75 2.205 1.98 1.8 155.1 91.0 127.8
166 5.1618 -7.9591 -24.925 0.91876 1.4 1.29 155.1 91.0 175.9
167 4.7242 -9.2273 -24.338 0.91876 1.4 1.29 155.1 91.0 170.0
168 5.3888 -10.495 -23.75 0.91876 1.4 1.29 155.1 91.0 164.3
169 5.457 -11.763 -23.163 0.91876 1.4 1.29 155.1 91.0 166.2
170 4.1013 -13.032 -22.575 0.91876 1.4 1.29 155.1 91.0 169.8
171 3.3365 -14.3 -21.988 0.91876 1.4 1.29 155.1 91.0 175.4
172 3.9033 -15.568 -21.4 0.91876 1.4 1.29 155.1 91.0 175.1
173 4.0315 -16.836 -20.813 0.91876 1.4 1.29 155.1 91.0 174.9
174 3.8532 -18.104 -20.225 0.91876 1.4 1.29 155.1 91.0 175.0
175 3.8967 -19.372 -19.638 0.91876 1.4 1.29 155.1 91.0 -178.2
176 3.6127 -20.64 -19.051 0.91876 1.4 1.29 155.1 91.0 -179.6
177 4.2347 -21.908 -18.463 0.91876 1.4 1.29 155.1 91.0 176.3
178 2.9552 -23.177 -17.876 0.91876 1.4 1.29 155.1 91.0 174.1
179 0.68766 -24.445 -17.288 0.91876 1.4 1.29 155.1 91.0 157.2
180 0.79218 -25.713 -16.701 0.91876 1.4 1.29 155.1 91.0 132.6
181 2.5776 -26.981 -16.113 0.91876 1.4 1.29 155.1 91.0 155.1
182 3.4791 -28.249 -15.526 0.91876 1.4 1.29 155.1 91.0 165.7
183 3.0218 -7.9659 -24.94 0 0.99 0.919 155.1 91.0 176.8
184 0.092013 -8.8641 -24.524 0 0.99 0.919 155.1 91.0 94.1
185 1.2851 -9.7624 -24.107 0 0.99 0.919 155.1 91.0 143.3
186 1.9462 -10.661 -23.691 0 0.99 0.919 155.1 91.0 130.8
187 1.581 -11.559 -23.275 0 0.99 0.919 155.1 91.0 106.0
188 1.34 -12.457 -22.859 0 0.99 0.919 155.1 91.0 139.2
189 0.43065 -13.355 -22.443 0 0.99 0.919 155.1 91.0 91.8
190 0.58089 -14.254 -22.027 0 0.99 0.919 155.1 91.0 -91.3
191 4.2456 -15.152 -21.611 0 0.99 0.919 155.1 91.0 -172.7
192 4.1809 -16.05 -21.195 0 0.99 0.919 155.1 91.0 -179.0
193 4.71 -16.948 -20.779 0 0.99 0.919 155.1 91.0 -176.1
194 0.54704 -17.847 -20.362 0 0.99 0.919 155.1 91.0 -128.1
195 1.6704 -18.745 -19.946 0 0.99 0.919 155.1 91.0 -175.7
196 5.5091 -19.643 -19.53 0 0.99 0.919 155.1 91.0 -169.5
197 4.3785 -20.541 -19.114 0 0.99 0.919 155.1 91.0 -165.7
198 1.5433 -21.44 -18.698 0 0.99 0.919 155.1 91.0 -174.7
199 1.8462 -22.338 -18.282 0 0.99 0.919 155.1 91.0 170.8
200 2.1803 -23.236 -17.866 0 0.99 0.919 155.1 91.0 169.6
201 0.054567 -24.134 -17.45 0 0.99 0.919 155.1 91.0 111.6
202 0.74211 -25.033 -17.033 0 0.99 0.919 155.1 91.0 167.5
203 0.12396 -25.931 -16.617 0 0.99 0.919 155.1 91.0 105.6
204 1.6743 -26.829 -16.201 0 0.99 0.919 155.1 91.0 156.7
205 0.99002 -27.727 -15.785 0 0.99 0.919 155.1 91.0 116.7
206 3.0638 -28.626 -15.369 0 0.99 0.919 155.1 91.0 169.9
207 3.4137 -24.344 -16.825 10.056 5.45 4.94 173.6 91.0 155.4
208 1.8634 -29.76 -16.219 10.056 5.45 4.94 173.6 91.0 144.9
209 2.2224 -35.176 -15.613 10.056 5.45 4.94 173.6 91.0 162.0
210 3.0044 -40.593 -15.007 10.056 5.45 4.94 173.6 91.0 158.5
211 5.418 -24.351 -16.886 6.5269 3.63 3.53 173.6 91.0 -175.0
212 1.7625 -27.961 -16.482 6.5269 3.63 3.53 173.6 91.0 174.4
213 1.9121 -31.572 -16.078 6.5269 3.63 3.53 173.6 91.0 177.7
214 2.2365 -35.183 -15.674 6.5269 3.63 3.53 173.6 91.0 -173.7
215 1.4487 -38.794 -15.27 6.5269 3.63 3.53 173.6 91.0 -160.0
216 2.0201 -42.405 -14.866 6.5269 3.63 3.53 173.6 91.0 -174.5
217 6.537 -24.355 -16.93 4.0058 2.73 2.52 173.6 91.0 -168.2
218 3.2458 -27.064 -16.627 4.0058 2.73 2.52 173.6 91.0 -172.5
219 3.0968 -29.772 -16.324 4.0058 2.73 2.52 173.6 91.0 -179.8
220 3.3949 -32.48 -16.021 4.0058 2.73 2.52 173.6 91.0 -179.4
221 3.1449 -35.188 -15.718 4.0058 2.73 2.52 173.6 91.0 -177.1
222 2.9154 -37.896 -15.415 4.0058 2.73 2.52 173.6 91.0 -170.5
223 1.8167 -40.604 -15.112 4.0058 2.73 2.52 173.6 91.0 -155.2
224 3.9446 -43.313 -14.809 4.0058 2.73 2.52 173.6 91.0 -178.0
225 6.5971 -24.359 -16.961 2.205 1.98 1.8 173.6 91.0 -166.8
226 4.4559 -26.329 -16.741 2.205 1.98 1.8 173.6 91.0 -172.3
227 4.2798 -28.298 -16.52 2.205 1.98 1.8 173.6 91.0 -178.0
228 5.6515 -30.268 -16.3 2.205 1.98 1.8 173.6 91.0 179.5
229 5.5716 -32.237 -16.08 2.205 1.98 1.8 173.6 91.0 178.6
230 4.6302 -34.207 -15.859 2.205 1.98 1.8 173.6 91.0 176.7
231 3.6476 -36.176 -15.639 2.205 1.98 1.8 173.6 91.0 177.3
232 4.1017 -38.146 -15.419 2.205 1.98 1.8 173.6 91.0 177.8
233 3.3488 -40.116 -15.198 2.205 1.98 1.8 173.6 91.0 -174.4
234 2.7548 -42.085 -14.978 2.205 1.98 1.8 173.6 91.0 -170.3
235 5.5995 -44.055 -14.757 2.205 1.98 1.8 173.6 91.0 174.7
236 5.7398 -24.361 -16.983 0.91876 1.36 1.29 173.6 91.0 -163.6
237 3.8519 -25.716 -16.832 0.91876 1.36 1.29 173.6 91.0 -164.7
238 3.2288 -27.07 -16.68 0.91876 1.36 1.29 173.6 91.0 -172.7
239 3.3095 -28.424 -16.529 0.91876 1.36 1.29 173.6 91.0 -177.0
240 4.7332 -29.778 -16.377 0.91876 1.36 1.29 173.6 91.0 -178.1
241 5.1829 -31.132 -16.226 0.91876 1.36 1.29 173.6 91.0 -179.1
242 5.9437 -32.486 -16.074 0.91876 1.36 1.29 173.6 91.0 -179.7
243 4.1306 -33.84 -15.923 0.91876 1.36 1.29 173.6 91.0 178.9
244 4.2325 -35.194 -15.771 0.91876 1.36 1.29 173.6 91.0 179.2
245 3.1194 -36.548 -15.62 0.91876 1.36 1.29 173.6 91.0 -176.0
246 3.3847 -37.902 -15.468 0.91876 1.36 1.29 173.6 91.0 -179.8
247 2.6967 -39.256 -15.317 0.91876 1.36 1.29 173.6 91.0 178.0
248 3.6303 -40.61 -15.165 0.91876 1.36 1.29 173.6 91.0 -179.5
249 2.8975 -41.965 -15.014 0.91876 1.36 1.29 173.6 91.0 -174.9
250 2.8228 -43.319 -14.862 0.91876 1.36 1.29 173.6 91.0 -177.5
251 2.6276 -44.673 -14.711 0.91876 1.36 1.29 173.6 91.0 156.4
252 3.6794 -24.363 -16.999 0 0.991 0.919 173.6 91.0 -149.5
253 3.903 -25.348 -16.889 0 0.991 0.919 173.6 91.0 -158.3
254 1.9118 -26.333 -16.779 0 0.991 0.919 173.6 91.0 -152.5
255 1.4942 -27.318 -16.669 0 0.991 0.919 173.6 91.0 -157.2
256 1.1064 -28.302 -16.559 0 0.991 0.919 173.6 91.0 -167.9
257 0.033682 -29.287 -16.448 0 0.991 0.919 173.6 91.0 -97.1
258 2.5274 -30.272 -16.338 0 0.991 0.919 173.6 91.0 -174.1
259 0.89058 -31.257 -16.228 0 0.991 0.919 173.6 91.0 -174.1
260 1.7173 -32.242 -16.118 0 0.991 0.919 173.6 91.0 -176.8
261 3.0221 -33.226 -16.008 0 0.991 0.919 173.6 91.0 -176.3
262 0.85751 -34.211 -15.897 0 0.991 0.919 173.6 91.0 -155.2
263 1.3325 -35.196 -15.787 0 0.991 0.919 173.6 91.0 -169.7
264 2.5205 -36.181 -15.677 0 0.991 0.919 173.6 91.0 -168.8
265 2.5914 -37.165 -15.567 0 0.991 0.919 173.6 91.0 -169.1
266 1.8008 -38.15 -15.457 0 0.991 0.919 173.6 91.0 -176.2
267 0.29283 -39.135 -15.347 0 0.991 0.919 173.6 91.0 -174.0
268 2.0916 -40.12 -15.236 0 0.991 0.919 173.6 91.0 -179.7
269 2.5942 -41.105 -15.126 0 0.991 0.919 173.6 91.0 -172.4
270 1.1183 -42.089 -15.016 0 0.991 0.919 173.6 91.0 -168.6
271 1.9112 -43.074 -14.906 0 0.991 0.919 173.6 91.0 -171.8
272 3.6811 -44.059 -14.796 0 0.991 0.919 173.6 91.0 -178.6
273 1.7971 -45.044 -14.686 0 0.991 0.919 173.6 91.0 155.9
274 0.073286 -48.802 -53.084 7.4656 4 3.51 42.4 95.0 87.5
275 0.43368 -45.848 -50.387 7.4656 4 3.51 42.4 95.0 89.2
276 0.71009 -42.894 -47.69 7.4656 4 3.51 42.4 95.0 89.6
277 0.92055 -39.94 -44.992 7.4656 4 3.51 42.4 95.0 89.2
278 0.83887 -36.986 -42.295 7.4656 4 3.51 42.4 95.0 89.4
279 0.32071 -34.033 -39.598 7.4656 4 3.51 42.4 95.0 69.0
280 0.02884 -48.949 -52.923 4.9682 2.67 2.51 42.4 95.0 -16.4
281 0.048285 -46.98 -51.125 4.9682 2.67 2.51 42.4 95.0 42.7
282 0.10827 -45.011 -49.326 4.9682 2.67 2.51 42.4 95.0 71.3
283 0.068555 -43.041 -47.528 4.9682 2.67 2.51 42.4 95.0 66.3
284 0.16023 -41.072 -45.73 4.9682 2.67 2.51 42.4 95.0 86.8
285 0.19689 -39.103 -43.932 4.9682 2.67 2.51 42.4 95.0 88.2
286 0.28677 -37.134 -42.134 4.9682 2.67 2.51 42.4 95.0 86.8
287 0.26808 -35.164 -40.336 4.9682 2.67 2.51 42.4 95.0 87.8
288 0.20858 -33.195 -38.538 4.9682 2.67 2.51 42.4 95.0 -34.6
289 0.46329 -49.054 -52.807 3.1843 2 1.79 42.4 95.0 43.0
290 0.1227 -47.577 -51.459 3.1843 2 1.79 42.4 95.0 25.6
291 0.066533 -46.1 -50.11 3.1843 2 1.79 42.4 95.0 -76.3
292 0.059686 -44.623 -48.762 3.1843 2 1.79 42.4 95.0 -68.6
293 0.13103 -43.147 -47.413 3.1843 2 1.79 42.4 95.0 -82.6
294 0.18335 -41.67 -46.064 3.1843 2 1.79 42.4 95.0 -84.8
295 0.18514 -40.193 -44.716 3.1843 2 1.79 42.4 95.0 -84.0
296 0.22299 -38.716 -43.367 3.1843 2 1.79 42.4 95.0 -86.8
297 0.15775 -37.239 -42.019 3.1843 2 1.79 42.4 95.0 -90.0
298 0.016948 -35.762 -40.67 3.1843 2 1.79 42.4 95.0 -58.4
299 0.044994 -34.285 -39.321 3.1843 2 1.79 42.4 95.0 -62.0
300 0.7677 -32.808 -37.973 3.1843 2 1.79 42.4 95.0 -22.6
301 1.4157 -49.129 -52.725 1.9101 1.41 1.28 42.4 95.0 29.5
302 0.98511 -48.087 -51.773 1.9101 1.41 1.28 42.4 95.0 20.3
303 0.54139 -47.044 -50.821 1.9101 1.41 1.28 42.4 95.0 -3.7
304 0.14982 -46.002 -49.869 1.9101 1.41 1.28 42.4 95.0 -89.7
305 0.034591 -44.959 -48.917 1.9101 1.41 1.28 42.4 95.0 -86.3
306 0.0060647 -43.917 -47.965 1.9101 1.41 1.28 42.4 95.0 12.4
307 0.027014 -42.874 -47.013 1.9101 1.41 1.28 42.4 95.0 -87.8
308 0.025038 -41.832 -46.061 1.9101 1.41 1.28 42.4 95.0 31.6
309 0.17182 -40.789 -45.109 1.9101 1.41 1.28 42.4 95.0 0.6
310 0.32586 -39.747 -44.158 1.9101 1.41 1.28 42.4 95.0 -11.3
311 0.38745 -38.704 -43.206 1.9101 1.41 1.28 42.4 95.0 -38.4
312 0.28091 -37.662 -42.254 1.9101 1.41 1.28 42.4 95.0 -53.9
313 0.20135 -36.619 -41.302 1.9101 1.41 1.28 42.4 95.0 -31.4
314 0.07192 -35.577 -40.35 1.9101 1.41 1.28 42.4 95.0 -55.7
315 0.0047984 -34.534 -39.398 1.9101 1.41 1.28 42.4 95.0 50.9
316 0.13688 -33.491 -38.446 1.9101 1.41 1.28 42.4 95.0 -8.5
317 0.40899 -32.449 -37.494 1.9101 1.41 1.28 42.4 95.0 -24.7
318 2.244 -49.183 -52.666 1 1 0.914 42.4 95.0 23.9
319 1.8274 -48.445 -51.992 1 1 0.914 42.4 95.0 19.7
320 1.4918 -47.706 -51.318 1 1 0.914 42.4 95.0 7.5
321 1.184 -46.968 -50.643 1 1 0.914 42.4 95.0 -7.0
322 0.48564 -46.229 -49.969 1 1 0.914 42.4 95.0 -25.3
323 0.13961 -45.491 -49.295 1 1 0.914 42.4 95.0 -89.9
324 0.013849 -44.752 -48.621 1 1 0.914 42.4 95.0 26.8
325 0.068621 -44.014 -47.946 1 1 0.914 42.4 95.0 87.2
326 0.073081 -43.275 -47.272 1 1 0.914 42.4 95.0 75.7
327 0.1554 -42.537 -46.598 1 1 0.914 42.4 95.0 44.5
328 0.40581 -41.799 -45.923 1 1 0.914 42.4 95.0 35.4
329 0.47077 -41.06 -45.249 1 1 0.914 42.4 95.0 26.1
330 0.66254 -40.322 -44.575 1 1 0.914 42.4 95.0 15.8
331 0.9204 -39.583 -43.9 1 1 0.914 42.4 95.0 11.9
332 0.92896 -38.845 -43.226 1 1 0.914 42.4 95.0 -1.5
333 0.66997 -38.106 -42.552 1 1 0.914 42.4 95.0 -11.3
334 0.41035 -37.368 -41.877 1 1 0.914 42.4 95.0 -6.0
335 0.67659 -36.629 -41.203 1 1 0.914 42.4 95.0 2.8
336 0.37661 -35.891 -40.529 1 1 0.914 42.4 95.0 -7.6
337 0.057776 -35.152 -39.855 1 1 0.914 42.4 95.0 -53.5
338 0.26348 -34.414 -39.18 1 1 0.914 42.4 95.0 3.2
339 0.055221 -33.675 -38.506 1 1 0.914 42.4 95.0 84.8
340 0.07339 -32.937 -37.832 1 1 0.914 42.4 95.0 60.4
341 0.048853 -32.199 -37.157 1 1 0.914 42.4 95.0 -58.1
342 0.041834 3.2423 -35.248 6.4893 4 3.51 80.3 91.0 75.8
343 1.503 3.9128 -31.305 6.4893 4 3.51 80.3 91.0 -11.8
344 3.0404 4.5833 -27.361 6.4893 4 3.51 80.3 91.0 -5.7
345 2.0837 5.2539 -23.418 6.4893 4 3.51 80.3 91.0 -11.3
346 1.1532 3.1992 -35.241 3.9827 2.67 2.51 80.3 91.0 -4.2
347 1.0105 3.6462 -32.612 3.9827 2.67 2.51 80.3 91.0 -10.9
348 2.1851 4.0932 -29.983 3.9827 2.67 2.51 80.3 91.0 0.3
349 2.1751 4.5402 -27.354 3.9827 2.67 2.51 80.3 91.0 1.9
350 1.3098 4.9872 -24.725 3.9827 2.67 2.51 80.3 91.0 -2.8
351 0.60884 5.4342 -22.096 3.9827 2.67 2.51 80.3 91.0 -28.8
352 0.64197 3.1684 -35.235 2.1923 2 1.79 80.3 91.0 -80.5
353 0.88005 3.5036 -33.264 2.1923 2 1.79 80.3 91.0 -31.7
354 1.3387 3.8389 -31.292 2.1923 2 1.79 80.3 91.0 -5.2
355 1.7594 4.1741 -29.32 2.1923 2 1.79 80.3 91.0 9.2
356 1.3745 4.5094 -27.349 2.1923 2 1.79 80.3 91.0 12.0
357 0.80014 4.8447 -25.377 2.1923 2 1.79 80.3 91.0 4.3
358 0.098118 5.1799 -23.405 2.1923 2 1.79 80.3 91.0 -6.5
359 0.15065 5.5152 -21.433 2.1923 2 1.79 80.3 91.0 -83.8
360 2.0608 3.1464 -35.232 0.91347 1.45 1.28 80.3 91.0 -37.0
361 1.385 3.3902 -33.798 0.91347 1.45 1.28 80.3 91.0 -37.2
362 0.44255 3.634 -32.364 0.91347 1.45 1.28 80.3 91.0 -75.4
363 0.15393 3.8778 -30.93 0.91347 1.45 1.28 80.3 91.0 -82.8
364 0.13796 4.1217 -29.496 0.91347 1.45 1.28 80.3 91.0 -15.3
365 0.68129 4.3655 -28.062 0.91347 1.45 1.28 80.3 91.0 6.4
366 0.89526 4.6093 -26.628 0.91347 1.45 1.28 80.3 91.0 3.2
367 0.0056801 4.8531 -25.194 0.91347 1.45 1.28 80.3 91.0 23.6
368 0.010134 5.0969 -23.76 0.91347 1.45 1.28 80.3 91.0 -69.9
369 0.057374 5.3408 -22.326 0.91347 1.45 1.28 80.3 91.0 87.1
370 0.22465 5.5846 -20.892 0.91347 1.45 1.28 80.3 91.0 7.3
371 2.9832 3.1306 -35.229 0 1 0.914 80.3 91.0 -30.2
372 3.3915 3.2983 -34.243 0 1 0.914 80.3 91.0 -20.3
373 0.70196 3.4659 -33.257 0 1 0.914 80.3 91.0 -70.1
374 0.35222 3.6335 -32.271 0 1 0.914 80.3 91.0 -82.0
375 0.14868 3.8012 -31.286 0 1 0.914 80.3 91.0 76.9
376 0.068403 3.9688 -30.3 0 1 0.914 80.3 91.0 84.9
377 0.088797 4.1364 -29.314 0 1 0.914 80.3 91.0 89.9
378 0.34015 4.304 -28.328 0 1 0.914 80.3 91.0 55.6
379 0.33888 4.4717 -27.342 0 1 0.914 80.3 91.0 0.1
380 0.49792 4.6393 -26.356 0 1 0.914 80.3 91.0 -85.4
381 0.79699 4.8069 -25.37 0 1 0.914 80.3 91.0 -30.2
382 0.14453 4.9746 -24.385 0 1 0.914 80.3 91.0 -88.3
383 0.083684 5.1422 -23.399 0 1 0.914 80.3 91.0 11.3
384 0.047513 5.3098 -22.413 0 1 0.914 80.3 91.0 88.8
385 0.11486 5.4774 -21.427 0 1 0.914 80.3 91.0 89.1
386 0.13628 5.6451 -20.441 0 1 0.914 80.3 91.0 -78.1
387 2.9754 -3.3932 -27.542 6.4893 3.39 3.51 89.3 91.0 -9.6
388 2.9578 -3.3527 -24.149 6.4893 3.39 3.51 89.3 91.0 -2.1
389 1.658 -3.3123 -20.756 6.4893 3.39 3.51 89.3 91.0 11.5
390 0.77808 -3.2719 -17.362 6.4893 3.39 3.51 89.3 91.0 54.6
391 3.5478 -3.4369 -27.542 3.9827 2.71 2.51 89.3 91.0 -9.0
392 2.3377 -3.4046 -24.827 3.9827 2.71 2.51 89.3 91.0 -4.5
393 0.084965 -3.3722 -22.112 3.9827 2.71 2.51 89.3 91.0 22.8
394 0.23616 -3.3399 -19.398 3.9827 2.71 2.51 89.3 91.0 36.1
395 0.083681 -3.3076 -16.683 3.9827 2.71 2.51 89.3 91.0 37.4
396 2.6388 -3.4682 -27.541 2.1923 1.94 1.79 89.3 91.0 -4.7
397 1.1897 -3.4451 -25.602 2.1923 1.94 1.79 89.3 91.0 1.9
398 0.13436 -3.422 -23.663 2.1923 1.94 1.79 89.3 91.0 60.5
399 0.066143 -3.3989 -21.724 2.1923 1.94 1.79 89.3 91.0 -62.6
400 0.10627 -3.3758 -19.785 2.1923 1.94 1.79 89.3 91.0 -68.6
401 0.059833 -3.3527 -17.846 2.1923 1.94 1.79 89.3 91.0 -70.3
402 0.22767 -3.3296 -15.907 2.1923 1.94 1.79 89.3 91.0 -86.7
403 2.3668 -3.4905 -27.541 0.91347 1.36 1.28 89.3 91.0 -1.4
404 0.16001 -3.4743 -26.184 0.91347 1.36 1.28 89.3 91.0 72.0
405 0.026524 -3.4581 -24.826 0.91347 1.36 1.28 89.3 91.0 21.3
406 0.062217 -3.442 -23.469 0.91347 1.36 1.28 89.3 91.0 -60.2
407 0.084789 -3.4258 -22.112 0.91347 1.36 1.28 89.3 91.0 -78.4
408 0.097283 -3.4096 -20.754 0.91347 1.36 1.28 89.3 91.0 -84.7
409 0.096432 -3.3935 -19.397 0.91347 1.36 1.28 89.3 91.0 -89.6
410 0.029206 -3.3773 -18.04 0.91347 1.36 1.28 89.3 91.0 -2.3
411 0.056008 -3.3611 -16.682 0.91347 1.36 1.28 89.3 91.0 89.5
412 0.13937 -3.345 -15.325 0.91347 1.36 1.28 89.3 91.0 -34.0
413 4.1446 -3.5064 -27.541 0 0.97 0.914 89.3 91.0 -3.8
414 0.32321 -3.4949 -26.571 0 0.97 0.914 89.3 91.0 84.7
415 0.30054 -3.4833 -25.602 0 0.97 0.914 89.3 91.0 77.3
416 0.24911 -3.4718 -24.632 0 0.97 0.914 89.3 91.0 86.6
417 0.11491 -3.4602 -23.663 0 0.97 0.914 89.3 91.0 47.2
418 0.15045 -3.4487 -22.693 0 0.97 0.914 89.3 91.0 57.2
419 0.071049 -3.4371 -21.724 0 0.97 0.914 89.3 91.0 90.0
420 0.28105 -3.4256 -20.754 0 0.97 0.914 89.3 91.0 17.2
421 0.10014 -3.414 -19.785 0 0.97 0.914 89.3 91.0 90.0
422 0.062134 -3.4025 -18.815 0 0.97 0.914 89.3 91.0 90.0
423 0.0022769 -3.3909 -17.846 0 0.97 0.914 89.3 91.0 -90.0
424 0.092329 -3.3794 -16.876 0 0.97 0.914 89.3 91.0 17.2
425 0.16934 -3.3678 -15.907 0 0.97 0.914 89.3 91.0 28.6
426 0.0052909 -3.3563 -14.937 0 0.97 0.914 89.3 91.0 90.0
# number of tensile cracks
0
# number of dilatation sources
0
# number of surface traction
0
EOF

