#!/usr/bin/env ruby

=begin

=module (({CShadow}))

(({CShadow})) is a way of creating objects which transparently mix C data with Ruby data.

The (({CShadow})) module is a mix-in which adds a C struct to objects which derive from the class which includes it. The data members of the C struct, called shadow attributes, may vary in subclasses of this base class. Shadow attrs can be added to any class inheriting from the base class, and they will propagate along the ordinary ruby inheritance hierarchy. (For now, shadow attributes cannot be defined in modules.)

The (({CShadow})) module uses (({CGenerator}))'s structure templates to handle the code generation. (({CGenerator})) is also useful for inline C definitons of methods that operate on shadow attributes.

(({CShadow})) cooperates with the (({CShadow::Attribute})) subclasses defined in ((<attribute.rb>)) to handle the mark and free functions, type checking and conversion, serialization, and so on. New attribute types can be added easily. 

==Usage

  class MyClass
    include CShadow

Include (({CShadow})) in the base class(es) that need to have shadow attributes. The base class is assigned a (({Library})) (see cgen.rb), which can be accessed using the base class's (({shadow_library})) method. Each subclass of the base class will be associated with a struct defined in this library's .h files.

As usual, the (({initialize})) method of the class will be called when the object is created, and the arguments are whatever was passed to (({new})), including the block, if any. You can write your own (({initialize})) method to assign initial values to the shadow attrs. (Typically, shadow attrs are initialized by default to nil or 0. See ((<attribute.rb>)).)

The file name of the library is the same as the name of the class which includes (({CShadow})), by default, and the library directory is generated in the current dir when (({commit})) is called. To change the name, or to use a different library:

    shadow_library <Library, Class, or String>

The argument can be another library (instance of (({CShadow::Library}))), a class which includes (({CShadow})), in which case the library of the class is used, or a string, in which case a new library is created. Using this feature, several base classes (and their descendants) can be kept in the same library. It is not possible to split a tree (a base class which includes (({CShadow})) and its descendants) into more than one library.

Shadow classes that are placed in the same library can still be put in separate source files, using (({shadow_library_file})):

    shadow_library_file <CFile or String>

This setting is inherited (and can be overridden) by subclasses of the current class. If a class calls both (({shadow_library})) and (({shadow_library_file})) then they must be called in that order. Note that anything defined before the (({shadow_library_file})) statement will not be placed in the specifed file.

The include and source file for the current class can be accessed with (({shadow_library_include_file})) and (({shadow_library_source_file})). This is not required for normal use.

Behavior at commit time can be controlled by scheduling (({before_commit})) and (({after_commit})) procs. These procs are called immediately before and after the actual commit, which allows, for example, removing instances that would otherwise cause an exception, or creating the instance of a singleton class. The (({before_commit})) and (({after_commit})) class methods of (({CShadow})) delegate to the shadow_library's methods. See the cgen documentation for details.


===Struct members

All shadow structs have a (({self})) member of type (({VALUE})) which points to the Ruby object.

The subclass struct inherits members of the base class struct.

There are two types of shadow attributes:

(1) C data: (({:d => "double d", :x => "char *foo"}))
(2) Ruby value: (({:str => String, :obj => Object}))

In addition to shadow attributes, you can declare other struct members using

  shadow_struct.declare ...

as in cgen.

===Accessors

---CShadow.shadow_attr [options,] :var => decl, ...

Adds the specified declarations to the shadow struct of the current class without defining any accessors. The data can be accessed only in C code.

---CShadow.shadow_attr_reader :var => decl, ...
---CShadow.shadow_attr_writer :var => decl, ...
---CShadow.shadow_attr_accessor :var => decl, ...

Like (({CShadow#shadow_attr})), but adds a reader and/or writer named (({var})) or (({var=})).

Each (({:var => decl})) pair generates one C struct member ((({:x => "int x,y"})) will generate an exception).

The same symbol :var cannot be used in both a (({shadow_attr_*})) definition and an ordinary (({attr_*})) definition. (You can always get around this by manually defining accessor methods.)

The available options are:

: (({:persistent})), (({:nonpersistent}))
  Serialize or do not serialize this attribute when dumping with Marshal, YAML,
  etc. When loading, it is initialized using the attribute's (({init})) code
  (which, typically, sets it to zero or (({nil}))). Default is
  (({:persistent})).

: (({:reader})), (({:writer}))
  Creates a reader or writer method, just like (({shadow_attr_reader})) and
  (({shadow_attr_writer})). (This is mostly for internal use.)

===Type checking and conversion

In case (1) (C data attribute), assignments to int and double struct members are done with (({NUM2INT})) and (({NUM2DBL})), which convert between numeric types, but raise exceptions for unconvertible types. The (({char *})) case uses StringValue (formerly (({rb_str2cstr}))), which behaves analogously.

In case (2) (Ruby value attribute) the assigned value must always be a descendant of the specified type, except that (({nil})) is always accepted. The purpose of specifying a class (should also allow type predicate!) is to allow C code to assume that certain struct members are present in the shadow struct, and so it can be safely cast to the "ancestor" struct.

===Adding methods

(({CGenerator})) provides a general interface to the Ruby-C api. However, for simplicity, (({CShadow})) defines two methods in the client class for defining methods and class methods

---CShadow.define_c_method name, subclass = Method, &block
---CShadow.define_c_class_method name, subclass = SingletonMethod, &block
---CShadow.define_c_function name, subclass = Function, &block

The block is evaluated in a context that allows commands for listing arguments, declarations, C body code, etc. See ((<cgenerator.rb>)) for details. See examples in ((<examples/matrix.rb>)) and ((<examples/complex.rb>)). The (({subclass})) argument is optional and allows the template to belong to a subclass of the function template it would normally belong to.

In the case of (({define_c_method})), a pointer to the object's shadow struct is available in the C variable (({shadow})).

===Introspection

Each class which includes the (({CShadow})) module has the following methods to iterate over its shadow attributes.

---CShadow.each_shadow_attr(&bl)

Note that the shadow attributes dynamically include inherited ones. (Dynamically in the sense that subsequent changes to superclasses are automatically reflected.) The order is from root to leaf of the inheritance chain, and within each class in order of definition. (TEST THIS)

---CShadow.shadow_attrs

Returns a proxy (({Enumerable})) object referring to the same attributes as (({#each_shadow_attr})). For example:

  sub_class.shadow_attrs.collect { |attr| attr.var }

returns an array of variable names for all attributes of the class.

---CShadow#each_attr_value(&bl)
---CShadow#each_persistent_attr_value(&bl)

Iterate over each (persistent) shadow attr and instance var of (({self})), yielding the attr name and value in this instance to the block. Differs in three ways from the above introspection methods: it is an instance method of shadow objects, it iterates over both shadow attrs and instance vars, and it yields both the name and the value. (In the case of instance vars, the name does _not_ include the "@".)

===Memory management

Each type of attribute is responsible for managing any required marking of referenced Ruby objects or freeing of allocated bocks of memory. See ((<attribute.rb>)) for details.

===C attribute plug-ins

CShadow's (({shadow_attr})) methods check for one (no more, no less) matching attribute class. The details of the matching depend on the attribute class. See ((<attribute.rb>)) for details. Additional attribute classes can easily be added simply by subclassing (({CShadow::Attribute})).

===Namespace usage

prefix every attribute, method, constant name with "cshadow_"?

===Using CShadow classes with YAML

As of v0.11, CShadow classes can serialize via YAML. Both shadow attributes and ordinary ruby instance variables are serialized. No additional attribute code is needed, because of the generic load/dump mechanism used in cshadow (attributes are pushed into or shifted out of an array, which is passed to or from the serialization protocol, either Marshal or YAML). The ordering of shadow attributes is preserved when dumping to a YAML string.

The only user requirement is that, before attempting to load shadow class instances from YAML string, the shadow class types must be registered with YAML. This is simple:

  CShadow.allow_yaml

See examples/yaml.rb

===Common problems and their solutions

Do you get a NameError because accessor methods are not defined? Make sure you commit your class.

You assign to an attribute but it doesn't change? Ruby assumes that, in an assignment like "x=1", the left hand side is a local variable. For all writer methods, whether Ruby or C attributes, you need to do "self.x=1".

==Notes

* As with most modules, including CShadow more than once has no effect. However, CShadow cannot currently be included in another module.

==Limitations:

* Hash args are ordered unpredictably, so if struct member order is significant (for example, because you want to pass the struct to C code that expects it that way), use a separate declare statement for each member. Also, take note of the self pointer at the beginning of the struct.

* Creating a ruby+shadow object has a bit more time/space overhead than just a C object, so CShadow may not be the best mechansism for managing heap allocated data (for example, if you want lots of 2-element arrays). Also, CShadow objects are fixed in size (though their members can point to other structures).

* CShadow attributes are, of course, not dynamic. They are fixed at the time of (({commit})). Otherwise, they behave essentially like Ruby attributes, except that they can be accessed only with methods or from C code; they cannot be accessed with the @ notation. Of course, the reader and writer for a shadow attribute can be flagged as protected or private. However, a private writer cannot be used, since by definition private methods can only be called in the receiverless form.

==To do:

* Optimization: if class A<B, and their free func have the same content, use B's function in A, and don't generate a new function for A. Similarly for all the other kinds of functions.

* Allow

  shadow_attr "int x", "double y"

or even

  attr_accessor :a, :b, "int x", "double y"

and (in cgen)

  declare "int x", "double y"

The ruby name will be extracted from the string using the matching pattern.

* Generate documentation for the shadow class.

* Change name to CStruct? CStructure?

* Find a way to propagate append_features so that CShadow can be included in modules and modules can contribute shadow_attrs.

* option to omit the "self" pointer, or put it at the end of the struct
  automatically omit it in a class if no ShadowObjectAttributes point to it?

* shadow_struct_constructor class method to use DATA_WRAP_STRUCT

* Use CNativeAttribute as a default attribute? Or use the attr class hierarchy to supply defaults?

=end

require "cgen/cgen"
require "cgen/attribute"

module CShadow
  SHADOW_SUFFIX = "_Shadow"
  
  if RUBY_VERSION.to_f >= 1.8
    
    ALLOC_NAME = nil
    GC_CAST = nil
    
  elsif RUBY_VERSION.to_f >= 1.7
    
    require 'date'
    ruby_release_date =
      Date.new(*RUBY_RELEASE_DATE.scan(/(\d+)-(\d+)-(\d+)/)[0].map {|s|s.to_i})
      
    if ruby_release_date >= Date.new(2002, 12, 20)
      ALLOC_NAME = nil
    else
      ALLOC_NAME = :allocate
    end
    GC_CAST = nil
    
  else
    
    ALLOC_NAME = :_alloc # this requires a patch to marshal.c for <1.6.7
    GC_CAST = "(void *)"
    
  end
  
  class Library < CGenerator::Library
    def initialize(*args)
      super
      
      before_commit do
        @classes_to_commit = []
        ObjectSpace.each_object(CShadowClassMethods) do |cl|
          if cl.shadow_library == self
            @classes_to_commit << cl
          end
        end

        # This is done here, rather than in #inherited, to get around
        # the Ruby bug with names of nested classes. It's ugly...
        ## this may be fixed in 1.7
        # Better: register classes with libraries...
        
        classes = Library.sort_class_tree(@classes_to_commit)
        
        classes.each do |cl|
          # make sure the following have been defined by now
          cl.shadow_struct
          cl.new_method; cl.mark_function; cl.free_function
          cl._dump_data_method; cl._load_data_method; cl._alloc_method
        end
      end

      after_commit do
        for cl in @classes_to_commit
          cl.protect_shadow_attrs
        end
      end
    end
    
    # Sort a list of classes. Sorting has the following properties:
    #
    # Deterministic -- you get the same output no matter what order the input
    # is in, because we're using #sort_by.
    #
    # Compatible with original order on classes: superclass comes before
    # subclass.
    #
    # Unrelated classes are ordered alphabetically by name
    #
    # Note that
    #
    #   classes.sort {|c,d| (d <=> c) || (c.name <=> d.name)}
    #
    # is wrong.
    #
    def self.sort_class_tree(classes)
      classes.sort_by {|c| c.ancestors.reverse!.map!{|d|d.to_s}}
    end
  end
  
  module CShadowClassMethods
    def new
      raise Library::CommitError,
        "\nCannot create shadow objects before committing library"
    end

    # Primarily for loading yaml data. The hash is of the form
    #
    #   { 'attr' => value, ... }
    #
    # where <tt>attr</tt> is either the name of a shadow attr, or
    # the name (without @) of an attribute.
    #
    # Warning: The hash +h+ is modified.
    def new_from_hash(h)
      obj = allocate

      psa = shadow_attrs.select {|attr| attr.persists}
      shadow_vars = psa.map{|attr|attr.var.to_s}
      from_array = h.values_at(*shadow_vars)
      obj._load_data(from_array)
      shadow_vars.each {|v| h.delete(v) }

      h.each do |ivar, value|
        obj.instance_variable_set("@#{ivar}", h[ivar])
      end

      obj
    end
  
    def base_class
      @base_class ||= superclass.base_class
    end

    def persistent flag = true
      unless self == base_class
        raise "Persistence must be selected for the base class, " +
              "#{base_class}, and it applies to the tree of subclasses."
      end
      @persistent = flag
    end
    private :persistent

    def persistent?
      bc = @base_class
      if self == bc
        @persistent
      else
        bc.persistent?
      end
    end

    def commit
      shadow_library.commit
    end

    def committed?
      shadow_library.committed?
    end

    def before_commit(&block)
      shadow_library.before_commit(&block)
    end

    def after_commit(&block)
      shadow_library.after_commit(&block)
    end

    def each_shadow_attr(&bl)
      if superclass.respond_to? :each_shadow_attr
        superclass.each_shadow_attr(&bl)
      end
      @shadow_attrs.each(&bl) if @shadow_attrs
    end

    def shadow_attrs
      proxy = Object.new.extend Enumerable
      shadow_class = self
      proxy.instance_eval {@target = shadow_class}
      def proxy.each(&bl)
        @target.each_shadow_attr(&bl)
      end
      proxy
    end

    def shadow_library lib = nil
      bc = base_class
      if self == bc
        if @shadow_library
          if lib
            raise RuntimeError,
                  "\nClass #{name} is already associated" +
                  " with library #{@shadow_library.name}."
          end
        else
          case lib
          when Library
            @shadow_library = lib
          when Class
            begin
              @shadow_library = lib.shadow_library
            rescue NameError
              raise ScriptError, "\n#{lib} does not include CShadow."
            end
          when String
            @shadow_library = Library.new(lib)
          when nil
            n = name.dup
            n.gsub! /_/, '__'
            n.gsub! /::/, '_' # almost reversible
            @shadow_library = Library.new(n)
          else
            raise ArgumentError,
                  "\n#{lib} is not a CShadow::Library, String, or Class" +
                  "\nIts class is #{lib.class}"
          end
        end
        @shadow_library
      else
        bc.shadow_library lib
      end
    end

    def shadow_library_file file = nil
      if defined? @shadow_library_file
        if file
          raise RuntimeError,
                "\nClass #{self} is already associated" +
                " with file #{@shadow_library_file}."
        end
        @shadow_library_file
      elsif file
        case file
        when CGenerator::CFile
          file_name = file.name
        when String
          file_name = file
        else
          raise ArgumentError, "\n#{file} is not a String or CFile."
        end
        file_name = file_name.sub(/\.[ch]$/, "")
        @shadow_library_file = shadow_library.add_file file_name
      else
        if superclass.respond_to? :shadow_library_file
          superclass.shadow_library_file
        else
          [shadow_library.include_file, shadow_library.source_file]
        end
      end
    end

    def shadow_library_include_file
      shadow_library_file[0]
    end

    def shadow_library_source_file
      shadow_library_file[1]
    end

    #== Convenience methods ==#

    def define_c_method name, subclass = CGenerator::Method, &block
      sf = shadow_library_source_file
      m = sf.define_c_method self, name, subclass
      m.scope :extern
      m.declare :shadow => "#{shadow_struct.name} *shadow"
      m.setup :shadow =>
        "Data_Get_Struct(self, #{shadow_struct.name}, shadow)"
      m.instance_eval &block if block
      m
    end

    def define_c_class_method name,
          subclass = CGenerator::SingletonMethod, &block
      sf = shadow_library_source_file
      m = sf.define_c_singleton_method self, name, subclass
      m.scope :extern
      m.instance_eval &block if block
      m
    end
    
    def define_c_function name, subclass = CGenerator::Function, &block
      sf = shadow_library_source_file
      m = sf.define_c_function name, subclass
      m.scope :extern
      m.instance_eval &block if block
      m
    end

    #== Internal methods ==#

    def shadow_struct_name
      name.gsub(/_/, '__').gsub(/::/, '_o_') + CShadow::SHADOW_SUFFIX
      # To preserve differences.
    end
    private :shadow_struct_name

    def shadow_struct
      unless @shadow_struct
        sf = shadow_library_source_file
        ssn = shadow_struct_name
        @shadow_struct = sf.declare_extern_struct(ssn)
        if self == base_class
          @shadow_struct.declare :self => "VALUE self"
        else
          sss = superclass.shadow_struct
          shadow_struct.inherit\
            sss.inherit!,
            "/* #{superclass.shadow_struct.name} members */",
            sss.declare!, " "

          unless superclass.shadow_library_source_file ==
                 shadow_library_source_file
            shadow_library_include_file.include(
              superclass.shadow_library_include_file)
          end
        end
      end
      @shadow_struct
    end

    def new_method
      unless @new_method
        sf = shadow_library_source_file
        ssn = shadow_struct.name
        @new_method = sf.define_c_singleton_method self,
          :new, AttrClassMethod
        @new_method.instance_eval {
          scope :extern
          c_array_args
          declare :object => "VALUE object"
          declare :shadow => "#{ssn} *shadow"
          setup :shadow_struct => %{
            object = Data_Make_Struct(self,
                       #{ssn},
                       mark_#{ssn},
                       free_#{ssn},
                       shadow);
            shadow->self = object;
          }.tabto(0)
          body attr_code!
          body %{
            rb_obj_call_init(object, argc, argv);
          }
          returns "object"
        }
        if superclass.respond_to? :shadow_struct
          @new_method.attr_code superclass.new_method.attr_code!
        end
      end
      @new_method
    end

    def mark_function
      unless @mark_function
        sf = shadow_library_source_file
        ssn = shadow_struct.name
        @mark_function = sf.define_c_function("mark_#{ssn}", MarkFunction)
        @mark_function.instance_eval {
          scope :static
          arguments "#{ssn} *shadow"
          return_type "void"
        }
        if superclass.respond_to? :shadow_struct
          @mark_function.mark superclass.mark_function.mark!
        end
      end
      @mark_function
    end

    def free_function
      unless @free_function
        sf = shadow_library_source_file
        ssn = shadow_struct.name
        @free_function = sf.define_c_function("free_#{ssn}", FreeFunction)
        @free_function.instance_eval {
          scope :static
          arguments "#{ssn} *shadow"
          return_type "void"
        }
        if superclass.respond_to? :shadow_struct
          @free_function.free superclass.free_function.free!
        end
      end
      @free_function
    end

    def _dump_data_method
      return nil unless persistent?
      unless @_dump_data_method
        @_dump_data_method = define_c_method(:_dump_data, AttrMethod) {
          declare :result   => "VALUE   result"
          setup   :result   => "result  = rb_ary_new()"
          body pre_code!, attr_code!, post_code!
          returns "result"
        }
        if superclass.respond_to? :shadow_struct
          @_dump_data_method.attr_code superclass._dump_data_method.body!
        end
      end
      @_dump_data_method
    end

    def _load_data_method
      return nil unless persistent?
      unless @_load_data_method
        @_load_data_method = define_c_method(:_load_data, AttrMethod) {
          arguments :from_array
          declare :tmp  => "VALUE tmp"
          body pre_code!, attr_code!, post_code!
          returns "from_array"  ## needed?
        }
        if superclass.respond_to? :shadow_struct
          @_load_data_method.attr_code superclass._load_data_method.body!
        end
      end
      @_load_data_method
    end

    # same as new_method, but no initialize -- factor this
    # can we use define_c_class_method?
    def _alloc_method
      return nil unless persistent?
      unless @_alloc_method
        sf = shadow_library_source_file
        ssn = shadow_struct.name
        @_alloc_method =
          if ALLOC_NAME
            sf.define_c_singleton_method(self, ALLOC_NAME)
          else
            sf.define_alloc_func(self)
          end
        @_alloc_method.instance_eval {
          scope :extern
          if ALLOC_NAME
            c_array_args
            klass_c_name = "self"
          else
            arguments 'VALUE klass'
            return_type 'VALUE'
            klass_c_name = "klass"
          end
          declare :object => "VALUE object"
          declare :shadow => "#{ssn} *shadow"
          body %{
            object = Data_Make_Struct(#{klass_c_name},
                       #{ssn},
                       mark_#{ssn},
                       free_#{ssn},
                       shadow);
            shadow->self = object;
          }
          returns "object"
        }
      end
      @_alloc_method
    end

  private

    def check_overwrite_shadow_attrs(*symbols)
      for attr in shadow_attrs
        for sym in symbols
          if sym == attr.var
            raise NameError, "\n#{sym} is a shadow attr."
          end
        end
      end
    end

    def attr_accessor(*args)
      check_overwrite_shadow_attrs(*args)
      super
    end

    def attr_reader(*args)
      check_overwrite_shadow_attrs(*args)
      super
    end

    def attr_writer(*args)
      check_overwrite_shadow_attrs(*args)
      super
    end

    def shadow_attr_accessor(*args)
      shadow_attr :reader, :writer, *args
    end

    def shadow_attr_reader(*args)
      shadow_attr :reader, *args
    end

    def shadow_attr_writer(*args)
      shadow_attr :writer, *args
    end

    def shadow_attr(*args)
      attr_persists = true
      for arg in args
        case arg
        when Hash;            var_map = arg.sort_by {|var, decl| var.to_s}
        when :reader;         reader = true
        when :writer;         writer = true
        when :nonpersistent;  attr_persists = false
        when :persistent;     attr_persists = true
        else
          raise SyntaxError,
            "\nUnrecognized shadow_attr argument: #{arg.inspect}"
        end
      end

      source_file = shadow_library_source_file
      ssn = shadow_struct.name
      @shadow_attrs ||= []

      for var, decl in var_map
        var = var.intern if var.is_a? String

        if instance_methods(false).include?(var.to_s) or
           instance_methods(false).include?(var.to_s + '=')
          raise NameError, "\n#{var} already has a Ruby attribute."
        end

        matches = AttributeTypes.collect { |t|
           [t, t.match(decl)]
        }.select {|t, match| match}

        if matches.size > 1
          raise StandardError, %{
            No unique AttributeType for '#{decl}':
              each of #{matches.map{|t|t[0]}.join ", "} match.
          }.tabto(0)
        end

        if matches.size < 1
          raise StandardError, "\nNo Attribute type matches '#{decl}'."
        end

        attr_type, match = matches[0]
        attr = attr_type.new var, match, attr_persists
        each_shadow_attr { |a|
          if a.var == attr.var or a.cvar == attr.cvar
            raise NameError, "\nAttribute #{a.inspect} already exists."
          end
        }
        @shadow_attrs << attr

        shadow_struct.declare attr.cvar => attr.cdecl

        new_method.attr_code attr.init if attr.init

        mark_function.mark attr.mark
        free_function.free attr.free

        if persistent?
          if attr_persists
            _dump_data_method.attr_code attr.dump
            _load_data_method.attr_code attr.load
          else
            _load_data_method.attr_code attr.init if attr.init
          end
        end

        if reader
          unless attr.reader
            raise ScriptError, "\nCan't have a reader method for #{attr}."
          end
          source_file.define_c_method(self, var).instance_eval {
            scope :extern
            declare :shadow => "#{ssn} *shadow"
            declare :result => "VALUE result"
            body "Data_Get_Struct(self, #{ssn}, shadow)", attr.reader
            returns "result"
          }
        end

        if writer
          unless attr.writer
            raise ScriptError, "\nCan't have a writer method for #{attr}."
          end
          source_file.define_c_method(self, "#{var}=").instance_eval {
            scope :extern
            c_array_args {
              required  :arg
              typecheck :arg => attr.check
            }
            declare :shadow => "#{ssn} *shadow"
            body "Data_Get_Struct(self, #{ssn}, shadow)", attr.writer
            returns "arg"
          }
        end

      end

    end

  end

  def self.append_features base_class
    unless base_class.is_a? Class
      raise TypeError, "\nCShadow can be included only in a Class"
    end

    unless base_class.ancestors.include? self

      base_class.class_eval {@base_class = self; @persistent = true}
      base_class.extend CShadowClassMethods

      class << base_class
        ## why can't these be in CShadowClassMethods?

        alias really_protected protected
        def protected(*args)
          (@to_be_protected ||= []).concat args
        end

        alias really_private private
        def private(*args)
          (@to_be_private ||= []).concat args
        end

        def protect_shadow_attrs
          really_protected *@to_be_protected if @to_be_protected
          really_private *@to_be_private if @to_be_private
          ## should undo the aliasing
        end
        public :protect_shadow_attrs

      end

    end

    super
  end

  class AttrCodeAccumulator < CGenerator::CFragment::StatementAccumulator
    include CGenerator::SetAccumulator
  end
  
  class AttrMethod < CGenerator::Method
    accumulator(:attr_code) {AttrCodeAccumulator}
    accumulator(:pre_code)  {CGenerator::CFragment::StatementAccumulator}
    accumulator(:post_code) {CGenerator::CFragment::StatementAccumulator}
  end
  
  class AttrClassMethod < CGenerator::SingletonMethod
    accumulator(:attr_code) {AttrCodeAccumulator}
  end

  class MarkFunction < CGenerator::Function
    accumulator(:mark) {AttrCodeAccumulator}
    
    def initialize(*args)
      super
      body "rb_gc_mark(#{GC_CAST}shadow->self)", mark!
    end
  end
  
  class FreeFunction < CGenerator::Function
    accumulator(:free) {AttrCodeAccumulator}
    
    def initialize(*args)
      super
      body free!, "free(shadow)"  # free the struct last!
    end
  end
  
  def inspect
    attrs = []
    seen = {self => true}
    each_attr_value do |attr, value|
      if seen[value]
        attrs << "#{attr}=#{value}"
      else
        attrs << "#{attr}=#{value.inspect}"
      end
      seen[value] = true
    end
    super.sub(/(?=>\z)/, " " + attrs.join(", "))
  end
  
  # Iterates over attr_name, attr_value pairs, for both shadow attrs
  # and regular instance variables.
  def each_attr_value
    values = _dump_data
    self.class.shadow_attrs.each_with_index do |attr, i|
      yield attr.var.to_s, values[i]
    end
    instance_variables.each do |ivar|
      yield ivar[1..-1], instance_variable_get(ivar)
    end
  end

  # Iterates over attr_name, attr_value pairs, for both shadow attrs
  # and regular instance variables.
  def each_persistent_attr_value
    values = _dump_data
    psa = self.class.shadow_attrs.select {|attr| attr.persists}
    psa.each_with_index do |attr, i|
      yield attr.var.to_s, values[i]
    end
    instance_variables.each do |ivar|
      yield ivar[1..-1], instance_variable_get(ivar)
    end
  end

  # YAML methods for CShadow classes.
  def self.allow_yaml
    require 'yaml'
    
    if defined?(YAML.type_tag) # old version of YAML
      YAML.add_ruby_type(/^cshadow/) do |type, val|
        subtype, subclass = YAML.read_type_class(type, Object)
        subclass.new_from_hash(val)
      end
      include CShadow::YAML_Old ## yuck!
    
    else
      yaml_as "tag:path.berkeley.edu,2006:cshadow"
      
      def self.yaml_new( klass, tag, val )
        subtype, subclass = YAML.read_type_class(tag, Object)
        subclass.new_from_hash(val)
      end

      module_eval do
        def add_yaml_map_contents(map)
          each_persistent_attr_value do |attr, value|
            map.add(attr, value)
          end
        end

        def to_yaml( opts = {} )
          YAML.quick_emit(object_id, opts) do |out|
            out.map( taguri, to_yaml_style ) do |map|
              add_yaml_map_contents(map)
            end
          end
        end
      end
    end
  end

  module CShadow::YAML_Old
    def add_yaml_map_contents(map)
      each_persistent_attr_value do |attr, value|
        map.add(attr, value)
      end
    end

    def is_complex_yaml? 
      true
    end

    def to_yaml_type
      "!ruby/cshadow:#{self.class}"
    end

    def to_yaml( opts = {} )
      opts[:DocType] = self.class if Hash === opts
      YAML.quick_emit(self.object_id, opts) do |out|
        out.map(to_yaml_type) do |map|
          add_yaml_map_contents(map)
        end
      end
    end
  end

end # module CShadow
