require 'test/unit'
require 'divcal'

module DivCal

  class TestLogin < Test::Unit::TestCase
    def setup
      @login = Login.new
    end

    def test_select
      assert_equal('foo', @login.get_user('foo', 'bar'))
      assert_equal(nil, @login.get_user(nil, 'bar'))
      assert_equal(nil, @login.get_user('bar', nil))
    end
  end

  class TestEvent < Test::Unit::TestCase
    def test_new
      today = Date.today
      e = Event.new('message', 5.0)
      assert_equal('message', e.text)
      assert_equal(5.0, e.estimate)
      assert_equal(nil, e.actual)
    end

    def test_update
      e = Event.new('message', 5.0)
      assert_equal('message', e.text)
      e.update({:text => 'text'})
      assert_equal('text', e.text)
      assert_equal(5.0, e.estimate)
      e2 = e.dup
      assert_equal(e2, e)
      e.update({:date => Date.new(2002, 1, 1)})
      assert_equal(e2, e)
    end
  end

  class TestEventTable < Test::Unit::TestCase
    def setup
      @table = EventTable.new
    end

    def test_00
      today = Date.today
      assert_equal(nil, @table.fetch(today))
      event = store_dummy_event(today)
      assert_equal(event, @table.fetch(today))
    end

    def store_dummy_event(date)
      event = Event.new(date.to_s, 1)
      @table.store(date, event)
      event
    end

    def test_query
      date = Date.new(2003, 11)
      date.step(Date.new(2004, 1, 31), 1) do |x|
        store_dummy_event(x)
      end
      assert_equal(92, @table.event.size)
      ary = @table.query_month(2003, 12)
      assert_equal(31, ary.size)
      assert_equal(Date.new(2003, 12, 1).to_s, ary[0][1].text)
      assert_equal(Date.new(2003, 12, 31).to_s, ary[30][1].text)
    end
  end

  class TestEventDB < Test::Unit::TestCase
    def setup
      @db = EventDB.instance
    end

    def test_00
      foo = @db['foo']
      assert(! foo.nil?)
      assert_equal(foo, @db['foo'])
    end
  end

  class TestFront < Test::Unit::TestCase
    def setup
      @front = Front.new
      @user = 'foo'
    end

    def test_00
      @front.delete(@user, Date.today)
      assert_equal(nil, @front.fetch(@user, Date.today))

      date = Date.new(2003, 11)
      date.step(Date.new(2004, 1, 31), 1) do |x|
        @front.update(@user, x, {:text => x.to_s, :estimate => 3})
      end

      ary = @front.query_month(@user, 2003, 12)
      assert_equal(31, ary.size)
      assert_equal(Date.new(2003, 12, 1).to_s, ary[0][1].text)
      
      @front.update(@user, Date.new(2003, 12, 1), {:actual => 2})

      ary = @front.query_month(@user, 2003, 12)
      assert_equal(31, ary.size)
      assert_equal(3, ary[0][1].estimate)
      assert_equal(2, ary[0][1].actual)
    end
  end
end
