require 'fileutils'
require 'yaml'

# The Preferences class is an easy way to make variables in an application
# persist in a file. See intro.txt[link:files/intro_txt.html] for a general
# introduction.
class Preferences
  class Error < StandardError; end
  class LoadError < Error; end
  class SaveError < Error; end
  class ConfigError < Error; end
  class EnvError < Error; end
  
  # Client code should raise this in an accessor method when the current
  # value of a variable is not avaiable or for some other reason should
  # not be saved to the file.
  class DataUnavailable < StandardError; end
  
  # Name of file in which preferences persist. Can be assigned a new value
  # while application is open, in order to define a new location at which
  # preferences are stored when #save is called.
  attr_accessor :filename
  
  # Create a Preferences instance which persists at +filename+.
  # All directories containing +filename+ will be created if they do not
  # already exist.
  def initialize filename
    @filename = filename
  end

  # Utility method for guessing a suitable directory to store preferences.
  # On win32, tries +APPDATA+, +USERPROFILE+, and +HOME+. On other platforms,
  # tries +HOME+ and ~. Raises EnvError if preferences dir cannot be chosen.
  # Not called by any of the Preferences library code, but available to the
  # client code for constructing an argument to Preferences.new.
  def Preferences.dir
    unless @dir
      case RUBY_PLATFORM
      when /win32/
        @dir = 
          ENV['APPDATA'] ||  # C:\Documents and Settings\name\Application Data
          ENV['USERPROFILE'] || # C:\Documents and Settings\name
          ENV['HOME']

      else
        @dir =
          ENV['HOME'] ||
          File.expand_path('~')
      end

      unless @dir
        raise EnvError, "Can't determine a preferences directory."
      end
    end
    @dir
  end

  # Encapsulates the metadata that describes what preferences apply to
  # a particular object.
  class Metadata  # :nodoc:
    # The key under which all of the vars of the object are stored.
    # The key may be hierarchical: "foo/bar/baz"--in which case a var
    # is stored at <tt>['foo']['bar']['baz'][var]</tt> in the preferences.
    # Two objects may share a key (in particular, the empty key).
    attr_reader :key

    # The list of vars of the object that are to persist. Each var is a
    # a string or symbol naming a reader method. A corresponding writer
    # method must also exist.
    attr_reader :vars
    
    # Determine (if true) that the object will be used by #save as a source
    # of data, or (if false) that the previous data will be used instead.
    attr_accessor :connected

    # The +hierarchy+ is the complex structure (nested hashes) that is written
    # to the prefs file.
    def initialize(hierarchy, key = nil)
      @key = Metadata.normalize(key)
      @vars = []
      @hierarchy = hierarchy
      @connected = true
    end
    
    # Convert a key to an array of path components.
    def self.normalize(key)
      key.respond_to?(:split) ? key.split("/") : key
    end

    # Traverse the hierarchy to get the ground-level hash used for this
    # object's vars, filling in the hierarchy as needed with empty hashes.
    def prefs
      unless @prefs
        h = @hierarchy
        if key
          key.each do |k|
            h[k] ||= {}
            h = h[k]
          end
        end
        @prefs = h
      end
      @prefs
    
    rescue StandardError => e
      raise LoadError, "Bad pref hierarchy at key #{key.inspect}", e.backtrace
    end
  end

  # Reads preferences file from disk, in YAML format.
  # Can be overridden to keep preferences somewhere else (e.g. Windows
  # registry).
  def read_pref_file # :nodoc:
    FileUtils.mkdir_p(File.dirname(filename))
    File.open(filename) do |f|
      YAML.load(f)
    end
  end

  # Writes preferences file to disk, in YAML format.
  # Can be overridden to keep preferences somewhere else (e.g. Windows
  # registry). Could also do a lock-load-merge-unlock to reduce problems using
  # same pref file for multiple apps or instances of an app.
  def write_pref_file(pref) # :nodoc:
    FileUtils.mkdir_p(File.dirname(filename))
    File.open(filename, "w") do |f|
      YAML.dump(pref, f)
    end
  end

  # Returns the root of the preferences hierarchy, reading from disk or
  # creating if necessary.
  def preferences # :nodoc:
    unless @preferences
      begin
        @preferences = read_pref_file || {}
      rescue Errno::ENOENT
        @preferences = {}
      rescue StandardError => e
        raise LoadError, e.message, e.backtrace
      end
    end
    @preferences
  end

  def metadata # :nodoc:
    @metadata ||= {}
  end
  
  # Has _obj_ been registered?
  def registered?(obj)
    metadata[obj.object_id] != nil
  end
  
  class Registrar # :nodoc:
    def initialize prefs, obj
      @prefs = prefs
      @obj = obj
    end
    
    def var(*vs)
      @prefs.register_pref_var(@obj, *vs)
    end
  end
  
  # A simplified interface that combines #register_pref_key and
  # #register_pref_var in to one method call with a nice block.
  #
  # Simply call the #var method inside the block instead of calling
  # #register_pref_key.
  #
  # For example:
  #
  #   PREFS.register "my app/window" do |entry|
  #     entry.var "x", "y" => "default_y"
  #     entry.var "z"
  #   end
  #
  # This is recommended over using #register_pref_key and
  # #register_pref_var directly.
  #
  # The object whose preferences are regsitered is the self object in the
  # current context. If you need to register prefs for an object that is not
  # currently self, pass the object as a second argument to #register:
  #
  #   subwindow = SubWindow.new(self)
  #   PREFS.register "my app/window/subwindow", subwindow do |entry|
  #     entry.var :x, :y
  #   end
  #
  # It perfectly safe to call #register more than once with the same key.
  #
  def register key, obj = nil, &block # :yields: entry
    unless block
      raise ArgumentError, "Must supply a block with pref vars"
    end
    obj ||= eval("self", block)
    register_pref_key(obj, key)
    registrar = Registrar.new(self, obj)
    yield registrar
  end

  # Register use of +key+ for +obj+. The prefs of +obj+ will be stored under
  # this key. An object can have only one key, but it may be complex, like
  # <tt>"foo/bar"</tt> or, equivalently, <tt>["foo", "bar"]</tt>. Two objects
  # can share a key. Any object, even a class or a module, can be registered.
  # A suggested convention is to use strings for the key, so that there is no
  # conflict with the symbols used for var names. (This is optional.)
  def register_pref_key(obj, key)
    obj_id = obj.object_id
    if metadata[obj_id]
      unless Metadata.normalize(key) == metadata[obj_id].key
        raise ConfigError,
          "Cannot register #{key}. Already registered a preferences key " +
          "for this object, #{metadata[obj_id].key}."
      end
    else
      metadata[obj_id] = Metadata.new(preferences, key)
    end
    key
  end

  # Register the +vars+, which by convention are represented as symbols,
  # as part of the persistent preferences for +obj+. Default values may be
  # provided using hash syntax:
  #
  #   register_pref_var obj :v1, :v2, :v3 => default_for_v3
  #
  # This method is typically called in an +initialize+ method, after
  # register_pref_key. At the time of registration (that is, when this method is
  # called), the variable is updated from the persistent preferences, if the
  # preferences specify a value for it. Otherwise, the default, if any, is
  # applied. If the default is a Proc or Method, it is called to produce
  # the default value. (This is useful if getting the default is an expensive
  # operation.)
  def register_pref_var(obj, *vars)
    obj_id = obj.object_id
    md = metadata[obj_id]
    unless md
      raise ConfigError, "This object must register a preferences key."
    end

    obj_prefs = md.prefs
    obj_vars = md.vars

    defaults = vars[-1].is_a?(Hash) && vars.pop
    vars += defaults.keys if defaults

    vars.each do |var|
      obj_vars << var

      if obj_prefs.key?(var)
        obj.send "#{var}=", obj_prefs[var]
      elsif defaults and defaults.key?(var)
        value = defaults[var]
        value = value.call if value.is_a? Proc or value.is_a? Method
        obj.send "#{var}=", value
      end
    end
  end
  
  # If an object with preferences vars ceases to exist before prefs are saved
  # with #save, the app should dump the object's pref vars using this method.
  # Does not save prefs to disk, but only stores them so they can be saved
  # later after the object has been collected. (Note that the preferences
  # system keeps only a weak reference to each registered object, so they may
  # be garbage collected if no other references exist. A good place to call
  # this method is, for example, in the show/hide methods (or the SEL_MAP/
  # SEL_UNMAP handlers for FXWindow objects in Fox. See examples/life-cycle.rb)
  def dump_prefs_for(obj)
    md = metadata[obj.object_id]
    copy_prefs(obj, md)
  end
  
  def copy_prefs(obj, md)
    obj_prefs = md.prefs
    md.vars.each do |var|
      begin
        obj_prefs[var] = obj.send(var)
      rescue DataUnavailable
      end
    end
  end
  private :copy_prefs
  
  # If _state_ is +true+, _obj_ will be used by #save as a source of data.
  # Otherwise, the previous data will be used. Initially, the state is +true+.
  # If a block is given, the state change is in force only during the block.
  # This can be helpful when one knows that an object will lose access to 
  # the pref vars (e.g. a window is unmapped, so x and y become unreliable).
  def set_connect(obj, state)
    md = metadata[obj.object_id]
    if block_given?
      begin
        old_state = md.connected
        md.connected = state
        yield
      ensure
        md.connected = old_state
      end
    else
      return if md == nil and state == true
        # connect called before pref registered, ok if and only if state true
      
      md.connected = state
    end
  end
  
  # Same as calling #set_connect with the _state_ arg equal to +true+.
  def connect(obj, &block);     set_connect(obj, true, &block);   end

  # Same as calling #set_connect with the _state_ arg equal to +false+.
  def disconnect(obj, &block);  set_connect(obj, false, &block);  end

  # Should be called by the app when quitting or otherwise requested by the
  # user to save preferences. If a registered object has been GC-ed, uses
  # the last known values for the object (see #dump_prefs_for).
  def save
    metadata.each do |obj_id, md|
      obj = ObjectSpace._id2ref(obj_id) rescue nil
      copy_prefs(obj, md) if obj and md.connected
    end
    
    write_pref_file(preferences)
  rescue StandardError => e
    raise SaveError, e.message, e.backtrace
  end
  
  # Clear the preferences structure. All keys and values will be removed.
  def clear
    preferences.clear
  end
end
