/******************************************************************************
 * freebsd.c (host.c) - Ruby Extension
 *
 * Author: Daniel Berger
 *****************************************************************************/
#include "ruby.h"
#include "lib/constants.h"
#include <unistd.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/types.h>

#ifdef __cplusplus
extern "C"
{
#endif

VALUE cHostError;

static VALUE host_hostname()
{
   char host_name[MAXHOSTNAMELEN];
   int rv;
   rv = gethostname(host_name,sizeof(host_name));

   /* If you get this exception, you've got big problems */
   if(rv != 0){
      rb_raise(cHostError,"gethostname() call failed");
   }
   return rb_str_new2(host_name);
}

static VALUE host_ip_addr()
{
   char host_name[MAXHOSTNAMELEN];
   char str[INET_ADDRSTRLEN];
   char **pptr;
   struct hostent* hp;
   int rv;
   int err;
   VALUE rbResults = rb_ary_new();
#ifndef HAVE_INET_NTOP
   struct in_addr ipa;
   int n;
#endif

   rv = gethostname(host_name,sizeof(host_name));
   if(rv != 0){
      rb_raise(cHostError,"gethostname() call failed");
   }

#ifdef HAVE_GETIPNODEBYNAME
   hp = getipnodebyname(host_name,AF_INET,AI_DEFAULT,&err);
#else
   hp = gethostbyname(host_name)
#endif

   if(hp == NULL)
   {
      rb_raise(cHostError,"getipnodebyname() call failed");
   }

   pptr = hp->h_addr_list;

#ifdef HAVE_INET_NTOP
   for( ; *pptr != NULL; pptr++)
   {
      rb_ary_push(rbResults,
         rb_str_new2(inet_ntop(hp->h_addrtype,*pptr,str,sizeof(str))));
   }
#else
   for(n = 0; hp->h_addr_list[n] != NULL; n++)
   {
      memcpy(&ipa.s_addr,hp->h_addr_list[n],hp->h_length);
      rb_ary_push(rbResults,rb_str_new2(inet_ntoa(ipa)));
   }
#endif
   return rbResults;
}

void Init_host()
{
   VALUE sys_mSys, cHost;

   /* Modules and Classes */
   sys_mSys = rb_define_module("Sys");
   cHost = rb_define_class_under(sys_mSys, "Host", rb_cObject);

   /* Constants */
   rb_define_const(cHost,"VERSION",rb_str_new2(SYS_HOST_VERSION));

   /* Class Methods */
   rb_define_singleton_method(cHost, "hostname", host_hostname, 0);
   rb_define_singleton_method(cHost, "ip_addr", host_ip_addr, 0);

   /* Exceptions */
   cHostError = rb_define_class_under(sys_mSys,"HostError",rb_eStandardError);
}

#ifdef __cplusplus
}
#endif
