## Defining Modules to Control Scope and Privacy

In this section, we’ll talk about modules and other parts of the module system,
namely *paths* that allow you to name items; the `use` keyword that brings a
path into scope; and the `pub` keyword to make items public. We’ll also discuss
using the `as` keyword, external packages, and the glob operator. For now,
let’s focus on modules!

*Modules* let us organize code within a crate into groups for readability and
easy reuse. Modules also control the *privacy* of items, which is whether an
item can be used by outside code (*public*) or whether it’s an internal
implementation detail and not available for outside use (*private*).

As an example, let’s write a library crate that provides the functionality of a
restaurant. We’ll define the signatures of functions but leave their bodies
empty to concentrate on the organization of the code rather than actually
implementing a restaurant in code.

In the restaurant industry, parts of a restaurant are referred to as *front of
house* and others as *back of house*. Front of house is where customers are and
includes hosts seating customers, servers taking orders and payment, and
bartenders making drinks. Back of house includes the chefs and cooks in the
kitchen, dishwashers cleaning up, and managers doing administrative work.

To structure our crate in the same way that a real restaurant works, we can
organize the functions into nested modules. Create a new library named
`restaurant` by running `cargo new --lib restaurant`; then put the code in
Listing 7-1 into *src/lib.rs* to define some modules and function signatures.

<span class="filename">Filename: src/lib.rs</span>

```rust
mod front_of_house {
    mod hosting {
        fn add_to_waitlist() {}

        fn seat_at_table() {}
    }

    mod serving {
        fn take_order() {}

        fn serve_order() {}

        fn take_payment() {}
    }
}
```

<span class="caption">Listing 7-1: A `front_of_house` module containing other
modules that then contain functions</span>

We define a module by starting with the `mod` keyword, and then specify the
name of the module (in this case, `front_of_house`) and place curly brackets
around the body of the module. Inside modules, we can have other modules, as in
this case with the modules `hosting` and `serving`. Modules can also hold
definitions for other items, such as structs, enums, constants, traits, or as
in Listing 7-1, functions.

By using modules, we can group related definitions together and name why
they’re related. Programmers using this code would have an easier time finding
the definitions they want to use because they could navigate the code based on
the groups rather than having to read through all the definitions. Programmers
adding new functionality to this code would know where to place the code to
keep the program organized.

Earlier, we mentioned that *src/main.rs* and *src/lib.rs* are called *crate
roots*. The reason for their name is that the contents of either of these two
files form a module named `crate` at the root of the crate’s module structure,
known as the *module tree*.

Listing 7-2 shows the module tree for the structure in Listing 7-1.

```text
crate
 └── front_of_house
     ├── hosting
     │   ├── add_to_waitlist
     │   └── seat_at_table
     └── serving
         ├── take_order
         ├── serve_order
         └── take_payment
```

<span class="caption">Listing 7-2: The module tree for the code in Listing
7-1</span>

This tree shows how some of the modules nest inside one another (such as
`hosting` nests inside `front_of_house`). The tree also shows how some modules
are *siblings* to each other, meaning they’re defined in the same module
(`hosting` and `serving` are defined within `front_of_house`). To continue the
family metaphor, if module A is contained inside module B, we say that module A
is the *child* of module B, and that module B is the *parent* of module A.
Notice that the entire module tree is rooted under the implicit module named
`crate`.

The module tree might remind you of the filesystem’s directory tree on your
computer; this is a very apt comparison! Just like directories in a filesystem,
you use modules to organize your code. And just like files in a directory, we
need a way to find our modules.
