---
title: "Fortran 95 problems"
keywords: spherical harmonics software package, spherical harmonic transform, legendre functions, multitaper spectral analysis, fortran, Python, gravity, magnetic field
sidebar: fortran_sidebar
permalink: fortran-95-problems.html
summary: 
toc: true
folder: fortran
---

## SHTOOLS won't compile on my computer

If the command `make fortran` (or `make fortran-mp`) results in compilation errors:

* verify that you have the latest release of SHTOOLS,
* perform a `make clean` and try again,
* go to the [SHTOOLS issues at GitHub](https://github.com/SHTOOLS/SHTOOLS/issues) to see if this problem has already been documented,
* open an issue on the [GitHub SHTOOLS project page](https://github.com/SHTOOLS/SHTOOLS/issues) and include the **entire** output generated by the Makefile, along with information on the operating system and compilers that you are using.

If you are having problems linking an already compiled SHTOOLS library to your code, see the "linking problems" below. If you are instead receiving compilation errors in *your* code, only contact us after you have thoroughly debugged your program. If you are receiving non fatal warnings, please let us know so that we can clean up the relevant code.

## Can I compile SHTOOLS with Fortran 77?

No, you must use a Fortran 95 compiler. Two free Fortran 95 compilers are [gfortran](https://gcc.gnu.org/) and [g95](https://www.g95.org/). If you will be using the Python code, it will be easiest to use the `gfortran` compiler.

## Linking fails because of "Undefined symbols"

Linking your program to the SHTOOLS and LAPACK libraries might result in an error similar to the following:
```
ld: Undefined symbols:
_dgels
../../lib/libSHTOOLS.a(SHExpandLSQ.o) reference to undefined _dgels
link failed.
```
If the linker is correctly finding the LAPACK and FFTW libraries, the most likely cause of this is that the subroutine names in the SHTOOLS, LAPACK, and FFTW libraries are not exactly the same. If SHTOOLS is able to link to the FFTW library correctly, but not LAPACK, try rebuilding SHTOOLS using
```
make clean
make fortran LAPACK_UNDERSCORE=1
```
This command will compile SHTOOLS source files that append explicitly underscores to the LAPACK subroutine names. If you get similar link errors, but with an added underscore, this probably means that the linker can't find the LAPACK library.

## Linking fails because e_wsfe, z_abs, c_sqrt, s_cmp, etc., are undefined

Linking your program to the SHTOOLS, LAPACK, BLAS, and FFTW libraries might result in link errors similar to:
```
/usr/lib///liblapack.so: undefined reference to `e_wsfe'
/usr/lib///liblapack.so: undefined reference to `z_abs'
/usr/lib///liblapack.so: undefined reference to `c_sqrt'
/usr/lib///liblapack.so: undefined reference to `s_cmp'
/usr/lib///liblapack.so: undefined reference to `z_exp'
/usr/lib///liblapack.so: undefined reference to `c_exp'
/usr/lib///liblapack.so: undefined reference to `do_fio'
/usr/lib///liblapack.so: undefined reference to `z_sqrt'
/usr/lib///liblapack.so: undefined reference to `s_cat'
/usr/lib///liblapack.so: undefined reference to `s_stop'
/usr/lib///liblapack.so: undefined reference to `c_abs'
/usr/lib///liblapack.so: undefined reference to `s_wsfe'
/usr/lib///liblapack.so: undefined reference to `s_copy'
```
This can arise when the offending libraries were built using the g77 compiler. In order to rectify this, it should only be necessary to link to the additional library `libg2c.a` (i.e., g77 to c). Assuming that this library can be found by the linker, just append `-lg2c` to the list of libraries passed to the linker. If the `g2c` library can not be found by the linker, an easy way to find its location is by using either the "locate" or "find" shell commands
```
locate libg2c.a
find /usr -name libg2c.a
```
where the find command searches the directory `/usr`. This pathname can then be added to the linker's search path by using the option `-Ldirname`, by, for example,
```
-L/usr/lib/gcc/i586-mandrake-linux-gnu/3.4.3 -lg2c
```

## Linking fails because of undefined references to SHTOOLS routines

For some compilers, the location of the source file following the compiler name is important. When this is the case, if the source file is not in its correct position, you could receive link errors that resemble the following:
```
gfortran -L../lib -lSHTOOLS TimingAccuracy/TimingAccuracyGLQC.f95 -I../modules/ -L../lib -lfftw3 -lm -m64 -O3 -o TimingAccuracy/TimingAccuracyGLQC
/tmp/cchgdOpg.o: In function `MAIN__':
TimingAccuracyGLQC.f95:(.text+0x5b3): undefined reference to `randomgaussian_'
TimingAccuracyGLQC.f95:(.text+0xb9b): undefined reference to `shglq_'
TimingAccuracyGLQC.f95:(.text+0xd39): undefined reference to `makegridglqc_'
TimingAccuracyGLQC.f95:(.text+0xedd): undefined reference to `shexpandglqc_'
collect2: error: ld returned 1 exit status
```
For this example, successful compilation can be achieved by placing the source file before the library calls:
```
gfortran  TimingAccuracy/TimingAccuracyGLQC.f95 -L../lib -lSHTOOLS -I../modules/ -L../lib -lfftw3 -lm -m64 -O3 -o TimingAccuracy/TimingAccuracyGLQC
```

## The linker can't seem to find either the LAPACK, BLAS, or FFTW libraries

The linker `ld` generally searches for libraries in the directories `/lib`, `/usr/lib` and `/usr/local/lib`. When passing `-lname` to the linker, it will search for the filename `libname.a` in these directories. Additional directories can be added to the search path by passing one or more `-Ldirname` to the linker. For instance, to link to the library `/Users/Me/Lapack/liblapack.a` you would pass the following to the linker
```
-L/Users/Me/Lapack -llapack
```
If you do not know the location of a given library, it can be easily found using either the "locate" or "find" shell commands
```
locate lapack.a
find /usr -name lapack.a
```
where the find command searches the directory `/usr`.

Note that the LAPACK and BLAS library names might be somewhat different than assumed in this documentation. For instance, the LAPACK and BLAS libraries might be invoked by passing `-llapack3`, `-lf77BLAS`, or `-lcblas` to the linker. Note that if the library ATLAS is present, it can be substituted for the BLAS library. If you are certain that the library exists, and that the linker is attempting to link to it, then see [How do I know if I need to append underscores to external function and subroutine names?](#how-do-i-know-if-i-need-to-append-underscores-to-external-function-and-subroutine-names)

## How do I know if I need to append underscores to external function and subroutine names?

When making libraries such as SHTOOLS, LAPACK and FFTW, some compilers by default add trailing underscores to external function and subroutine names. If the function and subroutine names in the SHTOOLS library do not correspond exactly to what is in the LAPACK and FFTW libraries, the linker will not be able to resolve all of the symbols, and you will receive a linker "undefined symbols" error. Most compilers have options that allows you to "modify" the name of the subroutine and function name by specifying arguments such as "fold external function names to lower case", "do not append trailing underscores to external function names", and so on.

If you are certain that you have installed correctly the FFTW, LAPACK and BLAS libraries, and that the linker can indeed find them, then it is quite possible that a linking problem is related to a trailing underscore being present in the external library, but not in the function and subroutine names in the SHTOOLS library (or vice versa). It is simple to inspect these libraries to determine the underscoring convections. For instance, the unix command "display name list"
```
nm -j libSHTOOLS.a | grep makegridglq
```
yields
```
_makegridglq_
_makegridglq_
_makegridglq_
_makegridglq_
_makegridglqc_
```
Indicating that underscores were added to the SHTOOLS subroutine names.

The command
```
nm -j /Applications/Absoft13.0/lib/liblapack.a | grep dgels
```
yields
```
/Applications/Absoft/lib/liblapack.a(dgels.o):
_dgels_
/Applications/Absoft/lib/liblapack.a(dgelsd.o):
_dgelsd_
/Applications/Absoft/lib/liblapack.a(dgelss.o):
_dgelss_
/Applications/Absoft/lib/liblapack.a(dgelsx.o):
_dgelsx_
/Applications/Absoft/lib/liblapack.a(dgelsy.o):
_dgelsy_
```
indicating that trailing underscores where also appended to the LAPACK subroutine names. However, the command
```
nm -j /usr/local/lib/libfftw3.a | grep dfftw_plan_dft_r2c_1d
```
might yield
```
nm: no name list
_dfftw_plan_dft_r2c_1d
```
indicating that trailing underscores were not appended to the FFTW subroutine names. It is this discrepancy between the FFTW and LAPACK libraries that could make it difficult to link to both simultaneously. To resolve this issue, see [Linking fails because of "Undefined symbols"](#linking-fails-because-of-undefined-references-to-shtools-routines).

## Linking fails because SHTOOLS routines are undefined

Linking your program to the SHTOOLS library might result in errors such as
```
TestSHRotate.f95:(.text+0xdb6): undefined reference to `makegrid2d'
```
The most likely source of this error is that the SHTOOLS library was compiled with a different set of options than used when compiling your program. In particular, it is necessary to use the same compiler options that relate to the naming of external names, such as "fold to lower case" and the appending of underscores.

## I get a "segmentation fault" when running a program that links to the SHTOOLS library

If the error is related to SHTOOLS, and not your code, then there are a few possible ways to fix this. One possibility is that this problem is related to memory allocation and stack overflows. First, try recompiling your code with the option `-s`. This will try to allocate dynamic arrays in the dataspace, as opposed to the stack. Second, try increasing the stacksize by typing
```
limit stacksize unlimited
```
at the unix prompt. If this works, consider adding this to your default shell initialization file. Finally, try increasing the datasize by typing
```
limit datasize unlimited
```

## My program crashes when calling SHTOOLS routines

If your program still crashes after making the above modifications, then try the following:

* Read the version history to make certain that you are aware of any recent modifications.
* Verify that you have enough memory to perform the calculations. Be aware that some routines dynamically allocate arrays.
* Verify that the documentation describing the routine is correct by (1) comparing the man pages with the intent and argument ordering in the interface block of the file `SHTOOLS.f95`, and (2) reading the documentation in the actual source code.
* Try recompiling the SHTOOLS library *without* using optimization, i.e., by removing the fortran flag `-O3`.
* Use a debugger to find out where the error is occurring.

If this all fails, then let us know which routine is involved, as well as the error message you encounter. Any debugging that is done on your part will help greatly in finding the problem. Be aware that we are not being paid to solve your problems.

## How do I specify "optional" parameters in functions and subroutines?

In Fortran 95, this can be specified in one of two ways. First, if there is only one possible optional parameter, it is only necessary to include (or exclude) it at the end of the list of arguments. In contrast, if more than one optional parameter is allowed, it is necessary to use the syntax `OPTIONALVARIABLENAME=variable` at the end of the list of calling arguments. It is strongly suggested that you always directly specify the name of the optional parameter, because it is quite possible that additional optional parameters will be added to the routine at a later date.

## The LAPACK routine DSTEGR crashes with an arithmetic exception

The LAPACK routine DSTEGR is used to obtain the eigenvalues of a symmetric real tridiagonal matrix. Normal execution of this routine may generate infinities and NANs that are properly treated by ieee-754 floating point standards. If this routine crashes with the message
```
*** Arithmetic exception:  - aborting
Aborted
```
the LAPACK library was most likely not compiled using full ieee-754 standards. Try recompiling LAPACK with a compiler option such as `-ieee=full`.
