package Spidy;

use Spidy::Object;
@ISA = ('Spidy::Object');

use strict;
use vars '$VERSION';
$VERSION = '0.21';

use Spidy::HtmlBuilder;
use Spidy::GraphicsBuilder;
use Carp;
use File::Basename;

sub autoload_members {
  my $self = shift;
  return {
    %{$self->SUPER::autoload_members()},
    'tree'             => 'scalar:object',
    'database'         => 'scalar:object',
    'html_builder'     => 'scalar:object',
    'graphics_builder' => 'scalar:object',
  };
}

sub new {
  my $class = shift;
  my %args;
  %args = ('file' => $_[0] ) if @_ == 1;
  %args =  @_ if @_ % 2 == 0;

  %args = $class->command_options( %args )
    unless $ENV{GATEWAY_INTERFACE} =~ /CGI/;
  
  my $self = $class->SUPER::new();
  $self->{'args'} = \%args;
  
  return $self if ($args{show_sizes} && !$args{file}) || $args{templates};

  #
  # set some the output image defaults
  #
  unless( $args{'output_image_path_type'} ) {
    $args{'output_image_path_type'} = 'fixed';
    unless( $args{'image_url'} ) {
      $args{'image_url'} = 'images';
    }
    $args{'file_path'} = '.' unless $args{'file_path'};
    $args{'output_image_path'} = "$args{'file_path'}/images" unless $args{'output_image_path'};
  }

  my $globals;
  if( $args{'database_name'} || $args{'database_server'} || $args{'database_port'} ||
      $args{'database_password'} || $args{'database_user'}   || $args{'database_type'}) {

    $args{'group'} ||= "Spidy::Database::Group";
    $args{'image'} ||= "Spidy::Database::Image";
    
    # dont create if we are in database mode
    $args{'html'} = 0;
      
    require Spidy::Database;
    $self->{'database'} = Spidy::Database->new(
      'server'              => $args{'database_server'},
      'database'            => $args{'database_name'},
      'user'                => $args{'database_user'},
      'password'            => $args{'database_password'},
      'type'                => $args{'database_type'},
      'port'                => $args{'database_port'},
      'warning_sql'         => $args{'warning_sql'},
      'group'               => $args{'group'},
      'image'               => $args{'image'},
    );
  } 
  if ( $args{'file'} ) {
    require Spidy::Tree;
    ($self->{'tree'}, $self->{'globals'}) = Spidy::Tree->new(
      'file'                   => $args{'file'},
      'input_image_path'       => $args{'input_image_path'},
      'output_image_path'      => $args{'output_image_path'},
      'output_image_type'      => $args{'output_image_type'},
      'output_image_path_type' => $args{'output_image_path_type'},
      'base_url'               => $args{'base_url'},
      'file_path'              => $args{'file_path'},
      'image_url'              => $args{'image_url'},
      'verbose'                => defined $args{'verbose'} ? $args{'verbose'} : ( $ENV{'SPIDY_DEBUG'} || 0 ),
      'test_file'              => $args{'test_file'},
      'bg_color'               => $args{'bg_color'},
      'fg_color'               => $args{'fg_color'},
      'vlink_color'            => $args{'vlink_color'},
      'link_color'             => $args{'link_color'},
      'bundle_download'        => $args{'bundle_download'},
      'input_file_path'        => $args{'input_file_path'},
      'parent_url'             => $args{'parent_url'},
      'template_path'          => $args{'template_path'},
      'image_template_file'    => $args{'image_template_file'},
      'group_template_file'    => $args{'group_template_file'},
      'group'                  => $args{'group'},
      'image'                  => $args{'image'},
      'keep_patterns'          => $args{'keep_patterns'},
      'keep_autogroups'        => $args{'keep_autogroups'},
      'keep_names'             => $args{'keep_names'},
    );
  } 

  $self->{'html_builder'} = new Spidy::HtmlBuilder(
    'default_size'        => $args{'default_size'},
  );
  
  $self->{'graphics_builder'} = new Spidy::GraphicsBuilder(
    'im_convert'          => $args{'im_convert'},
    'overwrite'           => defined $args{'overwrite'} ? $args{'overwrite'} : 0 ,
    'sizes'               => defined($args{'sizes'}) && @{$args{'sizes'}} 
                               ? $args{'sizes'} 
                               : [ qw( teeny small medium large ) ],
    'custom_sizes'        => $self->{'globals'}->{'sizes'},
    'watermark'           => $args{'watermark'},
    'gravity'             => $args{'gravity'},
    'type'                => $args{'graphics_builder'},
  );
  
  return $self;  
}

sub insert {
  my $self = shift;
  return $self->{'database'}->insert(
    $self->{'tree'},
    $self->{'graphics_builder'}
  );
}
sub drop {
  return shift->{'database'}->drop();
}
sub create {
  return shift->{'database'}->create();
}

sub get_html_builder {
  return shift->{'html_builder'};
}

sub get_graphics_builder {
  return shift->{'graphics_builder'};
}
  
sub get_html {
  my $self = shift;
  require CGI;
  my $cgi = CGI::new();
  my $hb = $self->{'html_builder'};
  my $size_name = $cgi->param('size') || $hb->{'default_size'};
  my $size = $self->{'graphics_builder'}->get_size($size_name);
  my $path = $cgi->param('path');
  my $type = $cgi->param('type');
  my $group = $self->get_group( $path, $size, $type );
  return CGI::html(
    CGI::center(
      CGI::h1(
        "There has been an error"
      )
    )
  ) unless $group;
#  $group->{'script_name'} = $cgi->script_name;
  return $hb->get_html($group, $size);
} 

sub get_group {
  my $self = shift;
  my $source = $self->{'database'} || $self->{'tree'};
  return $source->get_group(@_);
}

sub build_images {
  my $self = shift;
  my $source = $self->{'tree'} || $self->{'database'};
  return $source->build_images($self->{'graphics_builder'});
}

sub build_html {
  my $self = shift;
  my $source = $self->{'tree'} || $self->{'database'};

  my $gb = $self->{'graphics_builder'};

  return $source->build_html(
    $self->{'html_builder'},
    $gb->{'sizes'}->list()
  );
}

sub header {
  my $self = shift;
  require CGI;
  return CGI::header(@_);
}


sub command_options {

  my $self = shift;
  my %args = @_;
  
  require Getopt::Long;

  # pass all options though, we just want to get
  # out the --config option for now.
  # if a user uses --sizes small --config foobar.txt
  # then we want just pull out the --config foobar.txt
  # and leave the --sizes small for later.

  Getopt::Long::Configure( "pass_through" );
  my $config;

  #
  # first check to see if there is a config option;
  #
  Getopt::Long::GetOptions( 
    'config:s' => \$config,
  );

  #
  # reset the options to default.  So now it will
  # complain about invalid options like --foobar
  #

  # Getopt::Long::Configure( "default" );

  #
  # Check to see if the config file exists.  If it does not
  # then check to see it ./config exists.
  #

  if( defined $config && $config !~ /\S/ ) {
    $config = 0;  # skip config if user just has --config with no options
  } elsif( $config && !-f "$config" ) {
    die "Could not load config file \"$config\": File does not exists\n";
  } elsif( !$config && -f "config" ) {
    $config = 'config';
  }

  my @CONFIG;

  #
  # load up the @CONFIG array with all the options of the 
  # config file.
  #
  if( $config ) {
    print "Loading config file \"$config\"\n"
      unless $ENV{GATEWAY_INTERFACE} =~ /CGI/;
    open CONFIG, "$config" or die "Could not open \"$config\": $!\n";
    #local $/;
    # this does a few things ... obviously :)
    # First, for each line grep out everything that does not
    # start with a leading '#' (spaces optional) ... basically
    # remove the comments from the config files.
    # Next it joins it back together, then splits it up getting rid
    # of all white space and carriage returns
    # Finally it gets rid of the quote characters for arguments since this
    # is normally done by the shell during argument interpolation.  We must
    # mimic that effect here.
    # probably an easier way to do this, but I am not caring so much now.
    @CONFIG = map { 
      s/^("|')(.*?)\1$/$2/; $_  
    } split /\s+/, 
      join "\n", grep {
      !/^\s*#/
    } <CONFIG>;
    close CONFIG;
  }
  
  return wantarray ? %args : \%args if !@CONFIG && !@ARGV;
  
  #
  # parse the config file options first, if there are
  # any.  Save the real arguments in @TMP, then
  # run the GetOptions which reads from @ARGV.  
  # Then after that reset the @ARGV and rerun
  # the GetOptions with the real command arguments
  # ... the real command arguments should override
  #     what is in the config file.
  #
  
  if( @CONFIG ) {
    my @TMP = @ARGV;
    @ARGV = @CONFIG;
    %args = $self->parse_command_options(%args);
    @ARGV = @TMP;
  }
  %args = $self->parse_command_options(%args);

  return wantarray ? %args : \%args;
}

sub parse_command_options {
  my $self = shift;
  my %args = @_;
  
  $args{'html'}   = 1 unless defined $args{'html'};
  $args{'images'} = 1 unless defined $args{'images'};
  $args{'icons'}  = 1 unless defined $args{'icons'};
  
  Getopt::Long::GetOptions( 
      'file=s'                => \$args{'file'},
      'test'                  => \$args{'test'},

      'images!'               => \$args{'images'},
      'forceImages'           => \$args{'overwrite'},
      'inputImagePath=s'      => \$args{'input_image_path'},
      'outputImagePath=s'     => \$args{'output_image_path'},
      'outputImagePathType=s' => \$args{'output_image_path_type'},
      'outputImageType=s'     => \$args{'output_image_type'},

      'html!'                 => \$args{'html'},
      'baseUrl=s'             => \$args{'base_url'},
      'filePath=s'            => \$args{'file_path'},
      'imageUrl=s'            => \$args{'image_url'},
      'tmplPath=s'            => \$args{'template_path'},
      'imageTmpl=s'           => \$args{'image_template_file'},
      'groupTmpl=s'           => \$args{'group_template_file'},
      'nodeTmpl=s'            => \$args{'group_tmpl'},    #deprecated
      'defaultSize=s'         => \$args{'default_size'},
      'bgColor=s'             => \$args{'bg_color'},
      'fgColor=s'             => \$args{'fg_color'},
      'vlinkColor=s'          => \$args{'vlink_color'},
      'linkColor=s'           => \$args{'link_color'},
      'bundleDownload'        => \$args{'bundle_download'},
      'inputFilePath=s'       => \$args{'input_file_path'},
      'parentUrl=s'           => \$args{'parent_url'},

      'icons!'                => \$args{'icons'},
      'convert=s'             => \$args{'im_convert'},
      'sizes:s'               => sub {  
                                   my $var = shift;
                                   # sizes is defined at the top of the page
                                   local $" = ' ';
                                   $args{'sizes'} = [split /,|\s+/, "@_"];
                                   $args{'show_sizes'} = @{$args{'sizes'}} ? 0 : 1;
                                 },
      'watermark=s'           => \$args{'watermark'},
      'gravity=s'             => \$args{'gravity'},
      'graphicsBuilder=s'     => \$args{'graphics_builder'},

      'getTemplates'          => sub{ $args{'templates'}->{'all'} = 1 },
      'getXmlTemplate'        => sub{ $args{'templates'}->{'xml'} = 1 },
      'getImageTemplate'      => sub{ $args{'templates'}->{'image'} = 1 },
      'getGroupTemplate'      => sub{ $args{'templates'}->{'group'} = 1 },
      'getNodeTemplate'       => sub{ $args{'templates'}->{'group'} = 1 },   #deprecated

      'createDB'              => \$args{'create_db'},
      'loadDB'                => \$args{'load_db'},
      'dropDB'                => \$args{'drop_db'},
      'warnSql'               => \$args{'warning_sql'},
      'type:s'                => \$args{'database_type'},
      'user=s'                => \$args{'database_user'},
      'database=s'            => \$args{'database_name'},
      'server=s'              => \$args{'database_server'},
      'port=i'                => \$args{'database_port'},
      'password=s'            => \$args{'database_password'},

      'help'                  => \&usage,
      'verbose'               => \$args{'verbose'},
  ) or usage();
  
  return wantarray ? %args : \%args;
}

sub show_sizes {
  my $s = shift;
  require Spidy::GraphicsBuilder;
  my $gb = ref($s) && $s->{'graphics_builder'}
    ? $s->{'graphics_builder'}
    : Spidy::GraphicsBuilder->new();
  
  my @sizes = $gb->{'default_sizes'}->list();

  my @fields = qw( name max_width max_height quality padded watermark custom );
  my %largest;
  for my $size ( @sizes ) {
    for my $key ( @fields ) {
      my $title = $key;
      $title =~ s/max_//;
      $largest{$key} = length( $title )
        if $largest{$key} < length( $title );
      
      $largest{$key} = length( $size->{$key} )
        if $largest{$key} < length( $size->{$key} );
    }
  }
  print "+";
  print "-"x$largest{$_}, "+" for @fields;
  print "\n";

  for my $key ( @fields ) {
    my $title = $key;
    $title =~ s/max_//;
    print "|$title", " "x($largest{$key} - length($title));
  }
  print "|\n";
    
  print "+";
  print "-"x$largest{$_}, "+" for @fields;
  print "\n";
        
  for my $size ( @sizes ) {      
    for my $key ( @fields ) {
      my $val;
      if( $key =~ /padded|custom|watermark/ ) {
        $val = ($size->{$key} ? "yes" : "no "). " "x($largest{$key} - 3);
      } elsif( $key =~ /width|height|quality/ ) {
        my $tmp = $size->{$key} || "*";
        $val = " "x($largest{$key} - length($tmp)).$tmp;
      } else {
        my $tmp = $size->{$key} || "*";
        $val = $tmp." "x($largest{$key} - length($tmp));
      }
            
      print "|$val";
    }
    print "|\n";
  }
  print "+";
  print "-"x$largest{$_}, "+" for @fields;
  print "\n";
  exit -1;
}

sub usage {
    my $warning = shift;
    require File::Basename;
    
    my $dir = File::Basename::dirname( $INC{'Spidy.pm'} );
    warn $warning, "\n" if $warning;
    warn <<EOF;
Usage: spider [--file=<config file>] [--config=<file> | --config]
              [--help] [--verbose]
              [--sizes=<size1,size2,...>] [--images | --noimages]
              [--forceImages] [--inputImagePath=<dir>] [--outputImagePath=<dir>]
              [--outputImagePathType=<fixed|relative>]
              [--html | --nohtml] [--baseUrl=<url>] [--filePath=<dir>]
              [--imageUrl=<url>] [--tmplPath=<dir>] [--imageTmpl=<file>]
              [--groupTmpl=<file>] [--noicons] [--convert=<path to binary>]
              [--bgColor=<color>] [--fgColor=<color>] [--vlinkColor=<color>]
              [--linkColor=<color>] [--dropDB] [--createDB] [--loadDB]
              [--warnSql] [--server=<host>] [--database=<name>] [--user=<name>]
              [--password=<pass>] [--type=<Pg|Sybase|Mysql>] [--port=<num>]
              [--bundleDownload] [--inputFilePath=<dir>] [--parentUrl=<url>]
              [--outputImageType=<type>] [--watermark=<text>]
              [--gravity=<direction>] [--graphicsBuilder=<module>]
       spider --sizes      
       spider [--getTemplates] [--getXmlTemplate] [--getImageTemplate]
              [--getGroupTemplate]
       spider --file <config file> --test

file:             The Configuration XML document describing the site.
config:           Specify a file which holds 'spider' options to be used.
                  [default=./config]  If no argument to --config it will
                  not use any config file.
help:             This message.
verbose           Turns on the debug flag. The xml document tree will be output
                  to stderr, along with other interesting information as the
                  program runs.

sizes:            Without arguments - all the current sizes are printed.
                  With argument - only the images and html dealing with the
                  specified sizes will be created.
bundleDownload:   Will create pseudo tar.gz files that can be used to download
                  all of the input images for a give directory. [default=no]
images:           Create all the different sizes images that are used in
                  the XML file. [default=1]
noimages:         Do not create any images.
forceImages:      If the different sized images already exists, then
                  overwrite them. [default=0]
inputImagePath:   Directory where original images can be found. [default=.]
outputImagePath:  Directory root of where resized images will be placed.
                  [default='<filePath>/images' if outputImagePathType is 
                  'fixed', else the default is <filePath>]
outputImagePathType: This should be "fixed" or "relative".  The default is
                  "fixed".  Basically, should the images be placed "relative"
                  to the url path (include group names) or all "fixed" together
                  in one big directory. [default=fixed]
outputImageType:  This option will cause the program to force all output images
                  to be created with the specified filetype. 
                  [default=same as input image]
html:             Create all the html pages needed for the website. [default=1]
nohtml:           Do no create any html files.
baseUrl:          This will set the <BASE HREF...> tag in the html documents
filePath:         Directory root whre the html files will be placed.
                  [default=.]
imageUrl:         URL of where the images will be viewable from the web
                  browser. [default='images' if outputImagePathType is
                  'fixed', else the default is <baseUrl>]
tmplPath:         Directory containing the group.tmpl and image.tmpl html
                  templates.
                  [default=$dir/Spidy/Temlplates]
imageTmpl:        Template File to be used for the single image pages.
                  [default=<tmplPath>/Spidy/Templates/image.tmpl]
groupTmpl:         Template File to be used for the 'image group' aka group
                  pages. [default=<tmplPath>/Spidy/Templates/group.tmpl]
defaultSize:      Which image size pages will be linked to 'index.html'
noicons:          Do not install the default icons into <filePath>/icons
convert:          Which 'convert' Image Magick program to use on your system 
                  to create the various size images.
watermark:        Provide an image name that will be used as an overlay layer
                  on the output images. [default=<none>]
gravity:          Used for placement of the watermark text. [default=northwest]
graphicsBuilder:  Used to specify the graphics generation module to use.  The default
                  is "ImageMagick".  "Imager", and "PerlMagick" are also supported.
test:             Tries to parse the input file, but does nothing else.
getTemplates:     Retrieves all templates and puts them in <filePath>
getXmlTemplate:   Retrieves a generic Xml template and puts it in <filePath>
getImageTemplate: Retrieves the default image template and puts it in <filePath>
getGroupTemplate:  Retrieves the default group template and puts it in <filePath>
bgColor:          Background color [default=black]
fgColor:          Text color       [default=white]
vlinkColor:       Visited link color [default=7E26CE (purple)]
linkColor:        Unvisitied link color [defaul=4A4AFF (blue)]
inputFilePath:    Directory where to find the files named with the 'file'
                  attribute to the <comment> tags.  [default='.' but
                  ignored if absolute path used]
parentUrl         What the "Up one level" link goes to for the top level
                  group.  There is no default.
createDB:         Create the table structure to use a spider database.
loadDB:           Load the database from an xml config file specified by --file
dropDB:           Delete all the tables in the database
warnSql:          Print all the SQL statements executed to stderr.
server:           Database server
database:         Database name [default=spider]
user:             Database user
password:         Database password
type:             Database type, one of Pg (PostgreSql), Mysql, or Sybase.
                  [default=Pg]     
port:             Database server port.
EOF
    exit -1;
}


1;

__END__

=head1 NAME

Spidy - Spider Eyeballs modules to create image gallery website.

=head1 SYNOPSIS

Nothing yet

=head1 DESCRIPTION

Nothing yet, come back later.  See L<spider> for current documentation.

=head1 BUGS

There are probably bugs, but I do not know of any right now.  
Please contact me if you find one.

=head1 ACKNOWLEDGEMENTS

I would like the thank all the people who have helped write the required
modules for some very fine work.

This program was only made possible by Julian Cash's tremendous experience with
photography and web design, and his willingness to share these experiences
with the rest of the world.  He is responsible for the concepts and previous 
implemention of this program: Spider Eyeballs.

=head1 AUTHOR

Copyright 2001, Cory Bennett <coryb@cobracommander.com>

This module is distributed under the same terms as Perl itself.  Feel
free to use, modify and redistribute it as long as you retain the
correct attribution.

=head1 SEE ALSO

The official site for this program is http://www.spidereyeballs.com

L<spider>

=cut
