package Spidy::GraphicsBuilder;

use Spidy::Object;
@ISA = ('Spidy::Object');

use Spidy::Size;
use File::Path;
use File::Copy;
use File::Spec;
use Carp;
use strict;

use vars '$rxTrue';

$rxTrue = qr{^(?i)(?:1|yes|true|ok)$};

sub autoload_members {
  return {
    'sizes'         => 'list',
    'default_sizes' => 'list',
#     'im_convert'    => "scalar=$ENV{'IM_CONVERT'}",
#     'im_composite'  => "scalar=$ENV{'IM_COMPOSITE'}",
    'overwrite'     => 'bool=0',
    'custom_sizes'  => 'array',
    'watermark'     => 'scalar',
    'gravity'       => 'scalar=northwest',
    'type'          => 'scalar',
  }
}

sub new {
  my $package = shift;
  my %args = @_;
  
  $args{'type'} = lc($args{'type'}) || 'imagemagick';
  my $gb_package = "Spidy::GraphicsBuilder::$args{'type'}";
  eval "require $gb_package";
  croak "Error loading graphics builder module $gb_package: $@" if $@;
  #call the parent class of the GraphicsBuilder package should go to
  # Spidy::Object::new
  my $self = $gb_package->SUPER::new(%args);

  my @sizeNames;
  if( ref($self->{'sizes'}) eq 'ARRAY' ) {
    @sizeNames = @{$self->{'sizes'}};
  }
  #
  # there are no sizes already specified, so 
  # we will create a list of the default ones now.
  #
    
  my @sizes = (
      new Spidy::Size( 
        name       => 'thumb',
        max_width  => 60,
        max_height => 60,
        quality    => 80,
        padded     => 1,
        watermark  => 0,
      ),
      new Spidy::Size( 
        name       => 'teeny',
        max_width  => 250,
        max_height => 200,
        quality    => 80,
        watermark  => 1,
      ),
      new Spidy::Size( 
        name       => 'small',
        max_width  => 400,
        max_height => 300,
        quality    => 80,
        watermark  => 1,
      ),
      new Spidy::Size(
        name       => 'medium',
        max_width  => 500,
        max_height => 400,
        quality    => 80,
        watermark  => 1,
      ),
      new Spidy::Size(
        name       => 'large', 
        max_width  => 800,
        max_height => 600,
        quality    => 90,
        watermark  => 1,
      ),
      new Spidy::Size(
        name       => 'xlarge',
        max_width  => 1300,
        max_height => 1000,
        quality    => 97,
        watermark  => 1,
      ),
      new Spidy::Size( 
        name       => 'original',
        watermark  => 0,
        quality    => 95,
        symlink    => 1,
        max_width  => "w",
        max_height => "h"
      ) 
    );
   
  if( $self->{'custom_sizes'} ) {
    for my $cs ( @{$self->{'custom_sizes'}} ) {
      my $new = 1;
      for my $size ( @sizes ) {
        if( $cs->{'name'} eq $size->{'name'} ) {
          $new = 0;
          for my $k ( keys %$cs ) { 
            $size->{$k} = $cs->{$k} if defined $cs->{$k};
          }
        }
      }
      if( $new ) {
        push @sizes, $cs;
      }
    }
  }
  
  #
  # make sure thumb is in the sizeNames and it should
  # be first in the list.
  #
  unshift @sizeNames, "thumb" 
    unless grep { /thumb/ } @sizeNames or @sizeNames == 0;

  #
  # save the default sizes if the users has specifed a new list
  # of size names;
  #
  $self->{'default_sizes'} = Spidy::Size->link(@sizes);
  
  my @use;
  for my $name ( @sizeNames ) {
    for my $size ( @sizes ) {
      push @use, $size->copy() if $size->{'name'} eq $name;
    }
  }
      
  $self->{'sizes'} = Spidy::Size->link(@use ? @use : @sizes);

  return $self;
}
 
#
# This code was originally lifted from 
# Graphics::ColorNames::X.pm but has been slightly
# modified.  These are the 16 standard colors allowed
# in all HTML pages.  If it is not in here, the user
# will have to specify the HEX code.
#
use constant COLORS => {
  "snow"		    => "FFFAFA",
  "ghost white"             => "F8F8FF",
  "ghostwhite"              => "F8F8FF",
  "white smoke"             => "F5F5F5",
  "whitesmoke"              => "F5F5F5",
  "gainsboro"		    => "DCDCDC",
  "floral white"	    => "FFFAF0",
  "floralwhite"             => "FFFAF0",
  "old lace"		    => "FDF5E6",
  "oldlace"		    => "FDF5E6",
  "linen"		    => "FAF0E6",
  "antique white"	    => "FAEBD7",
  "antiquewhite"	    => "FAEBD7",
  "papaya whip"             => "FFEFD5",
  "papayawhip"              => "FFEFD5",
  "blanched almond"	    => "FFEBCD",
  "blanchedalmond"	    => "FFEBCD",
  "bisque"		    => "FFE4C4",
  "peach puff"              => "FFDAB9",
  "peachpuff"		    => "FFDAB9",
  "navajo white"	    => "FFDEAD",
  "navajowhite"             => "FFDEAD",
  "moccasin"		    => "FFE4B5",
  "cornsilk"		    => "FFF8DC",
  "ivory"		    => "FFFFF0",
  "lemon chiffon"	    => "FFFACD",
  "lemonchiffon"	    => "FFFACD",
  "seashell"		    => "FFF5EE",
  "honeydew"		    => "F0FFF0",
  "mint cream"              => "F5FFFA",
  "mintcream"		    => "F5FFFA",
  "azure"		    => "F0FFFF",
  "alice blue"              => "F0F8FF",
  "aliceblue"		    => "F0F8FF",
  "lavender"		    => "E6E6FA",
  "lavender blush"	    => "FFF0F5",
  "lavenderblush"	    => "FFF0F5",
  "misty rose"              => "FFE4E1",
  "mistyrose"		    => "FFE4E1",
  "white"		    => "FFFFFF",
  "black"		    => "000000",
  "dark slate gray"	    => "2F4F4F",
  "darkslategray"	    => "2F4F4F",
  "dark slate grey"	    => "2F4F4F",
  "darkslategrey"	    => "2F4F4F",
  "dim gray"		    => "696969",
  "dimgray"		    => "696969",
  "dim grey"		    => "696969",
  "dimgrey"		    => "696969",
  "slate gray"              => "708090",
  "slategray"		    => "708090",
  "slate grey"              => "708090",
  "slategrey"		    => "708090",
  "light slate gray"	    => "778899",
  "lightslategray"	    => "778899",
  "light slate grey"	    => "778899",
  "lightslategrey"	    => "778899",
  "gray"		    => "BEBEBE",
  "grey"		    => "BEBEBE",
  "light grey"              => "D3D3D3",
  "lightgrey"		    => "D3D3D3",
  "light gray"              => "D3D3D3",
  "lightgray"		    => "D3D3D3",
  "midnight blue"	    => "191970",
  "midnightblue"	    => "191970",
  "navy"		    => "000080",
  "navy blue"		    => "000080",
  "navyblue"		    => "000080",
  "cornflower blue"	    => "6495ED",
  "cornflowerblue"	    => "6495ED",
  "dark slate blue"	    => "483D8B",
  "darkslateblue"	    => "483D8B",
  "slate blue"              => "6A5ACD",
  "slateblue"		    => "6A5ACD",
  "medium slate blue"	    => "7B68EE",
  "mediumslateblue"	    => "7B68EE",
  "light slate blue"	    => "8470FF",
  "lightslateblue"	    => "8470FF",
  "medium blue"             => "0000CD",
  "mediumblue"              => "0000CD",
  "royal blue"              => "4169E1",
  "royalblue"		    => "4169E1",
  "blue"		    => "0000FF",
  "dodger blue"             => "1E90FF",
  "dodgerblue"              => "1E90FF",
  "deep sky blue"	    => "00BFFF",
  "deepskyblue"             => "00BFFF",
  "sky blue"		    => "87CEEB",
  "skyblue"		    => "87CEEB",
  "light sky blue"	    => "87CEFA",
  "lightskyblue"	    => "87CEFA",
  "steel blue"              => "4682B4",
  "steelblue"		    => "4682B4",
  "light steel blue"	    => "B0C4DE",
  "lightsteelblue"	    => "B0C4DE",
  "light blue"              => "ADD8E6",
  "lightblue"		    => "ADD8E6",
  "powder blue"             => "B0E0E6",
  "powderblue"              => "B0E0E6",
  "pale turquoise"	    => "AFEEEE",
  "paleturquoise"	    => "AFEEEE",
  "dark turquoise"	    => "00CED1",
  "darkturquoise"	    => "00CED1",
  "medium turquoise"	    => "48D1CC",
  "mediumturquoise"	    => "48D1CC",
  "turquoise"		    => "40E0D0",
  "cyan"		    => "00FFFF",
  "light cyan"              => "E0FFFF",
  "lightcyan"		    => "E0FFFF",
  "cadet blue"              => "5F9EA0",
  "cadetblue"		    => "5F9EA0",
  "medium aquamarine"	    => "66CDAA",
  "mediumaquamarine"	    => "66CDAA",
  "aquamarine"              => "7FFFD4",
  "dark green"              => "006400",
  "darkgreen"		    => "006400",
  "dark olive green"	    => "556B2F",
  "darkolivegreen"	    => "556B2F",
  "dark sea green"	    => "8FBC8F",
  "darkseagreen"	    => "8FBC8F",
  "sea green"		    => "2E8B57",
  "seagreen"		    => "2E8B57",
  "medium sea green"	    => "3CB371",
  "mediumseagreen"	    => "3CB371",
  "light sea green"	    => "20B2AA",
  "lightseagreen"	    => "20B2AA",
  "pale green"              => "98FB98",
  "palegreen"		    => "98FB98",
  "spring green"	    => "00FF7F",
  "springgreen"             => "00FF7F",
  "lawn green"              => "7CFC00",
  "lawngreen"		    => "7CFC00",
  "green"		    => "00FF00",
  "chartreuse"              => "7FFF00",
  "medium spring green"     => "00FA9A",
  "mediumspringgreen"	    => "00FA9A",
  "green yellow"	    => "ADFF2F",
  "greenyellow"             => "ADFF2F",
  "lime green"              => "32CD32",
  "limegreen"		    => "32CD32",
  "yellow green"	    => "9ACD32",
  "yellowgreen"             => "9ACD32",
  "forest green"	    => "228B22",
  "forestgreen"             => "228B22",
  "olive drab"              => "6B8E23",
  "olivedrab"		    => "6B8E23",
  "dark khaki"              => "BDB76B",
  "darkkhaki"		    => "BDB76B",
  "khaki"		    => "F0E68C",
  "pale goldenrod"	    => "EEE8AA",
  "palegoldenrod"	    => "EEE8AA",
  "light goldenrod yellow"  => "FAFAD2",
  "lightgoldenrodyellow"    => "FAFAD2",
  "light yellow"	    => "FFFFE0",
  "lightyellow"             => "FFFFE0",
  "yellow"		    => "FFFF00",
  "gold"		    => "FFD700",
  "light goldenrod"	    => "EEDD82",
  "lightgoldenrod"	    => "EEDD82",
  "goldenrod"		    => "DAA520",
  "dark goldenrod"	    => "B8860B",
  "darkgoldenrod"	    => "B8860B",
  "rosy brown"              => "BC8F8F",
  "rosybrown"		    => "BC8F8F",
  "indian red"              => "CD5C5C",
  "indianred"		    => "CD5C5C",
  "saddle brown"	    => "8B4513",
  "saddlebrown"             => "8B4513",
  "sienna"		    => "A0522D",
  "peru"		    => "CD853F",
  "burlywood"		    => "DEB887",
  "beige"		    => "F5F5DC",
  "wheat"		    => "F5DEB3",
  "sandy brown"             => "F4A460",
  "sandybrown"              => "F4A460",
  "tan"                     => "D2B48C",
  "chocolate"		    => "D2691E",
  "firebrick"		    => "B22222",
  "brown"		    => "A52A2A",
  "dark salmon"             => "E9967A",
  "darksalmon"              => "E9967A",
  "salmon"		    => "FA8072",
  "light salmon"	    => "FFA07A",
  "lightsalmon"             => "FFA07A",
  "orange"		    => "FFA500",
  "dark orange"             => "FF8C00",
  "darkorange"              => "FF8C00",
  "coral"		    => "FF7F50",
  "light coral"             => "F08080",
  "lightcoral"              => "F08080",
  "tomato"		    => "FF6347",
  "orange red"              => "FF4500",
  "orangered"		    => "FF4500",
  "red"		            => "FF0000",
  "hot pink"		    => "FF69B4",
  "hotpink"		    => "FF69B4",
  "deep pink"		    => "FF1493",
  "deeppink"		    => "FF1493",
  "pink"		    => "FFC0CB",
  "light pink"	            => "FFB6C1",
  "lightpink"		    => "FFB6C1",
  "pale violet red"	    => "DB7093",
  "palevioletred"	    => "DB7093",
  "maroon"		    => "B03060",
  "medium violet red"	    => "C71585",
  "mediumvioletred"	    => "C71585",
  "violet red"              => "D02090",
  "violetred"		    => "D02090",
  "magenta"		    => "FF00FF",
  "violet"		    => "EE82EE",
  "plum"		    => "DDA0DD",
  "orchid"		    => "DA70D6",
  "medium orchid"	    => "BA55D3",
  "mediumorchid"	    => "BA55D3",
  "dark orchid"             => "9932CC",
  "darkorchid"              => "9932CC",
  "dark violet"             => "9400D3",
  "darkviolet"              => "9400D3",
  "blue violet"             => "8A2BE2",
  "blueviolet"              => "8A2BE2",
  "purple"		    => "A020F0",
  "medium purple"	    => "9370DB",
  "mediumpurple"	    => "9370DB",
  "thistle"		    => "D8BFD8",
  "snow1"		    => "FFFAFA",
  "snow2"		    => "EEE9E9",
  "snow3"		    => "CDC9C9",
  "snow4"		    => "8B8989",
  "seashell1"		    => "FFF5EE",
  "seashell2"		    => "EEE5DE",
  "seashell3"		    => "CDC5BF",
  "seashell4"		    => "8B8682",
  "antiquewhite1"	    => "FFEFDB",
  "antiquewhite2"	    => "EEDFCC",
  "antiquewhite3"	    => "CDC0B0",
  "antiquewhite4"	    => "8B8378",
  "bisque1"		    => "FFE4C4",
  "bisque2"		    => "EED5B7",
  "bisque3"		    => "CDB79E",
  "bisque4"		    => "8B7D6B",
  "peachpuff1"              => "FFDAB9",
  "peachpuff2"              => "EECBAD",
  "peachpuff3"              => "CDAF95",
  "peachpuff4"              => "8B7765",
  "navajowhite1"	    => "FFDEAD",
  "navajowhite2"	    => "EECFA1",
  "navajowhite3"	    => "CDB38B",
  "navajowhite4"	    => "8B795E",
  "lemonchiffon1"	    => "FFFACD",
  "lemonchiffon2"	    => "EEE9BF",
  "lemonchiffon3"	    => "CDC9A5",
  "lemonchiffon4"	    => "8B8970",
  "cornsilk1"		    => "FFF8DC",
  "cornsilk2"		    => "EEE8CD",
  "cornsilk3"		    => "CDC8B1",
  "cornsilk4"		    => "8B8878",
  "ivory1"		    => "FFFFF0",
  "ivory2"		    => "EEEEE0",
  "ivory3"		    => "CDCDC1",
  "ivory4"		    => "8B8B83",
  "honeydew1"		    => "F0FFF0",
  "honeydew2"		    => "E0EEE0",
  "honeydew3"		    => "C1CDC1",
  "honeydew4"		    => "838B83",
  "lavenderblush1"	    => "FFF0F5",
  "lavenderblush2"	    => "EEE0E5",
  "lavenderblush3"	    => "CDC1C5",
  "lavenderblush4"	    => "8B8386",
  "mistyrose1"              => "FFE4E1",
  "mistyrose2"              => "EED5D2",
  "mistyrose3"              => "CDB7B5",
  "mistyrose4"              => "8B7D7B",
  "azure1"		    => "F0FFFF",
  "azure2"		    => "E0EEEE",
  "azure3"		    => "C1CDCD",
  "azure4"		    => "838B8B",
  "slateblue1"              => "836FFF",
  "slateblue2"              => "7A67EE",
  "slateblue3"              => "6959CD",
  "slateblue4"              => "473C8B",
  "royalblue1"              => "4876FF",
  "royalblue2"              => "436EEE",
  "royalblue3"              => "3A5FCD",
  "royalblue4"              => "27408B",
  "blue1"		    => "0000FF",
  "blue2"		    => "0000EE",
  "blue3"		    => "0000CD",
  "blue4"		    => "00008B",
  "dodgerblue1"             => "1E90FF",
  "dodgerblue2"             => "1C86EE",
  "dodgerblue3"             => "1874CD",
  "dodgerblue4"             => "104E8B",
  "steelblue1"              => "63B8FF",
  "steelblue2"              => "5CACEE",
  "steelblue3"              => "4F94CD",
  "steelblue4"              => "36648B",
  "deepskyblue1"	    => "00BFFF",
  "deepskyblue2"	    => "00B2EE",
  "deepskyblue3"	    => "009ACD",
  "deepskyblue4"	    => "00688B",
  "skyblue1"		    => "87CEFF",
  "skyblue2"		    => "7EC0EE",
  "skyblue3"		    => "6CA6CD",
  "skyblue4"		    => "4A708B",
  "lightskyblue1"	    => "B0E2FF",
  "lightskyblue2"	    => "A4D3EE",
  "lightskyblue3"	    => "8DB6CD",
  "lightskyblue4"	    => "607B8B",
  "slategray1"              => "C6E2FF",
  "slategray2"              => "B9D3EE",
  "slategray3"              => "9FB6CD",
  "slategray4"              => "6C7B8B",
  "lightsteelblue1"	    => "CAE1FF",
  "lightsteelblue2"	    => "BCD2EE",
  "lightsteelblue3"	    => "A2B5CD",
  "lightsteelblue4"	    => "6E7B8B",
  "lightblue1"              => "BFEFFF",
  "lightblue2"              => "B2DFEE",
  "lightblue3"              => "9AC0CD",
  "lightblue4"              => "68838B",
  "lightcyan1"              => "E0FFFF",
  "lightcyan2"              => "D1EEEE",
  "lightcyan3"              => "B4CDCD",
  "lightcyan4"              => "7A8B8B",
  "paleturquoise1"	    => "BBFFFF",
  "paleturquoise2"	    => "AEEEEE",
  "paleturquoise3"	    => "96CDCD",
  "paleturquoise4"	    => "668B8B",
  "cadetblue1"              => "98F5FF",
  "cadetblue2"              => "8EE5EE",
  "cadetblue3"              => "7AC5CD",
  "cadetblue4"              => "53868B",
  "turquoise1"              => "00F5FF",
  "turquoise2"              => "00E5EE",
  "turquoise3"              => "00C5CD",
  "turquoise4"              => "00868B",
  "cyan1"		    => "00FFFF",
  "cyan2"		    => "00EEEE",
  "cyan3"		    => "00CDCD",
  "cyan4"		    => "008B8B",
  "darkslategray1"	    => "97FFFF",
  "darkslategray2"	    => "8DEEEE",
  "darkslategray3"	    => "79CDCD",
  "darkslategray4"	    => "528B8B",
  "aquamarine1"             => "7FFFD4",
  "aquamarine2"             => "76EEC6",
  "aquamarine3"             => "66CDAA",
  "aquamarine4"             => "458B74",
  "darkseagreen1"	    => "C1FFC1",
  "darkseagreen2"	    => "B4EEB4",
  "darkseagreen3"	    => "9BCD9B",
  "darkseagreen4"	    => "698B69",
  "seagreen1"		    => "54FF9F",
  "seagreen2"		    => "4EEE94",
  "seagreen3"		    => "43CD80",
  "seagreen4"		    => "2E8B57",
  "palegreen1"              => "9AFF9A",
  "palegreen2"              => "90EE90",
  "palegreen3"              => "7CCD7C",
  "palegreen4"              => "548B54",
  "springgreen1"	    => "00FF7F",
  "springgreen2"	    => "00EE76",
  "springgreen3"	    => "00CD66",
  "springgreen4"	    => "008B45",
  "green1"		    => "00FF00",
  "green2"		    => "00EE00",
  "green3"		    => "00CD00",
  "green4"		    => "008B00",
  "chartreuse1"             => "7FFF00",
  "chartreuse2"             => "76EE00",
  "chartreuse3"             => "66CD00",
  "chartreuse4"             => "458B00",
  "olivedrab1"              => "C0FF3E",
  "olivedrab2"              => "B3EE3A",
  "olivedrab3"              => "9ACD32",
  "olivedrab4"              => "698B22",
  "darkolivegreen1"	    => "CAFF70",
  "darkolivegreen2"	    => "BCEE68",
  "darkolivegreen3"	    => "A2CD5A",
  "darkolivegreen4"	    => "6E8B3D",
  "khaki1"		    => "FFF68F",
  "khaki2"		    => "EEE685",
  "khaki3"		    => "CDC673",
  "khaki4"		    => "8B864E",
  "lightgoldenrod1"	    => "FFEC8B",
  "lightgoldenrod2"	    => "EEDC82",
  "lightgoldenrod3"	    => "CDBE70",
  "lightgoldenrod4"	    => "8B814C",
  "lightyellow1"	    => "FFFFE0",
  "lightyellow2"	    => "EEEED1",
  "lightyellow3"	    => "CDCDB4",
  "lightyellow4"	    => "8B8B7A",
  "yellow1"		    => "FFFF00",
  "yellow2"		    => "EEEE00",
  "yellow3"		    => "CDCD00",
  "yellow4"		    => "8B8B00",
  "gold1"		    => "FFD700",
  "gold2"		    => "EEC900",
  "gold3"		    => "CDAD00",
  "gold4"		    => "8B7500",
  "goldenrod1"              => "FFC125",
  "goldenrod2"              => "EEB422",
  "goldenrod3"              => "CD9B1D",
  "goldenrod4"              => "8B6914",
  "darkgoldenrod1"	    => "FFB90F",
  "darkgoldenrod2"	    => "EEAD0E",
  "darkgoldenrod3"	    => "CD950C",
  "darkgoldenrod4"	    => "8B6508",
  "rosybrown1"              => "FFC1C1",
  "rosybrown2"              => "EEB4B4",
  "rosybrown3"              => "CD9B9B",
  "rosybrown4"              => "8B6969",
  "indianred1"              => "FF6A6A",
  "indianred2"              => "EE6363",
  "indianred3"              => "CD5555",
  "indianred4"              => "8B3A3A",
  "sienna1"		    => "FF8247",
  "sienna2"		    => "EE7942",
  "sienna3"		    => "CD6839",
  "sienna4"		    => "8B4726",
  "burlywood1"              => "FFD39B",
  "burlywood2"              => "EEC591",
  "burlywood3"              => "CDAA7D",
  "burlywood4"              => "8B7355",
  "wheat1"		    => "FFE7BA",
  "wheat2"		    => "EED8AE",
  "wheat3"		    => "CDBA96",
  "wheat4"		    => "8B7E66",
  "tan1"		    => "FFA54F",
  "tan2"		    => "EE9A49",
  "tan3"		    => "CD853F",
  "tan4"		    => "8B5A2B",
  "chocolate1"              => "FF7F24",
  "chocolate2"              => "EE7621",
  "chocolate3"              => "CD661D",
  "chocolate4"              => "8B4513",
  "firebrick1"              => "FF3030",
  "firebrick2"              => "EE2C2C",
  "firebrick3"              => "CD2626",
  "firebrick4"              => "8B1A1A",
  "brown1"		    => "FF4040",
  "brown2"		    => "EE3B3B",
  "brown3"		    => "CD3333",
  "brown4"		    => "8B2323",
  "salmon1"		    => "FF8C69",
  "salmon2"		    => "EE8262",
  "salmon3"		    => "CD7054",
  "salmon4"		    => "8B4C39",
  "lightsalmon1"	    => "FFA07A",
  "lightsalmon2"	    => "EE9572",
  "lightsalmon3"	    => "CD8162",
  "lightsalmon4"	    => "8B5742",
  "orange1"		    => "FFA500",
  "orange2"		    => "EE9A00",
  "orange3"		    => "CD8500",
  "orange4"		    => "8B5A00",
  "darkorange1"             => "FF7F00",
  "darkorange2"             => "EE7600",
  "darkorange3"             => "CD6600",
  "darkorange4"             => "8B4500",
  "coral1"		    => "FF7256",
  "coral2"		    => "EE6A50",
  "coral3"		    => "CD5B45",
  "coral4"		    => "8B3E2F",
  "tomato1"		    => "FF6347",
  "tomato2"		    => "EE5C42",
  "tomato3"		    => "CD4F39",
  "tomato4"		    => "8B3626",
  "orangered1"              => "FF4500",
  "orangered2"              => "EE4000",
  "orangered3"              => "CD3700",
  "orangered4"              => "8B2500",
  "red1"		    => "FF0000",
  "red2"		    => "EE0000",
  "red3"		    => "CD0000",
  "red4"		    => "8B0000",
  "deeppink1"		    => "FF1493",
  "deeppink2"		    => "EE1289",
  "deeppink3"		    => "CD1076",
  "deeppink4"		    => "8B0A50",
  "hotpink1"		    => "FF6EB4",
  "hotpink2"		    => "EE6AA7",
  "hotpink3"		    => "CD6090",
  "hotpink4"		    => "8B3A62",
  "pink1"		    => "FFB5C5",
  "pink2"		    => "EEA9B8",
  "pink3"		    => "CD919E",
  "pink4"		    => "8B636C",
  "lightpink1"              => "FFAEB9",
  "lightpink2"              => "EEA2AD",
  "lightpink3"              => "CD8C95",
  "lightpink4"              => "8B5F65",
  "palevioletred1"	    => "FF82AB",
  "palevioletred2"	    => "EE799F",
  "palevioletred3"	    => "CD6889",
  "palevioletred4"	    => "8B475D",
  "maroon1"		    => "FF34B3",
  "maroon2"		    => "EE30A7",
  "maroon3"		    => "CD2990",
  "maroon4"		    => "8B1C62",
  "violetred1"              => "FF3E96",
  "violetred2"              => "EE3A8C",
  "violetred3"              => "CD3278",
  "violetred4"              => "8B2252",
  "magenta1"		    => "FF00FF",
  "magenta2"		    => "EE00EE",
  "magenta3"		    => "CD00CD",
  "magenta4"		    => "8B008B",
  "orchid1"		    => "FF83FA",
  "orchid2"		    => "EE7AE9",
  "orchid3"		    => "CD69C9",
  "orchid4"		    => "8B4789",
  "plum1"		    => "FFBBFF",
  "plum2"		    => "EEAEEE",
  "plum3"		    => "CD96CD",
  "plum4"		    => "8B668B",
  "mediumorchid1"	    => "E066FF",
  "mediumorchid2"	    => "D15FEE",
  "mediumorchid3"	    => "B452CD",
  "mediumorchid4"	    => "7A378B",
  "darkorchid1"             => "BF3EFF",
  "darkorchid2"             => "B23AEE",
  "darkorchid3"             => "9A32CD",
  "darkorchid4"             => "68228B",
  "purple1"		    => "9B30FF",
  "purple2"		    => "912CEE",
  "purple3"		    => "7D26CD",
  "purple4"		    => "551A8B",
  "mediumpurple1"	    => "AB82FF",
  "mediumpurple2"	    => "9F79EE",
  "mediumpurple3"	    => "8968CD",
  "mediumpurple4"	    => "5D478B",
  "thistle1"		    => "FFE1FF",
  "thistle2"		    => "EED2EE",
  "thistle3"		    => "CDB5CD",
  "thistle4"		    => "8B7B8B",
  "gray0"		    => "000000",
  "gray1"		    => "030303",
  "gray2"		    => "050505",
  "gray3"		    => "080808",
  "grey4"		    => "0A0A0A",
  "grey5"		    => "0D0D0D",
  "grey6"		    => "0F0F0F",
  "grey7"		    => "121212",
  "grey8"		    => "141414",
  "grey9"		    => "171717",
  "grey10"		    => "1A1A1A",
  "grey11"		    => "1C1C1C",
  "grey12"		    => "1F1F1F",
  "grey13"		    => "212121",
  "grey14"		    => "242424",
  "grey15"		    => "262626",
  "grey16"		    => "292929",
  "grey17"		    => "2B2B2B",
  "grey18"		    => "2E2E2E",
  "grey19"		    => "303030",
  "grey20"		    => "333333",
  "grey21"		    => "363636",
  "grey22"		    => "383838",
  "grey23"		    => "3B3B3B",
  "grey24"		    => "3D3D3D",
  "grey25"		    => "404040",
  "grey26"		    => "424242",
  "gray27"		    => "454545",
  "gray28"		    => "474747",
  "gray29"		    => "4A4A4A",
  "gray30"		    => "4D4D4D",
  "gray31"		    => "4F4F4F",
  "gray32"		    => "525252",
  "gray33"		    => "545454",
  "gray34"		    => "575757",
  "gray35"		    => "595959",
  "gray36"		    => "5C5C5C",
  "gray37"		    => "5E5E5E",
  "gray38"		    => "616161",
  "gray39"		    => "636363",
  "gray40"		    => "666666",
  "gray41"		    => "696969",
  "gray42"		    => "6B6B6B",
  "gray43"		    => "6E6E6E",
  "gray44"		    => "707070",
  "gray45"		    => "737373",
  "gray46"		    => "757575",
  "gray47"		    => "787878",
  "gray48"		    => "7A7A7A",
  "gray49"		    => "7D7D7D",
  "gray50"		    => "7F7F7F",
  "gray51"		    => "828282",
  "gray52"		    => "858585",
  "gray53"		    => "878787",
  "gray54"		    => "8A8A8A",
  "gray55"		    => "8C8C8C",
  "gray56"		    => "8F8F8F",
  "gray57"		    => "919191",
  "gray58"		    => "949494",
  "gray59"		    => "969696",
  "gray60"		    => "999999",
  "gray61"		    => "9C9C9C",
  "gray62"		    => "9E9E9E",
  "gray63"		    => "A1A1A1",
  "gray64"		    => "A3A3A3",
  "gray65"		    => "A6A6A6",
  "gray66"		    => "A8A8A8",
  "gray67"		    => "ABABAB",
  "gray68"		    => "ADADAD",
  "gray69"		    => "B0B0B0",
  "gray70"		    => "B3B3B3",
  "gray71"		    => "B5B5B5",
  "gray72"		    => "B8B8B8",
  "grey73"		    => "BABABA",
  "gray74"		    => "BDBDBD",
  "gray75"		    => "BFBFBF",
  "gray76"		    => "C2C2C2",
  "gray77"		    => "C4C4C4",
  "gray78"		    => "C7C7C7",
  "gray79"		    => "C9C9C9",
  "gray80"		    => "CCCCCC",
  "gray81"		    => "CFCFCF",
  "gray82"		    => "D1D1D1",
  "gray83"		    => "D4D4D4",
  "gray84"		    => "D6D6D6",
  "gray85"		    => "D9D9D9",
  "gray86"		    => "DBDBDB",
  "gray87"		    => "DEDEDE",
  "gray88"		    => "E0E0E0",
  "gray89"		    => "E3E3E3",
  "gray90"		    => "E5E5E5",
  "gray91"		    => "E8E8E8",
  "gray92"		    => "EBEBEB",
  "gray93"		    => "EDEDED",
  "gray94"		    => "F0F0F0",
  "gray95"		    => "F2F2F2",
  "gray96"		    => "F5F5F5",
  "gray97"		    => "F7F7F7",
  "gray98"		    => "FAFAFA",
  "gray99"		    => "FCFCFC",
  "gray100"		    => "FFFFFF",
  "dark grey"		    => "A9A9A9",
  "darkgrey"		    => "A9A9A9",
  "dark gray"		    => "A9A9A9",
  "darkgray"		    => "A9A9A9",
  "dark blue"		    => "00008B",
  "darkblue"		    => "00008B",
  "dark cyan"		    => "008B8B",
  "darkcyan"		    => "008B8B",
  "dark magenta"	    => "8B008B",
  "darkmagenta"	            => "8B008B",
  "dark red"		    => "8B0000",
  "darkred"		    => "8B0000",
  "light green"	            => "90EE90",
  "lightgreen"	            => "90EE90",
};

sub get_rgb {
  my $self = shift;
  my $color_name = shift;
  # check to see if the $color_name is already 
  # in a hex format.
  if( $color_name =~ /^(?:#|0[xX])?[a-fA-F0-9]{6}$/ ) {
    #strip off leading "#" as in #FFFFFF and
    #strip off leading "0x" as in 0xFFFFFF
    $color_name =~ s/^(?:#|0[xX])//g;
    return uc($color_name);
  }
  my $color = COLORS->{lc($color_name)};
  return $color if $color;
  confess( "Color Name \"$color_name\" not found" );  
}

sub get_size_names {
  my $self = shift;
  my @names;
  my $cursor = $self->{'sizes'};
  do {
    push @names, $cursor->{'name'};
    $cursor = $cursor->{'next'};
  } while( $cursor );
  
  return @names;
}

sub get_size {
  my $self = shift;
  my $size_name = shift;
  my $cursor = $self->{'sizes'};
  do {
    return $cursor if $cursor->{'name'} eq $size_name;
    $cursor = $cursor->{'next'};
  } while( $cursor );
  #
  # if not in the specified names, they try the defualt
  # sizes.  This is necessary if someone specifies
  # --sizes small on the command line ... then there
  # would be no 'thumb' in the sizes array ... but
  # group pages *need* 'thumb' to be there.
  #
  $cursor = $self->{'default_sizes'};
  do {
    return $cursor if $cursor->{'name'} eq $size_name;
    $cursor = $cursor->{'next'};
  } while( $cursor );
  croak( "Image Size '$size_name' not found!");
}

sub convert {
  my $self = shift;
  my $image = shift;
  my $size = shift;

  $self->_convert( 
    $image,
    $image->{'input_image'},
    $image->get_output_image($size),
    $size,
  );
  $self->_convert( 
    $image,
    $image->{'mouse_over'},
    $image->get_output_mouse_over($size),
    $size,
  ) if $image->{'mouse_over'};
  return;
}

sub _convert {
  my $self = shift;
  my $image = shift;
  my $input_image = shift;
  my $output_image = shift;
  my $size = shift;
  
  # Not sure when $output_image would ever be a directory.  DJM says
  # this is happening on windows.  Will have to find the source of this
  # bug later.
  if( -d $output_image ) {
    Carp::confess("The output_image '$output_image' is a directory!");
  }
  
  unless ( $input_image && 
           -e "$image->{'input_image_path'}/$input_image" ) {
    warn "Ack! Image $image->{'input_image_path'}/$input_image not found for $image->{'title'}\n";
    warn "Check your config file!\n";
    exit -1;
  }
  
  my $geom = $size->{'max_width'} ."x". $size->{'max_height'};
  my $qual = $size->{'quality'};
  
  my $dir = "$image->{'output_image_path'}/$size->{'name'}";
  unless ( -d $dir ) {
    mkpath($dir, 1, 0755);
  }    
  
  # for -> print "@array"
  local $" = " ";
      
  if( defined $size->{'padded'} && $size->{'padded'} =~ /$rxTrue/ ) {
  
    #
    # since this image is padded, we need to figure out all
    # the different colors that the image will be required to be
    # displayed as.  If the image has a bg_color of black, but
    # its parent's bg_color is white, then it will need to be
    # created with both colors so it can be displayed by the 
    # parent and current image
    #
    my %colors;
    # add the current color
    $colors{$image->{'bg_color'}}++;
    
    # add the parent color
    $colors{$image->{'parent'}->{'bg_color'}}++
      if $image->{'parent'};

    # if this is the first image in a group
    # many parents could be linking to it.
    # To find all the parent and add their
    # colors.
    unless( $image->{'previous'} ) {
      # add color for "previous group" links
      my $prev = $image->previous_group();
      $colors{$prev->{'bg_color'}}++ if $prev;

      # add color for "next group" links
      my $next = $image->next_group();
      $colors{$next->{'bg_color'}}++ if $next;

      my $cursor = $image->{'parent'};
      while( $cursor ) {
        $colors{$cursor->{'bg_color'}}++;
        $cursor = $cursor->{'parent'};
      }    
    }  
    # add color for "previous image" link
    $colors{$image->{'previous'}->{'bg_color'}}++
      if $image->{'previous'};
      
    # add color for "next image" link   
    $colors{$image->{'next'}->{'bg_color'}}++
      if $image->{'next'};
    
    for my $color ( keys %colors ) {
      $color = $self->get_rgb($color);

      next unless ( (stat( "$image->{'input_image_path'}/$input_image" ))[9] >
                  (stat( "$dir/$color/$output_image" ))[9] 
                  || $self->{'overwrite'});

      unless ( -d "$dir/$color" ) {
        mkpath("$dir/$color", 1, 0755);
      }
      
      unless( -f "$dir/$output_image" ) {
        $self->resize_padded_image(
          "size" => $size,
          "input_image" => "$image->{'input_image_path'}/$input_image",
          "output_image" => "$dir/$color/$output_image",
          "color" => $color,
        );
        $self->watermark($output_image, $dir, $size);
      }
    }
#     if( -f "$dir/$output_image" ) {
#       print "deleting temporary file $dir/$output_image\n" unless $ENV{'GATEWAY_INTERFACE'} =~ /CGI/;
#       unlink( "$dir/$output_image" ) or die "Could not remove \"$dir/$output_image\": $!";    
#     }
  } elsif( $size->{'name'} eq 'original' ) {
    if($size->{'symlink'} ) { 
      return unless ( (stat( "$image->{'input_image_path'}/$input_image" ))[9] >
                    (stat( "$dir/$input_image" ))[9] 
                    || $self->{'overwrite'});
      if( $^O eq 'MSWin32' ) {
        print "Copying \"$image->{'input_image_path'}/$input_image\" to \"$dir/$input_image\"\n"
          unless $ENV{'GATEWAY_INTERFACE'} =~ /CGI/;
        File::Copy::copy( "$image->{'input_image_path'}/$input_image", "$dir/$input_image" )
          or die "Could not create: $dir/$input_image: $!";
      } else {
        print "Linking \"$image->{'input_image_path'}/$input_image\" to \"$dir/$input_image\"\n"
          unless $ENV{'GATEWAY_INTERFACE'} =~ /CGI/;
        eval {
          symlink( "$image->{'input_image_path'}/$input_image", "$dir/$input_image" )
            or warn "symlink failed: $!";
        };
      }
    } else {
      my $input_time = (stat( "$image->{'input_image_path'}/$input_image" ))[9];
      my $output_time = (stat( "$dir/$output_image" ))[9];
      if( ( $input_time > $output_time ) || $self->{'overwrite'} ) {
        my($x,$y) = Image::Size::imgsize( "$image->{'input_image_path'}/$input_image" );
        $self->resize_image(
          "size" => $size,
          "input_image" => "$image->{'input_image_path'}/$input_image",
          "output_image" => "$dir/".$image->get_output_image($size),
          "width" => $x,
          "height" => $y,
        );
      }
      return;
    }
  } else {
    return unless ( (stat( "$image->{'input_image_path'}/$input_image" ))[9] >
                  (stat( "$dir/$output_image" ))[9] 
                  || $self->{'overwrite'});

    $self->resize_image(
      "size" => $size,
      "input_image" => "$image->{'input_image_path'}/$input_image",
      "output_image" => "$dir/$output_image",
    );
    $self->watermark(
      output_image => "$dir/$output_image",
      size => $size
    );        
  }
}  

1;
