package Spidy::Tree;

use Spidy::Group;
BEGIN {
  push @Spidy::Group::ISA, __PACKAGE__;
}

use strict;
use Spidy::Link;
use Spidy::Image;
use Spidy::Comment;

use XML::Parser;
use Carp;
use File::Path;
my $TEST_FILE;
my $KEEP_PATTERNS = 0;
my $KEEP_AUTOGROUPS = 0;
my $KEEP_NAMES = 0;

sub new {
  my $package = shift;
  my $file;
  my $expat = new XML::Parser(
    Handlers => {
      Init  => \&_init,
      Start => \&_start,
      End   => \&_end,
      Char  => \&_char,
      Final => \&_final,
      ExternEnt => \&_extern_ent,
    },
    ParseParamEnt => 0,
    ErrorContext=> 4,
  );
  
  if( @_ > 1 ) {
    my %args = @_;

    my $defaults;
    if( $args{ 'group' } ) {
      eval "require $args{'group'}";
      Carp::confess "Error loading $args{'group'}: $@" if $@;      
      $defaults = $args{'group'}->default();
    } else {
      $defaults = Spidy::Group->default ();
    }

    $file                                         = $args{'file'};
    $expat->{'spidy'}->{'bundle_download'}
      = defined( $args{'bundle_download'} ) 
        ? $args{'bundle_download'} 
        : $defaults->{'bundle_download'};
      
    $expat->{'spidy'}->{'input_file_path'}
      = $args{'input_file_path'}
      || $defaults->{'input_file_path'};
      
    $expat->{'spidy'}->{'template_path'}
      = $args{'template_path'}
      || $defaults->{'template_path'};

    $expat->{'spidy'}->{'output_html'}
      = $args{'output_html'}
      || $defaults->{'output_html'};

    $expat->{'spidy'}->{'rating'}
      = $args{'rating'}
      || $defaults->{'rating'};

    $TEST_FILE                                    = $args{'test_file'};
    $expat->{'spidy_base_url'}                    = $args{'base_url'};
    $expat->{'spidy_image_url'}                   = $args{'image_url'};
    $expat->{'spidy_file_path'}
      = $args{'file_path'}
      || $defaults->{'file_path'};
    
    $expat->{'spidy_verbose'}                     = $args{'verbose'};
    $expat->{'spidy_file_path_type'} 
      = $args{'file_path_type'}
      || $defaults->{'file_path_type'};
    
    $expat->{'spidy_output_image_path'}           = $args{'output_image_path'};
    $expat->{'spidy_output_image_type'}           = $args{'output_image_type'};

    $expat->{'spidy_output_image_path_type'}
      = $args{'output_image_path_type'}
      || $defaults->{'output_image_path_type'};
      
    $expat->{'spidy_input_image_path'}            = $args{'input_image_path'};
    
    $expat->{'spidy_input_image_path_type'}
      = $args{'input_image_path_type'}
      || $defaults->{'input_image_path_type'};
      
    $expat->{'spidy_parent_url'}                  = $args{'parent_url'};

    $expat->{'spidy_image_template_file'}
      = $args{'image_template_file'}
      || $defaults->{'image_template_file'};

    $expat->{'spidy_group_template_file'}
      = $args{'group_template_file'}
      || $defaults->{'group_template_file'};

    $expat->{'spidy_bg_color'}
      = $args{'bg_color'}
      || $defaults->{'bg_color'};

    $expat->{'spidy_fg_color'}
      = $args{'fg_color'}
      || $defaults->{'fg_color'};

    $expat->{'spidy_vlink_color'}
      = $args{'vlink_color'}
      || $defaults->{'vlink_color'};

    $expat->{'spidy_link_color'}
      = $args{'link_color'}
      || $defaults->{'link_color'};

    $expat->{'spidy_group_object'}                = $args{'group'};
    $expat->{'spidy_image_object'}                = $args{'image'};
    
    $KEEP_PATTERNS                                = $args{'keep_patterns'};
    $KEEP_AUTOGROUPS                              = $args{'keep_autogroups'};
    $KEEP_NAMES                                   = $args{'keep_names'};
    
  } elsif ( @_ == 1) {
    $file = shift;
  }

  my $image_stats;
  my $contents;
  if( $file ) {
    open(FILE, $file) or die "open $file: $!";
    {
      local $/ = undef;
      $contents = <FILE>;
    }
    close FILE;
  } else {
    $contents = "<group></group>";
  }
  
  #
  # cheesy '<html>...</html>' hack really
  # just means <![CDATA[...]]>
  #
  $contents = _cdata_wrapper($contents);
  
  my ( $tree, $globals ) = $expat->parse($contents);
  
  if( $TEST_FILE ) {
    warn "File '$file' is valid\n";
    exit;
  }
  
  return wantarray ? ( $tree, $globals ) : $tree;
}

my $words = qr{(?i)(?:autogroup|node|group|image|title|shorttitle|copyright|comment|link|input_image_path|output_image_path|html_path|file_path|var|image_url)$};
my $param_map = {
  'mouseover'           => 'mouse_over',
  'columncount'         => 'column_count',
  'bgcolor'             => 'bg_color',
  'fgcolor'             => 'fg_color',
  'linkcolor'           => 'link_color',
  'vlinkcolor'          => 'vlink_color',
  'outputimagepathtype' => 'output_image_path_type',
  'outputimagepath'     => 'output_image_path',
  'outputimagetype'     => 'output_image_type',
  'inputimagepathtype'  => 'input_image_path_type',
  'inputimagepath'      => 'input_image_path',
  'filepathtype'        => 'file_path_type',
  'filepath'            => 'file_path',
  'bundledownload'      => 'bundle_download',
  'baseurl'             => 'base_url',
  'imageurl'            => 'image_url',
  'shorttitle'          => 'short_title',
};

sub _init {
  my $expat = shift;
  $expat->{'spidy_depth'} = 0;
  $expat->{'spidy_root'} = { 
    'groups' => [],
    'globals' => {},
  };
  bless $expat->{'spidy_root'}, $expat->{'spidy_group_object'} || 'Spidy::Group';
  $expat->{'spidy_group'} = undef;
  $expat->{'spidy_last_element'} = undef;
  $expat->{'spidy_autogroup_size'} = undef;
  $expat->{'spidy_var'} = undef;
  $expat->{'spidy_id'} = 0;
  $expat->{'spidy_counts'} = undef;
  $expat->{'spidy_found_first_group'} = 0;
}

sub _start {
  my ($expat, $element, %attr) = @_;
  $element = lc($element) unless $element =~ /::/;
  
  for my $key ( keys %attr ) {
    if( exists $param_map->{lc($key)} ) {
      $attr{$param_map->{lc($key)}} = $attr{$key};
      delete $attr{$key};
    } elsif( $key ne lc($key) ) {
      $attr{lc($key)} = $attr{$key};
      delete $attr{$key};
    }
  }   

  print STDERR "   "x$expat->{'spidy_depth'}, "_start $element "
    if $expat->{'spidy_verbose'};
  while(($a,$b) = each %attr) {   
    print STDERR "$a=>$b " if $expat->{'spidy_verbose'};
  }
  warn "\n" if $expat->{'spidy_verbose'};
  $expat->{'spidy_depth'}++ if $element =~ /^(?:group|autogroup|node|gallery)$/;

  my $package;
  if( $element =~ /::/ ) {
    $package = $element;
  } else {
    $package = "Spidy::$element";
    $package =~ s/\b\w/uc($&)/eg;
  }
  
  $attr{'id'} = $expat->{'spidy_id'}++;

  $expat->{'spidy_last_element'} = $element if $element =~ $words;
  if($element =~ /(?:group|autogroup|node)$/i ) {
    undef $expat->{'spidy_counts'};
    if( $element =~ /autogroup/i ) {
      $expat->{'spidy_autogroup_size'} = $attr{'size'} || 24;
      delete $attr{'size'};
      if( $element =~ /::/ ) {
        $element =~ s/Auto//;
        $package = $element;
      } else {
        $package = $expat->{'spidy_group_object'} || "Spidy::Group";
      }
    }
    if( $element =~ /group|node/i ) {
      if( $element =~ /::/ ) {
        $package = $element;
      } else {
        $package = $expat->{'spidy_group_object'} || "Spidy::Group";
      }
    }
    
    eval "require $package";
    Carp::confess "Error loading $package: $@" if $@;      
    
    if( $expat->{'spidy_found_first_group'}) {
      if( defined $expat->{'spidy_group'}->{'images'} ) {
        $expat->xpcroak("There can not have Groups and Images at the same level");
      }
      unless( $attr{'name'} =~ /\S/ ) {
        $expat->xpcroak("name attribute required for <group> tags");
      }
      if( UNIVERSAL::isa($expat->{'spidy_group'}, "Spidy::Image" ) ) {
        $expat->xpcroak("<group> tags can not be nested withing an <image> tag");
      }
      
      $attr{'template'} = $expat->{'spidy_group_template_file'}
        unless $attr{'template'};
      my $n = eval ( "new $package(\%attr)" );
      Carp::confess "Error creating new $package: $@" if $@;      
      $n->{'parent'} = $expat->{'spidy_group'};
      $n->set_defaults( $expat->{'spidy'} );

      #add group
      if( $expat->{'spidy_group'}->{'groups'} ) {
        $expat->{'spidy_group'}->{'groups'}->append($n);
      } else {
        $expat->{'spidy_group'}->{'groups'} = $n;
      }
      
      $expat->{'spidy_group'} = $n;
    } else {
      if( $element =~ /autogroup/i ) {
        $expat->xpcroak("<autogroup> tags are not allowed to be top-level tags.\n".
                        "They must be within a <group> tag.");
      }

      $expat->{'spidy_found_first_group'}++;
      
      $attr{'template'} = $expat->{'spidy_group_template_file'}
        unless $attr{'template'};

      $expat->{'spidy_group'} = eval ( "new $package(\%attr)" );
      Carp::confess "Error creating new $package: $@" if $@;

      $expat->{'spidy_group'}->{'bg_color'}
        = $attr{'bg_color'} || $expat->{'spidy_bg_color'};
      $expat->{'spidy_group'}->{'fg_color'}
        = $attr{'fg_color'} || $expat->{'spidy_fg_color'};
      $expat->{'spidy_group'}->{'link_color'}
        = $attr{'link_color'} || $expat->{'spidy_link_color'};
      $expat->{'spidy_group'}->{'vlink_color'}
        = $attr{'vlink_color'} || $expat->{'spidy_vlink_color'};
        
      $expat->{'spidy_group'}->{'base_url'}
        = $expat->{'spidy_base_url'};
      $expat->{'spidy_group'}->{'image_url'}
        = $expat->{'spidy_image_url'} || '.';

      $expat->{'spidy_group'}->set_defaults( $expat->{'spidy'} );
      
      # set one time default for the top level node.
      $expat->{'spidy_group'}->{'parent_path'} = $expat->{'spidy_parent_url'};
      $expat->{'spidy_group'}->{'output_image_path'}
        = $expat->{'spidy_output_image_path'}
        || $expat->{'spidy_file_path'};     
      $expat->{'spidy_group'}->{'output_image_path_type'}
        = $expat->{'spidy_output_image_path_type'};
      $expat->{'spidy_group'}->{'output_image_type'}
        = $expat->{'spidy_output_image_type'};
      $expat->{'spidy_group'}->{'input_image_path'}
        = $expat->{'spidy_input_image_path'};
      $expat->{'spidy_group'}->{'file_path_type'}
        = $expat->{'spidy_file_path_type'};
      $expat->{'spidy_group'}->{'file_path'}
        = $expat->{'spidy_file_path'};
      $expat->{'spidy_group'}->{'html_path'}
        = ".";
      
      #add group
      $expat->{'spidy_root'}->{'groups'} = $expat->{'spidy_group'}; 
    }
  } elsif($element eq 'image') {
    undef $expat->{'spidy_counts'};
    if( defined $expat->{'spidy_group'}->{'groups'} ) {
      $expat->xpcroak("There can not be a <group> and <image> at the same level");
    }
    unless( $attr{'name'} =~ /\S/ ) {
      $expat->xpcroak("name attribute required for <image> tags");
    }
    if( UNIVERSAL::isa($expat->{'spidy_group'}, "Spidy::Image") ) {
      $expat->xpcroak("<image> tags can not be nested withing another <image> tag");
    }

    $attr{'template'} = $expat->{'spidy_image_template_file'}
      unless $attr{'template'};
    
    $package = $expat->{'spidy_image_object'} || "Spidy::Image";
    eval "require $package";
    Carp::confess "Error loading $package: $@" if $@;      
    my $i = eval ( "new $package(\%attr)" );
    Carp::confess "Error creating new $package: $@" if $@;
    $i->{'parent'} = $expat->{'spidy_group'};
    $i->set_defaults($expat->{'spidy'});

    # add image
    if( $expat->{'spidy_group'}->{'images'} ) {
      $expat->{'spidy_group'}->{'images'}->append($i);
    } else {
      $expat->{'spidy_group'}->{'images'} = $i;
    }
    
    $expat->{'spidy_group'} = $i;
  } elsif( $element eq 'gallery' && $expat->current_element() ) {
    $expat->xpcroak("<gallery> must be the top-level outside tag" );
  } elsif($element eq 'size') {
    unless( $attr{'name'} =~ /\S/ ) {
      $expat->xpcroak("name attribute required for <size> tags");
    }
    unless( $expat->in_element('gallery') || $expat->in_element('custom-sizes') ) {
      $expat->xpcroak("You cannot use <size> inside <". $expat->current_element(). ">\n");
    }
    
    $attr{'custom'} = 1;
    my $s = eval ( "new $package(\%attr)" );
    Carp::confess "Error creating new $package: $@" if $@;

    push @{$expat->{'spidy_root'}->{'globals'}->{'sizes'}}, $s;
    
  } elsif($element eq 'comment') {
    if( ref($expat->{'spidy_group'}->{'comment'}) ) {
      $expat->{'spidy_group'}->{'comment'}->{'file'} = $attr{'file'};
      $expat->{'spidy_group'}->{'comment'}->{'prepend'} = $attr{'prepend'};
    } else {
      #set the comment object
      $expat->{'spidy_group'}->{'comment'} = eval ( "new $package(\%attr)" );
      Carp::confess "Error creating new $package: $@" if $@;
    }
    
    if( $attr{'file'} =~ /\S/ ) {
      my $file = substr($attr{'file'}, 0, 1 ) eq '/'
        ? $attr{'file'}
        : "$expat->{'spidy'}->{'input_file_path'}/$attr{'file'}";
      if( -f $file ) {
        if( $attr{'prepend'} ) {
          open FILE, $file
            or $expat->xpcroak( "Cannot open file $file: $!" );
          {
            local $/;
            $expat->{'spidy_group'}->{'comment'}->{'text'} =
              <FILE> . $expat->{'spidy_group'}->{'comment'}->{'text'};
          }
          close FILE;
        }
      } else {
        $expat->xpcroak( "File does not exist: $file" );
      }
    }
  } elsif($element eq 'copyright') {
    if( $expat->{'spidy_counts'}->{$element} ) {
      $expat->xpcroak("Only one <$element> per <group> or <image>");
    }
    $expat->{'spidy_counts'}->{$element}++;
    undef $expat->{'spidy_group'}->{'copyright'};
  } elsif($element eq 'link') {
    #add link
    push @{$expat->{'spidy_group'}->{'links'}}, eval ( "new $package(\%attr)" );
    Carp::confess "Error creating new $package: $@" if $@;
  } elsif( $element eq 'output_image_path' || $element eq 'image_url' ) {
    if( $expat->{'spidy_counts'}->{$element} ) {
      $expat->xpcroak("Only one <$element> per <group> or <image>");
    }
    $expat->{'spidy_counts'}->{$element}++;

    if( $attr{'type'} ) {
      $expat->{'spidy_group'}->{'output_image_path_type'} = $attr{'type'};
    }
  } elsif( $element eq 'input_image_path' ) {
    if( $expat->{'spidy_counts'}->{$element} ) {
      $expat->xpcroak("Only one <$element> per <group> or <image>");
    }
    $expat->{'spidy_counts'}->{$element}++;

    if( $attr{'type'} ) {
      $expat->{'spidy_group'}->{'input_image_path_type'} =  $attr{'type'} ;
    } 
  } elsif( $element eq 'file_path' || $element eq 'html_path') {
    if( $expat->{'spidy_counts'}->{$element} ) {
      $expat->xpcroak("Only one <$element> per <group> or <image>");
    }
    $expat->{'spidy_counts'}->{$element}++;

    if( $attr{'type'} ) {
      $expat->{'spidy_group'}->{'file_path_type'} =  $attr{'type'} ;
    }
  } elsif( $element eq 'var' ) {
    $expat->xpcroak("'name' attribute required for 'var' tag")
      unless $attr{'name'};
    if( $attr{'value'} ) {
      $expat->{'spidy_group'}->{'vars'}->{$attr{'name'}} = $attr{'value'};
    } else {
      if( exists $expat->{'spidy_group'}->{'vars'}->{$attr{'name'}} ) {
        delete $expat->{'spidy_group'}->{'vars'}->{$attr{'name'}};
      }
      $expat->{'spidy_var'} = $attr{'name'};
    }
  } elsif( $element eq 'title' ) {
    if( $expat->{'spidy_counts'}->{$element} ) {
      $expat->xpcroak("Only one <$element> per <group> or <image>");
    }
    $expat->{'spidy_counts'}->{$element}++;
    $expat->{'spidy_group'}->{'title'} = "";
  } elsif( $element eq 'shorttitle' || $element eq 'short_title' ) {
    if( $expat->{'spidy_counts'}->{$element} ) {
      $expat->xpcroak("Only one <$element> per <group> or <image>");
    }
    $expat->{'spidy_counts'}->{$element}++;
    $expat->{'spidy_group'}->{'short_title'} = "";
  }
  
  
  $expat->{'spidy_depth'}++ if $element =~ /^(?:image)$/;
}

sub _char {
  my ($expat, $string) = @_;
  if( $string =~ /\S/ ) {
    if( exists $expat->{'spidy_group'}->{vars} && $string =~ /\&\w+;/ ) {
      while( my( $var, $val ) = each %{$expat->{'spidy_group'}->{vars}} ) {
        $string =~ s/\&$var;/$val/sg;
      }
    }
    if( $expat->{'spidy_last_element'} eq 'title' ) {
      #
      # tag on a new line if this is a continuation from the last line
      # (_char is called for every line, and it does not contain the 
      # new line at the end of $string)
      #
      $expat->{'spidy_group'}->{'title'} .= "\n"
        if $expat->{'spidy_group'}->{'title'};
      #
      # append text to title
      #
      $expat->{'spidy_group'}->{'title'} .= $string;
    } elsif( $expat->{'spidy_last_element'} eq 'shorttitle' ) {
      #
      # tag on a new line if this is a continuation from the last line
      # (_char is called for every line, and it does not contain the 
      # new line at the end of $string)
      #
      $expat->{'spidy_group'}->{'short_title'} .= "\n"
        if $expat->{'spidy_group'}->{'short_title'};
      #
      # append text to short title
      #
      $expat->{'spidy_group'}->{'short_title'} .= $string;
    } elsif( $expat->{'spidy_last_element'} eq 'comment' ) {
      #
      # tag on a new line if this is a continuation from the last line
      # (_char is called for every line, and it does not contain the 
      # new line at the end of $string)
      #
      $expat->{'spidy_group'}->{'comment'}->{'text'} .= "\n"
        if $expat->{'spidy_group'}->{'comment'}->{'text'};
      #
      # append text to comment
      #
      $expat->{'spidy_group'}->{'comment'}->{'text'} .= $string;
    } elsif( $expat->{'spidy_last_element'} eq 'copyright' ) {
      #append text to copyright text
      $expat->{'spidy_group'}->{'copyright'} .= $string;
    } elsif( $expat->{'spidy_last_element'} eq 'link' ) {
      my $links = $expat->{'spidy_group'}->{'links'};
      #append text to links text
      $links->[-1]->{'text'} .= $string;
    } elsif( $expat->{'spidy_last_element'} eq 'html_path' ) {
      $string =~ s/^\s+//;
      $string =~ s/\s+$//;
      $expat->{'spidy_group'}->{'html_path'} =  $string  if $string =~ /\S/;
    } elsif( $expat->{'spidy_last_element'} eq 'image_url' ) {
      $string =~ s/^\s+//;
      $string =~ s/\s+$//;
      $expat->{'spidy_group'}->{'image_url'} =  $string  if $string =~ /\S/;
    } elsif( $expat->{'spidy_last_element'} eq 'file_path' ) {
      $string =~ s/^\s+//;
      $string =~ s/\s+$//;
      $expat->{'spidy_group'}->{'file_path'} =  $string  if $string =~ /\S/;
    } elsif( $expat->{'spidy_last_element'} eq 'input_image_path' ) {
      $string =~ s/^\s+//;
      $string =~ s/\s+$//;
      $expat->{'spidy_group'}->{'input_image_path'} =  $string  if $string =~ /\S/;
    } elsif( $expat->{'spidy_last_element'} eq 'output_image_path' ) {
      $string =~ s/^\s+//;
      $string =~ s/\s+$//;
      $expat->{'spidy_group'}->{'output_image_path'} =  $string  if $string =~ /\S/;
    } elsif( $expat->{'spidy_last_element'} eq 'var' ) {
      $expat->{'spidy_group'}->{vars}->{$expat->{'spidy_var'}} .= $string;
    }
  }
  warn "   "x$expat->{'spidy_depth'}, "_char $expat->{'spidy_last_element'} ##$string##\n" if $string =~ /\S/ and $expat->{'spidy_verbose'};
}

sub _end {
  my ($expat, $element) = @_;
  $element = lc($element);
  
  
  my $current = $expat->{'spidy_group'};
  my $parent;
  if ( $element =~ /(?:group|image|autogroup|node|gallery)$/ ) {
    $parent = $current->{'parent'};
    $expat->{'spidy_group'} = $parent;
    $expat->{'spidy_depth'}--;
  }

  if( $element eq 'comment' ) {
    unless( $current->{'comment'}->{'prepend'} ) {
      if( $current->{'comment'}->{'file'} =~ /\S/ ) {
        my $file = substr($current->{'comment'}->{'file'}, 0, 1 ) eq '/'
          ? $current->{'comment'}->{'file'}
          : "$expat->{'spidy'}->{'input_file_path'}/$current->{'comment'}->{'file'}";
        if( $file ) {
          if( -f $file ) {
            open FILE, $file
              or $expat->xpcroak( "Cannot open file $file: $!" );
            {
              local $/;
              $current->{'comment'}->{'text'} .= <FILE>;
            }
            close FILE;
          } else {
            $expat->xpcroak( "File does not exist: $file" );
          }
        }
      }
    }
  }

  if( $element =~ /image$/ ) {
    if($current->{'name'} =~ /^pattern:/ && !$TEST_FILE && !$KEEP_PATTERNS) {
      $current->{'name'} =~ s/^pattern://;
      my ($imgpath, $imgfile) = ($current->{'name'} =~ /^(.+\/)?([^\/]+)$/);
      unless( $imgpath =~ m!^/! ) {
        if( $imgpath  ) {
          $imgpath = "$current->{'input_image_path'}/$imgpath";
        } else {
          $imgpath = $current->{'input_image_path'};
        }
      }

      my @files = $current->find_pattern_files( $imgpath, $imgfile );

      my @new_images;
      my $counter = 1;
      my $total = @files;
      foreach my $file (@files) {
        unless( -e "$imgpath/$file" ) {
          $expat->xpcroak("Image '$imgpath/$file' found for pattern match ... but does not exist?");
        }
        my $output = $file;
        $output =~ s/\s/_/g;
        if( $current->{'output_image_type'} ) {
          $output =~ s/[.][^.]+$/.$current->{'output_image_type'}/;
        } else {
          $output =~ s/tif$/jpg/i;
        }
        my $html = $file;
        $html =~ s/\..*$/.html/;
        push( @new_images, $current->copy( 
          'name'             => $file,
          'input_image'      => $file,
          'input_image_path' => $imgpath,
          'output_image'     => $output,
          'output_html'      => $html,
          'title'            => $current->{'title'} && $total > 1 && !$KEEP_NAMES
                                ? "$current->{'title'} ($counter of $total)"
                                : $current->{'title'},
          'short_title'            => $current->{'short_title'} && $total > 1 && !$KEEP_NAMES 
                                ? "$current->{'short_title'} ($counter of $total)"
                                : $current->{'short_title'},
         ));
         $counter++;
      }
      if( @new_images ) {
        $current = $current->replace( @new_images );
        $parent->{'images'} = $current unless $current->{'previous'};
      } else {
        $expat->xpcroak("No images matching the pattern $imgfile");
      }
    } elsif ( $current->{'name'} =~ /^pattern:/ && ($TEST_FILE || $KEEP_PATTERNS) ) {
      $current->{'name'} =~ s/^pattern://;
      $current->{'is_pattern'} = 1;
    } else {
      my $imgpath;
      ($imgpath, $current->{'input_image'}) = ($current->{'name'} =~ /^(.+\/)?([^\/]+?)$/);
      if( $imgpath =~ m!^/! ) {
        $current->{'input_image_path'} =  $imgpath ;
      } else {
        #append image path
        $current->{'input_image_path'} .= "/". $imgpath if $imgpath;
      }
      $current->{'output_image'} = $current->{'output_html'} = $current->{'input_image'};
      $current->{'output_mouse_over'} = $current->{'mouse_over'};
      $current->{'output_image'} =~ s/\s/_/g;

      if( $current->{'output_image_type'} ) {
        $current->{'output_image'} =~ s/[.][^.]+$/.$current->{'output_image_type'}/;
        $current->{'output_mouse_over'} =~ s/[.][^.]+$/.$current->{'output_image_type'}/;
      } else {
        $current->{'output_image'} =~ s/tif$/jpg/i;
        $current->{'output_mouse_over'} =~ s/tif$/jpg/i;
      }
      $current->{'output_html'} =~ s/[.][^.]*$/.html/;
      $current->{'output_html'} =~ s/\s/_/g;
    }  
  }

  if( $element =~ /autogroup$/ ) {
    if( $KEEP_AUTOGROUPS || $TEST_FILE) {
      $current->{'is_autogroup'} = 1;
      $current->{'size'} = $expat->{'spidy_autogroup_size'};
    } else {
      my @images = $current->{'images'}->list();
      my $size = $expat->{'spidy_autogroup_size'};
      my $total = int(@images / $size);
      $total++ unless @images % $size == 0 || @images < $size;

      my @autogroups;
      $expat->{'spidy_depth'}++;
      for my $counter ( 1 .. $total ) {
        my @set = grep{ defined $_ } @images[$size * ($counter - 1) .. $size * ($counter) - 1];
        my $title = $current->{'title'};
        my $shorttitle = $current->{'short_title'} || $title;
        my $output_image_path = $current->{'output_image_path_type'} eq 'relative'
          ? "$current->{'output_image_path'}$counter"
          : $current->{'output_image_path'};
        my $n = $current->copy(
          'name' => "$current->{'name'}$counter",
          'output_image_path' => $output_image_path,
          'file_path' => "$current->{'file_path'}$counter",
          'html_path' => "$current->{'html_path'}$counter",
          'title' => $total > 1 && !$KEEP_NAMES ? "$title ($counter of $total)" : $title,
          'short_title' => $total > 1 && !$KEEP_NAMES ? "$shorttitle ($counter of $total)" : $shorttitle,
          'images' => Spidy::Image->link(@set),
        );
        $n->set_defaults($expat->{'spidy'});
        my $cur = $n->{'images'};
        while( $cur ) {
          $cur->{'output_image_path'} = $output_image_path;
          $cur->{'file_path'}  = "$current->{'file_path'}$counter",
          $cur->{'html_path'} = "$current->{'html_path'}$counter",
          $cur->{'image_url'} = $current->{'output_image_path_type'} eq 'relative'
                              ? "$current->{'image_url'}$counter"
                              : $current->{'image_url'},
          $cur->{'parent'} = $n;
          $cur = $cur->{'next'};
        }
        push @autogroups, $n;
      }
      $expat->{'spidy_depth'}--;

      if( @autogroups ) {
        $current = $current->replace(@autogroups);
        $parent->{'groups'} = $current unless $current->{'previous'};
      } 
    }
  } 
  warn "   "x$expat->{'spidy_depth'}, "_end $element\n" if $expat->{'spidy_verbose'};
}

sub _final { 
  my $expat = shift;
  return ( $expat->{'spidy_root'}->{'groups'}, $expat->{'spidy_root'}->{'globals'} );
}

sub _extern_ent {
  my( $expat, $file, $sysid, $pubid) = @_;

  my $include_contents;
  open( INCLUDE, $file || $sysid );# or die "Cant include $file: $!";
  {
    local $/ = undef;
    $include_contents = <INCLUDE>;
  }

  return _cdata_wrapper($include_contents);
}
    
sub _cdata_wrapper {
  my $contents = shift;
  my $text_fields = "comment|copyright|link|title|shorttitle|var";
  # we hve deprecated the use of <html>...</html> if favor or wrapping
  # CDATA tags around all text fields.  This should prevent some headaches
  if( $contents =~ s/<\s*\/?\s*html\s*>//ig ) {
    warn( <<EOF );
=========================================================================
Warning: The use of <html>...</html> blocks are now deprecated.
Now all comment, copyright, link shortTitle, and title blocks are preset
to accept any data, even html (which looks like xml to the config file
reader.)

For the xml guru's out there, this means that I now add <![CDATA[ ... ]]>
wrappers around all data inside the blocks mentioned above so the xml
parser will not parse them.
=========================================================================
EOF
  }
  
  if( $contents =~ /<\s*\/?\s*node\s*>/ig ) {
    warn( <<EOF );
=========================================================================
Warning: The keyword <node> has been deprecated in favor of <group>.
Replace all <node> tags with <group> to get rid of this warning.  There
is no functional change between <group> and <node> yet, but eventually
support for the <node> tag will be removed.
=========================================================================
EOF
  }
  

  #
  # Warning: crazy magic ahead
  #
  # First try to match something like <comment>...</comment>
  #
  while( $contents =~ /<\s*([\w:]*(?:$text_fields))(.*?)>(.*?)<\s*\/\s*(\1.*?)>/isg ) {
    #
    # upon match, capture the length of the matched string to be used 
    # in calculating the replacement offset later.
    #
    my $length = length($&);
    
    #
    # capture the open tag, the close tag, and that data between the tags
    #
    my $open = "<$1$2>";
    my $close = "</$4>";
    my $data = $3;
    
    #
    # if it looks like <var name="..." value="..."/> 
    # then the above regex will match way too much, so we need to reset
    # the matching pos to just beyond where we just matched.
    #
    if( (my $args = $2 ) =~ /\s*\/\s*$/ ) {
      pos($contents) = pos($contents) - $length +1;
      next;
    }
    if( (my $content = $3 ) =~ /-->/ ) {
      my $start_count = ( $content =~ s/<!--/<!--/g );
      my $end_count = ( $content =~ s/-->/-->/g );
      
      if( $end_count > $start_count ) {
        pos($contents) = pos($contents) - $length +1;
        next;
      }
    }
    
    #
    # Strip out any existing CDATA tags ... the xml parser will puke
    # on nested CDATA tags.
    #
    $data =~ s/<!\[CDATA\[(.*?)\]\]>/$1/ig;
    #
    # capture the postion of the last character in the string of 
    # $contents where we just matched in the while loop
    #
    my $pos = pos($contents);
    #
    # replace the part of $contents that was matched with our new CDATA hacked
    # sting.
    # We literally replace $conents from position $pos - $lenght (where pos is 
    # the end of the string and $pos - $length will be be beginning of the 
    # string, and we replace $length of it (the entire string) with the 
    # new string wrapped with CDATA.
    #
    substr($contents, $pos - $length, $length) = "$open<![CDATA[$data]]>$close";
    #
    # Now reset the pos to be the the last place in the string we just modified
    # so we do not skip anything.  We simple take the current pos and add on the
    # difference in length between the old string and the new one.
    #
    pos($contents) = $pos - $length + length("$open<![CDATA[$data]]>$close");
  }

  return $contents;
}

#
# NEEDED FOR CGI MODE!!!!!
#
sub get_group {
  my $self = shift;
  my $path = shift;

  return $self if $path eq '' || $path eq '.';
  my $cur = $self->{'groups'} || $self->{'images'};
  my $group;
  OUTER: for my $part ( split /\//, $path ) {
    next if $part eq '' || $part eq '.';
    while( $cur ) {
      if( $cur->{'name'} eq $part ) {
        $group = $cur;
        $cur = $cur->{'groups'} || $cur->{'images'};
        next OUTER;
      }
      $cur = $cur->{'next'};
    }
  }
  return $group;
}

sub recurse_bottom_up {
  my $group = shift;
  my $routine = shift;
  my @args = @_;
  my $cur = $group->{'groups'};
  while( $cur ) {
    my $return_val =  $_->recurse_bottom_up($routine, @args);
    return $return_val if $return_val;
    $cur = $cur->{'next'}
  }
  if ( ref($routine) eq 'ARRAY' ) {
    foreach my $func (@$routine) {
      no strict 'refs';
      my $ret_val = $group->$func(@args);
      return $ret_val;
    }
  } else {
    no strict 'refs';
    my $ret_val = $group->$routine(@args);
    return $ret_val;
  }
}

sub recurse_top_down {
  my $group = shift;
  my $routine = shift;
  my @args = @_;

  if ( ref($routine) eq 'ARRAY' ) {
    foreach my $func (@$routine) {
      no strict 'refs';
      my $ret_val = $group->$func(@args);
      return $ret_val if $ret_val;
    }
  } else {
    no strict 'refs';
    my $ret_val = $group->$routine(@args);
    return $ret_val if $ret_val;
  }

  return if $group->isa("Spidy::Image");

  my $groups;
  $groups = $group->{'groups'}->list() if $group->{'groups'};
  $groups = $group->{'images'}->list() if $group->{'images'};
  foreach( @$groups ) {
    my $ret_val = $_->recurse_top_down($routine, @args);
    return $ret_val if $ret_val;
  }
  return;
}

sub delete_unless {
  my $self = shift;
  my $var = shift;
  my $unless = shift;
  
  unless ( $self->{$unless} ) {
    delete $self->{$var};
  }
  return;
}

sub build_images {
  my $self = shift;
  my $gb = shift;
  my $cur = $self->{'images'};
  while( $cur ) {
      for my $size ( $gb->{'sizes'}->list() ) {
      $gb->convert($cur, $size);
    }
    $cur = $cur->{'next'};
  }
  if( $self->{'groups'} ) {
    $_->build_images($gb) for $self->{'groups'}->list();
  }
}



sub build_html {
  my $self = shift;
  my $hb = shift;
  my @sizes = @_;
  
  if( $self->{'bundle_download'} 
    && ( $self->isa("Spidy::Image") 
      || ( $self->{'images'} && $self->{'images'}->count() > 0 ) ) ) {
    my $name = $self->{'name'} || "everything";
    $name =~ s/[.][^.]+$//;
    $name =~ s/\s/_/g;
    my $files;
    if( $self->isa("Spidy::Image" ) ) {
      $files = "$self->{'input_image_path'}/$self->{'input_image'}";
      $self->{'bundle_size'} = -s $files
    } else {
      $files = join "','", 
      map { "$_->{'input_image_path'}/$_->{'input_image'}" }
      $self->{'images'}->list();
      $self->{'bundle_size'} += -s "$_->{'input_image_path'}/$_->{'input_image'}"
        for $self->{'images'}->list();
    }
    $files = "'$files'";

    $self->{'bundle_size'} /= 1_048_576; # B/MB
    $self->{'bundle_size'} = sprintf( "%6.1f", $self->{'bundle_size'});
    $self->{'bundle_name'} = $name;

    unless( -d $self->{'file_path'} ) {
      mkpath( $self->{'file_path'}, 1 );
    }

    open FILE, ">$self->{'file_path'}/$name.tar.gz" 
      or die "Counld not open $self->{'file_path'}/$name.tar.gz: $!";
    print "Creating $self->{'file_path'}/$name.tar.gz\n";
    print FILE <<EOF;
#!/usr/bin/perl
use Archive::Tar;
use Compress::Zlib;
use strict;
\$| = 1;
my \$tar = Archive::Tar->new();
for my \$path ( $files ) {
  my \$file = \$path;
  \$file =~ s/.*\\///;
  open FILE, \$path or die "Could not open \$path: \$!";
  { local \$/;
    \$tar->add_files("images/\$file");
    \$tar->add_data( "images/\$file", <FILE> );
  }
  close FILE;
}
print "Content-Type: application/x-gzip\n\n" if \$ENV{'GATEWAY_INTERFACE'};
my \$gz = gzopen(\*STDOUT, "wb");
\$gz->gzwrite( \$tar->write() );
EOF
    close FILE;
    chmod 0755, "$self->{'file_path'}/$name.tar.gz";

    open FILE, ">$self->{'file_path'}/$name.zip" 
      or die "Counld not open $self->{'file_path'}/$name.zip: $!";
    print "Creating $self->{'file_path'}/$name.zip\n";
    print FILE <<EOF;
#!/usr/bin/perl
use Archive::Zip;
use strict;
\$| = 1;
my \$zip = Archive::Zip->new();
\$zip->addDirectory( "images" );
for my \$path ( $files ) {
  my \$file = \$path;
  \$file =~ s/.*\\///;
  \$zip->addFile( \$path, "images/\$file");
}
print "Content-Type: application/x-zip\n\n" if \$ENV{'GATEWAY_INTERFACE'};
\$zip->writeToFileHandle( *STDOUT, 0 );
EOF
    close FILE;
    chmod 0755, "$self->{'file_path'}/$name.zip";
  }
  
  #
  # make sure the default size actuall exists
  # if it does not, then just default to the
  # second size (skipping the first size of
  # 'thumb'
  #
  my $default_size;
  for my $size ( @sizes ) {
    $default_size = $hb->{'default_size'}
      if $size->{'name'} eq $hb->{'default_size'};
  }
  
  unless( $default_size ) {
    if( $sizes[0] && $sizes[0]->{'next'} ) {
      $default_size = $sizes[0]->{'next'}->{'name'};
    }
  }
      
  for my $size ( @sizes ) {
    next if $size->{name} eq 'thumb';
    unless( -d $self->{'file_path'} ) {
      mkpath( $self->{'file_path'}, 1 );
    }
    my $size_name = $size->{'name'};
    open FILE, ">".$self->{'file_path'}."/${size_name}_".$self->{'output_html'};
    print "Creating $self->{'file_path'}/${size_name}_$self->{'output_html'}\n";
    print FILE $hb->get_html($self, $size);
    close FILE;
    if( not $self->isa("Spidy::Image") ) {
      if( $size_name eq $default_size && !-e $self->{'file_path'}."/index.html" ) {
        print "Creating $self->{'file_path'}/index.html\n";
        if( $^O eq 'MSWin32' ) {
          require File::Copy;
          File::Copy::copy( "${size_name}_index.html", $self->{'file_path'}."/index.html" );
        } else {
          eval {
            symlink( "${size_name}_index.html", $self->{'file_path'}."/index.html" )
              or warn "symlink failed: $!";
          };
        }
      }
    }
  }
  if( not $self->isa("Spidy::Image") ) {
    if( $self->{'groups'} ) {
      $_->build_html($hb, @sizes) for $self->{'groups'}->list();
    }
    if( $self->{'images'} ) {
      $_->build_html($hb, @sizes) for $self->{'images'}->list();
    }
  }
  return;
}

sub get_group_from_path {
  my $group = shift;
  return if $group->isa("Spidy::Image");
  my $lookup_path = shift;
  return $group if $group->{'html_path'} eq $lookup_path;
  return;
}

sub find_pattern_files {
  my $self = shift;
  my $path = shift;
  my $pattern = shift;

  $pattern =~ /\[(\d+-\d+)\]/;
  my ($from,$to) = split /-/,$1;
  my @matches = split /\|/, $pattern;
  $_ =~ s/\[(?:\d+-\d+)\]/'(?:'.join('|',($from .. $to)).')'/e for @matches;
  opendir(DIR, $path) or die "opendir $path: $!";
  # get the files in alphabetical order, and skip the directories
  my @files = sort grep { -f "$path/$_" } readdir(DIR);
  closedir(DIR);

  #
  # This code is a kludge.  Basically I am trying to
  # order the files based on the regex order, rather
  # than the file order from the readdir.
  #
  # This code needs to be replaced but I dont have
  # time to research a better algorithm.
  #
  my @newfiles;
  for my $regex ( @matches ) {
    next unless $regex =~ /\S/;
    my $rx = qr{$regex};
    for my $file ( @files ) {
      push @newfiles, $file
        if $file =~ /$rx/;
    }
  }
  return @newfiles;
}
1;  
