package SpidyTest;

use strict;

use lib "t/lib";

use vars '%URL_CACHE', '$TEST_URL';

BEGIN {
  $|++; #autoflush output buffers
  
  $TEST_URL = 1;
  eval {
    require LWP::UserAgent;
  };
  $TEST_URL = 0 if $@;
  eval {
    require HTML::TreeBuilder;
  };
  $TEST_URL = 0 if $@;
} 

use Test::More;

use File::Basename;
use File::Find;
use File::Spec;

sub skipping {
  my $count = shift;
  warn "\nSkipping $count tests.\nLWP::UserAgent and HTML::TreeBuilder are needed for more complete testing.\n";
}

sub check_url {
  my $file = shift;
  return unless $file;

  # remove groups of "stuff/../" from the path
  # so t/gallery/set1/../medium_index.html will turn into
  # t/gallery/medium_index.html
  while( $file =~ /[.][.]\// ) {
    $file =~ s/[^\/]+\/[.][.]\///;
  }
  
  # clean up the "./" in the middle path.  This is needtd
  # to make sure the path names are unique for the cache.
  
  $file =~ s/[.]\///g;
  
  
  #
  # cache the file name so we do not end up checking the
  # same file over and over.
  #
  return if $URL_CACHE{$file}++;
  
  my $ua = LWP::UserAgent->new();
   
  # make a request using the file:// protocol since we are not
  # using a web server for testing.
  $file =~ s|^file://||;
  my $req = HTTP::Request->new( 'GET', "file://$file" );
  my $res = $ua->request($req);
  
  #
  #  the file must be greater than 0 length and the http status must be
  #  successful
  #
  my $length = $res->content_length();
  ok( $file !~  m://: && defined($length) && $length > 0 && $res->is_success, "file://$file" );
  
  # dont try to recurse into graphics
  if( $res->content_type =~ /text\/html/ ) {
    
    my $root = HTML::TreeBuilder->new_from_content( $res->content() );
    # get all image tags
    my @images = $root->find_by_tag_name('img');
    my $dir = dirname( $file );
    
    # for all images, recurse to check the image
    for my $i ( @images ) {
      my $src = $i->attr('src');
      next if $src =~ /^http/;
      my $url = $src =~ /^file:/ ? $src : "$dir/$src";
      check_url( $url );
    }
    
    # get all anchor tags
    my @anchors = $root->find_by_tag_name('a');
    
    # for all anchors, recurse to check the page
    for my $a ( @anchors ) {
      my $href = $a->attr('href');
      # skip non-local urls
      next if $href =~ /^http/;
      next unless $href =~ /\S/;
      my $url = $href =~ /^file:/ ? $href : "$dir/$href";
      check_url( $url );
    }
    
    # delete the tree as required by HTML::TreeBuilder
    $root = $root->delete();
  }
}

sub which {
  my $prog = shift;
  my $uc_prog = uc($prog);
  
  my $binprog = $^O eq 'MSWin32' ? "$prog.exe" : $prog;
  
  my $bin = $ENV{"IM_$uc_prog"};
  unless( defined($bin) && -f $bin ) {
    my @paths = File::Spec->path();
    foreach my $path (@paths) {
      if( -e "$path/$binprog" ) {
        $bin = "$path/$binprog";
        last;
      }
    }
  }

  unless( -f $bin ) {
    die <<EOF;
I could not find the Image Magick '$prog' program which is necessary to
create all the various size images.  You can specify where the program is
by setting the IM_$uc_prog environment variable or by using the 
--$prog option if you are using the 'spider' program.

If you do not have Image Magick installed, then you can get it from
http://www.imagemagick.org
EOF
  }
  return $bin;
}

sub count {
  my $path = shift;
  my $expected = shift;
  my $count;
  File::Find::find( sub { 
      $count++ 
        if -f "$File::Find::dir/$_" 
          || -l "$File::Find::dir/$_"
    },
    $path,
  );

  # there should *only* be the expected count of
  # files installed for this gallery
  is( $count, $expected, "checking file count" );
}

sub check_files {
  my $files = shift;
  my $dirs = shift;
  # there should be 94 files and directories to test in the 
  # next two block

  for my $dir ( @$dirs ) {
    ok( -d $dir, "checking directory $dir");
  }

  for my $file ( @$files ) {
    ok( (-f $file && -s $file) || -l $file, "checking file $file");
  }
}

sub cleanup {
  my $files = shift;
  my $dirs = shift;
  for my $file ( grep { -f $_ || -l $_  } @$files ) {
    unlink $file or die "Could not remove $file: $!";
  }
  for my $dir ( grep { -d $_ } @$dirs ) {
    rmdir $dir or die "Could not remove $dir: $!";
  }
  undef %URL_CACHE;
}
  
sub check_image_count {
  my $file = shift;
  my @counts = @_;
  
  open( FILE, $file ) or warn "Could not open $file: $!";
  local $/;
  my $content = <FILE>;
  for my $count ( @counts ) {
    ok( $content =~ s/\b$count images//, "Checking for \"$count images\" text" );
  }
  close FILE;
}

1;
