#
# Copyright (C) 2005, 2006, 2007, 2008 Nikolas Zimmermann <zimmermann@kde.org>
# Copyright (C) 2006 Anders Carlsson <andersca@mac.com>
# Copyright (C) 2006, 2007 Samuel Weinig <sam@webkit.org>
# Copyright (C) 2006 Alexey Proskuryakov <ap@webkit.org>
# Copyright (C) 2006-2010, 2013-2016 Apple Inc. All rights reserved.
# Copyright (C) 2009 Cameron McCormack <cam@mcc.id.au>
# Copyright (C) Research In Motion Limited 2010. All rights reserved.
# Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies)
# Copyright (C) 2011 Patrick Gansterer <paroga@webkit.org>
# Copyright (C) 2012 Ericsson AB. All rights reserved.
# Copyright (C) 2007, 2008, 2009, 2012 Google Inc.
# Copyright (C) 2013 Samsung Electronics. All rights reserved.
# Copyright (C) 2015, 2016 Canon Inc. All rights reserved.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public License
# along with this library; see the file COPYING.LIB.  If not, write to
# the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA 02110-1301, USA.


package CodeGeneratorJS;

use strict;
use constant FileNamePrefix => "JS";
use Carp qw<longmess>;
use Data::Dumper;
use Hasher;

my $codeGenerator;
my $writeDependencies;

my @headerContentHeader = ();
my @headerContent = ();
my %headerIncludes = ();
my %headerTrailingIncludes = ();

my @implContentHeader = ();
my @implContent = ();
my %implIncludes = ();
my @depsContent = ();
my $numCachedAttributes = 0;
my $currentCachedAttribute = 0;

my $beginAppleCopyrightForHeaderFiles = <<END;
// ------- Begin Apple Copyright -------
/*
 * Copyright (C) 2008, Apple Inc. All rights reserved.
 *
 * Permission is granted by Apple to use this file to the extent
 * necessary to relink with LGPL WebKit files.
 *
 * No license or rights are granted by Apple expressly or by
 * implication, estoppel, or otherwise, to Apple patents and
 * trademarks. For the sake of clarity, no license or rights are
 * granted by Apple expressly or by implication, estoppel, or otherwise,
 * under any Apple patents, copyrights and trademarks to underlying
 * implementations of any application programming interfaces (APIs)
 * or to any functionality that is invoked by calling any API.
 */

END
my $beginAppleCopyrightForSourceFiles = <<END;
// ------- Begin Apple Copyright -------
/*
 * Copyright (C) 2008, Apple Inc. All rights reserved.
 *
 * No license or rights are granted by Apple expressly or by implication,
 * estoppel, or otherwise, to Apple copyrights, patents, trademarks, trade
 * secrets or other rights.
 */

END
my $endAppleCopyright   = <<END;
// ------- End Apple Copyright   -------

END

# Default .h template
my $headerTemplate = << "EOF";
/*
    This file is part of the WebKit open source project.
    This file has been generated by generate-bindings.pl. DO NOT MODIFY!

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
EOF

sub assert
{
    my $message = shift;

    my $mess = longmess();
    print Dumper($mess);

    die $message;
}

# Default constructor
sub new
{
    my $object = shift;
    my $reference = { };

    $codeGenerator = shift;
    $writeDependencies = shift;

    bless($reference, $object);
    return $reference;
}

sub GenerateDictionary
{
    my ($object, $dictionary, $enumerations) = @_;

    my $className = GetDictionaryClassName($dictionary->name);
    $object->GenerateDictionaryHeader($dictionary, $className, $enumerations);
    $object->GenerateDictionaryImplementation($dictionary, $className, $enumerations);
}

sub GenerateInterface
{
    my ($object, $interface, $defines, $enumerations, $dictionaries) = @_;

    $codeGenerator->LinkOverloadedFunctions($interface);
    AddStringifierOperationIfNeeded($interface);

    if ($interface->isCallback) {
        $object->GenerateCallbackHeader($interface, $enumerations, $dictionaries);
        $object->GenerateCallbackImplementation($interface, $enumerations, $dictionaries);
    } else {
        $object->GenerateHeader($interface, $enumerations, $dictionaries);
        $object->GenerateImplementation($interface, $enumerations, $dictionaries);
    }
}

sub AddStringifierOperationIfNeeded
{
    my $interface = shift;

    foreach my $attribute (@{$interface->attributes}) {
        next unless $attribute->isStringifier;

        my $stringifier = domFunction->new();
        $stringifier->signature(domSignature->new());
        my $extendedAttributeList = {};
        $extendedAttributeList->{ImplementedAs} = $attribute->signature->name;
        $stringifier->signature->extendedAttributes($extendedAttributeList);
        $stringifier->signature->name("toString");
        die "stringifier can only be used on attributes of String types" unless $codeGenerator->IsStringType($attribute->signature->type);
        
        # FIXME: This should use IDLParser's cloneType.
        my $type = domType->new();
        $type->name($attribute->signature->type);

        $stringifier->signature->idlType($type);
        $stringifier->signature->type($type->name);

        push(@{$interface->functions}, $stringifier);
        last;
    }
}

sub EventHandlerAttributeEventName
{
    my $attribute = shift;
    my $eventType = $attribute->signature->extendedAttributes->{ImplementedAs} || $attribute->signature->name;

    # Remove the "on" prefix.
    $eventType = substr($eventType, 2);

    return "eventNames().${eventType}Event";
}

sub GetParentClassName
{
    my $interface = shift;

    return $interface->extendedAttributes->{JSLegacyParent} if $interface->extendedAttributes->{JSLegacyParent};
    return "JSDOMObject" unless NeedsImplementationClass($interface);
    return "JSDOMWrapper<" . GetImplClassName($interface->name) . ">" unless $interface->parent;
    return "JS" . $interface->parent;
}

sub GetCallbackClassName
{
    my $className = shift;

    return "JS$className";
}

sub GetJSCallbackDataType
{
    my $callbackInterface = shift;

    return $callbackInterface->extendedAttributes->{IsWeakCallback} ? "JSCallbackDataWeak" : "JSCallbackDataStrong";
}

sub AddIncludesForTypeInImpl
{
    my $type = shift;
    my $isCallback = @_ ? shift : 0;
    
    AddIncludesForType($type, $isCallback, \%implIncludes);
}

sub AddIncludesForTypeInHeader
{
    my $type = shift;
    my $isCallback = @_ ? shift : 0;
    
    AddIncludesForType($type, $isCallback, \%headerIncludes);
}

sub GetExportMacroForJSClass
{
    my $interface = shift;

    return $interface->extendedAttributes->{ExportMacro} . " " if $interface->extendedAttributes->{ExportMacro};
    return "";
}

sub AddIncludesForType
{
    my $type = shift;
    my $isCallback = shift;
    my $includesRef = shift;

    return if $codeGenerator->SkipIncludeHeader($type);
    
    # When we're finished with the one-file-per-class reorganization, we won't need these special cases.
    if ($isCallback && $codeGenerator->IsWrapperType($type)) {
        $includesRef->{"JS${type}.h"} = 1;
    } elsif ($codeGenerator->IsSequenceOrFrozenArrayType($type)) {
        my $innerType = $codeGenerator->GetSequenceOrFrozenArrayInnerType($type);
        if ($codeGenerator->IsRefPtrType($innerType)) {
            $includesRef->{"JS${innerType}.h"} = 1;
            $includesRef->{"${innerType}.h"} = 1;
        }
        $includesRef->{"<runtime/JSArray.h>"} = 1;
    } else {
        # default, include the same named file
        $includesRef->{"${type}.h"} = 1;
    }
}

sub AddToImplIncludesForIDLType
{
    my ($idlType, $conditional) = @_;

    return if $codeGenerator->SkipIncludeHeader($idlType->name);

    if ($idlType->isUnion) {
        AddToImplIncludes("<wtf/Variant.h>", $conditional);

        foreach my $memberType (@{$idlType->subtypes}) {
            AddToImplIncludesForIDLType($memberType, $conditional);
        }

        return;
    }

    if ($codeGenerator->IsSequenceOrFrozenArrayType($idlType->name)) {
        AddToImplIncludesForIDLType(@{$idlType->subtypes}[0], $conditional);
        return;
    }

    if ($codeGenerator->IsExternalDictionaryType($idlType->name)) {
        AddToImplIncludes("JS" . $idlType->name . ".h", $conditional);
        return;
    }

    if ($codeGenerator->IsWrapperType($idlType->name)) {
        AddToImplIncludes("JS" . $idlType->name . ".h", $conditional);
        return;
    }
}

sub AddToImplIncludes
{
    my $header = shift;
    my $conditional = shift;

    if (not $conditional) {
        $implIncludes{$header} = 1;
    } elsif (not exists($implIncludes{$header})) {
        $implIncludes{$header} = $conditional;
    } else {
        my $oldValue = $implIncludes{$header};
        $implIncludes{$header} = "$oldValue|$conditional" if $oldValue ne 1;
    }
}

sub IsReadonly
{
    my $attribute = shift;
    return $attribute->isReadOnly && !$attribute->signature->extendedAttributes->{Replaceable} && !$attribute->signature->extendedAttributes->{PutForwards};
}

sub AddClassForwardIfNeeded
{
    my $interfaceName = shift;

    # SVGAnimatedLength/Number/etc. are not classes so they can't be forward declared as classes.
    return if $codeGenerator->IsSVGAnimatedType($interfaceName);

    return if $codeGenerator->IsTypedArrayType($interfaceName);
    return if $interfaceName eq "BufferSource";

    push(@headerContent, "class $interfaceName;\n\n");
}

sub GetGenerateIsReachable
{
    my $interface = shift;
    return $interface->extendedAttributes->{GenerateIsReachable};
}

sub GetCustomIsReachable
{
    my $interface = shift;
    return $interface->extendedAttributes->{CustomIsReachable};
}

sub IsDOMGlobalObject
{
    my $interface = shift;
    return $interface->name eq "DOMWindow" || $codeGenerator->InheritsInterface($interface, "WorkerGlobalScope") || $interface->name eq "TestGlobalObject";
}

sub ShouldUseGlobalObjectPrototype
{
    my $interface = shift;

    # For workers, the global object is a DedicatedWorkerGlobalScope.
    return 0 if $interface->name eq "WorkerGlobalScope";

    return IsDOMGlobalObject($interface);
}

sub GenerateGetOwnPropertySlotBody
{
    my ($interface, $className, $inlined) = @_;

    my $namespaceMaybe = ($inlined ? "JSC::" : "");
    my $namedGetterFunction = GetNamedGetterFunction($interface);
    my $indexedGetterFunction = GetIndexedGetterFunction($interface);

    my @getOwnPropertySlotImpl = ();

    my $ownPropertyCheck = sub {
        push(@getOwnPropertySlotImpl, "    if (Base::getOwnPropertySlot(thisObject, state, propertyName, slot))\n");
        push(@getOwnPropertySlotImpl, "        return true;\n");
    };

    # FIXME: As per the Web IDL specification, the prototype check is supposed to skip "named properties objects":
    # https://heycam.github.io/webidl/#dfn-named-property-visibility
    # https://heycam.github.io/webidl/#dfn-named-properties-object
    my $prototypeCheck = sub {
        push(@getOwnPropertySlotImpl, "    ${namespaceMaybe}JSValue proto = thisObject->getPrototypeDirect();\n");
        push(@getOwnPropertySlotImpl, "    if (proto.isObject() && jsCast<${namespaceMaybe}JSObject*>(proto)->hasProperty(state, propertyName))\n");
        push(@getOwnPropertySlotImpl, "        return false;\n\n");
    };

    if ($indexedGetterFunction) {
        push(@getOwnPropertySlotImpl, "    Optional<uint32_t> optionalIndex = parseIndex(propertyName);\n");

        # If the item function returns a string then we let the TreatReturnedNullStringAs handle the cases
        # where the index is out of range.
        if ($indexedGetterFunction->signature->type eq "DOMString") {
            push(@getOwnPropertySlotImpl, "    if (optionalIndex) {\n");
        } else {
            push(@getOwnPropertySlotImpl, "    if (optionalIndex && optionalIndex.value() < thisObject->wrapped().length()) {\n");
        }
        push(@getOwnPropertySlotImpl, "        unsigned index = optionalIndex.value();\n");
        # Assume that if there's a setter, the index will be writable
        if ($interface->extendedAttributes->{CustomIndexedSetter}) {
            push(@getOwnPropertySlotImpl, "        unsigned attributes = 0;\n");
        } else {
            push(@getOwnPropertySlotImpl, "        unsigned attributes = ${namespaceMaybe}ReadOnly;\n");
        }
        push(@getOwnPropertySlotImpl, "        slot.setValue(thisObject, attributes, " . GetIndexedGetterExpression($indexedGetterFunction) . ");\n");
        push(@getOwnPropertySlotImpl, "        return true;\n");
        push(@getOwnPropertySlotImpl, "    }\n");
    }

    my $hasNamedGetter = $namedGetterFunction || $interface->extendedAttributes->{CustomNamedGetter};
    if ($hasNamedGetter) {
        if (!$interface->extendedAttributes->{OverrideBuiltins}) {
            &$ownPropertyCheck();
            &$prototypeCheck();
        }

        # The "thisObject->classInfo() == info()" check is to make sure we use the subclass' named getter
        # instead of the base class one when possible.
        if ($indexedGetterFunction) {
            # Indexing an object with an integer that is not a supported property index should not call the named property getter.
            # https://heycam.github.io/webidl/#idl-indexed-properties
            push(@getOwnPropertySlotImpl, "    if (!optionalIndex && thisObject->classInfo() == info()) {\n");
        } else {
            push(@getOwnPropertySlotImpl, "    if (thisObject->classInfo() == info()) {\n");
        }
        push(@getOwnPropertySlotImpl, "        JSValue value;\n");
        push(@getOwnPropertySlotImpl, "        if (thisObject->nameGetter(state, propertyName, value)) {\n");
        push(@getOwnPropertySlotImpl, "            slot.setValue(thisObject, ReadOnly | DontEnum, value);\n");
        push(@getOwnPropertySlotImpl, "            return true;\n");
        push(@getOwnPropertySlotImpl, "        }\n");
        push(@getOwnPropertySlotImpl, "    }\n");
        if ($inlined) {
            $headerIncludes{"wtf/text/AtomicString.h"} = 1;
        } else {
            $implIncludes{"wtf/text/AtomicString.h"} = 1;
        }
    }

    if ($interface->extendedAttributes->{JSCustomGetOwnPropertySlotAndDescriptor}) {
        push(@getOwnPropertySlotImpl, "    if (thisObject->getOwnPropertySlotDelegate(state, propertyName, slot))\n");
        push(@getOwnPropertySlotImpl, "        return true;\n");
    }

    if (!$hasNamedGetter || $interface->extendedAttributes->{OverrideBuiltins}) {
        &$ownPropertyCheck();
    }

    push(@getOwnPropertySlotImpl, "    return false;\n");

    return @getOwnPropertySlotImpl;
}

sub GenerateHeaderContentHeader
{
    my $interface = shift;
    my $className = "JS" . $interface->name;

    my @headerContentHeader;
    if ($interface->extendedAttributes->{AppleCopyright}) {
        @headerContentHeader = split("\r", $beginAppleCopyrightForHeaderFiles);
    } else {
        @headerContentHeader = split("\r", $headerTemplate);
    }

    push(@headerContentHeader, "\n#pragma once\n\n");

    my $conditionalString = $codeGenerator->GenerateConditionalString($interface);
    push(@headerContentHeader, "#if ${conditionalString}\n\n") if $conditionalString;
    return @headerContentHeader;
}

sub GenerateImplementationContentHeader
{
    my $interface = shift;
    my $className = "JS" . $interface->name;

    my @implContentHeader;
    if ($interface->extendedAttributes->{AppleCopyright}) {
        @implContentHeader = split("\r", $beginAppleCopyrightForSourceFiles);
    } else {
        @implContentHeader = split("\r", $headerTemplate);
    }

    push(@implContentHeader, "\n#include \"config.h\"\n");
    my $conditionalString = $codeGenerator->GenerateConditionalString($interface);
    push(@implContentHeader, "\n#if ${conditionalString}\n\n") if $conditionalString;
    push(@implContentHeader, "#include \"$className.h\"\n\n");
    return @implContentHeader;
}

sub NeedsImplementationClass
{
    my ($interface) = @_;

    return 0 if $interface->extendedAttributes->{JSBuiltin};
    return 1;
}

sub ShouldGenerateToWrapped
{
    my ($hasParent, $interface) = @_;

    return 0 if not NeedsImplementationClass($interface);
    return 1 if !$hasParent or $interface->extendedAttributes->{JSGenerateToNativeObject};
    return 1 if $interface->parent && $interface->parent eq "EventTarget";
    return 0;
}

sub ShouldGenerateWrapperOwnerCode
{
    my ($hasParent, $interface) = @_;

    return 0 if not NeedsImplementationClass($interface);
    return 1 if !$hasParent;
    return 1 if GetGenerateIsReachable($interface);
    return 1 if GetCustomIsReachable($interface);
    return 1 if $interface->extendedAttributes->{JSCustomFinalize};
    return 1 if $codeGenerator->InheritsExtendedAttribute($interface, "ActiveDOMObject");
    return 0;
}

sub ShouldGenerateToJSDeclaration
{
    my ($hasParent, $interface) = @_;

    return 0 if ($interface->extendedAttributes->{SuppressToJSObject});
    return 0 if not NeedsImplementationClass($interface);
    return 0 if $interface->extendedAttributes->{CustomProxyToJSObject};
    return 1 if (!$hasParent or $interface->extendedAttributes->{JSGenerateToJSObject} or $interface->extendedAttributes->{CustomToJSObject});
    return 1 if $interface->parent && $interface->parent eq "EventTarget";
    return 1 if $interface->extendedAttributes->{Constructor} or $interface->extendedAttributes->{NamedConstructor};
    return 0;
}

sub ShouldGenerateToJSImplementation
{
    my ($hasParent, $interface) = @_;

    return 0 if not ShouldGenerateToJSDeclaration($hasParent, $interface);
    return 1 if not $interface->extendedAttributes->{CustomToJSObject};
    return 0;
}

sub GetAttributeGetterName
{
    my ($interface, $className, $attribute) = @_;

    return $codeGenerator->WK_lcfirst($className) . "Constructor" . $codeGenerator->WK_ucfirst($attribute->signature->name) if $attribute->isStatic;
    return GetJSBuiltinFunctionName($className, $attribute) if IsJSBuiltin($interface, $attribute);
    return "js" . $interface->name . $codeGenerator->WK_ucfirst($attribute->signature->name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
}

sub GetAttributeSetterName
{
    my ($interface, $className, $attribute) = @_;

    return "set" . $codeGenerator->WK_ucfirst($className) . "Constructor" . $codeGenerator->WK_ucfirst($attribute->signature->name) if $attribute->isStatic;
    return "set" . $codeGenerator->WK_ucfirst(GetJSBuiltinFunctionName($className, $attribute)) if IsJSBuiltin($interface, $attribute);
    return "setJS" . $interface->name . $codeGenerator->WK_ucfirst($attribute->signature->name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
}

sub GetFunctionName
{
    my ($interface, $className, $function) = @_;

    return GetJSBuiltinFunctionName($className, $function) if IsJSBuiltin($interface, $function);

    my $functionName = $function->signature->name;
    $functionName = "SymbolIterator" if $functionName eq "[Symbol.Iterator]";

    my $kind = $function->isStatic ? "Constructor" : (OperationShouldBeOnInstance($interface, $function) ? "Instance" : "Prototype");
    return $codeGenerator->WK_lcfirst($className) . $kind . "Function" . $codeGenerator->WK_ucfirst($functionName);
}

sub GetSpecialAccessorFunctionForType
{
    my $interface = shift;
    my $special = shift;
    my $firstParameterType = shift;
    my $numberOfParameters = shift;

    foreach my $function (@{$interface->functions}, @{$interface->anonymousFunctions}) {
        my $specials = $function->signature->specials;
        my $specialExists = grep { $_ eq $special } @$specials;
        my $parameters = $function->parameters;
        if ($specialExists and scalar(@$parameters) == $numberOfParameters and $parameters->[0]->type eq $firstParameterType) {
            return $function;
        }
    }

    return 0;
}

sub HasComplexGetOwnProperty
{
    my $interface = shift;
    return $interface->extendedAttributes->{CheckSecurity}
        || IsDOMGlobalObject($interface)
        || InstanceOverridesGetOwnPropertySlot($interface);
}

sub InterfaceRequiresAttributesOnInstance
{
    my $interface = shift;
    my $interfaceName = $interface->name;

    # FIXME: All these return 1 if ... should ideally be removed.
    # Some of them are unavoidable due to DOM weirdness, in which case we should
    # add an IDL attribute for them.

    # FIXME: We should be able to drop this once <rdar://problem/24466097> is fixed.
    return 1 if $interface->isException;

    # FIXME: Add support for [PrimaryGlobal] / [Global].
    return 1 if IsDOMGlobalObject($interface) && $interface->name ne "WorkerGlobalScope";

    return 0;
}

sub AttributeShouldBeOnInstance
{
    my $interface = shift;
    my $attribute = shift;

    # FIXME: The bindings generator does not support putting runtime-enabled attributes on the instance yet (except for global objects).
    return 0 if $attribute->signature->extendedAttributes->{EnabledAtRuntime} && !IsDOMGlobalObject($interface);

    return 1 if InterfaceRequiresAttributesOnInstance($interface);
    return 1 if $attribute->signature->type =~ /Constructor$/;

    # [Unforgeable] attributes should be on the instance.
    # https://heycam.github.io/webidl/#Unforgeable
    return 1 if IsUnforgeable($interface, $attribute);

    if ($interface->extendedAttributes->{CheckSecurity}) {
        return 0 if $attribute->signature->extendedAttributes->{DoNotCheckSecurity};
        return 0 if $attribute->signature->extendedAttributes->{DoNotCheckSecurityOnGetter};
        return 1;
    }

    return 0;
}

# https://heycam.github.io/webidl/#es-operations
sub OperationShouldBeOnInstance
{
    my $interface = shift;
    my $function = shift;

    # FIXME: Add support for [PrimaryGlobal] / [Global].
    return 1 if IsDOMGlobalObject($interface) && $interface->name ne "WorkerGlobalScope";

    # FIXME: The bindings generator does not support putting runtime-enabled operations on the instance yet (except for global objects).
    return 0 if $function->signature->extendedAttributes->{EnabledAtRuntime};

    # [Unforgeable] operations should be on the instance. https://heycam.github.io/webidl/#Unforgeable
    return 1 if IsUnforgeable($interface, $function);

    return 0;
}

sub GetJSCAttributesForAttribute
{
    my $interface = shift;
    my $attribute = shift;

    my @specials = ();
    push(@specials, "DontDelete") if IsUnforgeable($interface, $attribute);

    # As per Web IDL specification, constructor properties on the ECMAScript global object should not be enumerable.
    my $is_global_constructor = $attribute->signature->type =~ /Constructor$/;
    push(@specials, "DontEnum") if ($attribute->signature->extendedAttributes->{NotEnumerable} || $is_global_constructor);
    push(@specials, "ReadOnly") if IsReadonly($attribute);
    push(@specials, "CustomAccessor") unless $is_global_constructor or IsJSBuiltin($interface, $attribute);
    push(@specials, "DOMJITAttribute") if $attribute->signature->extendedAttributes->{"DOMJIT"};
    push(@specials, "Accessor | Builtin") if  IsJSBuiltin($interface, $attribute);
    return (@specials > 0) ? join(" | ", @specials) : "0";
}

sub GetIndexedGetterFunction
{
    my $interface = shift;
    return GetSpecialAccessorFunctionForType($interface, "getter", "unsigned long", 1);
}

sub GetNamedGetterFunction
{
    my $interface = shift;
    return GetSpecialAccessorFunctionForType($interface, "getter", "DOMString", 1);
}

sub InstanceFunctionCount
{
    my $interface = shift;
    my $count = 0;

    foreach my $function (@{$interface->functions}) {
        $count++ if OperationShouldBeOnInstance($interface, $function);
    }

    return $count;
}

sub PrototypeFunctionCount
{
    my $interface = shift;
    my $count = 0;

    foreach my $function (@{$interface->functions}) {
        $count++ if !$function->isStatic && !OperationShouldBeOnInstance($interface, $function);
    }

    $count += scalar @{$interface->iterable->functions} if $interface->iterable;
    $count += scalar @{$interface->serializable->functions} if $interface->serializable;

    return $count;
}

sub InstancePropertyCount
{
    my $interface = shift;
    my $count = 0;
    foreach my $attribute (@{$interface->attributes}) {
        $count++ if AttributeShouldBeOnInstance($interface, $attribute);
    }
    $count += InstanceFunctionCount($interface);
    return $count;
}

sub PrototypePropertyCount
{
    my $interface = shift;
    my $count = 0;
    foreach my $attribute (@{$interface->attributes}) {
        $count++ if !AttributeShouldBeOnInstance($interface, $attribute);
    }
    $count += PrototypeFunctionCount($interface);
    $count++ if NeedsConstructorProperty($interface);
    return $count;
}

sub InstanceOverridesGetOwnPropertySlot
{
    my $interface = shift;
    return $interface->extendedAttributes->{CustomGetOwnPropertySlot}
        || $interface->extendedAttributes->{CustomNamedGetter}
        || $interface->extendedAttributes->{JSCustomGetOwnPropertySlotAndDescriptor}
        || GetIndexedGetterFunction($interface)
        || GetNamedGetterFunction($interface);
}

sub PrototypeHasStaticPropertyTable
{
    my $interface = shift;
    my $numConstants = @{$interface->constants};
    return $numConstants > 0 || PrototypePropertyCount($interface) > 0;
}

sub InstanceOverridesPutImplementation
{
    my $interface = shift;
    return $interface->extendedAttributes->{CustomNamedSetter}
        || $interface->extendedAttributes->{CustomIndexedSetter};
}

sub InstanceOverridesPutDeclaration
{
    my $interface = shift;
    return $interface->extendedAttributes->{CustomPutFunction}
        || $interface->extendedAttributes->{CustomNamedSetter}
        || $interface->extendedAttributes->{CustomIndexedSetter};
}

sub InstanceNeedsVisitChildren
{
    my $interface = shift;
    return $interface->extendedAttributes->{JSCustomMarkFunction}
        || $codeGenerator->InheritsInterface($interface, "EventTarget")
        || $interface->name eq "EventTarget"
        || $interface->extendedAttributes->{ReportExtraMemoryCost}
        || IsJSBuiltinConstructor($interface)
}

sub InstanceNeedsEstimatedSize
{
    my $interface = shift;
    return $interface->extendedAttributes->{ReportExtraMemoryCost};
}

sub GetImplClassName
{
    my $name = shift;

    my ($svgPropertyType, $svgListPropertyType, $svgNativeType) = GetSVGPropertyTypes($name);
    return $svgNativeType if $svgNativeType;

    return $name;
}

sub IsClassNameWordBoundary
{
    my ($name, $i) = @_;

    # Interpret negative numbers as distance from end of string, just as the substr function does.
    $i += length($name) if $i < 0;

    return 0 if $i < 0;
    return 1 if $i == 0;
    return 1 if $i == length($name);
    return 0 if $i > length($name);

    my $checkString = substr($name, $i - 1);
    return $checkString =~ /^[^A-Z][A-Z]/ || $checkString =~ /^[A-Z][A-Z][^A-Z]/;
}

sub IsPrefixRemovable
{
    my ($class, $name, $i) = @_;

    return IsClassNameWordBoundary($name, $i)
        && (IsClassNameWordBoundary($class, $i) && substr($class, 0, $i) eq substr($name, 0, $i)
            || IsClassNameWordBoundary($class, -$i) && substr($class, -$i) eq substr($name, 0, $i));
}

sub GetNestedClassName
{
    my ($interface, $name) = @_;

    my $class = GetImplClassName($interface->name);
    my $member = $codeGenerator->WK_ucfirst($name);

    # Since the enumeration name will be nested in the class name's namespace, remove any words
    # that happen to match the start or end of the class name. If an enumeration is named TrackType or
    # TextTrackType, and the class is named TextTrack, then we will get a name like TextTrack::Type.
    my $memberLength = length($member);
    my $longestPrefixLength = 0;
    if ($member =~ /^[A-Z]./) {
        for (my $i = 2; $i < $memberLength - 1; $i++) {
            $longestPrefixLength = $i if IsPrefixRemovable($class, $member, $i);
        }
    }
    $member = substr($member, $longestPrefixLength);

    return "${class}::$member";
}

sub GetEnumerationClassName
{
    my ($name, $interface) = @_;

    if ($codeGenerator->HasEnumImplementationNameOverride($name)) {
        return $codeGenerator->GetEnumImplementationNameOverride($name);
    }

    return $name unless defined($interface);
    return GetNestedClassName($interface, $name);
}

sub GetEnumerationValueName
{
    my ($name) = @_;

    return "EmptyString" if $name eq "";
    $name = join("", map { $codeGenerator->WK_ucfirst($_) } split("-", $name));
    $name = "_$name" if $name =~ /^\d/;
    return $name;
}

sub GenerateEnumerationsImplementationContent
{
    my ($interface, $enumerations) = @_;

    return "" unless @$enumerations;

    my $result = "";
    foreach my $enumeration (@$enumerations) {
        my $name = $enumeration->name;

        my $className = GetEnumerationClassName($name, $interface);

        # FIXME: A little ugly to have this be a side effect instead of a return value.
        AddToImplIncludes("<runtime/JSString.h>");

        my $conditionalString = $codeGenerator->GenerateConditionalString($enumeration);
        $result .= "#if ${conditionalString}\n\n" if $conditionalString;

        # FIXME: Change to take VM& instead of ExecState*.
        $result .= "template<> JSString* convertEnumerationToJS(ExecState& state, $className enumerationValue)\n";
        $result .= "{\n";
        # FIXME: Might be nice to make this global be "const", but NeverDestroyed does not currently support that.
        # FIXME: Might be nice to make the entire array be NeverDestroyed instead of each value, but not sure what the syntax for that is.
        AddToImplIncludes("<wtf/NeverDestroyed.h>");
        $result .= "    static NeverDestroyed<const String> values[] = {\n";
        foreach my $value (@{$enumeration->values}) {
            if ($value eq "") {
                $result .= "        emptyString(),\n";
            } else {
                $result .= "        ASCIILiteral(\"$value\"),\n";
            }
        }
        $result .= "    };\n";
        my $index = 0;
        foreach my $value (@{$enumeration->values}) {
            my $enumerationValueName = GetEnumerationValueName($value);
            $result .= "    static_assert(static_cast<size_t>(${className}::$enumerationValueName) == $index, \"${className}::$enumerationValueName is not $index as expected\");\n";
            $index++;
        }
        $result .= "    ASSERT(static_cast<size_t>(enumerationValue) < WTF_ARRAY_LENGTH(values));\n";
        $result .= "    return jsStringWithCache(&state, values[static_cast<size_t>(enumerationValue)]);\n";
        $result .= "}\n\n";

        # FIXME: Change to take VM& instead of ExecState&.
        # FIXME: Consider using toStringOrNull to make exception checking faster.
        # FIXME: Consider finding a more efficient way to match against all the strings quickly.
        $result .= "template<> Optional<$className> parseEnumeration<$className>(ExecState& state, JSValue value)\n";
        $result .= "{\n";
        $result .= "    auto stringValue = value.toWTFString(&state);\n";
        foreach my $value (@{$enumeration->values}) {
            my $enumerationValueName = GetEnumerationValueName($value);
            if ($value eq "") {
                $result .= "    if (stringValue.isEmpty())\n";
            } else {
                $result .= "    if (stringValue == \"$value\")\n";
            }
            $result .= "        return ${className}::${enumerationValueName};\n";
        }
        $result .= "    return Nullopt;\n";
        $result .= "}\n\n";

        # FIXME: A little ugly to have this be a side effect instead of a return value.
        AddToImplIncludes("JSDOMConvert.h");

        $result .= "template<> $className convertEnumeration<$className>(ExecState& state, JSValue value)\n";
        $result .= "{\n";
        $result .= "    VM& vm = state.vm();\n";
        $result .= "    auto throwScope = DECLARE_THROW_SCOPE(vm);\n";
        $result .= "    auto result = parseEnumeration<$className>(state, value);\n";
        $result .= "    if (UNLIKELY(!result)) {\n";
        $result .= "        throwTypeError(&state, throwScope);\n";
        $result .= "        return { };\n";
        $result .= "    }\n";
        $result .= "    return result.value();\n";
        $result .= "}\n\n";

        $result .= "template<> const char* expectedEnumerationValues<$className>()\n";
        $result .= "{\n";
        $result .= "    return \"\\\"" . join ("\\\", \\\"", @{$enumeration->values}) . "\\\"\";\n";
        $result .= "}\n\n";

        $result .= "#endif\n\n" if $conditionalString;
    }
    return $result;
}

sub GenerateEnumerationsHeaderContent
{
    my ($interface, $enumerations) = @_;

    return "" unless @$enumerations;

    # FIXME: Could optimize this to only generate the parts of each enumeration that are actually
    # used, which would require iterating over everything in the interface.

    $headerIncludes{"JSDOMConvert.h"} = 1;

    my $result = "";

    foreach my $enumeration (@$enumerations) {
        my $name = $enumeration->name;

        my $className = GetEnumerationClassName($name, $interface);

        my $conditionalString = $codeGenerator->GenerateConditionalString($enumeration);
        $result .= "#if ${conditionalString}\n\n" if $conditionalString;

        $result .= "template<> JSC::JSString* convertEnumerationToJS(JSC::ExecState&, $className);\n\n";

        $result .= "template<> Optional<$className> parseEnumeration<$className>(JSC::ExecState&, JSC::JSValue);\n";
        $result .= "template<> $className convertEnumeration<$className>(JSC::ExecState&, JSC::JSValue);\n";
        $result .= "template<> const char* expectedEnumerationValues<$className>();\n\n";

        $result .= "#endif\n\n" if $conditionalString;
    }
    return $result;
}

sub GetDictionaryClassName
{
    my ($name, $interface) = @_;

    if ($codeGenerator->HasDictionaryImplementationNameOverride($name)) {
        return $codeGenerator->GetDictionaryImplementationNameOverride($name);
    }

    return $name if $codeGenerator->IsExternalDictionaryType($name);
    return $name unless defined($interface);
    return GetNestedClassName($interface, $name);
}

sub GenerateDefaultValue
{
    my ($interface, $signature) = @_;

    my $defaultValue = $signature->default;

    if ($codeGenerator->IsEnumType($signature->type)) {
        # FIXME: Would be nice to report an error if the value does not have quote marks around it.
        # FIXME: Would be nice to report an error if the value is not one of the enumeration values.
        my $className = GetEnumerationClassName($signature->type, $interface);
        my $enumerationValueName = GetEnumerationValueName(substr($defaultValue, 1, -1));
        return $className . "::" . $enumerationValueName;
    }
    if ($defaultValue eq "null") {
        if ($signature->idlType->isUnion) {
            return "Nullopt" if $signature->idlType->isNullable;

            my $IDLType = GetIDLType($interface, $signature->idlType);
            return "convert<${IDLType}>(state, jsNull());";
        }
        return "jsNull()" if $signature->type eq "any";
        return "nullptr" if $codeGenerator->IsWrapperType($signature->type) || $codeGenerator->IsTypedArrayType($signature->type);
        return "String()" if $codeGenerator->IsStringType($signature->type);
        return "Nullopt";
    }
    if ($defaultValue eq "[]") {
        my $nativeType = GetNativeTypeFromSignature($interface, $signature);
        return "$nativeType()"
    }

    return "jsUndefined()" if $defaultValue eq "undefined";
    return "PNaN" if $defaultValue eq "NaN";

    return $defaultValue;
}

sub ShouldAllowNonFiniteForFloatingPointType
{
    my $type = shift;

    die "Can only be called with floating point types" unless $codeGenerator->IsFloatingPointType($type);
    return $type eq "unrestricted double" || $type eq "unrestricted float";
}

sub GenerateConversionRuleWithLeadingComma
{
    my ($interface, $member) = @_;

    if ($codeGenerator->IsFloatingPointType($member->type)) {
        return ", " . (ShouldAllowNonFiniteForFloatingPointType($member->type) ? "ShouldAllowNonFinite::Yes" : "ShouldAllowNonFinite::No");
    }
    return ", " . GetIntegerConversionConfiguration($member) if $codeGenerator->IsIntegerType($member->type);
    return "";
}

sub GenerateDictionaryHeaderContent
{
    my ($dictionary, $className, $conditionalString) = @_;

    my $result = "";
    $result .= "#if ${conditionalString}\n\n" if $conditionalString;
    $result .= "template<> $className convertDictionary<$className>(JSC::ExecState&, JSC::JSValue);\n\n";
    $result .= "#endif\n\n" if $conditionalString;
    return $result;
}

sub GenerateDictionariesHeaderContent
{
    my ($interface, $allDictionaries) = @_;

    return "" unless @$allDictionaries;

    $headerIncludes{"JSDOMConvert.h"} = 1;

    my $result = "";
    foreach my $dictionary (@$allDictionaries) {
        $headerIncludes{$interface->name . ".h"} = 1;
        my $className = GetDictionaryClassName($dictionary->name, $interface);
        my $conditionalString = $codeGenerator->GenerateConditionalString($dictionary);
        $result .= GenerateDictionaryHeaderContent($dictionary, $className, $conditionalString);
    }
    return $result;
}

sub GenerateDictionaryImplementationContent
{
    my ($dictionary, $className, $interface, $conditionalString) = @_;

    my $result = "";

    my $name = $dictionary->name;

    $result .= "#if ${conditionalString}\n\n" if $conditionalString;

    # FIXME: A little ugly to have this be a side effect instead of a return value.
    AddToImplIncludes("JSDOMConvert.h");

    # https://heycam.github.io/webidl/#es-dictionary
    $result .= "template<> $className convertDictionary<$className>(ExecState& state, JSValue value)\n";
    $result .= "{\n";
    $result .= "    VM& vm = state.vm();\n";
    $result .= "    auto throwScope = DECLARE_THROW_SCOPE(vm);\n";
    $result .= "    bool isNullOrUndefined = value.isUndefinedOrNull();\n";
    $result .= "    auto* object = isNullOrUndefined ? nullptr : value.getObject();\n";
    # 1. If Type(V) is not Undefined, Null or Object, then throw a TypeError.
    $result .= "    if (UNLIKELY(!isNullOrUndefined && !object)) {\n";
    $result .= "        throwTypeError(&state, throwScope);\n";
    $result .= "        return { };\n";
    $result .= "    }\n";

    # 2. If V is a native RegExp object, then throw a TypeError.
    # FIXME: This RegExp special handling is likely to go away in the specification.
    $result .= "    if (UNLIKELY(object && object->type() == RegExpObjectType)) {\n";
    $result .= "        throwTypeError(&state, throwScope);\n";
    $result .= "        return { };\n";
    $result .= "    }\n";

    # 3. Let dict be an empty dictionary value of type D; every dictionary member is initially considered to be not present.

    # 4. Let dictionaries be a list consisting of D and all of D’s inherited dictionaries, in order from least to most derived.
    my @dictionaries;
    push(@dictionaries, $dictionary);
    my $parentName = $dictionary->parent;
    while (defined($parentName)) {
        my $parentDictionary = $codeGenerator->GetDictionaryByName($parentName);
        assert("Unable to find definition for dictionary named '" . $parentName . "'!") unless defined($parentDictionary);
        unshift(@dictionaries, $parentDictionary);
        $parentName = $parentDictionary->parent;
    }

    my $arguments = "";
    my $comma = "";

    $result .= "    $className result;\n";

    # 5. For each dictionary dictionary in dictionaries, in order:
    foreach my $dictionary (@dictionaries) {
        # For each dictionary member member declared on dictionary, in lexicographical order:
        my @sortedMembers = sort { $a->name cmp $b->name } @{$dictionary->members};
        foreach my $member (@sortedMembers) {
            $member->default("undefined") if $member->idlType->name eq "any" and !defined($member->default); # Use undefined as default value for member of type 'any' unless specified otherwise.

            my $idlType = $member->idlType;
            AddToImplIncludesForIDLType($idlType);

            # 5.1. Let key be the identifier of member.
            my $key = $member->name;

            # 5.2. Let value be an ECMAScript value, depending on Type(V):
            $result .= "    JSValue ${key}Value = isNullOrUndefined ? jsUndefined() : object->get(&state, Identifier::fromString(&state, \"${key}\"));\n";

            my $IDLType = GetIDLType($interface, $idlType);

            # 5.3. If value is not undefined, then:
            $result .= "    if (!${key}Value.isUndefined()) {\n";
            $result .= "        result.$key = convert<${IDLType}>(state, ${key}Value);\n";
            $result .= "        RETURN_IF_EXCEPTION(throwScope, { });\n";

            # Value is undefined.
            # 5.4. Otherwise, if value is undefined but the dictionary member has a default value, then:
            if ($member->isOptional && defined $member->default) {
                $result .= "    } else\n";
                $result .= "        result.$key = " . GenerateDefaultValue($interface, $member) . ";\n";
            } elsif (!$member->isOptional) {
                # 5.5. Otherwise, if value is undefined and the dictionary member is a required dictionary member, then throw a TypeError.
                $result .= "    } else {\n";
                $result .= "        throwRequiredMemberTypeError(state, throwScope, \"".$member->name."\", \"$name\", \"".$idlType->name."\");\n";
                $result .= "        return { };\n";
                $result .= "    }\n";
            } else {
                $result .= "    }\n";
            }
        }
    }

    $result .= "    return result;\n";
    $result .= "}\n\n";
    $result .= "#endif\n\n" if $conditionalString;

    return $result;
}

sub GenerateDictionariesImplementationContent
{
    my ($interface, $allDictionaries) = @_;

    my $result = "";
    foreach my $dictionary (@$allDictionaries) {
        my $className = GetDictionaryClassName($dictionary->name, $interface);
        my $conditionalString = $codeGenerator->GenerateConditionalString($dictionary);
        $result .= GenerateDictionaryImplementationContent($dictionary, $className, $interface, $conditionalString);
    }
    return $result;
}

sub GetJSTypeForNode
{
    my ($codeGenerator, $interface) = @_;

    if ($codeGenerator->InheritsInterface($interface, "Document")) {
        return "JSDocumentWrapperType";
    }
    if ($codeGenerator->InheritsInterface($interface, "DocumentFragment")) {
        return "JSDocumentFragmentNodeType";
    }
    if ($codeGenerator->InheritsInterface($interface, "DocumentType")) {
        return "JSDocumentTypeNodeType";
    }
    if ($codeGenerator->InheritsInterface($interface, "ProcessingInstruction")) {
        return "JSProcessingInstructionNodeType";
    }
    if ($codeGenerator->InheritsInterface($interface, "CDATASection")) {
        return "JSCDATASectionNodeType";
    }
    if ($codeGenerator->InheritsInterface($interface, "Attr")) {
        return "JSAttrNodeType";
    }
    if ($codeGenerator->InheritsInterface($interface, "Comment")) {
        return "JSCommentNodeType";
    }
    if ($codeGenerator->InheritsInterface($interface, "Text")) {
        return "JSTextNodeType";
    }
    if ($codeGenerator->InheritsInterface($interface, "Element")) {
        return "JSElementType";
    }
    return "JSNodeType";
}

sub GenerateHeader
{
    my ($object, $interface, $enumerations, $dictionaries) = @_;

    my $interfaceName = $interface->name;
    my $className = "JS$interfaceName";
    my %structureFlags = ();

    my $hasLegacyParent = $interface->extendedAttributes->{JSLegacyParent};
    my $hasRealParent = $interface->parent;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    my $parentClassName = GetParentClassName($interface);
    my $needsVisitChildren = InstanceNeedsVisitChildren($interface);

    # - Add default header template and header protection
    push(@headerContentHeader, GenerateHeaderContentHeader($interface));

    if ($hasParent) {
        $headerIncludes{"$parentClassName.h"} = 1;
    } else {
        $headerIncludes{"JSDOMWrapper.h"} = 1;
        if ($interface->isException) {
            $headerIncludes{"<runtime/ErrorPrototype.h>"} = 1;
        }
    }

    $headerIncludes{"<runtime/CallData.h>"} = 1 if $interface->extendedAttributes->{CustomCall};

    $headerIncludes{"$interfaceName.h"} = 1 if $hasParent && $interface->extendedAttributes->{JSGenerateToNativeObject};

    $headerIncludes{"SVGElement.h"} = 1 if $className =~ /^JSSVG/;

    my $implType = GetImplClassName($interfaceName);
    my ($svgPropertyType, $svgListPropertyType, $svgNativeType) = GetSVGPropertyTypes($interfaceName);
    my $svgPropertyOrListPropertyType;
    $svgPropertyOrListPropertyType = $svgPropertyType if $svgPropertyType;
    $svgPropertyOrListPropertyType = $svgListPropertyType if $svgListPropertyType;

    my $numConstants = @{$interface->constants};
    my $numAttributes = @{$interface->attributes};
    my $numFunctions = @{$interface->functions};

    push(@headerContent, "\nnamespace WebCore {\n\n");

    if ($codeGenerator->IsSVGAnimatedType($interfaceName)) {
        $headerIncludes{"$interfaceName.h"} = 1;
    } else {
        # Implementation class forward declaration
        if (IsDOMGlobalObject($interface)) {
            AddClassForwardIfNeeded($interfaceName) unless $svgPropertyOrListPropertyType;
        }
    }

    AddClassForwardIfNeeded("JSDOMWindowShell") if $interfaceName eq "DOMWindow";

    my $exportMacro = GetExportMacroForJSClass($interface);

    # Class declaration
    push(@headerContent, "class $exportMacro$className : public $parentClassName {\n");

    # Static create methods
    push(@headerContent, "public:\n");
    push(@headerContent, "    using Base = $parentClassName;\n");
    push(@headerContent, "    using DOMWrapped = $implType;\n") if $interface->parent;

    if ($interfaceName eq "DOMWindow") {
        push(@headerContent, "    static $className* create(JSC::VM& vm, JSC::Structure* structure, Ref<$implType>&& impl, JSDOMWindowShell* windowShell)\n");
        push(@headerContent, "    {\n");
        push(@headerContent, "        $className* ptr = new (NotNull, JSC::allocateCell<$className>(vm.heap)) ${className}(vm, structure, WTFMove(impl), windowShell);\n");
        push(@headerContent, "        ptr->finishCreation(vm, windowShell);\n");
        push(@headerContent, "        vm.heap.addFinalizer(ptr, destroy);\n");
        push(@headerContent, "        return ptr;\n");
        push(@headerContent, "    }\n\n");
    } elsif ($codeGenerator->InheritsInterface($interface, "WorkerGlobalScope")) {
        push(@headerContent, "    static $className* create(JSC::VM& vm, JSC::Structure* structure, Ref<$implType>&& impl, JSC::JSProxy* proxy)\n");
        push(@headerContent, "    {\n");
        push(@headerContent, "        $className* ptr = new (NotNull, JSC::allocateCell<$className>(vm.heap)) ${className}(vm, structure, WTFMove(impl));\n");
        push(@headerContent, "        ptr->finishCreation(vm, proxy);\n");
        push(@headerContent, "        vm.heap.addFinalizer(ptr, destroy);\n");
        push(@headerContent, "        return ptr;\n");
        push(@headerContent, "    }\n\n");
    } elsif ($interface->extendedAttributes->{MasqueradesAsUndefined}) {
        AddIncludesForTypeInHeader($implType) unless $svgPropertyOrListPropertyType;
        push(@headerContent, "    static $className* create(JSC::Structure* structure, JSDOMGlobalObject* globalObject, Ref<$implType>&& impl)\n");
        push(@headerContent, "    {\n");
        push(@headerContent, "        globalObject->masqueradesAsUndefinedWatchpoint()->fireAll(globalObject->vm(), \"Allocated masquerading object\");\n");
        push(@headerContent, "        $className* ptr = new (NotNull, JSC::allocateCell<$className>(globalObject->vm().heap)) $className(structure, *globalObject, WTFMove(impl));\n");
        push(@headerContent, "        ptr->finishCreation(globalObject->vm());\n");
        push(@headerContent, "        return ptr;\n");
        push(@headerContent, "    }\n\n");
    } elsif (!NeedsImplementationClass($interface)) {
        push(@headerContent, "    static $className* create(JSC::Structure* structure, JSDOMGlobalObject* globalObject)\n");
        push(@headerContent, "    {\n");
        push(@headerContent, "        $className* ptr = new (NotNull, JSC::allocateCell<$className>(globalObject->vm().heap)) $className(structure, *globalObject);\n");
        push(@headerContent, "        ptr->finishCreation(globalObject->vm());\n");
        push(@headerContent, "        return ptr;\n");
        push(@headerContent, "    }\n\n");  
    } else {
        AddIncludesForTypeInHeader($implType) unless $svgPropertyOrListPropertyType;
        push(@headerContent, "    static $className* create(JSC::Structure* structure, JSDOMGlobalObject* globalObject, Ref<$implType>&& impl)\n");
        push(@headerContent, "    {\n");
        push(@headerContent, "        $className* ptr = new (NotNull, JSC::allocateCell<$className>(globalObject->vm().heap)) $className(structure, *globalObject, WTFMove(impl));\n");
        push(@headerContent, "        ptr->finishCreation(globalObject->vm());\n");
        push(@headerContent, "        return ptr;\n");
        push(@headerContent, "    }\n\n");
    }

    push(@headerContent, "    static const bool needsDestruction = false;\n\n") if IsDOMGlobalObject($interface);

    $structureFlags{"JSC::HasStaticPropertyTable"} = 1 if InstancePropertyCount($interface) > 0;

    # Prototype
    unless (ShouldUseGlobalObjectPrototype($interface)) {
        push(@headerContent, "    static JSC::JSObject* createPrototype(JSC::VM&, JSC::JSGlobalObject*);\n");
        push(@headerContent, "    static JSC::JSObject* prototype(JSC::VM&, JSC::JSGlobalObject*);\n");
    }

    # JSValue to implementation type
    if (ShouldGenerateToWrapped($hasParent, $interface)) {
        my $nativeType = GetNativeType($interface, $implType);
        if ($interface->name eq "XPathNSResolver") {
            push(@headerContent, "    static $nativeType toWrapped(JSC::ExecState&, JSC::JSValue);\n");
        } else {
            my $export = "";
            $export = "WEBCORE_EXPORT " if $interface->extendedAttributes->{ExportToWrappedFunction};
            push(@headerContent, "    static $export$nativeType toWrapped(JSC::JSValue);\n");
        }
    }

    $headerTrailingIncludes{"${className}Custom.h"} = 1 if $interface->extendedAttributes->{JSCustomHeader};

    my $namedGetterFunction = GetNamedGetterFunction($interface);
    my $indexedGetterFunction = GetIndexedGetterFunction($interface);

    my $hasNamedGetter = $namedGetterFunction
        || $interface->extendedAttributes->{CustomNamedGetter};

    my $hasComplexGetter = $indexedGetterFunction
        || $interface->extendedAttributes->{JSCustomGetOwnPropertySlotAndDescriptor}
        || $interface->extendedAttributes->{CustomGetOwnPropertySlot}
        || $hasNamedGetter;

    my $hasGetter = InstanceOverridesGetOwnPropertySlot($interface);

    if ($hasNamedGetter) {
        if ($interface->extendedAttributes->{OverrideBuiltins}) {
            $structureFlags{"JSC::GetOwnPropertySlotIsImpure"} = 1;
        } else {
            $structureFlags{"JSC::GetOwnPropertySlotIsImpureForPropertyAbsence"} = 1;
        }
    }
    $structureFlags{"JSC::NewImpurePropertyFiresWatchpoints"} = 1 if $interface->extendedAttributes->{NewImpurePropertyFiresWatchpoints};
    $structureFlags{"JSC::TypeOfShouldCallGetCallData"} = 1 if $interface->extendedAttributes->{CustomCall};

    # Getters
    if ($hasGetter) {
        push(@headerContent, "    static bool getOwnPropertySlot(JSC::JSObject*, JSC::ExecState*, JSC::PropertyName, JSC::PropertySlot&);\n");
        push(@headerContent, "    bool getOwnPropertySlotDelegate(JSC::ExecState*, JSC::PropertyName, JSC::PropertySlot&);\n") if $interface->extendedAttributes->{JSCustomGetOwnPropertySlotAndDescriptor};
        $structureFlags{"JSC::OverridesGetOwnPropertySlot"} = 1;

        if ($hasComplexGetter) {
            push(@headerContent, "    static bool getOwnPropertySlotByIndex(JSC::JSObject*, JSC::ExecState*, unsigned propertyName, JSC::PropertySlot&);\n");
            $structureFlags{"JSC::InterceptsGetOwnPropertySlotByIndexEvenWhenLengthIsNotZero"} = 1;
        }
    }

    my $overridesPut = InstanceOverridesPutDeclaration($interface);

    # Getters
    if ($overridesPut) {
        push(@headerContent, "    static bool put(JSC::JSCell*, JSC::ExecState*, JSC::PropertyName, JSC::JSValue, JSC::PutPropertySlot&);\n");
        push(@headerContent, "    static bool putByIndex(JSC::JSCell*, JSC::ExecState*, unsigned propertyName, JSC::JSValue, bool shouldThrow);\n");
        push(@headerContent, "    bool putDelegate(JSC::ExecState*, JSC::PropertyName, JSC::JSValue, JSC::PutPropertySlot&, bool& putResult);\n") if $interface->extendedAttributes->{CustomNamedSetter};
    }

    if (!$hasParent) {
        push(@headerContent, "    static void destroy(JSC::JSCell*);\n");
    }

    # Class info
    if ($interfaceName eq "Node") {
        push(@headerContent, "\n");
        push(@headerContent, "protected:\n");
        push(@headerContent, "    static const JSC::ClassInfo s_info;\n");
        push(@headerContent, "public:\n");
        push(@headerContent, "    static const JSC::ClassInfo* info() { return &s_info; }\n\n");
    } else {
        push(@headerContent, "\n");
        push(@headerContent, "    DECLARE_INFO;\n\n");
    }

    # Structure ID
    $structureFlags{"JSC::ImplementsHasInstance | JSC::ImplementsDefaultHasInstance"} = 1 if $interfaceName eq "DOMWindow";
    push(@headerContent, "    static JSC::Structure* createStructure(JSC::VM& vm, JSC::JSGlobalObject* globalObject, JSC::JSValue prototype)\n");
    push(@headerContent, "    {\n");
    if (IsDOMGlobalObject($interface)) {
        push(@headerContent, "        return JSC::Structure::create(vm, globalObject, prototype, JSC::TypeInfo(JSC::GlobalObjectType, StructureFlags), info());\n");
    } elsif ($codeGenerator->InheritsInterface($interface, "Node")) {
        my $type = GetJSTypeForNode($codeGenerator, $interface);
        push(@headerContent, "        return JSC::Structure::create(vm, globalObject, prototype, JSC::TypeInfo(JSC::JSType($type), StructureFlags), info());\n");
    } else {
        push(@headerContent, "        return JSC::Structure::create(vm, globalObject, prototype, JSC::TypeInfo(JSC::ObjectType, StructureFlags), info());\n");
    }
    push(@headerContent, "    }\n\n");

    # Custom pushEventHandlerScope function
    push(@headerContent, "    JSC::JSScope* pushEventHandlerScope(JSC::ExecState*, JSC::JSScope*) const;\n\n") if $interface->extendedAttributes->{JSCustomPushEventHandlerScope};

    # Custom call functions
    push(@headerContent, "    static JSC::CallType getCallData(JSC::JSCell*, JSC::CallData&);\n\n") if $interface->extendedAttributes->{CustomCall};

    # Custom deleteProperty function
    push(@headerContent, "    static bool deleteProperty(JSC::JSCell*, JSC::ExecState*, JSC::PropertyName);\n") if $interface->extendedAttributes->{CustomDeleteProperty};
    push(@headerContent, "    static bool deletePropertyByIndex(JSC::JSCell*, JSC::ExecState*, unsigned);\n") if $interface->extendedAttributes->{CustomDeleteProperty};

    # Custom getPropertyNames function exists on DOMWindow
    if ($interfaceName eq "DOMWindow") {
        push(@headerContent, "    static void getPropertyNames(JSC::JSObject*, JSC::ExecState*, JSC::PropertyNameArray&, JSC::EnumerationMode = JSC::EnumerationMode());\n");
        push(@headerContent, "    static void getGenericPropertyNames(JSC::JSObject*, JSC::ExecState*, JSC::PropertyNameArray&, JSC::EnumerationMode = JSC::EnumerationMode());\n");
        push(@headerContent, "    static void getStructurePropertyNames(JSC::JSObject*, JSC::ExecState*, JSC::PropertyNameArray&, JSC::EnumerationMode = JSC::EnumerationMode());\n");
        push(@headerContent, "    static uint32_t getEnumerableLength(JSC::ExecState*, JSC::JSObject*);\n");
        $structureFlags{"JSC::OverridesGetPropertyNames"} = 1;
    }

    # Custom getOwnPropertyNames function
    if ($interface->extendedAttributes->{CustomEnumerateProperty} || $indexedGetterFunction || $namedGetterFunction) {
        push(@headerContent, "    static void getOwnPropertyNames(JSC::JSObject*, JSC::ExecState*, JSC::PropertyNameArray&, JSC::EnumerationMode = JSC::EnumerationMode());\n");
        $structureFlags{"JSC::OverridesGetPropertyNames"} = 1;       
    }

    # Custom defineOwnProperty function
    push(@headerContent, "    static bool defineOwnProperty(JSC::JSObject*, JSC::ExecState*, JSC::PropertyName, const JSC::PropertyDescriptor&, bool shouldThrow);\n") if $interface->extendedAttributes->{JSCustomDefineOwnProperty};

    # Custom getPrototype / setPrototype functions.
    push (@headerContent, "    static JSC::JSValue getPrototype(JSC::JSObject*, JSC::ExecState*);\n") if $interface->extendedAttributes->{CustomGetPrototype};
    push (@headerContent, "    static bool setPrototype(JSC::JSObject*, JSC::ExecState*, JSC::JSValue, bool shouldThrowIfCantSet);\n") if $interface->extendedAttributes->{CustomSetPrototype};

    # Custom preventExtensions function.
    push(@headerContent, "    static bool preventExtensions(JSC::JSObject*, JSC::ExecState*);\n") if $interface->extendedAttributes->{CustomPreventExtensions};
    
    $structureFlags{"JSC::MasqueradesAsUndefined"} = 1 if $interface->extendedAttributes->{MasqueradesAsUndefined};

    # Constructor object getter
    unless ($interface->extendedAttributes->{NoInterfaceObject}) {
        push(@headerContent, "    static JSC::JSValue getConstructor(JSC::VM&, const JSC::JSGlobalObject*);\n");
        push(@headerContent, "    static JSC::JSValue getNamedConstructor(JSC::VM&, JSC::JSGlobalObject*);\n") if $interface->extendedAttributes->{NamedConstructor};
    }

    my $numCustomFunctions = 0;
    my $numCustomAttributes = 0;

    my $hasForwardDeclaringFunctions = 0;
    my $hasForwardDeclaringAttributes = 0;

    my $hasDOMJITAttributes = 0;

    # Attribute and function enums
    if ($numAttributes > 0) {
        foreach (@{$interface->attributes}) {
            my $attribute = $_;
            $numCustomAttributes++ if HasCustomGetter($attribute->signature->extendedAttributes);
            $numCustomAttributes++ if HasCustomSetter($attribute->signature->extendedAttributes);
            if ($attribute->signature->extendedAttributes->{CachedAttribute}) {
                my $conditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
                push(@headerContent, "#if ${conditionalString}\n") if $conditionalString;
                push(@headerContent, "    mutable JSC::WriteBarrier<JSC::Unknown> m_" . $attribute->signature->name . ";\n");
                $numCachedAttributes++;
                $needsVisitChildren = 1;
                push(@headerContent, "#endif\n") if $conditionalString;
            }
            $hasDOMJITAttributes = 1 if $attribute->signature->extendedAttributes->{"DOMJIT"};

            $hasForwardDeclaringAttributes = 1 if $attribute->signature->extendedAttributes->{ForwardDeclareInHeader};
        }
    }

    # visit function
    if ($needsVisitChildren) {
        push(@headerContent, "    static void visitChildren(JSCell*, JSC::SlotVisitor&);\n");
        push(@headerContent, "    void visitAdditionalChildren(JSC::SlotVisitor&);\n") if $interface->extendedAttributes->{JSCustomMarkFunction};
        push(@headerContent, "\n");
    }

    if (InstanceNeedsEstimatedSize($interface)) {
        push(@headerContent, "    static size_t estimatedSize(JSCell*);\n");
    }

    if ($numCustomAttributes > 0) {
        push(@headerContent, "\n    // Custom attributes\n");

        foreach my $attribute (@{$interface->attributes}) {
            my $conditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
            if (HasCustomGetter($attribute->signature->extendedAttributes)) {
                push(@headerContent, "#if ${conditionalString}\n") if $conditionalString;
                my $methodName = $codeGenerator->WK_lcfirst($attribute->signature->name);
                push(@headerContent, "    JSC::JSValue " . $methodName . "(JSC::ExecState&) const;\n");
                push(@headerContent, "#endif\n") if $conditionalString;
            }
            if (HasCustomSetter($attribute->signature->extendedAttributes) && !IsReadonly($attribute)) {
                push(@headerContent, "#if ${conditionalString}\n") if $conditionalString;
                push(@headerContent, "    void set" . $codeGenerator->WK_ucfirst($attribute->signature->name) . "(JSC::ExecState&, JSC::JSValue);\n");
                push(@headerContent, "#endif\n") if $conditionalString;
            }
        }
    }

    foreach my $function (@{$interface->functions}) {
        $numCustomFunctions++ if HasCustomMethod($function->signature->extendedAttributes);
        $hasForwardDeclaringFunctions = 1 if $function->signature->extendedAttributes->{ForwardDeclareInHeader};
    }

    if ($numCustomFunctions > 0) {
        my $inAppleCopyright = 0;
        push(@headerContent, "\n    // Custom functions\n");
        foreach my $function (@{$interface->functions}) {
            if ($function->signature->extendedAttributes->{AppleCopyright}) {
                if (!$inAppleCopyright) {
                    push(@headerContent, $beginAppleCopyrightForHeaderFiles);
                    $inAppleCopyright = 1;
                }
            } elsif ($inAppleCopyright) {
                push(@headerContent, $endAppleCopyright);
                $inAppleCopyright = 0;
            }
            next unless HasCustomMethod($function->signature->extendedAttributes);
            next if $function->{overloads} && $function->{overloadIndex} != 1;
            my $conditionalString = $codeGenerator->GenerateConditionalString($function->signature);
            push(@headerContent, "#if ${conditionalString}\n") if $conditionalString;
            my $functionImplementationName = $function->signature->extendedAttributes->{ImplementedAs} || $codeGenerator->WK_lcfirst($function->signature->name);
            push(@headerContent, "    " . ($function->isStatic ? "static " : "") . "JSC::JSValue " . $functionImplementationName . "(JSC::ExecState&);\n");
            push(@headerContent, "#endif\n") if $conditionalString;
        }
        push(@headerContent, $endAppleCopyright) if $inAppleCopyright;
    }

    if (NeedsImplementationClass($interface)) {
        if ($hasParent) {
            push(@headerContent, "    $interfaceName& wrapped() const\n");
            push(@headerContent, "    {\n");
            push(@headerContent, "        return static_cast<$interfaceName&>(Base::wrapped());\n");
            push(@headerContent, "    }\n");
        }
    }

    # structure flags
    if (%structureFlags) {
        push(@headerContent, "public:\n");
        push(@headerContent, "    static const unsigned StructureFlags = ");
        foreach my $structureFlag (sort (keys %structureFlags)) {
            push(@headerContent, $structureFlag . " | ");
        }
        push(@headerContent, "Base::StructureFlags;\n");
    }

    push(@headerContent, "protected:\n");

    # Constructor
    if ($interfaceName eq "DOMWindow") {
        push(@headerContent, "    $className(JSC::VM&, JSC::Structure*, Ref<$implType>&&, JSDOMWindowShell*);\n");
    } elsif ($codeGenerator->InheritsInterface($interface, "WorkerGlobalScope")) {
        push(@headerContent, "    $className(JSC::VM&, JSC::Structure*, Ref<$implType>&&);\n");
    } elsif (!NeedsImplementationClass($interface)) {
        push(@headerContent, "    $className(JSC::Structure*, JSDOMGlobalObject&);\n\n");
     } else {
        push(@headerContent, "    $className(JSC::Structure*, JSDOMGlobalObject&, Ref<$implType>&&);\n\n");
        push(@headerContent, "    void finishCreation(JSC::VM& vm)\n");
        push(@headerContent, "    {\n");
        push(@headerContent, "        Base::finishCreation(vm);\n");
        push(@headerContent, "        ASSERT(inherits(info()));\n");
        push(@headerContent, "    }\n\n");
    }

    if (IsDOMGlobalObject($interface)) {
        if ($interfaceName eq "DOMWindow") {
            push(@headerContent, "    void finishCreation(JSC::VM&, JSDOMWindowShell*);\n");
        } else {
            push(@headerContent, "    void finishCreation(JSC::VM&, JSC::JSProxy*);\n");
        }
    }

    push(@headerContent, "    void indexSetter(JSC::ExecState*, unsigned index, JSC::JSValue);\n") if $interface->extendedAttributes->{CustomIndexedSetter};

    if ($namedGetterFunction || $interface->extendedAttributes->{CustomNamedGetter}) {
        push(@headerContent, "private:\n");
        push(@headerContent, "    bool nameGetter(JSC::ExecState*, JSC::PropertyName, JSC::JSValue&);\n");
    }

    push(@headerContent, "};\n\n");

    if (ShouldGenerateWrapperOwnerCode($hasParent, $interface)) {
        if ($interfaceName ne "Node" && $codeGenerator->InheritsInterface($interface, "Node")) {
            $headerIncludes{"JSNode.h"} = 1;
            push(@headerContent, "class JS${interfaceName}Owner : public JSNodeOwner {\n");
        } else {
            push(@headerContent, "class JS${interfaceName}Owner : public JSC::WeakHandleOwner {\n");
        }
        $headerIncludes{"<wtf/NeverDestroyed.h>"} = 1;
        push(@headerContent, "public:\n");
        push(@headerContent, "    virtual bool isReachableFromOpaqueRoots(JSC::Handle<JSC::Unknown>, void* context, JSC::SlotVisitor&);\n");
        push(@headerContent, "    virtual void finalize(JSC::Handle<JSC::Unknown>, void* context);\n");
        push(@headerContent, "};\n");
        push(@headerContent, "\n");
        push(@headerContent, "inline JSC::WeakHandleOwner* wrapperOwner(DOMWrapperWorld&, $implType*)\n");
        push(@headerContent, "{\n");
        push(@headerContent, "    static NeverDestroyed<JS${interfaceName}Owner> owner;\n");
        push(@headerContent, "    return &owner.get();\n");
        push(@headerContent, "}\n");
        push(@headerContent, "\n");
        push(@headerContent, "inline void* wrapperKey($implType* wrappableObject)\n");
        push(@headerContent, "{\n");
        push(@headerContent, "    return wrappableObject;\n");
        push(@headerContent, "}\n");
        push(@headerContent, "\n");
    }
    if (ShouldGenerateToJSDeclaration($hasParent, $interface)) {
        # Node and NodeList have custom inline implementations which thus cannot be exported.
        # FIXME: The special case for Node and NodeList should probably be implemented via an IDL attribute.
        if ($implType eq "Node" or $implType eq "NodeList") {
            push(@headerContent, "JSC::JSValue toJS(JSC::ExecState*, JSDOMGlobalObject*, $implType&);\n");
        } else {
            push(@headerContent, $exportMacro."JSC::JSValue toJS(JSC::ExecState*, JSDOMGlobalObject*, $implType&);\n");
        }
        push(@headerContent, "inline JSC::JSValue toJS(JSC::ExecState* state, JSDOMGlobalObject* globalObject, $implType* impl) { return impl ? toJS(state, globalObject, *impl) : JSC::jsNull(); }\n");

        push(@headerContent, "JSC::JSValue toJSNewlyCreated(JSC::ExecState*, JSDOMGlobalObject*, Ref<$implType>&&);\n");
        push(@headerContent, "inline JSC::JSValue toJSNewlyCreated(JSC::ExecState* state, JSDOMGlobalObject* globalObject, RefPtr<$implType>&& impl) { return impl ? toJSNewlyCreated(state, globalObject, impl.releaseNonNull()) : JSC::jsNull(); }\n");
   }

    push(@headerContent, "\n");

    GeneratePrototypeDeclaration(\@headerContent, $className, $interface) if HeaderNeedsPrototypeDeclaration($interface);

    if ($hasForwardDeclaringFunctions) {
        my $inAppleCopyright = 0;
        push(@headerContent,"// Functions\n\n");
        foreach my $function (@{$interface->functions}) {
            next if $function->{overloadIndex} && $function->{overloadIndex} > 1;
            next unless $function->signature->extendedAttributes->{ForwardDeclareInHeader};

            if ($function->signature->extendedAttributes->{AppleCopyright}) {
                if (!$inAppleCopyright) {
                    push(@headerContent, $beginAppleCopyrightForHeaderFiles);
                    $inAppleCopyright = 1;
                }
            } elsif ($inAppleCopyright) {
                push(@headerContent, $endAppleCopyright);
                $inAppleCopyright = 0;
            }

            my $conditionalAttribute = getConditionalForFunctionConsideringOverloads($function);
            my $conditionalString = $conditionalAttribute ? $codeGenerator->GenerateConditionalStringFromAttributeValue($conditionalAttribute) : undef;
            push(@headerContent, "#if ${conditionalString}\n") if $conditionalString;
            my $functionName = GetFunctionName($interface, $className, $function);
            push(@headerContent, "JSC::EncodedJSValue JSC_HOST_CALL ${functionName}(JSC::ExecState*);\n");
            push(@headerContent, "#endif\n") if $conditionalString;
        }

        push(@headerContent, $endAppleCopyright) if $inAppleCopyright;
        push(@headerContent,"\n");
    }

    if ($hasForwardDeclaringAttributes) {
        push(@headerContent,"// Attributes\n\n");
        foreach my $attribute (@{$interface->attributes}) {
            next unless $attribute->signature->extendedAttributes->{ForwardDeclareInHeader};

            my $conditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
            push(@headerContent, "#if ${conditionalString}\n") if $conditionalString;
            my $getter = GetAttributeGetterName($interface, $className, $attribute);
            push(@headerContent, "JSC::EncodedJSValue ${getter}(JSC::ExecState*, JSC::EncodedJSValue, JSC::PropertyName);\n");
            if (!IsReadonly($attribute)) {
                my $setter = GetAttributeSetterName($interface, $className, $attribute);
                push(@headerContent, "bool ${setter}(JSC::ExecState*, JSC::EncodedJSValue, JSC::EncodedJSValue);\n");
            }
            push(@headerContent, "#endif\n") if $conditionalString;
        }
    }

    if ($hasDOMJITAttributes) {
        $headerIncludes{"<domjit/DOMJITGetterSetter.h>"} = 1;
        push(@headerContent,"// DOMJIT emitters for attributes\n\n");
        foreach my $attribute (@{$interface->attributes}) {
            next unless $attribute->signature->extendedAttributes->{"DOMJIT"};

            my $interfaceName = $interface->name;
            my $className = $interfaceName . $codeGenerator->WK_ucfirst($attribute->signature->name);
            my $domJITClassName = $className . "DOMJIT";

            push(@headerContent, "JSC::DOMJIT::GetterSetter* domJITGetterSetterFor$className(void);\n");

            push(@headerContent, "class $domJITClassName : public JSC::DOMJIT::GetterSetter {\n");
            push(@headerContent, "public:\n");
            push(@headerContent, "    $domJITClassName();\n");
            push(@headerContent, "#if ENABLE(JIT)\n");
            push(@headerContent, "    Ref<JSC::DOMJIT::Patchpoint> checkDOM() override;\n");
            push(@headerContent, "    Ref<JSC::DOMJIT::CallDOMPatchpoint> callDOM() override;\n");
            push(@headerContent, "#endif\n");
            push(@headerContent, "};\n\n");
        }
    }

    if (HasCustomConstructor($interface)) {
        push(@headerContent, "// Custom constructor\n");
        push(@headerContent, "JSC::EncodedJSValue JSC_HOST_CALL construct${className}(JSC::ExecState&);\n\n");
    }

    if (NeedsImplementationClass($interface)) {
        push(@headerContent, "template<> struct JSDOMWrapperConverterTraits<${implType}> {\n");
        push(@headerContent, "    using WrapperClass = ${className};\n");
        push(@headerContent, "    using ToWrappedReturnType = ${implType}*;\n");
        push(@headerContent, "};\n");
    }

    push(@headerContent, GenerateEnumerationsHeaderContent($interface, $enumerations));
    push(@headerContent, GenerateDictionariesHeaderContent($interface, $dictionaries));

    my $conditionalString = $codeGenerator->GenerateConditionalString($interface);
    push(@headerContent, "\n} // namespace WebCore\n");
    push(@headerContent, "\n#endif // ${conditionalString}\n") if $conditionalString;

    if ($interface->extendedAttributes->{AppleCopyright}) {
        push(@headerContent, "\n");
        push(@headerContent, split("\r", $endAppleCopyright));
    }

    # - Generate dependencies.
    if ($writeDependencies) {
        my @ancestors;
        $codeGenerator->ForAllParents($interface, sub {
            my $currentInterface = shift;
            push(@ancestors, $currentInterface->name);
        }, 0);
        for my $dictionary (@$dictionaries) {
            my $parentName = $dictionary->parent;
            while (defined($parentName)) {
                push(@ancestors, $parentName) if $codeGenerator->IsExternalDictionaryType($parentName);
                my $parentDictionary = $codeGenerator->GetDictionaryByName($parentName);
                assert("Unable to find definition for dictionary named '" . $parentName . "'!") unless defined($parentDictionary);
                $parentName = $parentDictionary->parent;
            }
        }
        push(@depsContent, "$className.h : ", join(" ", map { "$_.idl" } @ancestors), "\n");
        push(@depsContent, map { "$_.idl :\n" } @ancestors);
    }
}

sub GeneratePropertiesHashTable
{
    my ($object, $interface, $isInstance, $hashKeys, $hashSpecials, $hashValue1, $hashValue2, $conditionals, $runtimeEnabledFunctions, $runtimeEnabledAttributes) = @_;

    # FIXME: These should be functions on $interface.
    my $interfaceName = $interface->name;
    my $className = "JS$interfaceName";
    
    # - Add all properties in a hashtable definition
    my $propertyCount = $isInstance ? InstancePropertyCount($interface) : PrototypePropertyCount($interface);

    if (!$isInstance && NeedsConstructorProperty($interface)) {
        die if !$propertyCount;
        push(@$hashKeys, "constructor");
        my $getter = "js" . $interfaceName . "Constructor";
        push(@$hashValue1, $getter);

        my $setter = "setJS" . $interfaceName . "Constructor";
        push(@$hashValue2, $setter);
        push(@$hashSpecials, "DontEnum");
    }

    return 0 if !$propertyCount;

    foreach my $attribute (@{$interface->attributes}) {
        next if ($attribute->isStatic);
        next if AttributeShouldBeOnInstance($interface, $attribute) != $isInstance;

        # Global objects add RuntimeEnabled attributes after creation so do not add them to the static table.
        if (IsDOMGlobalObject($interface) && $attribute->signature->extendedAttributes->{EnabledAtRuntime}) {
            $propertyCount -= 1;
            next;
        }

        my $name = $attribute->signature->name;
        push(@$hashKeys, $name);

        my $special = GetJSCAttributesForAttribute($interface, $attribute);
        push(@$hashSpecials, $special);

        if ($attribute->signature->extendedAttributes->{"DOMJIT"}) {
            push(@$hashValue1, "domJITGetterSetterFor" . $interface->name . $codeGenerator->WK_ucfirst($attribute->signature->name));
            push(@$hashValue2, "0");
        } else {
            my $getter = GetAttributeGetterName($interface, $className, $attribute);
            push(@$hashValue1, $getter);

            if (IsReadonly($attribute)) {
                push(@$hashValue2, "0");
            } else {
                my $setter = GetAttributeSetterName($interface, $className, $attribute);
                push(@$hashValue2, $setter);
            }
        }

        my $conditional = $attribute->signature->extendedAttributes->{Conditional};
        $conditionals->{$name} = $conditional if $conditional;

        if ($attribute->signature->extendedAttributes->{EnabledAtRuntime}) {
            die "We currently do not support [EnabledAtRuntime] attributes on the instance (except for global objects)." if $isInstance;
            push(@$runtimeEnabledAttributes, $attribute);
        }
    }

    my @functions = @{$interface->functions};
    push(@functions, @{$interface->iterable->functions}) if IsKeyValueIterableInterface($interface);
    push(@functions, @{$interface->serializable->functions}) if $interface->serializable;
    foreach my $function (@functions) {
        next if ($function->signature->extendedAttributes->{PrivateIdentifier} and not $function->signature->extendedAttributes->{PublicIdentifier});
        next if ($function->isStatic);
        next if $function->{overloadIndex} && $function->{overloadIndex} > 1;
        next if OperationShouldBeOnInstance($interface, $function) != $isInstance;
        next if $function->signature->name eq "[Symbol.Iterator]";

        # Global objects add RuntimeEnabled operations after creation so do not add them to the static table.
        if (IsDOMGlobalObject($interface) && $function->signature->extendedAttributes->{EnabledAtRuntime}) {
            $propertyCount -= 1;
            next;
        }

        my $name = $function->signature->name;
        push(@$hashKeys, $name);

        my $functionName = GetFunctionName($interface, $className, $function);
        push(@$hashValue1, $functionName);

        my $functionLength = GetFunctionLength($function);
        push(@$hashValue2, $functionLength);

        push(@$hashSpecials, ComputeFunctionSpecial($interface, $function));

        my $conditional = getConditionalForFunctionConsideringOverloads($function);
        $conditionals->{$name} = $conditional if $conditional;

        if ($function->signature->extendedAttributes->{EnabledAtRuntime}) {
            die "We currently do not support [EnabledAtRuntime] operations on the instance (except for global objects)." if $isInstance;
            push(@$runtimeEnabledFunctions, $function);
        }
    }

    return $propertyCount;
}

sub IsNullableType
{
    my $type = shift;

    return substr($type, -1) eq "?";
}

sub StripNullable
{
    my $type = shift;

    chop($type) if IsNullableType($type);
    return $type;
}

# This computes an effective overload set for a given operation / constructor,
# which represents the allowable invocations.This set is used as input for
# the Web IDL overload resolution algorithm.
# http://heycam.github.io/webidl/#dfn-effective-overload-set
sub ComputeEffectiveOverloadSet
{
    my ($overloads) = @_;

    my %allSets;
    my $addTuple = sub {
        my $tuple = shift;
        # The Web IDL specification uses a flat set of tuples but we use a hash where the key is the
        # number of parameters and the value is the set of tuples for the given number of parameters.
        my $length = scalar(@{@$tuple[1]});
        if (!exists($allSets{$length})) {
            $allSets{$length} = [ $tuple ];
        } else {
            push(@{$allSets{$length}}, $tuple);
        }
    };

    my $m = LengthOfLongestFunctionParameterList($overloads);
    foreach my $overload (@{$overloads}) {
        my $n = @{$overload->parameters};
        my @t;
        my @o;
        my $isVariadic = 0;
        foreach my $parameter (@{$overload->parameters}) {
            push(@t, $parameter->idlType);
            if ($parameter->isOptional) {
                push(@o, "optional");
            } elsif ($parameter->isVariadic) {
                push(@o, "variadic");
                $isVariadic = 1;
            } else {
                push(@o, "required");
            }
        }
        &$addTuple([$overload, [@t], [@o]]);
        if ($isVariadic) {
            my @newT = @t;
            my @newO = @o;
            for (my $i = $n; $i < $m; $i++) {
                push(@newT, $t[-1]);
                push(@newO, "variadic");
                &$addTuple([$overload, [@newT], [@newO]]);
            }
        }
        for (my $i = $n - 1; $i >= 0; $i--) {
            my $parameter = @{$overload->parameters}[$i];
            last unless ($parameter->isOptional || $parameter->isVariadic);
            pop(@t);
            pop(@o);
            &$addTuple([$overload, [@t], [@o]]);
        }
    }
    return %allSets;
}

sub IsIDLTypeDistinguishableWithUnionForOverloadResolution
{
    my ($idlType, $unionSubtypes) = @_;

    assert("First type should not be a union") if $idlType->isUnion;
    for my $unionSubType (@$unionSubtypes) {
        return 0 unless AreTypesDistinguishableForOverloadResolution($idlType, $unionSubType);
    }
    return 1;
}

# Determines if two types are distinguishable in the context of overload resolution,
# according to the Web IDL specification:
# http://heycam.github.io/webidl/#dfn-distinguishable
sub AreTypesDistinguishableForOverloadResolution
{
    my ($idlTypeA, $idlTypeB) = @_;

    my $isDictionary = sub {
        my $type = shift;
        return $type eq "Dictionary" || $codeGenerator->IsDictionaryType($type);
    };
    my $isCallbackFunctionOrDictionary = sub {
        my $type = shift;
        return $codeGenerator->IsFunctionOnlyCallbackInterface($type) || &$isDictionary($type);
    };

    # Two types are distinguishable for overload resolution if at most one of the two includes a nullable type.
    return 0 if $idlTypeA->isNullable && $idlTypeB->isNullable;

    # Union types: idlTypeA and idlTypeB  are distinguishable if:
    # - Both types are either a union type or nullable union type, and each member type of the one is
    #   distinguishable with each member type of the other.
    # - One type is a union type or nullable union type, the other is neither a union type nor a nullable
    #   union type, and each member type of the first is distinguishable with the second.
    if ($idlTypeA->isUnion && $idlTypeB->isUnion) {
        for my $unionASubType (@{$idlTypeA->subtypes}) {
            return 0 unless IsIDLTypeDistinguishableWithUnionForOverloadResolution($unionASubType, $idlTypeB->subtypes);
        }
        return 1;
    } elsif ($idlTypeA->isUnion) {
        return IsIDLTypeDistinguishableWithUnionForOverloadResolution($idlTypeB, $idlTypeA->subtypes);
    } elsif ($idlTypeB->isUnion) {
        return IsIDLTypeDistinguishableWithUnionForOverloadResolution($idlTypeA, $idlTypeB->subtypes);
    }

    return 0 if $idlTypeA->name eq $idlTypeB->name;
    return 0 if $idlTypeA->name eq "object" or $idlTypeB->name eq "object";
    return 0 if $codeGenerator->IsNumericType($idlTypeA->name) && $codeGenerator->IsNumericType($idlTypeB->name);
    return 0 if $codeGenerator->IsStringOrEnumType($idlTypeA->name) && $codeGenerator->IsStringOrEnumType($idlTypeB->name);
    return 0 if &$isDictionary($idlTypeA->name) && &$isDictionary($idlTypeB->name);
    return 0 if $codeGenerator->IsCallbackInterface($idlTypeA->name) && $codeGenerator->IsCallbackInterface($idlTypeB->name);
    return 0 if &$isCallbackFunctionOrDictionary($idlTypeA->name) && &$isCallbackFunctionOrDictionary($idlTypeB->name);
    return 0 if $codeGenerator->IsSequenceOrFrozenArrayType($idlTypeA->name) && $codeGenerator->IsSequenceOrFrozenArrayType($idlTypeB->name);
    # FIXME: return 0 if $idlTypeA and $idlTypeB are both exception types.
    return 1;
}

# If there is more than one entry in an effective overload set that has a given type list length,
# then for those entries there must be an index i such that for each pair of entries the types
# at index i are distinguishable. The lowest such index is termed the distinguishing argument index.
# http://heycam.github.io/webidl/#dfn-distinguishing-argument-index
sub GetDistinguishingArgumentIndex
{
    my ($function, $S) = @_;

    # FIXME: Consider all the tuples, not just the 2 first ones?
    my $firstTupleIDLTypes = @{@{$S}[0]}[1];
    my $secondTupleIDLTypes = @{@{$S}[1]}[1];
    for (my $index = 0; $index < scalar(@$firstTupleIDLTypes); $index++) {
        return $index if AreTypesDistinguishableForOverloadResolution(@{$firstTupleIDLTypes}[$index], @{$secondTupleIDLTypes}[$index]);
    }
    die "Undistinguishable overloads for operation " . $function->signature->name . " with length: " . scalar(@$firstTupleIDLTypes);
}

sub GetOverloadThatMatches
{
    my ($S, $parameterIndex, $matches) = @_;

    for my $tuple (@{$S}) {
        my $idlType = @{@{$tuple}[1]}[$parameterIndex];
        my $optionality = @{@{$tuple}[2]}[$parameterIndex];
        if ($idlType->isUnion) {
            for my $idlSubtype (GetFlattenedMemberTypes($idlType)) {
                return @{$tuple}[0] if $matches->($idlSubtype, $optionality);
            }
        } else {
            return @{$tuple}[0] if $matches->($idlType, $optionality);
        }
    }
}

sub GetOverloadThatMatchesIgnoringUnionSubtypes
{
    my ($S, $parameterIndex, $matches) = @_;

    for my $tuple (@{$S}) {
        my $idlType = @{@{$tuple}[1]}[$parameterIndex];
        my $optionality = @{@{$tuple}[2]}[$parameterIndex];
        return @{$tuple}[0] if $matches->($idlType, $optionality);
    }
}

sub getConditionalForFunctionConsideringOverloads
{
    my $function = shift;

    return $function->signature->extendedAttributes->{Conditional} unless $function->{overloads};

    my %conditions;
    foreach my $overload (@{$function->{overloads}}) {
        my $conditional = $overload->signature->extendedAttributes->{Conditional};
        return unless $conditional;
        $conditions{$conditional} = 1;
    }
    return join("|", keys %conditions);
}

# Implements the overload resolution algorithm, as defined in the Web IDL specification:
# http://heycam.github.io/webidl/#es-overloads
sub GenerateOverloadedFunctionOrConstructor
{
    my ($function, $interface, $isConstructor) = @_;
    my %allSets = ComputeEffectiveOverloadSet($function->{overloads});

    my $interfaceName = $interface->name;
    my $className = "JS$interfaceName";
    my $functionName;
    if ($isConstructor) {
        $functionName = "construct${className}";
    } else {
        my $kind = $function->isStatic ? "Constructor" : (OperationShouldBeOnInstance($interface, $function) ? "Instance" : "Prototype");
        $functionName = "js${interfaceName}${kind}Function" . $codeGenerator->WK_ucfirst($function->signature->name);
    }

    my $generateOverloadCallIfNecessary = sub {
        my ($overload, $condition) = @_;
        return unless $overload;
        my $conditionalString = $codeGenerator->GenerateConditionalString($overload->signature);
        push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
        push(@implContent, "        if ($condition)\n    ") if $condition;
        push(@implContent, "        return ${functionName}$overload->{overloadIndex}(state);\n");
        push(@implContent, "#endif\n") if $conditionalString;
    };
    my $isOptionalParameter = sub {
        my ($idlType, $optionality) = @_;
        return $optionality eq "optional";
    };
    my $isDictionaryParameter = sub {
        my ($idlType, $optionality) = @_;
        return $idlType->name eq "Dictionary" || $codeGenerator->IsDictionaryType($idlType->name);
    };
    my $isNullableOrDictionaryParameterOrUnionContainingOne = sub {
        my ($idlType, $optionality) = @_;
        return 1 if $idlType->isNullable;
        if ($idlType->isUnion) {
            for my $idlSubtype (GetFlattenedMemberTypes($idlType)) {
                return 1 if $idlType->isNullable || &$isDictionaryParameter($idlSubtype, $optionality);
            }
            return 0;
        } else {
            return &$isDictionaryParameter($idlType, $optionality);
        }
    };
    my $isRegExpOrObjectParameter = sub {
        my ($idlType, $optionality) = @_;
        return $idlType->name eq "RegExp" || $idlType->name eq "object";
    };
    my $isObjectOrErrorParameter = sub {
        my ($idlType, $optionality) = @_;
        return $idlType->name eq "object" || $idlType->name eq "Error";
    };
    my $isObjectOrErrorOrDOMExceptionParameter = sub {
        my ($idlType, $optionality) = @_;
        return 1 if &$isObjectOrErrorParameter($idlType, $optionality);
        return $idlType->name eq "DOMException";
    };
    my $isObjectOrCallbackFunctionParameter = sub {
        my ($idlType, $optionality) = @_;
        return $idlType->name eq "object" || $codeGenerator->IsFunctionOnlyCallbackInterface($idlType->name);
    };
    my $isSequenceOrFrozenArrayParameter = sub {
        my ($idlType, $optionality) = @_;
        return $codeGenerator->IsSequenceOrFrozenArrayType($idlType->name);
    };
    my $isDictionaryOrObjectOrCallbackInterfaceParameter = sub {
        my ($idlType, $optionality) = @_;
        return 1 if &$isDictionaryParameter($idlType, $optionality);
        return 1 if $idlType->name eq "object";
        return 1 if $codeGenerator->IsCallbackInterface($idlType->name) && !$codeGenerator->IsFunctionOnlyCallbackInterface($idlType->name);
        return 0;
    };
    my $isBooleanParameter = sub {
        my ($idlType, $optionality) = @_;
        return $idlType->name eq "boolean";
    };
    my $isNumericParameter = sub {
        my ($idlType, $optionality) = @_;
        return $codeGenerator->IsNumericType($idlType->name);
    };
    my $isStringOrEnumParameter = sub {
        my ($idlType, $optionality) = @_;
        return $codeGenerator->IsStringOrEnumType($idlType->name);
    };
    my $isAnyParameter = sub {
        my ($idlType, $optionality) = @_;
        return $idlType->name eq "any";
    };

    my $maxArgCount = LengthOfLongestFunctionParameterList($function->{overloads});

    my $conditionalAttribute = getConditionalForFunctionConsideringOverloads($function);
    my $conditionalString = $conditionalAttribute ? $codeGenerator->GenerateConditionalStringFromAttributeValue($conditionalAttribute) : undef;
    push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
    if ($isConstructor) {
        push(@implContent, "template<> EncodedJSValue JSC_HOST_CALL ${className}Constructor::construct(ExecState* state)\n");
    } else {
        push(@implContent, "EncodedJSValue JSC_HOST_CALL ${functionName}(ExecState* state)\n");
    }
    push(@implContent, <<END);    
{
    VM& vm = state->vm();
    auto throwScope = DECLARE_THROW_SCOPE(vm);
    UNUSED_PARAM(throwScope);
    size_t argsCount = std::min<size_t>($maxArgCount, state->argumentCount());
END

    for my $length ( sort keys %allSets ) {
        push(@implContent, <<END);
    if (argsCount == $length) {
END
        my $S = $allSets{$length};
        if (scalar(@$S) > 1) {
            my $d = GetDistinguishingArgumentIndex($function, $S);
            push(@implContent, "        JSValue distinguishingArg = state->uncheckedArgument($d);\n");

            my $overload = GetOverloadThatMatchesIgnoringUnionSubtypes($S, $d, \&$isOptionalParameter);
            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isUndefined()");

            $overload = GetOverloadThatMatchesIgnoringUnionSubtypes($S, $d, \&$isNullableOrDictionaryParameterOrUnionContainingOne);
            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isUndefinedOrNull()");

            for my $tuple (@{$S}) {
                my $overload = @{$tuple}[0];
                my $idlType = @{@{$tuple}[1]}[$d];

                my @idlSubtypes = $idlType->isUnion ? GetFlattenedMemberTypes($idlType) : ( $idlType );
                for my $idlSubtype (@idlSubtypes) {
                    my $type = $idlSubtype->name;
                    if ($codeGenerator->IsWrapperType($type) || $codeGenerator->IsTypedArrayType($type)) {
                        if ($type eq "DOMWindow") {
                            AddToImplIncludes("JSDOMWindowShell.h");
                            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isObject() && (asObject(distinguishingArg)->inherits(JSDOMWindowShell::info()) || asObject(distinguishingArg)->inherits(JSDOMWindow::info()))");
                        } else {
                            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isObject() && asObject(distinguishingArg)->inherits(JS${type}::info())");
                        }
                    }
                }
            }

            $overload = GetOverloadThatMatches($S, $d, \&$isRegExpOrObjectParameter);
            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isObject() && asObject(distinguishingArg)->type() == RegExpObjectType");

            $overload = GetOverloadThatMatches($S, $d, \&$isObjectOrErrorOrDOMExceptionParameter);
            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isObject() && asObject(distinguishingArg)->inherits(JSDOMCoreException::info())");

            $overload = GetOverloadThatMatches($S, $d, \&$isObjectOrErrorParameter);
            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isObject() && asObject(distinguishingArg)->type() == ErrorInstanceType");

            $overload = GetOverloadThatMatches($S, $d, \&$isObjectOrCallbackFunctionParameter);
            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isFunction()");

            # FIXME: Avoid invoking GetMethod(object, Symbol.iterator) again in convert<IDLSequence<T>>(...).
            $overload = GetOverloadThatMatches($S, $d, \&$isSequenceOrFrozenArrayParameter);
            &$generateOverloadCallIfNecessary($overload, "hasIteratorMethod(*state, distinguishingArg)");

            $overload = GetOverloadThatMatches($S, $d, \&$isDictionaryOrObjectOrCallbackInterfaceParameter);
            &$generateOverloadCallIfNecessary($overload, "distinguishingArg.isObject() && asObject(distinguishingArg)->type() != RegExpObjectType");

            my $booleanOverload = GetOverloadThatMatches($S, $d, \&$isBooleanParameter);
            &$generateOverloadCallIfNecessary($booleanOverload, "distinguishingArg.isBoolean()");

            my $numericOverload = GetOverloadThatMatches($S, $d, \&$isNumericParameter);
            &$generateOverloadCallIfNecessary($numericOverload, "distinguishingArg.isNumber()");

            # Fallbacks.
            $overload = GetOverloadThatMatches($S, $d, \&$isStringOrEnumParameter);
            if ($overload) {
                &$generateOverloadCallIfNecessary($overload);
            } elsif ($numericOverload) {
                &$generateOverloadCallIfNecessary($numericOverload);
            } elsif ($booleanOverload) {
                &$generateOverloadCallIfNecessary($booleanOverload);
            } else {
                $overload = GetOverloadThatMatches($S, $d, \&$isAnyParameter);
                &$generateOverloadCallIfNecessary($overload);
            }
        } else {
            # Only 1 overload with this number of parameters.
            my $overload = @{@{$S}[0]}[0];
            &$generateOverloadCallIfNecessary($overload);
        }
        push(@implContent, <<END);
    }
END
    }
    my $minArgCount = GetFunctionLength($function);
    if ($minArgCount > 0) {
        push(@implContent, "    return argsCount < $minArgCount ? throwVMError(state, throwScope, createNotEnoughArgumentsError(state)) : throwVMTypeError(state, throwScope);\n")
    } else {
        push(@implContent, "    return throwVMTypeError(state, throwScope);\n")
    }
    push(@implContent, "}\n");
    push(@implContent, "#endif\n") if $conditionalString;
    push(@implContent, "\n");
}

# As per Web IDL specification, the length of a function Object is its number of mandatory parameters.
sub GetFunctionLength
{
    my $function = shift;

    my $getOverloadLength = sub {
        my $function = shift;

        my $length = 0;
        foreach my $parameter (@{$function->parameters}) {
            last if $parameter->isOptional || $parameter->isVariadic;
            $length++;
        }
        return $length;
    };

    my $length = &$getOverloadLength($function);
    foreach my $overload (@{$function->{overloads}}) {
        my $newLength = &$getOverloadLength($overload);
        $length = $newLength if $newLength < $length;
    }
    return $length;
}

sub LengthOfLongestFunctionParameterList
{
    my ($overloads) = @_;
    my $result = 0;
    foreach my $overload (@{$overloads}) {
        my @parameters = @{$overload->parameters};
        $result = @parameters if $result < @parameters;
    }
    return $result;
}

sub GetNativeTypeForConversions
{
    my $interface = shift;
    my $interfaceName = $interface->name;
    $interfaceName = $codeGenerator->GetSVGTypeNeedingTearOff($interfaceName) if $codeGenerator->IsSVGTypeNeedingTearOff($interfaceName);
    return $interfaceName;
}

# See http://refspecs.linux-foundation.org/cxxabi-1.83.html.
sub GetGnuVTableRefForInterface
{
    my $interface = shift;
    my $vtableName = GetGnuVTableNameForInterface($interface);
    if (!$vtableName) {
        return "0";
    }
    my $typename = GetNativeTypeForConversions($interface);
    my $offset = GetGnuVTableOffsetForType($typename);
    return "&" . $vtableName . "[" . $offset . "]";
}

sub GetGnuVTableNameForInterface
{
    my $interface = shift;
    my $typename = GetNativeTypeForConversions($interface);
    my $templatePosition = index($typename, "<");
    return "" if $templatePosition != -1;
    return "" if GetImplementationLacksVTableForInterface($interface);
    return "" if GetSkipVTableValidationForInterface($interface);
    return "_ZTV" . GetGnuMangledNameForInterface($interface);
}

sub GetGnuMangledNameForInterface
{
    my $interface = shift;
    my $typename = GetNativeTypeForConversions($interface);
    my $templatePosition = index($typename, "<");
    if ($templatePosition != -1) {
        return "";
    }
    my $mangledType = length($typename) . $typename;
    my $namespace = GetNamespaceForInterface($interface);
    my $mangledNamespace =  "N" . length($namespace) . $namespace;
    return $mangledNamespace . $mangledType . "E";
}

sub GetGnuVTableOffsetForType
{
    my $typename = shift;
    if ($typename eq "SVGAElement"
        || $typename eq "SVGCircleElement"
        || $typename eq "SVGClipPathElement"
        || $typename eq "SVGDefsElement"
        || $typename eq "SVGEllipseElement"
        || $typename eq "SVGForeignObjectElement"
        || $typename eq "SVGGElement"
        || $typename eq "SVGImageElement"
        || $typename eq "SVGLineElement"
        || $typename eq "SVGPathElement"
        || $typename eq "SVGPolyElement"
        || $typename eq "SVGPolygonElement"
        || $typename eq "SVGPolylineElement"
        || $typename eq "SVGRectElement"
        || $typename eq "SVGSVGElement"
        || $typename eq "SVGGraphicsElement"
        || $typename eq "SVGSwitchElement"
        || $typename eq "SVGTextElement"
        || $typename eq "SVGUseElement") {
        return "3";
    }
    return "2";
}

# See http://en.wikipedia.org/wiki/Microsoft_Visual_C%2B%2B_Name_Mangling.
sub GetWinVTableRefForInterface
{
    my $interface = shift;
    my $vtableName = GetWinVTableNameForInterface($interface);
    return 0 if !$vtableName;
    return "__identifier(\"" . $vtableName . "\")";
}

sub GetWinVTableNameForInterface
{
    my $interface = shift;
    my $typename = GetNativeTypeForConversions($interface);
    my $templatePosition = index($typename, "<");
    return "" if $templatePosition != -1;
    return "" if GetImplementationLacksVTableForInterface($interface);
    return "" if GetSkipVTableValidationForInterface($interface);
    return "??_7" . GetWinMangledNameForInterface($interface) . "6B@";
}

sub GetWinMangledNameForInterface
{
    my $interface = shift;
    my $typename = GetNativeTypeForConversions($interface);
    my $namespace = GetNamespaceForInterface($interface);
    return $typename . "@" . $namespace . "@@";
}

sub GetNamespaceForInterface
{
    my $interface = shift;
    return "WebCore";
}

sub GetImplementationLacksVTableForInterface
{
    my $interface = shift;
    return $interface->extendedAttributes->{ImplementationLacksVTable};
}

sub GetSkipVTableValidationForInterface
{
    my $interface = shift;
    return $interface->extendedAttributes->{SkipVTableValidation};
}

# URL becomes url, but SetURL becomes setURL.
sub ToMethodName
{
    my $param = shift;
    my $ret = lcfirst($param);
    $ret =~ s/cSS/css/ if $ret =~ /^cSS/;
    $ret =~ s/dOM/dom/ if $ret =~ /^dOM/;
    $ret =~ s/hTML/html/ if $ret =~ /^hTML/;
    $ret =~ s/jS/js/ if $ret =~ /^jS/;
    $ret =~ s/uRL/url/ if $ret =~ /^uRL/;
    $ret =~ s/xML/xml/ if $ret =~ /^xML/;
    $ret =~ s/xSLT/xslt/ if $ret =~ /^xSLT/;

    # For HTML5 FileSystem API Flags attributes.
    # (create is widely used to instantiate an object and must be avoided.)
    $ret =~ s/^create/isCreate/ if $ret =~ /^create$/;
    $ret =~ s/^exclusive/isExclusive/ if $ret =~ /^exclusive$/;

    return $ret;
}

# Returns the RuntimeEnabledFeatures function name that is hooked up to check if a method/attribute is enabled.
sub GetRuntimeEnableFunctionName
{
    my $signature = shift;

    # If a parameter is given (e.g. "EnabledAtRuntime=FeatureName") return the RuntimeEnabledFeatures::sharedFeatures().{FeatureName}Enabled() method.
    return "RuntimeEnabledFeatures::sharedFeatures()." . ToMethodName($signature->extendedAttributes->{EnabledAtRuntime}) . "Enabled" if ($signature->extendedAttributes->{EnabledAtRuntime} && $signature->extendedAttributes->{EnabledAtRuntime} ne "VALUE_IS_MISSING");

    # Otherwise return a function named RuntimeEnabledFeatures::sharedFeatures().{methodName}Enabled().
    return "RuntimeEnabledFeatures::sharedFeatures()." . ToMethodName($signature->name) . "Enabled";
}

sub GetCastingHelperForThisObject
{
    my $interface = shift;
    my $interfaceName = $interface->name;

    return "jsNodeCast" if $interfaceName eq "Node";
    return "jsElementCast" if $interfaceName eq "Element";
    return "jsDocumentCast" if $interfaceName eq "Document";
    return "jsEventTargetCast" if $interfaceName eq "EventTarget";
    return "jsDynamicCast<JS$interfaceName*>";
}

sub GetIndexedGetterExpression
{
    my $indexedGetterFunction = shift;
    return "jsStringOrUndefined(state, thisObject->wrapped().item(index))" if $indexedGetterFunction->signature->type eq "DOMString";
    return "toJS(state, thisObject->globalObject(), thisObject->wrapped().item(index))";
}

# http://heycam.github.io/webidl/#Unscopable
sub addUnscopableProperties
{
    my $interface = shift;

    my @unscopables;
    foreach my $functionOrAttribute (@{$interface->functions}, @{$interface->attributes}) {
        push(@unscopables, $functionOrAttribute->signature->name) if $functionOrAttribute->signature->extendedAttributes->{Unscopable};
    }
    return if scalar(@unscopables) == 0;

    AddToImplIncludes("<runtime/ObjectConstructor.h>");
    push(@implContent, "    JSObject& unscopables = *constructEmptyObject(globalObject()->globalExec(), globalObject()->nullPrototypeObjectStructure());\n");
    foreach my $unscopable (@unscopables) {
        push(@implContent, "    unscopables.putDirect(vm, Identifier::fromString(&vm, \"$unscopable\"), jsBoolean(true));\n");
    }
    push(@implContent, "    putDirectWithoutTransition(vm, vm.propertyNames->unscopablesSymbol, &unscopables, DontEnum | ReadOnly);\n");
}

sub GetResultTypeFilter
{
    my ($signature) = @_;
    my $idlType = $signature->idlType;

    my %TypeFilters = (
        "any" => "SpecHeapTop",
        "boolean" => "SpecBoolean",
        "byte" => "SpecInt32Only",
        "octet" => "SpecInt32Only",
        "short" => "SpecInt32Only",
        "unsigned short" => "SpecInt32Only",
        "long" => "SpecInt32Only",
        "unsigned long" => "SpecBytecodeNumber",
        "long long" => "SpecBytecodeNumber",
        "unsigned long long" => "SpecBytecodeNumber",
        "float" => "SpecBytecodeNumber",
        "unrestricted float" => "SpecBytecodeNumber",
        "double" => "SpecBytecodeNumber",
        "unrestricted double" => "SpecBytecodeNumber",
        "DOMString" => "SpecString",
        "ByteString" => "SpecString",
        "USVString" => "SpecString",
    );

    if (exists $TypeFilters{$idlType->name}) {
        my $resultType = "JSC::$TypeFilters{$idlType->name}";
        if ($idlType->isNullable) {
            die "\"any\" type must not become nullable." if $idlType->name eq "any";
            $resultType = "($resultType | JSC::SpecOther)";
        }
        return $resultType;
    }
    return "SpecHeapTop";
}

sub GenerateImplementation
{
    my ($object, $interface, $enumerations, $dictionaries) = @_;

    my $interfaceName = $interface->name;
    my $className = "JS$interfaceName";

    my $hasLegacyParent = $interface->extendedAttributes->{JSLegacyParent};
    my $hasRealParent = $interface->parent;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    my $parentClassName = GetParentClassName($interface);
    my $visibleInterfaceName = $codeGenerator->GetVisibleInterfaceName($interface);
    my $eventTarget = $codeGenerator->InheritsInterface($interface, "EventTarget") && $interface->name ne "EventTarget";
    my $needsVisitChildren = InstanceNeedsVisitChildren($interface);

    my $namedGetterFunction = GetNamedGetterFunction($interface);
    my $indexedGetterFunction = GetIndexedGetterFunction($interface);

    # - Add default header template
    push(@implContentHeader, GenerateImplementationContentHeader($interface));

    $implIncludes{"JSDOMBinding.h"} = 1;
    $implIncludes{"<wtf/GetPtr.h>"} = 1;
    $implIncludes{"<runtime/PropertyNameArray.h>"} = 1 if $indexedGetterFunction;

    my $implType = GetImplClassName($interfaceName);

    AddJSBuiltinIncludesIfNeeded($interface);

    @implContent = ();

    push(@implContent, "\nusing namespace JSC;\n\n");
    push(@implContent, "namespace WebCore {\n\n");

    push(@implContent, GenerateEnumerationsImplementationContent($interface, $enumerations));
    push(@implContent, GenerateDictionariesImplementationContent($interface, $dictionaries));

    my @functions = @{$interface->functions};
    push(@functions, @{$interface->iterable->functions}) if IsKeyValueIterableInterface($interface);
    push(@functions, @{$interface->serializable->functions}) if $interface->serializable;

    my $numConstants = @{$interface->constants};
    my $numFunctions = @functions;
    my $numAttributes = @{$interface->attributes};

    if ($numFunctions > 0) {
        my $inAppleCopyright = 0;
        push(@implContent,"// Functions\n\n");
        foreach my $function (@functions) {
            next if $function->{overloadIndex} && $function->{overloadIndex} > 1;
            next if $function->signature->extendedAttributes->{ForwardDeclareInHeader};
            next if IsJSBuiltin($interface, $function);

            if ($function->signature->extendedAttributes->{AppleCopyright}) {
                if (!$inAppleCopyright) {
                    push(@implContent, $beginAppleCopyrightForHeaderFiles);
                    $inAppleCopyright = 1;
                }
            } elsif ($inAppleCopyright) {
                push(@implContent, $endAppleCopyright);
                $inAppleCopyright = 0;
            }

            my $conditionalAttribute = getConditionalForFunctionConsideringOverloads($function);
            my $conditionalString = $conditionalAttribute ? $codeGenerator->GenerateConditionalStringFromAttributeValue($conditionalAttribute) : undef;
            push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
            my $functionName = GetFunctionName($interface, $className, $function);
            push(@implContent, "JSC::EncodedJSValue JSC_HOST_CALL ${functionName}(JSC::ExecState*);\n");
            push(@implContent, "#endif\n") if $conditionalString;
        }

        push(@implContent, $endAppleCopyright) if $inAppleCopyright;
        push(@implContent, "\n");
    }

    if ($numAttributes > 0 || NeedsConstructorProperty($interface)) {
        push(@implContent, "// Attributes\n\n");

        foreach my $attribute (@{$interface->attributes}) {
            next if $attribute->signature->extendedAttributes->{ForwardDeclareInHeader};
            next if IsJSBuiltin($interface, $attribute);

            my $conditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
            push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
            my $getter = GetAttributeGetterName($interface, $className, $attribute);
            push(@implContent, "JSC::EncodedJSValue ${getter}(JSC::ExecState*, JSC::EncodedJSValue, JSC::PropertyName);\n");
            if (!IsReadonly($attribute)) {
                my $setter = GetAttributeSetterName($interface, $className, $attribute);
                push(@implContent, "bool ${setter}(JSC::ExecState*, JSC::EncodedJSValue, JSC::EncodedJSValue);\n");
            }
            push(@implContent, "#endif\n") if $conditionalString;
        }
        
        if (NeedsConstructorProperty($interface)) {
            my $getter = "js" . $interfaceName . "Constructor";
            push(@implContent, "JSC::EncodedJSValue ${getter}(JSC::ExecState*, JSC::EncodedJSValue, JSC::PropertyName);\n");
        }

        my $constructorFunctionName = "setJS" . $interfaceName . "Constructor";
        push(@implContent, "bool ${constructorFunctionName}(JSC::ExecState*, JSC::EncodedJSValue, JSC::EncodedJSValue);\n");

        push(@implContent, "\n");
    }

    GeneratePrototypeDeclaration(\@implContent, $className, $interface) if !HeaderNeedsPrototypeDeclaration($interface);

    GenerateConstructorDeclaration(\@implContent, $className, $interface) if NeedsConstructorProperty($interface);

    my @hashKeys = ();
    my @hashValue1 = ();
    my @hashValue2 = ();
    my @hashSpecials = ();
    my %conditionals = ();
    my $hashName = $className . "Table";
    my @runtimeEnabledFunctions = ();
    my @runtimeEnabledAttributes = ();

    # Generate hash table for properties on the instance.
    my $numInstanceProperties = GeneratePropertiesHashTable($object, $interface, 1,
        \@hashKeys, \@hashSpecials,
        \@hashValue1, \@hashValue2,
        \%conditionals, \@runtimeEnabledFunctions, \@runtimeEnabledAttributes);

    $object->GenerateHashTable($hashName, $numInstanceProperties,
        \@hashKeys, \@hashSpecials,
        \@hashValue1, \@hashValue2,
        \%conditionals, 0) if $numInstanceProperties > 0;

    # - Add all interface object (aka constructor) properties (constants, static attributes, static operations).
    if (NeedsConstructorProperty($interface)) {
        my $hashSize = 0;
        my $hashName = $className . "ConstructorTable";

        my @hashKeys = ();
        my @hashValue1 = ();
        my @hashValue2 = ();
        my @hashSpecials = ();
        my %conditionals = ();

        my $needsConstructorTable = 0;

        foreach my $constant (@{$interface->constants}) {
            my $name = $constant->name;
            push(@hashKeys, $name);
            push(@hashValue1, $constant->value);
            push(@hashValue2, "0");
            push(@hashSpecials, "DontDelete | ReadOnly | ConstantInteger");

            my $implementedBy = $constant->extendedAttributes->{ImplementedBy};
            $implIncludes{"${implementedBy}.h"} = 1 if $implementedBy;

            my $conditional = $constant->extendedAttributes->{Conditional};
            $conditionals{$name} = $conditional if $conditional;

            $hashSize++;
        }

        foreach my $attribute (@{$interface->attributes}) {
            next unless ($attribute->isStatic);
            my $name = $attribute->signature->name;
            push(@hashKeys, $name);

            my @specials = ();
            push(@specials, "DontDelete") if IsUnforgeable($interface, $attribute);
            push(@specials, "ReadOnly") if IsReadonly($attribute);
            push(@specials, "DOMJITAttribute") if $attribute->signature->extendedAttributes->{"DOMJIT"};
            my $special = (@specials > 0) ? join(" | ", @specials) : "0";
            push(@hashSpecials, $special);

            if ($attribute->signature->extendedAttributes->{"DOMJIT"}) {
                push(@hashValue1, "domJITGetterSetterFor" . $interface->name . $codeGenerator->WK_ucfirst($attribute->signature->name));
                push(@hashValue2, "0");
            } else {
                my $getter = GetAttributeGetterName($interface, $className, $attribute);
                push(@hashValue1, $getter);

                if (IsReadonly($attribute)) {
                    push(@hashValue2, "0");
                } else {
                    my $setter = GetAttributeSetterName($interface, $className, $attribute);
                    push(@hashValue2, $setter);
                }
            }

            my $conditional = $attribute->signature->extendedAttributes->{Conditional};
            $conditionals{$name} = $conditional if $conditional;

            $hashSize++;
        }

        foreach my $function (@{$interface->functions}) {
            next unless ($function->isStatic);
            next if $function->{overloadIndex} && $function->{overloadIndex} > 1;
            my $name = $function->signature->name;
            push(@hashKeys, $name);

            my $functionName = GetFunctionName($interface, $className, $function);
            push(@hashValue1, $functionName);

            my $functionLength = GetFunctionLength($function);
            push(@hashValue2, $functionLength);

            push(@hashSpecials, ComputeFunctionSpecial($interface, $function));

            my $conditional = $function->signature->extendedAttributes->{Conditional};
            $conditionals{$name} = $conditional if $conditional;

            $hashSize++;
        }

        $object->GenerateHashTable($hashName, $hashSize,
                                   \@hashKeys, \@hashSpecials,
                                   \@hashValue1, \@hashValue2,
                                   \%conditionals, 1) if $hashSize > 0;

        push(@implContent, $codeGenerator->GenerateCompileTimeCheckForEnumsIfNeeded($interface));

        my $protoClassName = "${className}Prototype";
        GenerateConstructorDefinitions(\@implContent, $className, $protoClassName, $visibleInterfaceName, $interface);

        my $namedConstructor = $interface->extendedAttributes->{NamedConstructor};
        GenerateConstructorDefinitions(\@implContent, $className, $protoClassName, $namedConstructor, $interface, "GeneratingNamedConstructor") if $namedConstructor;
    }

    # - Add functions and constants to a hashtable definition

    $hashName = $className . "PrototypeTable";

    @hashKeys = ();
    @hashValue1 = ();
    @hashValue2 = ();
    @hashSpecials = ();
    %conditionals = ();
    @runtimeEnabledFunctions = ();
    @runtimeEnabledAttributes = ();

    # Generate hash table for properties on the prototype.
    my $numPrototypeProperties = GeneratePropertiesHashTable($object, $interface, 0,
        \@hashKeys, \@hashSpecials,
        \@hashValue1, \@hashValue2,
        \%conditionals, \@runtimeEnabledFunctions, \@runtimeEnabledAttributes);
    my $hashSize = $numPrototypeProperties;

    foreach my $constant (@{$interface->constants}) {
        my $name = $constant->name;

        push(@hashKeys, $name);
        push(@hashValue1, $constant->value);
        push(@hashValue2, "0");
        push(@hashSpecials, "DontDelete | ReadOnly | ConstantInteger");

        my $conditional = $constant->extendedAttributes->{Conditional};
        $conditionals{$name} = $conditional if $conditional;

        $hashSize++;
    }

    my $justGenerateValueArray = !IsDOMGlobalObject($interface);

    $object->GenerateHashTable($hashName, $hashSize,
                               \@hashKeys, \@hashSpecials,
                               \@hashValue1, \@hashValue2,
                               \%conditionals, $justGenerateValueArray);

    if ($justGenerateValueArray) {
        push(@implContent, "const ClassInfo ${className}Prototype::s_info = { \"${visibleInterfaceName}Prototype\", &Base::s_info, 0, CREATE_METHOD_TABLE(${className}Prototype) };\n\n");
    } else {
        push(@implContent, "const ClassInfo ${className}Prototype::s_info = { \"${visibleInterfaceName}Prototype\", &Base::s_info, &${className}PrototypeTable, CREATE_METHOD_TABLE(${className}Prototype) };\n\n");
    }

    if (PrototypeHasStaticPropertyTable($interface) && !IsDOMGlobalObject($interface)) {
        push(@implContent, "void ${className}Prototype::finishCreation(VM& vm)\n");
        push(@implContent, "{\n");
        push(@implContent, "    Base::finishCreation(vm);\n");
        push(@implContent, "    reifyStaticProperties(vm, ${className}PrototypeTableValues, *this);\n");

        my @runtimeEnabledProperties = @runtimeEnabledFunctions;
        push(@runtimeEnabledProperties, @runtimeEnabledAttributes);
        foreach my $functionOrAttribute (@runtimeEnabledProperties) {
            my $signature = $functionOrAttribute->signature;
            my $conditionalString = $codeGenerator->GenerateConditionalString($signature);
            push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
            AddToImplIncludes("RuntimeEnabledFeatures.h");
            my $enable_function = GetRuntimeEnableFunctionName($signature);
            my $name = $signature->name;
            push(@implContent, "    if (!${enable_function}()) {\n");
            push(@implContent, "        Identifier propertyName = Identifier::fromString(&vm, reinterpret_cast<const LChar*>(\"$name\"), strlen(\"$name\"));\n");
            push(@implContent, "        VM::DeletePropertyModeScope scope(vm, VM::DeletePropertyMode::IgnoreConfigurable);\n");
            push(@implContent, "        JSObject::deleteProperty(this, globalObject()->globalExec(), propertyName);\n");
            push(@implContent, "    }\n");
            push(@implContent, "#endif\n") if $conditionalString;
        }

        my $firstPrivateFunction = 1;
        foreach my $function (@{$interface->functions}) {
            next unless ($function->signature->extendedAttributes->{PrivateIdentifier});
            AddToImplIncludes("WebCoreJSClientData.h");
            push(@implContent, "    JSVMClientData& clientData = *static_cast<JSVMClientData*>(vm.clientData);\n") if $firstPrivateFunction;
            $firstPrivateFunction = 0;
            push(@implContent, "    putDirect(vm, clientData.builtinNames()." . $function->signature->name . "PrivateName(), JSFunction::create(vm, globalObject(), 0, String(), " . GetFunctionName($interface, $className, $function) . "), ReadOnly | DontEnum);\n");
        }

        if ($interface->iterable) {
            addIterableProperties($interface, $className);
        }

        addUnscopableProperties($interface);

        push(@implContent, "}\n\n");
    }

    if ($interface->extendedAttributes->{JSCustomNamedGetterOnPrototype}) {
        push(@implContent, "bool ${className}Prototype::put(JSCell* cell, ExecState* state, PropertyName propertyName, JSValue value, PutPropertySlot& slot)\n");
        push(@implContent, "{\n");
        push(@implContent, "    auto* thisObject = jsCast<${className}Prototype*>(cell);\n");
        push(@implContent, "    bool putResult = false;\n");
        push(@implContent, "    if (thisObject->putDelegate(state, propertyName, value, slot, putResult))\n");
        push(@implContent, "        return putResult;\n");
        push(@implContent, "    return Base::put(thisObject, state, propertyName, value, slot);\n");
        push(@implContent, "}\n\n");
    }

    # - Initialize static ClassInfo object
    push(@implContent, "const ClassInfo $className" . "::s_info = { \"${visibleInterfaceName}\", &Base::s_info, ");

    if ($numInstanceProperties > 0) {
        push(@implContent, "&${className}Table");
    } else {
        push(@implContent, "0");
    }
    push(@implContent, ", CREATE_METHOD_TABLE($className) };\n\n");

    my ($svgPropertyType, $svgListPropertyType, $svgNativeType) = GetSVGPropertyTypes($interfaceName);
    my $svgPropertyOrListPropertyType;
    $svgPropertyOrListPropertyType = $svgPropertyType if $svgPropertyType;
    $svgPropertyOrListPropertyType = $svgListPropertyType if $svgListPropertyType;

    # Constructor
    if ($interfaceName eq "DOMWindow") {
        AddIncludesForTypeInImpl("JSDOMWindowShell");
        push(@implContent, "${className}::$className(VM& vm, Structure* structure, Ref<$implType>&& impl, JSDOMWindowShell* shell)\n");
        push(@implContent, "    : $parentClassName(vm, structure, WTFMove(impl), shell)\n");
        push(@implContent, "{\n");
        push(@implContent, "}\n\n");
    } elsif ($codeGenerator->InheritsInterface($interface, "WorkerGlobalScope")) {
        AddIncludesForTypeInImpl($interfaceName);
        push(@implContent, "${className}::$className(VM& vm, Structure* structure, Ref<$implType>&& impl)\n");
        push(@implContent, "    : $parentClassName(vm, structure, WTFMove(impl))\n");
        push(@implContent, "{\n");
        push(@implContent, "}\n\n");
    } elsif (!NeedsImplementationClass($interface)) {
        push(@implContent, "${className}::$className(Structure* structure, JSDOMGlobalObject& globalObject)\n");
        push(@implContent, "    : $parentClassName(structure, globalObject) { }\n\n");
    } else {
        push(@implContent, "${className}::$className(Structure* structure, JSDOMGlobalObject& globalObject, Ref<$implType>&& impl)\n");
        push(@implContent, "    : $parentClassName(structure, globalObject, WTFMove(impl))\n");
        push(@implContent, "{\n");
        push(@implContent, "}\n\n");
    }

    if (IsDOMGlobalObject($interface)) {
        if ($interfaceName eq "DOMWindow") {
            push(@implContent, "void ${className}::finishCreation(VM& vm, JSDOMWindowShell* shell)\n");
            push(@implContent, "{\n");
            push(@implContent, "    Base::finishCreation(vm, shell);\n\n");
        } else {
            push(@implContent, "void ${className}::finishCreation(VM& vm, JSProxy* proxy)\n");
            push(@implContent, "{\n");
            push(@implContent, "    Base::finishCreation(vm, proxy);\n\n");
        }
        # Support for RuntimeEnabled attributes on global objects.
        foreach my $attribute (@{$interface->attributes}) {
            next unless $attribute->signature->extendedAttributes->{EnabledAtRuntime};

            AddToImplIncludes("RuntimeEnabledFeatures.h");
            my $conditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
            push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
            my $enable_function = GetRuntimeEnableFunctionName($attribute->signature);
            my $attributeName = $attribute->signature->name;
            push(@implContent, "    if (${enable_function}()) {\n");
            my $getter = GetAttributeGetterName($interface, $className, $attribute);
            my $setter = IsReadonly($attribute) ? "nullptr" : GetAttributeSetterName($interface, $className, $attribute);
            push(@implContent, "        auto* customGetterSetter = CustomGetterSetter::create(vm, $getter, $setter);\n");
            my $jscAttributes = GetJSCAttributesForAttribute($interface, $attribute);
            push(@implContent, "        putDirectCustomAccessor(vm, vm.propertyNames->$attributeName, customGetterSetter, attributesForStructure($jscAttributes));\n");
            push(@implContent, "    }\n");
            push(@implContent, "#endif\n") if $conditionalString;
        }

        # Support PrivateIdentifier attributes on global objects
        foreach my $attribute (@{$interface->attributes}) {
            next unless $attribute->signature->extendedAttributes->{PrivateIdentifier};

            AddToImplIncludes("WebCoreJSClientData.h");
            my $conditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
            my $attributeName = $attribute->signature->name;
            my $getter = GetAttributeGetterName($interface, $className, $attribute);

            push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
            push(@implContent, "    putDirectCustomAccessor(vm, static_cast<JSVMClientData*>(vm.clientData)->builtinNames()." . $attributeName . "PrivateName(), CustomGetterSetter::create(vm, $getter, nullptr), attributesForStructure(DontDelete | ReadOnly));\n");
            push(@implContent, "#endif\n") if $conditionalString;
        }

        # Support for RuntimeEnabled operations on global objects.
        foreach my $function (@{$interface->functions}) {
            next unless $function->signature->extendedAttributes->{EnabledAtRuntime};
            next if $function->{overloadIndex} && $function->{overloadIndex} > 1;

            AddToImplIncludes("RuntimeEnabledFeatures.h");
            my $conditionalString = $codeGenerator->GenerateConditionalString($function->signature);
            push(@implContent, "#if ${conditionalString}\n") if $conditionalString;
            my $enable_function = GetRuntimeEnableFunctionName($function->signature);
            my $functionName = $function->signature->name;
            my $implementationFunction = GetFunctionName($interface, $className, $function);
            my $functionLength = GetFunctionLength($function);
            my $jsAttributes = ComputeFunctionSpecial($interface, $function);
            push(@implContent, "    if (${enable_function}())\n");

            my $propertyName = "vm.propertyNames->$functionName";
            $propertyName = "static_cast<JSVMClientData*>(vm.clientData)->builtinNames()." . $functionName . "PrivateName()" if $function->signature->extendedAttributes->{PrivateIdentifier};
            if (IsJSBuiltin($interface, $function)) {
                push(@implContent, "        putDirectBuiltinFunction(vm, this, $propertyName, $implementationFunction(vm), attributesForStructure($jsAttributes));\n");
            } else {
                push(@implContent, "        putDirectNativeFunction(vm, this, $propertyName, $functionLength, $implementationFunction, NoIntrinsic, attributesForStructure($jsAttributes));\n");
            }
            push(@implContent, "#endif\n") if $conditionalString;
        }
        push(@implContent, "}\n\n");
    }

    unless (ShouldUseGlobalObjectPrototype($interface)) {
        push(@implContent, "JSObject* ${className}::createPrototype(VM& vm, JSGlobalObject* globalObject)\n");
        push(@implContent, "{\n");
        if ($interface->parent) {
            my $parentClassNameForPrototype = "JS" . $interface->parent;
            push(@implContent, "    return ${className}Prototype::create(vm, globalObject, ${className}Prototype::createStructure(vm, globalObject, ${parentClassNameForPrototype}::prototype(vm, globalObject)));\n");
        } else {
            my $prototype = $interface->isException ? "errorPrototype" : "objectPrototype";
            push(@implContent, "    return ${className}Prototype::create(vm, globalObject, ${className}Prototype::createStructure(vm, globalObject, globalObject->${prototype}()));\n");
        }
        push(@implContent, "}\n\n");

        push(@implContent, "JSObject* ${className}::prototype(VM& vm, JSGlobalObject* globalObject)\n");
        push(@implContent, "{\n");
        push(@implContent, "    return getDOMPrototype<${className}>(vm, globalObject);\n");
        push(@implContent, "}\n\n");
    }

    if (!$hasParent) {
        push(@implContent, "void ${className}::destroy(JSC::JSCell* cell)\n");
        push(@implContent, "{\n");
        push(@implContent, "    ${className}* thisObject = static_cast<${className}*>(cell);\n");
        push(@implContent, "    thisObject->${className}::~${className}();\n");
        push(@implContent, "}\n\n");
    }

    my $hasGetter = InstanceOverridesGetOwnPropertySlot($interface);

    # Attributes
    if ($hasGetter) {
        if (!$interface->extendedAttributes->{CustomGetOwnPropertySlot}) {
            push(@implContent, "bool ${className}::getOwnPropertySlot(JSObject* object, ExecState* state, PropertyName propertyName, PropertySlot& slot)\n");
            push(@implContent, "{\n");
            push(@implContent, "    auto* thisObject = jsCast<${className}*>(object);\n");
            push(@implContent, "    ASSERT_GC_OBJECT_INHERITS(thisObject, info());\n");
            push(@implContent, GenerateGetOwnPropertySlotBody($interface, $className, 0));
            push(@implContent, "}\n\n");
        }

        if ($indexedGetterFunction || $namedGetterFunction
                || $interface->extendedAttributes->{CustomNamedGetter}
                || $interface->extendedAttributes->{JSCustomGetOwnPropertySlotAndDescriptor}) {
            push(@implContent, "bool ${className}::getOwnPropertySlotByIndex(JSObject* object, ExecState* state, unsigned index, PropertySlot& slot)\n");
            push(@implContent, "{\n");
            push(@implContent, "    auto* thisObject = jsCast<${className}*>(object);\n");
            push(@implContent, "    ASSERT_GC_OBJECT_INHERITS(thisObject, info());\n");

            # Sink the int-to-string conversion that happens when we create a PropertyName
            # to the point where we actually need it.
            my $generatedPropertyName = 0;
            my $propertyNameGeneration = sub {
                if ($generatedPropertyName) {
                    return;
                }
                push(@implContent, "    Identifier propertyName = Identifier::from(state, index);\n");
                $generatedPropertyName = 1;
            };

            if ($indexedGetterFunction) {
                if ($indexedGetterFunction->signature->type eq "DOMString") {
                    push(@implContent, "    if (LIKELY(index <= MAX_ARRAY_INDEX)) {\n");
                } else {
                    push(@implContent, "    if (LIKELY(index < thisObject->wrapped().length())) {\n");
                }
                # Assume that if there's a setter, the index will be writable
                if ($interface->extendedAttributes->{CustomIndexedSetter}) {
                    push(@implContent, "        unsigned attributes = DontDelete;\n");
                } else {
                    push(@implContent, "        unsigned attributes = DontDelete | ReadOnly;\n");
                }
                push(@implContent, "        slot.setValue(thisObject, attributes, " . GetIndexedGetterExpression($indexedGetterFunction) . ");\n");
                push(@implContent, "        return true;\n");
                push(@implContent, "    }\n");
            }

            # Indexing an object with an integer that is not a supported property index should not call the named property getter.
            # https://heycam.github.io/webidl/#idl-indexed-properties
            if (!$indexedGetterFunction && ($namedGetterFunction || $interface->extendedAttributes->{CustomNamedGetter})) {
                &$propertyNameGeneration();

                # This condition is to make sure we use the subclass' named getter instead of the base class one when possible.
                push(@implContent, "    if (thisObject->classInfo() == info()) {\n");
                push(@implContent, "        JSValue value;\n");
                push(@implContent, "        if (thisObject->nameGetter(state, propertyName, value)) {\n");
                push(@implContent, "            slot.setValue(thisObject, ReadOnly | DontDelete | DontEnum, value);\n");
                push(@implContent, "            return true;\n");
                push(@implContent, "        }\n");
                push(@implContent, "    }\n");
                $implIncludes{"wtf/text/AtomicString.h"} = 1;
            }

            if ($interface->extendedAttributes->{JSCustomGetOwnPropertySlotAndDescriptor}) {
                &$propertyNameGeneration();
                push(@implContent, "    if (thisObject->getOwnPropertySlotDelegate(state, propertyName, slot))\n");
                push(@implContent, "        return true;\n");
            }

            push(@implContent, "    return Base::getOwnPropertySlotByIndex(thisObject, state, index, slot);\n");
            push(@implContent, "}\n\n");
        }

    }

    if ($numAttributes > 0) {
        my $castingFunction = $interface->extendedAttributes->{"CustomProxyToJSObject"} ? "to${className}" : GetCastingHelperForThisObject($interface);
        # FIXME: Remove ImplicitThis keyword as it is no longer defined by WebIDL spec and is only used in DOMWindow.
        if ($interface->extendedAttributes->{"ImplicitThis"}) {
            push(@implContent, "template<> inline ${className}* BindingCaller<${className}>::castForAttribute(ExecState& state, EncodedJSValue thisValue)\n");
            push(@implContent, "{\n");
            push(@implContent, "    auto decodedThisValue = JSValue::decode(thisValue);\n");
            push(@implContent, "    if (decodedThisValue.isUndefinedOrNull())\n");
            push(@implContent, "        decodedThisValue = state.thisValue().toThis(&state, NotStrictMode);\n");
            push(@implContent, "    return $castingFunction(decodedThisValue);");
            push(@implContent, "}\n\n");
        } else {
            push(@implContent, "template<> inline ${className}* BindingCaller<${className}>::castForAttribute(ExecState&, EncodedJSValue thisValue)\n");
            push(@implContent, "{\n");
            push(@implContent, "    return $castingFunction(JSValue::decode(thisValue));\n");
            push(@implContent, "}\n\n");
        }
    }

    if ($numFunctions > 0 && $interfaceName ne "EventTarget") {
        # FIXME: Make consistent castForAttibute and castForOperation in case of CustomProxyToJSObject.
        my $castingFunction = $interface->extendedAttributes->{"CustomProxyToJSObject"} ? "to${className}" : GetCastingHelperForThisObject($interface);
        my $thisValue = $interface->extendedAttributes->{"CustomProxyToJSObject"} ? "state.thisValue().toThis(&state, NotStrictMode)" : "state.thisValue()";
        push(@implContent, "template<> inline ${className}* BindingCaller<${className}>::castForOperation(ExecState& state)\n");
        push(@implContent, "{\n");
        push(@implContent, "    return $castingFunction($thisValue);\n");
        push(@implContent, "}\n\n");
    }

    $numAttributes = $numAttributes + 1 if NeedsConstructorProperty($interface);
    if ($numAttributes > 0) {
        foreach my $attribute (@{$interface->attributes}) {
            next if IsJSBuiltin($interface, $attribute);

            my $name = $attribute->signature->name;
            my $type = $attribute->signature->type;
            my $getFunctionName = GetAttributeGetterName($interface, $className, $attribute);
            my $implGetterFunctionName = $codeGenerator->WK_lcfirst($attribute->signature->extendedAttributes->{ImplementedAs} || $name);
            my $getterMayThrowLegacyException = $attribute->signature->extendedAttributes->{GetterMayThrowLegacyException};

            $implIncludes{"ExceptionCode.h"} = 1 if $getterMayThrowLegacyException;

            my $attributeConditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
            push(@implContent, "#if ${attributeConditionalString}\n") if $attributeConditionalString;

            if (!$attribute->isStatic || $attribute->signature->type =~ /Constructor$/) {
                my $templateParameters = "${getFunctionName}Getter";
                if ($attribute->signature->extendedAttributes->{LenientThis}) {
                    $templateParameters .= ", CastedThisErrorBehavior::ReturnEarly";
                } elsif (IsReturningPromise($attribute)) {
                    $templateParameters .= ", CastedThisErrorBehavior::RejectPromise";
                }

                push(@implContent, "static inline JSValue ${getFunctionName}Getter(ExecState&, ${className}&, ThrowScope& throwScope);\n\n");

                push(@implContent, "EncodedJSValue ${getFunctionName}(ExecState* state, EncodedJSValue thisValue, PropertyName)\n");
                push(@implContent, "{\n");
                push(@implContent, "    return BindingCaller<${className}>::attribute<${templateParameters}>(state, thisValue, \"$name\");\n");
                push(@implContent, "}\n\n");

                push(@implContent, "static inline JSValue ${getFunctionName}Getter(ExecState& state, ${className}& thisObject, ThrowScope& throwScope)\n");
                push(@implContent, "{\n");
                push(@implContent, "    UNUSED_PARAM(throwScope);\n");
            } else {
                push(@implContent, "static inline JSValue ${getFunctionName}Getter(ExecState&);\n\n");

                push(@implContent, "EncodedJSValue ${getFunctionName}(ExecState* state, EncodedJSValue, PropertyName)\n");
                push(@implContent, "{\n");
                push(@implContent, "    ASSERT(state);\n");
                push(@implContent, "    return JSValue::encode(${getFunctionName}Getter(*state));\n");
                push(@implContent, "}\n\n");

                push(@implContent, "static inline JSValue ${getFunctionName}Getter(ExecState& state)\n");
                push(@implContent, "{\n");
            }
            push(@implContent, "    UNUSED_PARAM(state);\n");

            my @arguments = ();
            if ($getterMayThrowLegacyException && !HasCustomGetter($attribute->signature->extendedAttributes)) {
                push(@arguments, "ec");
                push(@implContent, "    ExceptionCode ec = 0;\n");
            }

            # Global constructors can be disabled at runtime.
            if ($attribute->signature->type =~ /Constructor$/) {
                if ($attribute->signature->extendedAttributes->{EnabledBySetting}) {
                    AddToImplIncludes("Frame.h");
                    AddToImplIncludes("Settings.h");
                    my $enable_function = ToMethodName($attribute->signature->extendedAttributes->{EnabledBySetting}) . "Enabled";
                    push(@implContent, "    if (UNLIKELY(!thisObject.wrapped().frame()))\n");
                    push(@implContent, "        return jsUndefined();\n");
                    push(@implContent, "    Settings& settings = thisObject.wrapped().frame()->settings();\n");
                    push(@implContent, "    if (!settings.$enable_function())\n");
                    push(@implContent, "        return jsUndefined();\n");
                }
            }

            $needsVisitChildren = 1 if $attribute->signature->extendedAttributes->{CachedAttribute};

            if ($interface->extendedAttributes->{CheckSecurity} &&
                !$attribute->signature->extendedAttributes->{DoNotCheckSecurity} &&
                !$attribute->signature->extendedAttributes->{DoNotCheckSecurityOnGetter}) {
                if ($interfaceName eq "DOMWindow") {
                    push(@implContent, "    if (!BindingSecurity::shouldAllowAccessToDOMWindow(&state, thisObject.wrapped(), ThrowSecurityError))\n");
                } else {
                    push(@implContent, "    if (!BindingSecurity::shouldAllowAccessToFrame(&state, thisObject.wrapped().frame(), ThrowSecurityError))\n");
                }
                push(@implContent, "        return jsUndefined();\n");
            }

            if ($attribute->signature->extendedAttributes->{Nondeterministic}) {
                AddToImplIncludes("MemoizedDOMResult.h", "WEB_REPLAY");
                AddToImplIncludes("<replay/InputCursor.h>", "WEB_REPLAY");
                AddToImplIncludes("<wtf/NeverDestroyed.h>", "WEB_REPLAY");

                push(@implContent, "#if ENABLE(WEB_REPLAY)\n");
                push(@implContent, "    JSGlobalObject* globalObject = state.lexicalGlobalObject();\n");
                push(@implContent, "    InputCursor& cursor = globalObject->inputCursor();\n");

                my $nativeType = GetNativeType($interface, $type);
                my $memoizedType = GetNativeTypeForMemoization($interface, $type);
                my $exceptionCode = $getterMayThrowLegacyException ? "ec" : "0";
                push(@implContent, "    static NeverDestroyed<const AtomicString> bindingName(\"$interfaceName.$name\", AtomicString::ConstructFromLiteral);\n");
                push(@implContent, "    if (cursor.isCapturing()) {\n");
                push(@implContent, "        $memoizedType memoizedResult = thisObject.wrapped().$implGetterFunctionName(" . join(", ", @arguments) . ");\n");
                push(@implContent, "        cursor.appendInput<MemoizedDOMResult<$memoizedType>>(bindingName.get().string(), memoizedResult, $exceptionCode);\n");
                push(@implContent, "        JSValue result = " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "memoizedResult", "thisObject") . ";\n");
                push(@implContent, "        setDOMException(&state, throwScope, ec);\n") if $getterMayThrowLegacyException;
                push(@implContent, "        return result;\n");
                push(@implContent, "    }\n");
                push(@implContent, "\n");
                push(@implContent, "    if (cursor.isReplaying()) {\n");
                push(@implContent, "        $memoizedType memoizedResult;\n");
                push(@implContent, "        MemoizedDOMResultBase* input = cursor.fetchInput<MemoizedDOMResultBase>();\n");
                push(@implContent, "        if (input && input->convertTo<$memoizedType>(memoizedResult)) {\n");
                # FIXME: the generated code should report an error if an input cannot be fetched or converted.
                push(@implContent, "            JSValue result = " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "memoizedResult", "thisObject") . ";\n");
                push(@implContent, "            setDOMException(&state, throwScope, input->exceptionCode());\n") if $getterMayThrowLegacyException;
                push(@implContent, "            return result;\n");
                push(@implContent, "        }\n");
                push(@implContent, "    }\n");
                push(@implContent, "#endif\n");
            } # attribute Nondeterministic

            if (HasCustomGetter($attribute->signature->extendedAttributes)) {
                push(@implContent, "    return thisObject.$implGetterFunctionName(state);\n");
            } elsif ($attribute->signature->extendedAttributes->{CheckSecurityForNode}) {
                $implIncludes{"JSDOMBinding.h"} = 1;
                push(@implContent, "    auto& impl = thisObject.wrapped();\n");
                push(@implContent, "    return shouldAllowAccessToNode(&state, impl." . $attribute->signature->name . "()) ? " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "impl.$implGetterFunctionName()", "thisObject") . " : jsNull();\n");
            } elsif ($type eq "EventHandler") {
                $implIncludes{"EventNames.h"} = 1;
                my $getter = $attribute->signature->extendedAttributes->{WindowEventHandler} ? "windowEventHandlerAttribute"
                    : $attribute->signature->extendedAttributes->{DocumentEventHandler} ? "documentEventHandlerAttribute"
                    : "eventHandlerAttribute";
                my $eventName = EventHandlerAttributeEventName($attribute);
                push(@implContent, "    return $getter(thisObject.wrapped(), $eventName);\n");
            } elsif ($attribute->signature->type =~ /Constructor$/) {
                my $constructorType = $attribute->signature->type;
                $constructorType =~ s/Constructor$//;
                # When Constructor attribute is used by DOMWindow.idl, it's correct to pass thisObject as the global object
                # When JSDOMWrappers have a back-pointer to the globalObject we can pass thisObject->globalObject()
                if ($interfaceName eq "DOMWindow") {
                    my $named = ($constructorType =~ /Named$/) ? "Named" : "";
                    $constructorType =~ s/Named$//;
                    push(@implContent, "    return JS" . $constructorType . "::get${named}Constructor(state.vm(), &thisObject);\n");
                } else {
                    AddToImplIncludes("JS" . $constructorType . ".h", $attribute->signature->extendedAttributes->{Conditional});
                    push(@implContent, "    return JS" . $constructorType . "::getConstructor(state.vm(), thisObject.globalObject());\n");
                }
            } elsif (!$attribute->signature->extendedAttributes->{GetterMayThrowLegacyException}) {
                my $cacheIndex = 0;
                if ($attribute->signature->extendedAttributes->{CachedAttribute}) {
                    $cacheIndex = $currentCachedAttribute;
                    $currentCachedAttribute++;
                    push(@implContent, "    if (JSValue cachedValue = thisObject.m_" . $attribute->signature->name . ".get())\n");
                    push(@implContent, "        return cachedValue;\n");
                }

                my @callWithArgs = GenerateCallWithUsingReferences($attribute->signature->extendedAttributes->{CallWith}, \@implContent, "jsUndefined()");

                if ($svgListPropertyType) {
                    push(@implContent, "    JSValue result =  " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "thisObject.wrapped().$implGetterFunctionName(" . (join ", ", @callWithArgs) . ")", "thisObject") . ";\n");
                } elsif ($svgPropertyOrListPropertyType) {
                    push(@implContent, "    $svgPropertyOrListPropertyType& impl = thisObject.wrapped().propertyReference();\n");
                    if ($svgPropertyOrListPropertyType eq "float") { # Special case for JSSVGNumber
                        push(@implContent, "    JSValue result = " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "impl", "thisObject") . ";\n");
                    } else {
                        push(@implContent, "    JSValue result = " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "impl.$implGetterFunctionName(" . (join ", ", @callWithArgs) . ")", "thisObject") . ";\n");

                    }
                } else {
                    my ($functionName, @arguments) = $codeGenerator->GetterExpression(\%implIncludes, $interfaceName, $attribute);
                    my $implementedBy = $attribute->signature->extendedAttributes->{ImplementedBy};
                    if ($implementedBy) {
                        $implIncludes{"${implementedBy}.h"} = 1;
                        $functionName = "WebCore::${implementedBy}::${functionName}";
                        unshift(@arguments, "impl") if !$attribute->isStatic;
                    } elsif ($attribute->isStatic) {
                        $functionName = "${interfaceName}::${functionName}";
                    } else {
                        $functionName = "impl.${functionName}";
                    }

                    unshift(@arguments, @callWithArgs);
                    my $jsType = NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "${functionName}(" . join(", ", @arguments) . ")", "thisObject");
                    push(@implContent, "    auto& impl = thisObject.wrapped();\n") if !$attribute->isStatic;
                    if ($codeGenerator->IsSVGAnimatedType($type)) {
                        push(@implContent, "    auto obj = $jsType;\n");
                        push(@implContent, "    JSValue result = toJS(&state, thisObject.globalObject(), obj.get());\n");
                    } else {
                        push(@implContent, "    JSValue result = $jsType;\n");
                    }
                }

                push(@implContent, "    thisObject.m_" . $attribute->signature->name . ".set(state.vm(), &thisObject, result);\n") if $attribute->signature->extendedAttributes->{CachedAttribute};
                push(@implContent, "    return result;\n");

            } else {
                unshift(@arguments, GenerateCallWithUsingReferences($attribute->signature->extendedAttributes->{CallWith}, \@implContent, "jsUndefined()", 0));

                if ($svgPropertyOrListPropertyType) {
                    push(@implContent, "    $svgPropertyOrListPropertyType impl(*thisObject.wrapped());\n");
                    push(@implContent, "    JSValue result = " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "impl.$implGetterFunctionName(" . join(", ", @arguments) . ")", "thisObject") . ";\n");
                } else {
                    push(@implContent, "    auto& impl = thisObject.wrapped();\n");
                    push(@implContent, "    JSValue result = " . NativeToJSValueUsingReferences($attribute->signature, 0, $interface, "impl.$implGetterFunctionName(" . join(", ", @arguments) . ")", "thisObject") . ";\n");
                }

                push(@implContent, "    setDOMException(&state, throwScope, ec);\n");

                push(@implContent, "    return result;\n");
            }

            push(@implContent, "}\n\n");

            if ($attribute->signature->extendedAttributes->{"DOMJIT"}) {
                $implIncludes{"<wtf/NeverDestroyed.h>"} = 1;
                my $interfaceName = $interface->name;
                my $attributeName = $attribute->signature->name;
                my $generatorName = $interfaceName . $codeGenerator->WK_ucfirst($attribute->signature->name);
                my $domJITClassName = $generatorName . "DOMJIT";
                my $getter = GetAttributeGetterName($interface, $generatorName, $attribute);
                my $setter = IsReadonly($attribute) ? "nullptr" : GetAttributeSetterName($interface, $generatorName, $attribute);
                my $resultType = GetResultTypeFilter($attribute->signature);
                push(@implContent, "$domJITClassName::$domJITClassName()\n");
                push(@implContent, "    : JSC::DOMJIT::GetterSetter($getter, $setter, ${className}::info(), $resultType)\n");
                push(@implContent, "{\n");
                push(@implContent, "}\n\n");

                push(@implContent, "JSC::DOMJIT::GetterSetter* domJITGetterSetterFor" . $generatorName . "()\n");
                push(@implContent, "{\n");
                push(@implContent, "    static NeverDestroyed<$domJITClassName> compiler;\n");
                push(@implContent, "    return &compiler.get();\n");
                push(@implContent, "}\n\n");
            }

            push(@implContent, "#endif\n\n") if $attributeConditionalString;
        }

        if (NeedsConstructorProperty($interface)) {
            my $constructorFunctionName = "js" . $interfaceName . "Constructor";

            push(@implContent, "EncodedJSValue ${constructorFunctionName}(ExecState* state, EncodedJSValue thisValue, PropertyName)\n");
            push(@implContent, "{\n");
            push(@implContent, "    VM& vm = state->vm();\n");
            push(@implContent, "    auto throwScope = DECLARE_THROW_SCOPE(vm);\n");
            push(@implContent, "    ${className}Prototype* domObject = jsDynamicCast<${className}Prototype*>(JSValue::decode(thisValue));\n");
            push(@implContent, "    if (UNLIKELY(!domObject))\n");
            push(@implContent, "        return throwVMTypeError(state, throwScope);\n");

            if (!$interface->extendedAttributes->{NoInterfaceObject}) {
                push(@implContent, "    return JSValue::encode(${className}::getConstructor(state->vm(), domObject->globalObject()));\n");
            } else {
                push(@implContent, "    JSValue constructor = ${className}Constructor::create(state->vm(), ${className}Constructor::createStructure(state->vm(), *domObject->globalObject(), domObject->globalObject()->objectPrototype()), *jsCast<JSDOMGlobalObject*>(domObject->globalObject()));\n");
                push(@implContent, "    // Shadowing constructor property to ensure reusing the same constructor object\n");
                push(@implContent, "    domObject->putDirect(state->vm(), state->propertyNames().constructor, constructor, DontEnum | ReadOnly);\n");
                push(@implContent, "    return JSValue::encode(constructor);\n");
            }
            push(@implContent, "}\n\n");
        }

        my $constructorFunctionName = "setJS" . $interfaceName . "Constructor";

        push(@implContent, "bool ${constructorFunctionName}(ExecState* state, EncodedJSValue thisValue, EncodedJSValue encodedValue)\n");
        push(@implContent, "{\n");
        push(@implContent, "    VM& vm = state->vm();\n");
        push(@implContent, "    auto throwScope = DECLARE_THROW_SCOPE(vm);\n");
        push(@implContent, "    JSValue value = JSValue::decode(encodedValue);\n");
        push(@implContent, "    ${className}Prototype* domObject = jsDynamicCast<${className}Prototype*>(JSValue::decode(thisValue));\n");
        push(@implContent, "    if (UNLIKELY(!domObject)) {\n");
        push(@implContent, "        throwVMTypeError(state, throwScope);\n");
        push(@implContent, "        return false;\n");
        push(@implContent, "    }\n");

        push(@implContent, "    // Shadowing a built-in constructor\n");

        push(@implContent, "    return domObject->putDirect(state->vm(), state->propertyNames().constructor, value);\n");
        push(@implContent, "}\n\n");
    }

    my $hasCustomSetter = $interface->extendedAttributes->{CustomNamedSetter} || $interface->extendedAttributes->{CustomIndexedSetter};
    if ($hasCustomSetter) {
        if (!$interface->extendedAttributes->{CustomPutFunction}) {
            push(@implContent, "bool ${className}::put(JSCell* cell, ExecState* state, PropertyName propertyName, JSValue value, PutPropertySlot& slot)\n");
            push(@implContent, "{\n");
            push(@implContent, "    auto* thisObject = jsCast<${className}*>(cell);\n");
            push(@implContent, "    ASSERT_GC_OBJECT_INHERITS(thisObject, info());\n");
            if ($interface->extendedAttributes->{CustomIndexedSetter}) {
                push(@implContent, "    if (Optional<uint32_t> index = parseIndex(propertyName)) {\n");
                push(@implContent, "        thisObject->indexSetter(state, index.value(), value);\n");
                push(@implContent, "        return true;\n");
                push(@implContent, "    }\n");
            }
            if ($interface->extendedAttributes->{CustomNamedSetter}) {
                push(@implContent, "    bool putResult = false;\n");
                push(@implContent, "    if (thisObject->putDelegate(state, propertyName, value, slot, putResult))\n");
                push(@implContent, "        return putResult;\n");
            }

            push(@implContent, "    return Base::put(thisObject, state, propertyName, value, slot);\n");
            push(@implContent, "}\n\n");

            if ($interface->extendedAttributes->{CustomIndexedSetter} || $interface->extendedAttributes->{CustomNamedSetter}) {
                push(@implContent, "bool ${className}::putByIndex(JSCell* cell, ExecState* state, unsigned index, JSValue value, bool shouldThrow)\n");
                push(@implContent, "{\n");
                push(@implContent, "    auto* thisObject = jsCast<${className}*>(cell);\n");
                push(@implContent, "    ASSERT_GC_OBJECT_INHERITS(thisObject, info());\n");

                if ($interface->extendedAttributes->{CustomIndexedSetter}) {
                    push(@implContent, "    if (LIKELY(index <= MAX_ARRAY_INDEX)) {\n");
                    push(@implContent, "        thisObject->indexSetter(state, index, value);\n");
                    push(@implContent, "        return true;\n");
                    push(@implContent, "    }\n");
                }

                if ($interface->extendedAttributes->{CustomNamedSetter}) {
                    push(@implContent, "    Identifier propertyName = Identifier::from(state, index);\n");
                    push(@implContent, "    PutPropertySlot slot(thisObject, shouldThrow);\n");
                    push(@implContent, "    bool putResult = false;\n");
                    push(@implContent, "    if (thisObject->putDelegate(state, propertyName, value, slot, putResult))\n");
                    push(@implContent, "        return putResult;\n");
                }

                push(@implContent, "    return Base::putByIndex(cell, state, index, value, shouldThrow);\n");
                push(@implContent, "}\n\n");
            }
        }
    }

    foreach my $attribute (@{$interface->attributes}) {
        if (!IsReadonly($attribute)) {
            next if IsJSBuiltin($interface, $attribute);

            my $name = $attribute->signature->name;
            my $type = $attribute->signature->type;
            my $putFunctionName = GetAttributeSetterName($interface, $className, $attribute);
            my $implSetterFunctionName = $codeGenerator->WK_ucfirst($name);
            my $setterMayThrowLegacyException = $attribute->signature->extendedAttributes->{SetterMayThrowLegacyException};

            $implIncludes{"ExceptionCode.h"} = 1 if $setterMayThrowLegacyException;

            my $attributeConditionalString = $codeGenerator->GenerateConditionalString($attribute->signature);
            push(@implContent, "#if ${attributeConditionalString}\n") if $attributeConditionalString;

            if (!$attribute->isStatic) {
                my $setterFunction = "${putFunctionName}Function";
                my $templateParameters = $setterFunction;
                $templateParameters .= ", CastedThisErrorBehavior::ReturnEarly" if $attribute->signature->extendedAttributes->{LenientThis};

                push(@implContent, "static inline bool ${setterFunction}(ExecState&, ${className}&, JSValue, ThrowScope&);\n\n");

                push(@implContent, "bool ${putFunctionName}(ExecState* state, EncodedJSValue thisValue, EncodedJSValue encodedValue)\n");
                push(@implContent, "{\n");
                push(@implContent, "    return BindingCaller<${className}>::setAttribute<${templateParameters}>(state, thisValue, encodedValue, \"$name\");\n");
                push(@implContent, "}\n\n");

                push(@implContent, "static inline bool ${setterFunction}(ExecState& state, ${className}& thisObject, JSValue value, ThrowScope& throwScope)\n");
                push(@implContent, "{\n");
                push(@implContent, "    UNUSED_PARAM(state);\n");
                push(@implContent, "    UNUSED_PARAM(throwScope);\n");
            } else {
                push(@implContent, "bool ${putFunctionName}(ExecState* statePointer, EncodedJSValue, EncodedJSValue encodedValue)\n");
                push(@implContent, "{\n");
                push(@implContent, "    ASSERT(statePointer);\n");
                push(@implContent, "    auto& state = *statePointer;\n");
                push(@implContent, "    UNUSED_PARAM(state);\n");
                push(@implContent, "    auto value = JSValue::decode(encodedValue);\n");
            }

            if ($attribute->signature->extendedAttributes->{CEReactions}) {
                push(@implContent, "#if ENABLE(CUSTOM_ELEMENTS)\n");
                push(@implContent, "    CustomElementReactionStack customElementReactionStack;\n");
                push(@implContent, "#endif\n");
                $implIncludes{"CustomElementReactionQueue.h"} = 1;
            }

            if ($interface->extendedAttributes->{CheckSecurity} && !$attribute->signature->extendedAttributes->{DoNotCheckSecurity} && !$attribute->signature->extendedAttributes->{DoNotCheckSecurityOnSetter}) {
                if ($interfaceName eq "DOMWindow") {
                    push(@implContent, "    if (!BindingSecurity::shouldAllowAccessToDOMWindow(&state, thisObject.wrapped(), ThrowSecurityError))\n");
                } else {
                    push(@implContent, "    if (!BindingSecurity::shouldAllowAccessToFrame(&state, thisObject.wrapped().frame(), ThrowSecurityError))\n");
                }
                push(@implContent, "        return false;\n");
            }

            if (HasCustomSetter($attribute->signature->extendedAttributes)) {
                push(@implContent, "    thisObject.set$implSetterFunctionName(state, value);\n");
                push(@implContent, "    return true;\n");
            } elsif ($type eq "EventHandler") {
                $implIncludes{"JSEventListener.h"} = 1;
                my $eventName = EventHandlerAttributeEventName($attribute);
                # FIXME: Find a way to do this special case without hardcoding the class and attribute names here.
                if ((($interfaceName eq "DOMWindow") or ($interfaceName eq "WorkerGlobalScope")) and $name eq "onerror") {
                    $implIncludes{"JSErrorHandler.h"} = 1;
                    push(@implContent, "    thisObject.wrapped().setAttributeEventListener($eventName, createJSErrorHandler(&state, value, &thisObject));\n");
                } else {
                    $implIncludes{"JSEventListener.h"} = 1;
                    my $setter = $attribute->signature->extendedAttributes->{WindowEventHandler} ? "setWindowEventHandlerAttribute"
                        : $attribute->signature->extendedAttributes->{DocumentEventHandler} ? "setDocumentEventHandlerAttribute"
                        : "setEventHandlerAttribute";
                    push(@implContent, "    $setter(state, thisObject, thisObject.wrapped(), $eventName, value);\n");
                }
                push(@implContent, "    return true;\n");
            } elsif ($type =~ /Constructor$/) {
                my $constructorType = $type;
                $constructorType =~ s/Constructor$//;
                # $constructorType ~= /Constructor$/ indicates that it is NamedConstructor.
                # We do not generate the header file for NamedConstructor of class XXXX,
                # since we generate the NamedConstructor declaration into the header file of class XXXX.
                if ($constructorType ne "any" and $constructorType !~ /Named$/) {
                    AddToImplIncludes("JS" . $constructorType . ".h", $attribute->signature->extendedAttributes->{Conditional});
                }
                push(@implContent, "    // Shadowing a built-in constructor.\n");
                push(@implContent, "    return thisObject.putDirect(state.vm(), Identifier::fromString(&state, \"$name\"), value);\n");
            } elsif ($attribute->signature->extendedAttributes->{Replaceable}) {
                push(@implContent, "    // Shadowing a built-in property.\n");
                if (AttributeShouldBeOnInstance($interface, $attribute)) {
                    push(@implContent, "    return replaceStaticPropertySlot(state.vm(), &thisObject, Identifier::fromString(&state, \"$name\"), value);\n");
                } else {
                    push(@implContent, "    return thisObject.putDirect(state.vm(), Identifier::fromString(&state, \"$name\"), value);\n");
                }
            } else {
                if (!$attribute->isStatic) {
                    my $putForwards = $attribute->signature->extendedAttributes->{PutForwards};
                    if ($putForwards) {
                        my $implGetterFunctionName = $codeGenerator->WK_lcfirst($attribute->signature->extendedAttributes->{ImplementedAs} || $name);
                        my $forwardedAttribute = $codeGenerator->GetAttributeFromInterface($interface, $type, $putForwards);

                        if ($forwardedAttribute->signature->extendedAttributes->{CEReactions}) {
                            push(@implContent, "#if ENABLE(CUSTOM_ELEMENTS)\n");
                            push(@implContent, "    CustomElementReactionStack customElementReactionStack;\n");
                            push(@implContent, "#endif\n");
                            $implIncludes{"CustomElementReactionQueue.h"} = 1;
                        }

                        if ($attribute->signature->isNullable) {
                            push(@implContent, "    RefPtr<${type}> forwardedImpl = thisObject.wrapped().${implGetterFunctionName}();\n");
                            push(@implContent, "    if (!forwardedImpl)\n");
                            push(@implContent, "        return false;\n");
                            push(@implContent, "    auto& impl = *forwardedImpl;\n");
                        } else {
                            # Attribute is not nullable, the implementation is expected to return a reference.
                            push(@implContent, "    Ref<${type}> forwardedImpl = thisObject.wrapped().${implGetterFunctionName}();\n");
                            push(@implContent, "    auto& impl = forwardedImpl.get();\n");
                        }
                        $attribute = $forwardedAttribute;
                        $type = $attribute->signature->type;
                    } else {
                        push(@implContent, "    auto& impl = thisObject.wrapped();\n");
                    }
                }

                push(@implContent, "    ExceptionCode ec = 0;\n") if $setterMayThrowLegacyException;

                my $shouldPassByReference = ShouldPassWrapperByReference($attribute->signature, $interface);

                my ($nativeValue, $mayThrowException) = JSValueToNative($interface, $attribute->signature, "value", $attribute->signature->extendedAttributes->{Conditional}, "&state", "state", "thisObject");
                if (!$shouldPassByReference && ($codeGenerator->IsWrapperType($type) || $codeGenerator->IsTypedArrayType($type))) {
                    $implIncludes{"<runtime/Error.h>"} = 1;
                    push(@implContent, "    " . GetNativeTypeFromSignature($interface, $attribute->signature) . " nativeValue = nullptr;\n");
                    push(@implContent, "    if (!value.isUndefinedOrNull()) {\n");
                    push(@implContent, "        nativeValue = $nativeValue;\n");
                    push(@implContent, "        RETURN_IF_EXCEPTION(throwScope, false);\n") if $mayThrowException;
                    push(@implContent, "        if (UNLIKELY(!nativeValue)) {\n");
                    push(@implContent, "            throwAttributeTypeError(state, throwScope, \"$visibleInterfaceName\", \"$name\", \"$type\");\n");
                    push(@implContent, "            return false;\n");
                    push(@implContent, "        }\n");
                    push(@implContent, "    }\n");
                } else {
                    push(@implContent, "    auto nativeValue = $nativeValue;\n");
                    push(@implContent, "    RETURN_IF_EXCEPTION(throwScope, false);\n") if $mayThrowException;
                }

                if ($codeGenerator->IsEnumType($type)) {
                    push (@implContent, "    if (UNLIKELY(!nativeValue))\n");
                    push (@implContent, "        return false;\n");
                }

                if ($shouldPassByReference) {
                    push(@implContent, "    if (UNLIKELY(!nativeValue)) {\n");
                    push(@implContent, "        throwAttributeTypeError(state, throwScope, \"$visibleInterfaceName\", \"$name\", \"$type\");\n");
                    push(@implContent, "        return false;\n");
                    push(@implContent, "    }\n");
                }

                if ($svgPropertyOrListPropertyType) {
                    if ($svgPropertyType) {
                        push(@implContent, "    if (impl.isReadOnly()) {\n");
                        push(@implContent, "        setDOMException(&state, throwScope, NO_MODIFICATION_ALLOWED_ERR);\n");
                        push(@implContent, "        return false;\n");
                        push(@implContent, "    }\n");
                        $implIncludes{"ExceptionCode.h"} = 1;
                    }
                    push(@implContent, "    $svgPropertyOrListPropertyType& podImpl = impl.propertyReference();\n");
                    if ($svgPropertyOrListPropertyType eq "float") { # Special case for JSSVGNumber
                        push(@implContent, "    podImpl = nativeValue;\n");
                    } else {
                        my $functionString = "podImpl.set$implSetterFunctionName(nativeValue";
                        $functionString .= ", ec" if $setterMayThrowLegacyException;
                        $functionString .= ")";
                        $functionString = "propagateException(state, throwScope, $functionString)" if $attribute->signature->extendedAttributes->{SetterMayThrowException};
                        push(@implContent, "    $functionString;\n");
                        push(@implContent, "    setDOMException(&state, throwScope, ec);\n") if $setterMayThrowLegacyException;
                    }
                    if ($svgPropertyType) {
                        if ($setterMayThrowLegacyException) {
                            push(@implContent, "    if (LIKELY(!ec))\n");
                            push(@implContent, "        impl.commitChange();\n");
                        } else {
                            push(@implContent, "    impl.commitChange();\n");
                        }
                    }
                    push(@implContent, "    return true;\n");
                } else {
                    my ($functionName, @arguments) = $codeGenerator->SetterExpression(\%implIncludes, $interfaceName, $attribute);

                    if ($codeGenerator->IsTypedArrayType($type) and not $type eq "ArrayBuffer") {
                        push(@arguments, "nativeValue.get()");
                    } elsif ($codeGenerator->IsEnumType($type)) {
                        push(@arguments, "nativeValue.value()");
                    } else {
                        push(@arguments, $shouldPassByReference ? "*nativeValue" : "WTFMove(nativeValue)");
                    }
                    my $implementedBy = $attribute->signature->extendedAttributes->{ImplementedBy};
                    if ($implementedBy) {
                        AddToImplIncludes("${implementedBy}.h", $attribute->signature->extendedAttributes->{Conditional});
                        unshift(@arguments, "impl") if !$attribute->isStatic;
                        $functionName = "WebCore::${implementedBy}::${functionName}";
                    } elsif ($attribute->isStatic) {
                        $functionName = "${interfaceName}::${functionName}";
                    } else {
                        $functionName = "impl.${functionName}";
                    }

                    unshift(@arguments, GenerateCallWithUsingReferences($attribute->signature->extendedAttributes->{SetterCallWith}, \@implContent, "false"));
                    unshift(@arguments, GenerateCallWithUsingReferences($attribute->signature->extendedAttributes->{CallWith}, \@implContent, "false"));

                    push(@arguments, "ec") if $setterMayThrowLegacyException;

                    my $functionString = "$functionName(" . join(", ", @arguments) . ")";
                    $functionString = "propagateException(state, throwScope, $functionString)" if $attribute->signature->extendedAttributes->{SetterMayThrowException};

                    push(@implContent, "    $functionString;\n");
                    push(@implContent, "    setDOMException(&state, throwScope, ec);\n") if $setterMayThrowLegacyException;
                    push(@implContent, "    return true;\n");
                }
            }

            push(@implContent, "}\n\n");
            push(@implContent, "#endif\n") if $attributeConditionalString;
            push(@implContent, "\n");
        }
    }

    if (($indexedGetterFunction || $namedGetterFunction) && !$interface->extendedAttributes->{CustomEnumerateProperty}) {
        push(@implContent, "void ${className}::getOwnPropertyNames(JSObject* object, ExecState* state, PropertyNameArray& propertyNames, EnumerationMode mode)\n");
        push(@implContent, "{\n");
        push(@implContent, "    auto* thisObject = jsCast<${className}*>(object);\n");
        push(@implContent, "    ASSERT_GC_OBJECT_INHERITS(thisObject, info());\n");
        if ($indexedGetterFunction) {
            push(@implContent, "    for (unsigned i = 0, count = thisObject->wrapped().length(); i < count; ++i)\n");
            push(@implContent, "        propertyNames.add(Identifier::from(state, i));\n");
        }
        if ($namedGetterFunction) {
            # FIXME: We may need to add an IDL extended attribute at some point if an interface needs enumerable named properties.
            push(@implContent, "    if (mode.includeDontEnumProperties()) {\n");
            push(@implContent, "        for (auto& propertyName : thisObject->wrapped().supportedPropertyNames())\n");
            push(@implContent, "            propertyNames.add(Identifier::fromString(state, propertyName));\n");
            push(@implContent, "    }\n");
        }
        push(@implContent, "    Base::getOwnPropertyNames(thisObject, state, propertyNames, mode);\n");
        push(@implContent, "}\n\n");
    }

    if (!$interface->extendedAttributes->{NoInterfaceObject}) {
        push(@implContent, "JSValue ${className}::getConstructor(VM& vm, const JSGlobalObject* globalObject)\n{\n");
        push(@implContent, "    return getDOMConstructor<${className}Constructor>(vm, *jsCast<const JSDOMGlobalObject*>(globalObject));\n");
        push(@implContent, "}\n\n");
        if ($interface->extendedAttributes->{NamedConstructor}) {
            push(@implContent, "JSValue ${className}::getNamedConstructor(VM& vm, JSGlobalObject* globalObject)\n{\n");
            push(@implContent, "    return getDOMConstructor<${className}NamedConstructor>(vm, *jsCast<JSDOMGlobalObject*>(globalObject));\n");
            push(@implContent, "}\n\n");
        }
    }

    # Functions
    if ($numFunctions > 0) {
        my $inAppleCopyright = 0;
        foreach my $function (@{$interface->functions}) {
            next if IsJSBuiltin($interface, $function);
            if ($function->signature->extendedAttributes->{AppleCopyright}) {
                if (!$inAppleCopyright) {
                    push(@implContent, $beginAppleCopyrightForSourceFiles);
                    $inAppleCopyright = 1;
                }
            } elsif ($inAppleCopyright) {
                push(@implContent, $endAppleCopyright);
                $inAppleCopyright = 0;
            }

            my $isCustom = HasCustomMethod($function->signature->extendedAttributes);
            my $isOverloaded = $function->{overloads} && @{$function->{overloads}} > 1;

            my $mayThrowLegacyException = $function->signature->extendedAttributes->{MayThrowLegacyException};

            next if $isCustom && $isOverloaded && $function->{overloadIndex} > 1;

            AddIncludesForTypeInImpl($function->signature->type) unless $isCustom or IsReturningPromise($function);

            my $functionName = GetFunctionName($interface, $className, $function);

            my $conditional = $function->signature->extendedAttributes->{Conditional};
            if ($conditional) {
                my $conditionalString = $codeGenerator->GenerateConditionalStringFromAttributeValue($conditional);
                push(@implContent, "#if ${conditionalString}\n");
            }

            my $functionReturn = "EncodedJSValue JSC_HOST_CALL";
            if (!$isCustom && $isOverloaded) {
                # Append a number to an overloaded method's name to make it unique:
                $functionName = $functionName . $function->{overloadIndex};
                # Make this function static to avoid compiler warnings, since we don't generate a prototype for it in the header.
                $functionReturn = "static inline EncodedJSValue";
            }

            my $functionImplementationName = $function->signature->extendedAttributes->{ImplementedAs} || $codeGenerator->WK_lcfirst($function->signature->name);

            AddToImplIncludes("JSDOMPromise.h") if IsReturningPromise($function);

            if (!$function->isStatic) {
                my $classParameterType = $className eq "JSEventTarget" ? "JSEventTargetWrapper*" : "${className}*";
                my $optionalPromiseParameter = (IsReturningPromise($function) && !$isCustom) ? " Ref<DeferredPromise>&&," : "";
                push(@implContent, "static inline JSC::EncodedJSValue ${functionName}Caller(JSC::ExecState*, ${classParameterType},${optionalPromiseParameter} JSC::ThrowScope&);\n");
                push(@implContent, "\n");
            }

            if (IsReturningPromise($function) && !$isCustom) {
                my $scope = $interface->extendedAttributes->{Exposed} ? "WindowOrWorker" : "WindowOnly";
                push(@implContent, <<END);
static EncodedJSValue ${functionName}Promise(ExecState*, Ref<DeferredPromise>&&);

${functionReturn} ${functionName}(ExecState* state)
{
    ASSERT(state);
    return JSValue::encode(callPromiseFunction<${functionName}Promise, PromiseExecutionScope::${scope}>(*state));
}

static inline EncodedJSValue ${functionName}Promise(ExecState* state, Ref<DeferredPromise>&& promise)
END
            } else {
                push(@implContent, "${functionReturn} ${functionName}(ExecState* state)\n");
            }

            push(@implContent, "{\n");

            $implIncludes{"<runtime/Error.h>"} = 1;

            if ($function->signature->extendedAttributes->{CEReactions}) {
                push(@implContent, "#if ENABLE(CUSTOM_ELEMENTS)\n");
                push(@implContent, "    CustomElementReactionStack customElementReactionStack;\n");
                push(@implContent, "#endif\n");
                $implIncludes{"CustomElementReactionQueue.h"} = 1;
            }

            if ($function->isStatic) {
                if ($isCustom) {
                    GenerateArgumentsCountCheck(\@implContent, $function, $interface);
                    push(@implContent, "    return JSValue::encode(${className}::" . $functionImplementationName . "(state));\n");
                } else {
                    push(@implContent, "    VM& vm = state->vm();\n");
                    push(@implContent, "    auto throwScope = DECLARE_THROW_SCOPE(vm);\n");
                    push(@implContent, "    UNUSED_PARAM(throwScope);\n");

                    GenerateArgumentsCountCheck(\@implContent, $function, $interface);

                    push(@implContent, "    ExceptionCode ec = 0;\n") if $mayThrowLegacyException;

                    my ($functionString, $dummy) = GenerateParametersCheck(\@implContent, $function, $interface, $functionImplementationName, $svgPropertyType, $svgPropertyOrListPropertyType, $svgListPropertyType);
                    GenerateImplementationFunctionCall($function, $functionString, "    ", $svgPropertyType, $interface);
                }
            } else {
                my $methodName = $function->signature->name;
                if (IsReturningPromise($function) && !$isCustom) {
                    my $templateParameters = "${functionName}Caller";
                    $templateParameters .= ", CastedThisErrorBehavior::Assert" if ($function->signature->extendedAttributes->{PrivateIdentifier} and not $function->signature->extendedAttributes->{PublicIdentifier});
                    push(@implContent, "    return BindingCaller<$className>::callPromiseOperation<${templateParameters}>(state, WTFMove(promise), \"${methodName}\");\n");
                    push(@implContent, "}\n");
                    push(@implContent, "\n");
                    push(@implContent, "static inline JSC::EncodedJSValue ${functionName}Caller(JSC::ExecState* state, ${className}* castedThis, Ref<DeferredPromise>&& promise, JSC::ThrowScope& throwScope)\n");
                } else {
                    my $classParameterType = $className eq "JSEventTarget" ? "JSEventTargetWrapper*" : "${className}*";
                    my $templateParameters = "${functionName}Caller";
                    if ($function->signature->extendedAttributes->{PrivateIdentifier} and not $function->signature->extendedAttributes->{PublicIdentifier}) {
                        $templateParameters .= ", CastedThisErrorBehavior::Assert";
                    } elsif (IsReturningPromise($function)) {
                        # FIXME: We need this specific handling for custom promise-returning functions.
                        # It would be better to have the casted-this code calling the promise-specific code.
                        $templateParameters .= ", CastedThisErrorBehavior::RejectPromise" if IsReturningPromise($function);
                    }

                    push(@implContent, "    return BindingCaller<$className>::callOperation<${templateParameters}>(state, \"${methodName}\");\n");
                    push(@implContent, "}\n");
                    push(@implContent, "\n");
                    push(@implContent, "static inline JSC::EncodedJSValue ${functionName}Caller(JSC::ExecState* state, ${classParameterType} castedThis, JSC::ThrowScope& throwScope)\n");
                }

                push(@implContent, "{\n");
                push(@implContent, "    UNUSED_PARAM(state);\n");
                push(@implContent, "    UNUSED_PARAM(throwScope);\n");

                if ($interface->extendedAttributes->{CheckSecurity} and !$function->signature->extendedAttributes->{DoNotCheckSecurity}) {
                    if ($interfaceName eq "DOMWindow") {
                        push(@implContent, "    if (!BindingSecurity::shouldAllowAccessToDOMWindow(state, castedThis->wrapped(), ThrowSecurityError))\n");
                    } else {
                        push(@implContent, "    if (!BindingSecurity::shouldAllowAccessToFrame(state, castedThis->wrapped().frame(), ThrowSecurityError))\n");
                    }
                    push(@implContent, "        return JSValue::encode(jsUndefined());\n");
                }

                if ($isCustom) {
                    push(@implContent, "    return JSValue::encode(castedThis->" . $functionImplementationName . "(*state));\n");
                } else {
                    push(@implContent, "    auto& impl = castedThis->wrapped();\n");
                    if ($svgPropertyType) {
                        push(@implContent, "    if (impl.isReadOnly()) {\n");
                        push(@implContent, "        setDOMException(state, throwScope, NO_MODIFICATION_ALLOWED_ERR);\n");
                        push(@implContent, "        return JSValue::encode(jsUndefined());\n");
                        push(@implContent, "    }\n");
                        push(@implContent, "    $svgPropertyType& podImpl = impl.propertyReference();\n");
                        $implIncludes{"ExceptionCode.h"} = 1;
                    }

                    GenerateArgumentsCountCheck(\@implContent, $function, $interface);

                    push(@implContent, "    ExceptionCode ec = 0;\n") if $mayThrowLegacyException;

                    if ($function->signature->extendedAttributes->{CheckSecurityForNode}) {
                        push(@implContent, "    if (!shouldAllowAccessToNode(state, impl." . $function->signature->name . "(" . ($mayThrowLegacyException ? "ec" : "") .")))\n");
                        push(@implContent, "        return JSValue::encode(jsNull());\n");
                        $implIncludes{"JSDOMBinding.h"} = 1;
                    }

                    my ($functionString, $dummy) = GenerateParametersCheck(\@implContent, $function, $interface, $functionImplementationName, $svgPropertyType, $svgPropertyOrListPropertyType, $svgListPropertyType);
                    GenerateImplementationFunctionCall($function, $functionString, "    ", $svgPropertyType, $interface);
                }
            }

            push(@implContent, "}\n\n");
            push(@implContent, "#endif\n\n") if $conditional;

            # Generate a function dispatching call to the rest of the overloads.
            GenerateOverloadedFunctionOrConstructor($function, $interface, 0) if !$isCustom && $isOverloaded && $function->{overloadIndex} == @{$function->{overloads}};
        }

        push(@implContent, $endAppleCopyright) if $inAppleCopyright;

    }


    GenerateImplementationIterableFunctions($interface) if $interface->iterable;
    GenerateSerializerFunction($interface, $className) if $interface->serializable;

    if ($needsVisitChildren) {
        push(@implContent, "void ${className}::visitChildren(JSCell* cell, SlotVisitor& visitor)\n");
        push(@implContent, "{\n");
        push(@implContent, "    auto* thisObject = jsCast<${className}*>(cell);\n");
        push(@implContent, "    ASSERT_GC_OBJECT_INHERITS(thisObject, info());\n");
        push(@implContent, "    Base::visitChildren(thisObject, visitor);\n");
        if ($codeGenerator->InheritsInterface($interface, "EventTarget")) {
            push(@implContent, "    thisObject->wrapped().visitJSEventListeners(visitor);\n");
        }
        push(@implContent, "    thisObject->visitAdditionalChildren(visitor);\n") if $interface->extendedAttributes->{JSCustomMarkFunction};
        if ($interface->extendedAttributes->{ReportExtraMemoryCost}) {
            push(@implContent, "    visitor.reportExtraMemoryVisited(thisObject->wrapped().memoryCost());\n");
            if ($interface->extendedAttributes->{ReportExternalMemoryCost}) {;
                push(@implContent, "#if ENABLE(RESOURCE_USAGE)\n");
                push(@implContent, "    visitor.reportExternalMemoryVisited(thisObject->wrapped().externalMemoryCost());\n");
                push(@implContent, "#endif\n");
            }
        }
        if ($numCachedAttributes > 0) {
            foreach (@{$interface->attributes}) {
                my $attribute = $_;
                if ($attribute->signature->extendedAttributes->{CachedAttribute}) {
                    push(@implContent, "    visitor.append(&thisObject->m_" . $attribute->signature->name . ");\n");
                }
            }
        }
        push(@implContent, "}\n\n");
    }

    if (InstanceNeedsEstimatedSize($interface)) {
        push(@implContent, "size_t ${className}::estimatedSize(JSCell* cell)\n");
        push(@implContent, "{\n");
        push(@implContent, "    auto* thisObject = jsCast<${className}*>(cell);\n");
        push(@implContent, "    return Base::estimatedSize(thisObject) + thisObject->wrapped().memoryCost();\n");
        push(@implContent, "}\n\n");
    }

    # Cached attributes are indeed allowed when there is a custom mark/visitChildren function.
    # The custom function must make sure to account for the cached attribute.
    # Uncomment the below line to temporarily enforce generated mark functions when cached attributes are present.
    # die "Can't generate binding for class with cached attribute and custom mark." if $numCachedAttributes > 0 and $interface->extendedAttributes->{JSCustomMarkFunction};

    if ($indexedGetterFunction) {
        $implIncludes{"URL.h"} = 1 if $indexedGetterFunction->signature->type eq "DOMString";
        if ($interfaceName =~ /^HTML\w*Collection$/ or $interfaceName eq "RadioNodeList") {
            $implIncludes{"JSNode.h"} = 1;
            $implIncludes{"Node.h"} = 1;
        }
    }

    if (ShouldGenerateWrapperOwnerCode($hasParent, $interface) && !GetCustomIsReachable($interface)) {
        push(@implContent, "bool JS${interfaceName}Owner::isReachableFromOpaqueRoots(JSC::Handle<JSC::Unknown> handle, void*, SlotVisitor& visitor)\n");
        push(@implContent, "{\n");
        # All ActiveDOMObjects implement hasPendingActivity(), but not all of them
        # increment their C++ reference counts when hasPendingActivity() becomes
        # true. As a result, ActiveDOMObjects can be prematurely destroyed before
        # their pending activities complete. To wallpaper over this bug, JavaScript
        # wrappers unconditionally keep ActiveDOMObjects with pending activity alive.
        # FIXME: Fix this lifetime issue in the DOM, and move this hasPendingActivity
        # check just above the (GetGenerateIsReachable($interface) eq "Impl") check below.
        my $emittedJSCast = 0;
        if ($codeGenerator->InheritsExtendedAttribute($interface, "ActiveDOMObject")) {
            push(@implContent, "    auto* js${interfaceName} = jsCast<JS${interfaceName}*>(handle.slot()->asCell());\n");
            $emittedJSCast = 1;
            push(@implContent, "    if (js${interfaceName}->wrapped().hasPendingActivity())\n");
            push(@implContent, "        return true;\n");
        }
        if ($codeGenerator->InheritsInterface($interface, "EventTarget")) {
            if (!$emittedJSCast) {
                push(@implContent, "    auto* js${interfaceName} = jsCast<JS${interfaceName}*>(handle.slot()->asCell());\n");
                $emittedJSCast = 1;
            }
            push(@implContent, "    if (js${interfaceName}->wrapped().isFiringEventListeners())\n");
            push(@implContent, "        return true;\n");
        }
        if ($codeGenerator->InheritsInterface($interface, "Node")) {
            if (!$emittedJSCast) {
                push(@implContent, "    auto* js${interfaceName} = jsCast<JS${interfaceName}*>(handle.slot()->asCell());\n");
                $emittedJSCast = 1;
            }
            push(@implContent, "    if (JSNodeOwner::isReachableFromOpaqueRoots(handle, 0, visitor))\n");
            push(@implContent, "        return true;\n");
        }
        if (GetGenerateIsReachable($interface)) {
            if (!$emittedJSCast) {
                push(@implContent, "    auto* js${interfaceName} = jsCast<JS${interfaceName}*>(handle.slot()->asCell());\n");
                $emittedJSCast = 1;
            }

            my $rootString;
            if (GetGenerateIsReachable($interface) eq "Impl") {
                $rootString  = "    ${implType}* root = &js${interfaceName}->wrapped();\n";
            } elsif (GetGenerateIsReachable($interface) eq "ImplWebGLRenderingContext") {
                $rootString  = "    WebGLRenderingContextBase* root = WTF::getPtr(js${interfaceName}->wrapped().context());\n";
            } elsif (GetGenerateIsReachable($interface) eq "ImplFrame") {
                $rootString  = "    Frame* root = WTF::getPtr(js${interfaceName}->wrapped().frame());\n";
                $rootString .= "    if (!root)\n";
                $rootString .= "        return false;\n";
            } elsif (GetGenerateIsReachable($interface) eq "ImplDocument") {
                $rootString  = "    Document* root = WTF::getPtr(js${interfaceName}->wrapped().document());\n";
                $rootString .= "    if (!root)\n";
                $rootString .= "        return false;\n";
            } elsif (GetGenerateIsReachable($interface) eq "ImplElementRoot") {
                $implIncludes{"Element.h"} = 1;
                $implIncludes{"JSNodeCustom.h"} = 1;
                $rootString  = "    Element* element = WTF::getPtr(js${interfaceName}->wrapped().element());\n";
                $rootString .= "    if (!element)\n";
                $rootString .= "        return false;\n";
                $rootString .= "    void* root = WebCore::root(element);\n";
            } elsif (GetGenerateIsReachable($interface) eq "ImplOwnerNodeRoot") {
                $implIncludes{"Element.h"} = 1;
                $implIncludes{"JSNodeCustom.h"} = 1;
                $rootString  = "    void* root = WebCore::root(js${interfaceName}->wrapped().ownerNode());\n";
            } elsif (GetGenerateIsReachable($interface) eq "ImplScriptExecutionContext") {
                $rootString  = "    ScriptExecutionContext* root = WTF::getPtr(js${interfaceName}->wrapped().scriptExecutionContext());\n";
                $rootString .= "    if (!root)\n";
                $rootString .= "        return false;\n";
            } else {
                $rootString  = "    void* root = WebCore::root(&js${interfaceName}->wrapped());\n";
            }

            push(@implContent, $rootString);
            push(@implContent, "    return visitor.containsOpaqueRoot(root);\n");
        } else {
            if (!$emittedJSCast) {
                push(@implContent, "    UNUSED_PARAM(handle);\n");
            }
            push(@implContent, "    UNUSED_PARAM(visitor);\n");
            push(@implContent, "    return false;\n");
        }
        push(@implContent, "}\n\n");
    }

    if (ShouldGenerateWrapperOwnerCode($hasParent, $interface) && !$interface->extendedAttributes->{JSCustomFinalize}) {
        push(@implContent, "void JS${interfaceName}Owner::finalize(JSC::Handle<JSC::Unknown> handle, void* context)\n");
        push(@implContent, "{\n");
        push(@implContent, "    auto* js${interfaceName} = jsCast<JS${interfaceName}*>(handle.slot()->asCell());\n");
        push(@implContent, "    auto& world = *static_cast<DOMWrapperWorld*>(context);\n");
        push(@implContent, "    uncacheWrapper(world, &js${interfaceName}->wrapped(), js${interfaceName});\n");
        push(@implContent, "}\n\n");
    }

    if (ShouldGenerateToJSImplementation($hasParent, $interface)) {
        my $vtableNameGnu = GetGnuVTableNameForInterface($interface);
        my $vtableRefGnu = GetGnuVTableRefForInterface($interface);
        my $vtableRefWin = GetWinVTableRefForInterface($interface);

        push(@implContent, <<END) if $vtableNameGnu;
#if ENABLE(BINDING_INTEGRITY)
#if PLATFORM(WIN)
#pragma warning(disable: 4483)
extern "C" { extern void (*const ${vtableRefWin}[])(); }
#else
extern "C" { extern void* ${vtableNameGnu}[]; }
#endif
#endif

END

        push(@implContent, "JSC::JSValue toJSNewlyCreated(JSC::ExecState*, JSDOMGlobalObject* globalObject, Ref<$implType>&& impl)\n");
        push(@implContent, "{\n");
        push(@implContent, <<END) if $vtableNameGnu;

#if ENABLE(BINDING_INTEGRITY)
    void* actualVTablePointer = *(reinterpret_cast<void**>(impl.ptr()));
#if PLATFORM(WIN)
    void* expectedVTablePointer = reinterpret_cast<void*>(${vtableRefWin});
#else
    void* expectedVTablePointer = ${vtableRefGnu};
#if COMPILER(CLANG)
    // If this fails $implType does not have a vtable, so you need to add the
    // ImplementationLacksVTable attribute to the interface definition
    static_assert(__is_polymorphic($implType), "${implType} is not polymorphic");
#endif
#endif
    // If you hit this assertion you either have a use after free bug, or
    // $implType has subclasses. If $implType has subclasses that get passed
    // to toJS() we currently require $interfaceName you to opt out of binding hardening
    // by adding the SkipVTableValidation attribute to the interface IDL definition
    RELEASE_ASSERT(actualVTablePointer == expectedVTablePointer);
#endif
END
        push(@implContent, <<END) if $interface->extendedAttributes->{ImplementationLacksVTable};
#if COMPILER(CLANG)
    // If you hit this failure the interface definition has the ImplementationLacksVTable
    // attribute. You should remove that attribute. If the class has subclasses
    // that may be passed through this toJS() function you should use the SkipVTableValidation
    // attribute to $interfaceName.
    static_assert(!__is_polymorphic($implType), "${implType} is polymorphic but the IDL claims it is not");
#endif
END
        push(@implContent, <<END) if $interface->extendedAttributes->{ReportExtraMemoryCost};
    globalObject->vm().heap.reportExtraMemoryAllocated(impl->memoryCost());
END

        push(@implContent, "    return createWrapper<$implType>(globalObject, WTFMove(impl));\n");
        push(@implContent, "}\n\n");

        push(@implContent, "JSC::JSValue toJS(JSC::ExecState* state, JSDOMGlobalObject* globalObject, $implType& impl)\n");
        push(@implContent, "{\n");
        push(@implContent, "    return wrap(state, globalObject, impl);\n");
        push(@implContent, "}\n\n");
    }

    if (ShouldGenerateToWrapped($hasParent, $interface) and !$interface->extendedAttributes->{JSCustomToNativeObject}) {
        push(@implContent, "$implType* ${className}::toWrapped(JSC::JSValue value)\n");
        push(@implContent, "{\n");
        push(@implContent, "    if (auto* wrapper = " . GetCastingHelperForThisObject($interface) . "(value))\n");
        push(@implContent, "        return &wrapper->wrapped();\n");
        push(@implContent, "    return nullptr;\n");
        push(@implContent, "}\n");
    }

    push(@implContent, "\n}\n");

    my $conditionalString = $codeGenerator->GenerateConditionalString($interface);
    push(@implContent, "\n#endif // ${conditionalString}\n") if $conditionalString;
}

sub GenerateSerializerFunction
{
    my ($interface, $className) = @_;

    my $interfaceName = $interface->name;

    my $serializerFunctionName = "toJSON";
    my $serializerNativeFunctionName = $codeGenerator->WK_lcfirst($className) . "PrototypeFunction" . $codeGenerator->WK_ucfirst($serializerFunctionName);

    AddToImplIncludes("<runtime/ObjectConstructor.h>");
    push(@implContent, "static inline EncodedJSValue ${serializerNativeFunctionName}Caller(ExecState* state, JS$interfaceName* thisObject, JSC::ThrowScope& throwScope)\n");
    push(@implContent, "{\n");
    push(@implContent, "    auto& vm = state->vm();\n");
    push(@implContent, "    auto* result = constructEmptyObject(state);\n");
    push(@implContent, "\n");

    my @serializedAttributes = ();
    foreach my $attribute_name (@{$interface->serializable->attributes}) {
        my $found_attribute = 0;
        foreach my $attribute (@{$interface->attributes}) {
            if ($attribute_name eq $attribute->signature->name) {
                push @serializedAttributes, $attribute;
                $found_attribute = 1;
                last;
            }
        }
        
        die "Failed to find \"serializer\" attribute \"$attribute_name\" in $interfaceName" if !$found_attribute;
    }

    foreach my $attribute (@serializedAttributes) {
        my $name = $attribute->signature->name;

        my $getFunctionName = GetAttributeGetterName($interface, $className, $attribute);
        push(@implContent, "    auto ${name}Value = ${getFunctionName}Getter(*state, *thisObject, throwScope);\n");
        push(@implContent, "    ASSERT(!throwScope.exception());\n");
        push(@implContent, "    result->putDirect(vm, Identifier::fromString(&vm, \"${name}\"), ${name}Value);\n");
        push(@implContent, "\n");
    }

    push(@implContent, "    return JSValue::encode(result);\n");
    push(@implContent, "}\n");
    push(@implContent, "\n");
    push(@implContent, "EncodedJSValue JSC_HOST_CALL ${serializerNativeFunctionName}(ExecState* state)\n");
    push(@implContent, "{\n");
    push(@implContent, "    return BindingCaller<JS$interfaceName>::callOperation<${serializerNativeFunctionName}Caller>(state, \"$serializerFunctionName\");\n");
    push(@implContent, "}\n");
    push(@implContent, "\n");
}

sub GenerateCallWithUsingReferences
{
    my ($callWith, $outputArray, $returnValue, $function) = @_;

    my $statePointer = "&state";
    my $stateReference = "state";
    my $globalObject = "jsCast<JSDOMGlobalObject*>(state.lexicalGlobalObject())";

    return GenerateCallWith($callWith, $outputArray, $returnValue, $function, $statePointer, $stateReference, $globalObject);
}

# FIXME: We should remove GenerateCallWithUsingPointers and combine GenerateCallWithUsingReferences and GenerateCallWith
sub GenerateCallWithUsingPointers
{
    my ($callWith, $outputArray, $returnValue, $function) = @_;

    my $statePointer = "state";
    my $stateReference = "*state";
    my $globalObject = "jsCast<JSDOMGlobalObject*>(state->lexicalGlobalObject())";

    return GenerateCallWith($callWith, $outputArray, $returnValue, $function, $statePointer, $stateReference, $globalObject);
}

sub GenerateCallWith
{
    my $callWith = shift;
    return () unless $callWith;
    my $outputArray = shift;
    my $returnValue = shift;
    my $function = shift;
    my $statePointer = shift;
    my $stateReference = shift;
    my $globalObject = shift;

    my @callWithArgs;
    push(@callWithArgs, $stateReference) if $codeGenerator->ExtendedAttributeContains($callWith, "ScriptState");
    if ($codeGenerator->ExtendedAttributeContains($callWith, "ScriptExecutionContext")) {
        push(@$outputArray, "    auto* context = $globalObject->scriptExecutionContext();\n");
        push(@$outputArray, "    if (!context)\n");
        push(@$outputArray, "        return" . ($returnValue ? " " . $returnValue : "") . ";\n");
        push(@callWithArgs, "*context");
    }
    if ($codeGenerator->ExtendedAttributeContains($callWith, "Document")) {
        $implIncludes{"Document.h"} = 1;
        push(@$outputArray, "    auto* context = $globalObject->scriptExecutionContext();\n");
        push(@$outputArray, "    if (!context)\n");
        push(@$outputArray, "        return" . ($returnValue ? " " . $returnValue : "") . ";\n");
        push(@$outputArray, "    ASSERT(context->isDocument());\n");
        push(@$outputArray, "    auto& document = downcast<Document>(*context);\n");
        push(@callWithArgs, "document");
    }
    if ($codeGenerator->ExtendedAttributeContains($callWith, "CallerDocument")) {
        $implIncludes{"Document.h"} = 1;
        push(@$outputArray, "    auto* document = callerDOMWindow($statePointer).document();\n");
        push(@$outputArray, "    if (!document)\n");
        push(@$outputArray, "        return" . ($returnValue ? " " . $returnValue : "") . ";\n");
        push(@callWithArgs, "*document");
    }
    if ($function and $codeGenerator->ExtendedAttributeContains($callWith, "ScriptArguments")) {
        push(@$outputArray, "    RefPtr<Inspector::ScriptArguments> scriptArguments(Inspector::createScriptArguments($statePointer, " . @{$function->parameters} . "));\n");
        $implIncludes{"<inspector/ScriptArguments.h>"} = 1;
        $implIncludes{"<inspector/ScriptCallStackFactory.h>"} = 1;
        push(@callWithArgs, "WTFMove(scriptArguments)");
    }
    push(@callWithArgs, "activeDOMWindow($statePointer)") if $codeGenerator->ExtendedAttributeContains($callWith, "ActiveWindow");
    push(@callWithArgs, "firstDOMWindow($statePointer)") if $codeGenerator->ExtendedAttributeContains($callWith, "FirstWindow");
    push(@callWithArgs, "callerDOMWindow($statePointer)") if $codeGenerator->ExtendedAttributeContains($callWith, "CallerWindow");

    return @callWithArgs;
}

sub GenerateArgumentsCountCheck
{
    my $outputArray = shift;
    my $function = shift;
    my $interface = shift;

    my $numMandatoryParams = @{$function->parameters};
    foreach my $param (reverse(@{$function->parameters})) {
        if ($param->isOptional or $param->isVariadic) {
            $numMandatoryParams--;
        } else {
            last;
        }
    }
    if ($numMandatoryParams >= 1)
    {
        push(@$outputArray, "    if (UNLIKELY(state->argumentCount() < $numMandatoryParams))\n");
        push(@$outputArray, "        return throwVMError(state, throwScope, createNotEnoughArgumentsError(state));\n");
    }
}

my %automaticallyGeneratedDefaultValues = (
    "any" => "undefined",

    # toString() will convert undefined to the string "undefined";
    # (note that this optimizes a behavior that is almost never useful)
    "DOMString" => "\"undefined\"",
    "USVString" => "\"undefined\"",

    # Dictionary(state, undefined) will construct an empty Dictionary.
    "Dictionary" => "[]",

    # JSValue::toBoolean() will convert undefined to false.
    "boolean" => "false",

    # JSValue::toInt*() / JSValue::toUint*() will convert undefined to 0.
    "byte" => "0",
    "long long" => "0",
    "long" => "0",
    "octet" => "0",
    "short" => "0",
    "unsigned long long" => "0",
    "unsigned long" => "0",
    "unsigned short" => "0",

    # toNumber() / toFloat() convert undefined to NaN.
    "double" => "NaN",
    "float" => "NaN",
    "unrestricted double" => "NaN",
    "unrestricted float" => "NaN",
);

sub WillConvertUndefinedToDefaultParameterValue
{
    my $parameterType = shift;
    my $defaultValue = shift;

    my $automaticallyGeneratedDefaultValue = $automaticallyGeneratedDefaultValues{$parameterType};
    return 1 if defined $automaticallyGeneratedDefaultValue && $automaticallyGeneratedDefaultValue eq $defaultValue;

    return 1 if $defaultValue eq "null" && $codeGenerator->IsWrapperType($parameterType);
    return 1 if $defaultValue eq "[]" && $codeGenerator->IsDictionaryType($parameterType);

    return 0;
}

sub GenerateParametersCheck
{
    my ($outputArray, $function, $interface, $functionImplementationName, $svgPropertyType, $svgPropertyOrListPropertyType, $svgListPropertyType) = @_;

    my $interfaceName = $interface->name;
    my $visibleInterfaceName = $codeGenerator->GetVisibleInterfaceName($interface);
    my @arguments;
    my $functionName;
    my $implementedBy = $function->signature->extendedAttributes->{ImplementedBy};
    my $numParameters = @{$function->parameters};

    if ($implementedBy) {
        AddToImplIncludes("${implementedBy}.h", $function->signature->extendedAttributes->{Conditional});
        unshift(@arguments, "impl") if !$function->isStatic;
        $functionName = "WebCore::${implementedBy}::${functionImplementationName}";
    } elsif ($function->isStatic) {
        $functionName = "${interfaceName}::${functionImplementationName}";
    } elsif ($svgPropertyOrListPropertyType and !$svgListPropertyType) {
        $functionName = "podImpl.${functionImplementationName}";
    } else {
        $functionName = "impl.${functionImplementationName}";
    }

    my $quotedFunctionName;
    if (!$function->signature->extendedAttributes->{Constructor}) {
        my $name = $function->signature->name;
        $quotedFunctionName = "\"$name\"";
        push(@arguments, GenerateCallWithUsingPointers($function->signature->extendedAttributes->{CallWith}, \@$outputArray, "JSValue::encode(jsUndefined())", $function));
    } else {
        $quotedFunctionName = "nullptr";
    }

    $implIncludes{"ExceptionCode.h"} = 1;
    $implIncludes{"JSDOMBinding.h"} = 1;

    my $argumentIndex = 0;
    foreach my $parameter (@{$function->parameters}) {
        my $type = $parameter->type;
        my $idlType = $parameter->idlType;

        die "Optional parameters of non-nullable wrapper types are not supported" if $parameter->isOptional && !$parameter->isNullable && $codeGenerator->IsWrapperType($type);
        die "Optional parameters preceding variadic parameters are not supported" if ($parameter->isOptional &&  @{$function->parameters}[$numParameters - 1]->isVariadic);

        if ($parameter->isOptional && !defined($parameter->default)) {
            # As per Web IDL, optional dictionary parameters are always considered to have a default value of an empty dictionary, unless otherwise specified.
            $parameter->default("[]") if $type eq "Dictionary" or $codeGenerator->IsDictionaryType($type);

            # We use undefined as default value for optional parameters of type 'any' unless specified otherwise.
            $parameter->default("undefined") if $type eq "any";

            # We use the null string as default value for parameters of type DOMString unless specified otherwise.
            $parameter->default("null") if $codeGenerator->IsStringType($type);

            # As per Web IDL, passing undefined for a nullable parameter is treated as null. Therefore, use null as
            # default value for nullable parameters unless otherwise specified.
            $parameter->default("null") if $parameter->isNullable;

            # For callback parameters, the generated bindings treat undefined as null, so use null as implicit default value.
            $parameter->default("null") if $codeGenerator->IsCallbackInterface($type);
        }

        my $name = $parameter->name;
        my $value = $name;

        if ($codeGenerator->IsCallbackInterface($type)) {
            my $callbackClassName = GetCallbackClassName($type);
            $implIncludes{"$callbackClassName.h"} = 1;
            if ($parameter->isOptional) {
                push(@$outputArray, "    RefPtr<$type> $name;\n");
                push(@$outputArray, "    if (!state->argument($argumentIndex).isUndefinedOrNull()) {\n");
                if ($codeGenerator->IsFunctionOnlyCallbackInterface($type)) {
                    push(@$outputArray, "        if (!state->uncheckedArgument($argumentIndex).isFunction())\n");
                } else {
                    push(@$outputArray, "        if (!state->uncheckedArgument($argumentIndex).isObject())\n");
                }
                push(@$outputArray, "            return throwArgumentMustBeFunctionError(*state, throwScope, $argumentIndex, \"$name\", \"$visibleInterfaceName\", $quotedFunctionName);\n");
                if ($function->isStatic) {
                    AddToImplIncludes("CallbackFunction.h");
                    push(@$outputArray, "        $name = createFunctionOnlyCallback<${callbackClassName}>(state, jsCast<JSDOMGlobalObject*>(state->lexicalGlobalObject()), state->uncheckedArgument($argumentIndex));\n");
                } else {
                    push(@$outputArray, "        $name = ${callbackClassName}::create(asObject(state->uncheckedArgument($argumentIndex)), castedThis->globalObject());\n");
                }
                push(@$outputArray, "    }\n");
            } else {
                die "CallbackInterface does not support Variadic parameter" if $parameter->isVariadic;
                if ($codeGenerator->IsFunctionOnlyCallbackInterface($type)) {
                    push(@$outputArray, "    if (UNLIKELY(!state->uncheckedArgument($argumentIndex).isFunction()))\n");
                } else {
                    push(@$outputArray, "    if (UNLIKELY(!state->uncheckedArgument($argumentIndex).isObject()))\n");
                }
                push(@$outputArray, "        return throwArgumentMustBeFunctionError(*state, throwScope, $argumentIndex, \"$name\", \"$visibleInterfaceName\", $quotedFunctionName);\n");
                if ($function->isStatic) {
                    AddToImplIncludes("CallbackFunction.h");
                    push(@$outputArray, "    auto $name = createFunctionOnlyCallback<${callbackClassName}>(state, jsCast<JSDOMGlobalObject*>(state->lexicalGlobalObject()), state->uncheckedArgument($argumentIndex));\n");
                } else {
                    push(@$outputArray, "    auto $name = ${callbackClassName}::create(asObject(state->uncheckedArgument($argumentIndex)), castedThis->globalObject());\n");
                }
            }
            $value = "WTFMove($name)";
        } elsif ($parameter->isVariadic) {
            $implIncludes{"JSDOMConvert.h"} = 1;
            AddToImplIncludesForIDLType($idlType, $function->signature->extendedAttributes->{Conditional});
        
            my $metaType = GetIDLType($interface, $idlType);
            push(@$outputArray, "    auto $name = convertVariadicArguments<$metaType>(*state, $argumentIndex);\n");
            push(@$outputArray, "    RETURN_IF_EXCEPTION(throwScope, encodedJSValue());\n");

            $value = "WTFMove($name.arguments.value())";

        } elsif ($codeGenerator->IsEnumType($type)) {
            my $className = GetEnumerationClassName($type, $interface);

            $implIncludes{"<runtime/Error.h>"} = 1;

            my $nativeType = $className;
            my $optionalValue = "optionalValue";
            my $defineOptionalValue = "auto optionalValue";
            my $indent = "";

            die "Variadic parameter is already handled here" if $parameter->isVariadic;
            my $argumentLookupMethod = $parameter->isOptional ? "argument" : "uncheckedArgument";

            if ($parameter->isOptional && !defined($parameter->default)) {
                $nativeType = "Optional<$className>";
                $optionalValue = $name;
                $defineOptionalValue = $name;
            }

            push(@$outputArray, "    auto ${name}Value = state->$argumentLookupMethod($argumentIndex);\n");
            push(@$outputArray, "    $nativeType $name;\n");

            if ($parameter->isOptional) {
                if (!defined $parameter->default) {
                    push(@$outputArray, "    if (!${name}Value.isUndefined()) {\n");
                } else {
                    push(@$outputArray, "    if (${name}Value.isUndefined())\n");
                    push(@$outputArray, "        $name = " . GenerateDefaultValue($interface, $parameter) . ";\n");
                    push(@$outputArray, "    else {\n");
                }
                $indent = "    ";
            }

            push(@$outputArray, "$indent    $defineOptionalValue = parseEnumeration<$className>(*state, ${name}Value);\n");
            push(@$outputArray, "$indent    RETURN_IF_EXCEPTION(throwScope, encodedJSValue());\n");
            push(@$outputArray, "$indent    if (UNLIKELY(!$optionalValue))\n");
            push(@$outputArray, "$indent        return throwArgumentMustBeEnumError(*state, throwScope, $argumentIndex, \"$name\", \"$visibleInterfaceName\", $quotedFunctionName, expectedEnumerationValues<$className>());\n");
            push(@$outputArray, "$indent    $name = optionalValue.value();\n") if $optionalValue ne $name;

            push(@$outputArray, "    }\n") if $indent ne "";
        } else {
            my $outer;
            my $inner;
            my $nativeType = GetNativeTypeFromSignature($interface, $parameter);
            my $isTearOff = $codeGenerator->IsSVGTypeNeedingTearOff($type) && $interfaceName !~ /List$/;
            my $shouldPassByReference = $isTearOff || ShouldPassWrapperByReference($parameter, $interface);

            die "Variadic parameter is already handled here" if $parameter->isVariadic;
            my $argumentLookupMethod = $parameter->isOptional ? "argument" : "uncheckedArgument";

            if (!$shouldPassByReference && ($codeGenerator->IsWrapperType($type) || $codeGenerator->IsTypedArrayType($type))) {
                $implIncludes{"<runtime/Error.h>"} = 1;
                my $checkedArgument = "state->$argumentLookupMethod($argumentIndex)";
                my $uncheckedArgument = "state->uncheckedArgument($argumentIndex)";
                my ($nativeValue, $mayThrowException) = JSValueToNative($interface, $parameter, $uncheckedArgument, $function->signature->extendedAttributes->{Conditional});
                push(@$outputArray, "    $nativeType $name = nullptr;\n");
                push(@$outputArray, "    if (!$checkedArgument.isUndefinedOrNull()) {\n");
                push(@$outputArray, "        $name = $nativeValue;\n");
                push(@$outputArray, "        RETURN_IF_EXCEPTION(throwScope, encodedJSValue());\n") if $mayThrowException;
                push(@$outputArray, "        if (UNLIKELY(!$name))\n");
                push(@$outputArray, "            return throwArgumentTypeError(*state, throwScope, $argumentIndex, \"$name\", \"$visibleInterfaceName\", $quotedFunctionName, \"$type\");\n");
                push(@$outputArray, "    }\n");
                $value = "WTFMove($name)";
            } else {
                if ($parameter->isOptional && defined($parameter->default) && !WillConvertUndefinedToDefaultParameterValue($type, $parameter->default)) {
                    my $defaultValue = $parameter->default;

                    # String-related optimizations.
                    if ($codeGenerator->IsStringType($type)) {
                        my $useAtomicString = $parameter->extendedAttributes->{AtomicString};
                        if ($defaultValue eq "null") {
                            $defaultValue = $useAtomicString ? "nullAtom" : "String()";
                        } elsif ($defaultValue eq "\"\"") {
                            $defaultValue = $useAtomicString ? "emptyAtom" : "emptyString()";
                        } else {
                            $defaultValue = $useAtomicString ? "AtomicString($defaultValue, AtomicString::ConstructFromLiteral)" : "ASCIILiteral($defaultValue)";
                        }
                    } else {
                        $defaultValue = GenerateDefaultValue($interface, $parameter);
                    }

                    $outer = "state->$argumentLookupMethod($argumentIndex).isUndefined() ? $defaultValue : ";
                    $inner = "state->uncheckedArgument($argumentIndex)";
                } elsif ($parameter->isOptional && !defined($parameter->default)) {
                    # Use WTF::Optional<>() for optional parameters that are missing or undefined and that do not have a default value in the IDL.
                    $outer = "state->$argumentLookupMethod($argumentIndex).isUndefined() ? Optional<$nativeType>() : ";
                    $inner = "state->uncheckedArgument($argumentIndex)";
                } else {
                    $outer = "";
                    $inner = "state->$argumentLookupMethod($argumentIndex)";
                }

                my ($nativeValue, $mayThrowException) = JSValueToNative($interface, $parameter, $inner, $function->signature->extendedAttributes->{Conditional});
                push(@$outputArray, "    auto $name = ${outer}${nativeValue};\n");
                $value = "WTFMove($name)";
                push(@$outputArray, "    RETURN_IF_EXCEPTION(throwScope, encodedJSValue());\n") if $mayThrowException;
            }

            if ($shouldPassByReference) {
                push(@$outputArray, "    if (UNLIKELY(!$name))\n");
                push(@$outputArray, "        return throwArgumentTypeError(*state, throwScope, $argumentIndex, \"$name\", \"$visibleInterfaceName\", $quotedFunctionName, \"$type\");\n");
                $value = $isTearOff ? "$name->propertyReference()" : "*$name";
            }

            if ($codeGenerator->IsTypedArrayType($type) and $parameter->type ne "ArrayBuffer") {
               $value = $shouldPassByReference ? "$name.releaseNonNull()" : "WTFMove($name)";
            } elsif ($codeGenerator->IsDictionaryType($type)) {
                $value = "WTFMove($name)";
            }
        }

        push(@arguments, $value);
        $argumentIndex++;
    }

    push @arguments, GenerateReturnParameters($function);
    my $functionString = "$functionName(" . join(", ", @arguments) . ")";
    $functionString = "propagateException(*state, throwScope, $functionString)" if $function->signature->type && $function->signature->type eq "void" && $function->signature->extendedAttributes->{MayThrowException};

    return ($functionString, scalar @arguments);
}

sub GenerateReturnParameters
{
    my $function = shift;

    my @arguments;
    push(@arguments, "WTFMove(promise)") if IsReturningPromise($function);
    push(@arguments, "ec") if $function->signature->extendedAttributes->{MayThrowLegacyException};
    return @arguments;
}

sub GenerateDictionaryHeader
{
    my ($object, $dictionary, $className, $enumerations) = @_;

    my $dictionaryName = $dictionary->name;

    # - Add default header template and header protection.
    push(@headerContentHeader, GenerateHeaderContentHeader($dictionary));

    $headerIncludes{"$className.h"} = 1;
    $headerIncludes{"JSDOMConvert.h"} = 1;

    push(@headerContent, "\nnamespace WebCore {\n\n");
    push(@headerContent, GenerateDictionaryHeaderContent($dictionary, $className));
    push(@headerContent, GenerateEnumerationsHeaderContent($dictionary, $enumerations));
    push(@headerContent, "} // namespace WebCore\n");

    my $conditionalString = $codeGenerator->GenerateConditionalString($dictionary);
    push(@headerContent, "\n#endif // ${conditionalString}\n") if $conditionalString;
    
    # - Generate dependencies.
    if ($writeDependencies) {
        my @ancestors;
        my $parentName = $dictionary->parent;
        while (defined($parentName)) {
            push(@ancestors, $parentName) if $codeGenerator->IsExternalDictionaryType($parentName);
            my $parentDictionary = $codeGenerator->GetDictionaryByName($parentName);
            assert("Unable to find definition for dictionary named '" . $parentName . "'!") unless $parentDictionary;
            $parentName = $parentDictionary->parent;
        }
        push(@depsContent, "$className.h : ", join(" ", map { "$_.idl" } @ancestors), "\n");
        push(@depsContent, map { "$_.idl :\n" } @ancestors);
    }
}

sub GenerateDictionaryImplementation
{
    my ($object, $dictionary, $className, $enumerations) = @_;

    # - Add default header template
    push(@implContentHeader, GenerateImplementationContentHeader($dictionary));

    push(@implContent, "\nusing namespace JSC;\n\n");
    push(@implContent, "namespace WebCore {\n\n");
    push(@implContent, GenerateEnumerationsImplementationContent($dictionary, $enumerations));
    push(@implContent, GenerateDictionaryImplementationContent($dictionary, $className));
    push(@implContent, "} // namespace WebCore\n");

    my $conditionalString = $codeGenerator->GenerateConditionalString($dictionary);
    push(@implContent, "\n#endif // ${conditionalString}\n") if $conditionalString;
}

sub GenerateCallbackHeader
{
    my ($object, $interface, $enumerations, $dictionaries) = @_;

    my $interfaceName = $interface->name;
    my $className = "JS$interfaceName";

    # - Add default header template and header protection
    push(@headerContentHeader, GenerateHeaderContentHeader($interface));

    $headerIncludes{"ActiveDOMCallback.h"} = 1;
    $headerIncludes{"$interfaceName.h"} = 1;
    $headerIncludes{"JSCallbackData.h"} = 1;
    $headerIncludes{"<wtf/Forward.h>"} = 1;

    push(@headerContent, "\nnamespace WebCore {\n\n");
    push(@headerContent, "class $className : public $interfaceName, public ActiveDOMCallback {\n");
    push(@headerContent, "public:\n");

    # The static create() method.
    push(@headerContent, "    static Ref<$className> create(JSC::JSObject* callback, JSDOMGlobalObject* globalObject)\n");
    push(@headerContent, "    {\n");
    push(@headerContent, "        return adoptRef(*new $className(callback, globalObject));\n");
    push(@headerContent, "    }\n\n");

    push(@headerContent, "    virtual ScriptExecutionContext* scriptExecutionContext() const { return ContextDestructionObserver::scriptExecutionContext(); }\n\n");

    push(@headerContent, "    virtual ~$className();\n");

    push(@headerContent, "    " . GetJSCallbackDataType($interface) . "* callbackData() { return m_data; }\n");

    push(@headerContent, "    static JSC::JSValue getConstructor(JSC::VM&, const JSC::JSGlobalObject*);\n") if @{$interface->constants};

    push(@headerContent, "    virtual bool operator==(const $interfaceName&) const;\n\n") if $interface->extendedAttributes->{CallbackNeedsOperatorEqual};

    # Functions
    my $numFunctions = @{$interface->functions};
    if ($numFunctions > 0) {
        push(@headerContent, "\n    // Functions\n");
        foreach my $function (@{$interface->functions}) {
            my @arguments = ();
            foreach my $parameter (@{$function->parameters}) {
                push(@arguments, GetNativeTypeForCallbacks($interface, $parameter->type) . " " . $parameter->name);
            }
            push(@headerContent, "    virtual " . GetNativeTypeForCallbacks($interface, $function->signature->type) . " " . $function->signature->name . "(" . join(", ", @arguments) . ");\n");
        }
    }

    push(@headerContent, "\nprivate:\n");

    # Constructor
    push(@headerContent, "    $className(JSC::JSObject* callback, JSDOMGlobalObject*);\n\n");

    # Private members
    push(@headerContent, "    " . GetJSCallbackDataType($interface) . "* m_data;\n");
    push(@headerContent, "};\n\n");

    # toJS().
    push(@headerContent, "JSC::JSValue toJS(JSC::ExecState*, JSDOMGlobalObject*, $interfaceName&);\n");
    push(@headerContent, "inline JSC::JSValue toJS(JSC::ExecState* state, JSDOMGlobalObject* globalObject, $interfaceName* impl) { return impl ? toJS(state, globalObject, *impl) : JSC::jsNull(); }\n\n");

    push(@headerContent, GenerateEnumerationsHeaderContent($interface, $enumerations));
    push(@headerContent, GenerateDictionariesHeaderContent($interface, $dictionaries));

    push(@headerContent, "} // namespace WebCore\n");

    my $conditionalString = $codeGenerator->GenerateConditionalString($interface);
    push(@headerContent, "\n#endif // ${conditionalString}\n") if $conditionalString;
}

sub GenerateCallbackImplementation
{
    my ($object, $interface, $enumerations, $dictionaries) = @_;

    my $interfaceName = $interface->name;
    my $visibleInterfaceName = $codeGenerator->GetVisibleInterfaceName($interface);
    my $className = "JS$interfaceName";

    # - Add default header template
    push(@implContentHeader, GenerateImplementationContentHeader($interface));

    $implIncludes{"ScriptExecutionContext.h"} = 1;
    $implIncludes{"<runtime/JSLock.h>"} = 1;

    @implContent = ();

    push(@implContent, "\nusing namespace JSC;\n\n");
    push(@implContent, "namespace WebCore {\n\n");

    push(@implContent, GenerateEnumerationsImplementationContent($interface, $enumerations));
    push(@implContent, GenerateDictionariesImplementationContent($interface, $dictionaries));

    # Constructor
    push(@implContent, "${className}::${className}(JSObject* callback, JSDOMGlobalObject* globalObject)\n");
    if ($interface->extendedAttributes->{CallbackNeedsOperatorEqual}) {
        push(@implContent, "    : ${interfaceName}(${className}Type)\n");
    } else {
        push(@implContent, "    : ${interfaceName}()\n");
    }
    push(@implContent, "    , ActiveDOMCallback(globalObject->scriptExecutionContext())\n");
    push(@implContent, "    , m_data(new " . GetJSCallbackDataType($interface) . "(callback, this))\n");
    push(@implContent, "{\n");
    push(@implContent, "}\n\n");

    # Destructor
    push(@implContent, "${className}::~${className}()\n");
    push(@implContent, "{\n");
    push(@implContent, "    ScriptExecutionContext* context = scriptExecutionContext();\n");
    push(@implContent, "    // When the context is destroyed, all tasks with a reference to a callback\n");
    push(@implContent, "    // should be deleted. So if the context is 0, we are on the context thread.\n");
    push(@implContent, "    if (!context || context->isContextThread())\n");
    push(@implContent, "        delete m_data;\n");
    push(@implContent, "    else\n");
    push(@implContent, "        context->postTask(DeleteCallbackDataTask(m_data));\n");
    push(@implContent, "#ifndef NDEBUG\n");
    push(@implContent, "    m_data = nullptr;\n");
    push(@implContent, "#endif\n");
    push(@implContent, "}\n\n");

    if ($interface->extendedAttributes->{CallbackNeedsOperatorEqual}) {
        push(@implContent, "bool ${className}::operator==(const ${interfaceName}& other) const\n");
        push(@implContent, "{\n");
        push(@implContent, "    if (other.type() != type())\n");
        push(@implContent, "        return false;\n");
        push(@implContent, "    return static_cast<const ${className}*>(&other)->m_data->callback() == m_data->callback();\n");
        push(@implContent, "}\n\n");
    }

    # Constants.
    my $numConstants = @{$interface->constants};
    if ($numConstants > 0) {
        GenerateConstructorDeclaration(\@implContent, $className, $interface, $interfaceName);

        my $hashSize = 0;
        my $hashName = $className . "ConstructorTable";

        my @hashKeys = ();
        my @hashValue1 = ();
        my @hashValue2 = ();
        my @hashSpecials = ();
        my %conditionals = ();

        foreach my $constant (@{$interface->constants}) {
            my $name = $constant->name;
            push(@hashKeys, $name);
            push(@hashValue1, $constant->value);
            push(@hashValue2, "0");
            push(@hashSpecials, "DontDelete | ReadOnly | ConstantInteger");

            my $implementedBy = $constant->extendedAttributes->{ImplementedBy};
            $implIncludes{"${implementedBy}.h"} = 1 if $implementedBy;

            my $conditional = $constant->extendedAttributes->{Conditional};
            $conditionals{$name} = $conditional if $conditional;

            $hashSize++;
        }
        $object->GenerateHashTable($hashName, $hashSize, \@hashKeys, \@hashSpecials, \@hashValue1, \@hashValue2, \%conditionals, 1) if $hashSize > 0;

        push(@implContent, $codeGenerator->GenerateCompileTimeCheckForEnumsIfNeeded($interface));

        GenerateConstructorDefinitions(\@implContent, $className, "", $visibleInterfaceName, $interface);

        push(@implContent, "JSValue ${className}::getConstructor(VM& vm, const JSGlobalObject* globalObject)\n{\n");
        push(@implContent, "    return getDOMConstructor<${className}Constructor>(vm, *jsCast<const JSDOMGlobalObject*>(globalObject));\n");
        push(@implContent, "}\n\n");
    }

    # Functions.
    my $numFunctions = @{$interface->functions};
    if ($numFunctions > 0) {
        push(@implContent, "\n// Functions\n");
        foreach my $function (@{$interface->functions}) {
            my @params = @{$function->parameters};
            if ($function->signature->extendedAttributes->{Custom} || GetNativeType($interface, $function->signature->type) ne "bool") {
                next;
            }

            AddIncludesForTypeInImpl($function->signature->type);
            my $functionName = $function->signature->name;
            push(@implContent, "\n" . GetNativeTypeForCallbacks($interface, $function->signature->type) . " ${className}::${functionName}(");

            my @args = ();
            my @argsCheck = ();
            foreach my $param (@params) {
                my $paramName = $param->name;
                AddIncludesForTypeInImpl($param->type, 1);
                push(@args, GetNativeTypeForCallbacks($interface, $param->type) . " " . $paramName);
            }
            push(@implContent, join(", ", @args));
            push(@implContent, ")\n");

            push(@implContent, "{\n");
            push(@implContent, @argsCheck) if @argsCheck;
            push(@implContent, "    if (!canInvokeCallback())\n");
            push(@implContent, "        return true;\n\n");
            push(@implContent, "    Ref<$className> protectedThis(*this);\n\n");
            push(@implContent, "    JSLockHolder lock(m_data->globalObject()->vm());\n\n");
            push(@implContent, "    ExecState* state = m_data->globalObject()->globalExec();\n");
            push(@implContent, "    MarkedArgumentBuffer args;\n");

            foreach my $param (@params) {
                my $paramName = $param->name;
                push(@implContent, "    args.append(" . NativeToJSValueUsingPointers($param, 1, $interface, $paramName, "m_data") . ");\n");
            }

            push(@implContent, "\n    NakedPtr<JSC::Exception> returnedException;\n");

            my $propertyToLookup = "Identifier::fromString(state, \"${functionName}\")";
            my $invokeMethod = "JSCallbackData::CallbackType::FunctionOrObject";
            if ($codeGenerator->ExtendedAttributeContains($interface->extendedAttributes->{Callback}, "FunctionOnly")) {
                # For callback functions, do not look up callable property on the user object.
                # https://heycam.github.io/webidl/#es-callback-function
                $invokeMethod = "JSCallbackData::CallbackType::Function";
                $propertyToLookup = "Identifier()";
                push(@implContent, "    UNUSED_PARAM(state);\n");
            } elsif ($numFunctions > 1) {
                # The callback interface has more than one operation so we should not call the user object as a function.
                # instead, we should look for a property with the same name as the operation on the user object.
                # https://heycam.github.io/webidl/#es-user-objects
                $invokeMethod = "JSCallbackData::CallbackType::Object";
            }
            push(@implContent, "    m_data->invokeCallback(args, $invokeMethod, $propertyToLookup, returnedException);\n");

            # FIXME: We currently just report the exception. We should probably add an extended attribute to indicate when
            # we want the exception to be rethrown instead.
            push(@implContent, "    if (returnedException)\n");
            push(@implContent, "        reportException(state, returnedException);\n");
            push(@implContent, "    return !returnedException;\n");
            push(@implContent, "}\n");
        }
    }

    # toJS() implementation.
    push(@implContent, "\nJSC::JSValue toJS(JSC::ExecState*, JSDOMGlobalObject*, $interfaceName& impl)\n");
    push(@implContent, "{\n");
    push(@implContent, "    if (!static_cast<${className}&>(impl).callbackData())\n");
    push(@implContent, "        return jsNull();\n\n");
    push(@implContent, "    return static_cast<${className}&>(impl).callbackData()->callback();\n\n");
    push(@implContent, "}\n");

    push(@implContent, "\n}\n");
    my $conditionalString = $codeGenerator->GenerateConditionalString($interface);
    push(@implContent, "\n#endif // ${conditionalString}\n") if $conditionalString;

    push(@implContent, split("\r", $endAppleCopyright)) if $interface->extendedAttributes->{AppleCopyright};
}

sub GenerateImplementationFunctionCall()
{
    my ($function, $functionString, $indent, $svgPropertyType, $interface) = @_;

    my $nondeterministic = $function->signature->extendedAttributes->{Nondeterministic};
    my $mayThrowLegacyException = $function->signature->extendedAttributes->{MayThrowLegacyException};

    if ($function->signature->type eq "void" || IsReturningPromise($function)) {
        if ($nondeterministic) {
            AddToImplIncludes("<replay/InputCursor.h>", "WEB_REPLAY");
            push(@implContent, "#if ENABLE(WEB_REPLAY)\n");
            push(@implContent, $indent . "InputCursor& cursor = state->lexicalGlobalObject()->inputCursor();\n");
            push(@implContent, $indent . "if (!cursor.isReplaying()) {\n");
            push(@implContent, $indent . "    $functionString;\n");
            push(@implContent, $indent . "    setDOMException(state, throwScope, ec);\n") if $mayThrowLegacyException;
            push(@implContent, $indent . "}\n");
            push(@implContent, "#else\n");
            push(@implContent, $indent . "$functionString;\n");
            push(@implContent, $indent . "setDOMException(state, throwScope, ec);\n") if $mayThrowLegacyException;
            push(@implContent, "#endif\n");
        } else {
            push(@implContent, $indent . "$functionString;\n");
            push(@implContent, $indent . "setDOMException(state, throwScope, ec);\n") if $mayThrowLegacyException;
        }

        if ($svgPropertyType and !$function->isStatic) {
            if ($mayThrowLegacyException) {
                push(@implContent, $indent . "if (LIKELY(!ec))\n");
                push(@implContent, $indent . "    impl.commitChange();\n");
            } else {
                push(@implContent, $indent . "impl.commitChange();\n");
            }
        }

        push(@implContent, $indent . "return JSValue::encode(jsUndefined());\n");
    } else {
        my $thisObject = $function->isStatic ? 0 : "castedThis";
        if ($nondeterministic) {
            AddToImplIncludes("MemoizedDOMResult.h", "WEB_REPLAY");
            AddToImplIncludes("<replay/InputCursor.h>", "WEB_REPLAY");
            AddToImplIncludes("<wtf/NeverDestroyed.h>", "WEB_REPLAY");

            my $nativeType = GetNativeTypeFromSignature($interface, $function->signature);
            my $memoizedType = GetNativeTypeForMemoization($interface, $function->signature->type);
            my $bindingName = $interface->name . "." . $function->signature->name;
            push(@implContent, $indent . "JSValue result;\n");
            push(@implContent, "#if ENABLE(WEB_REPLAY)\n");
            push(@implContent, $indent . "InputCursor& cursor = state->lexicalGlobalObject()->inputCursor();\n");
            push(@implContent, $indent . "static NeverDestroyed<const AtomicString> bindingName(\"$bindingName\", AtomicString::ConstructFromLiteral);\n");
            push(@implContent, $indent . "if (cursor.isCapturing()) {\n");
            push(@implContent, $indent . "    $nativeType memoizedResult = $functionString;\n");
            my $exceptionCode = $mayThrowLegacyException ? "ec" : "0";
            push(@implContent, $indent . "    cursor.appendInput<MemoizedDOMResult<$memoizedType>>(bindingName.get().string(), memoizedResult, $exceptionCode);\n");
            push(@implContent, $indent . "    result = " . NativeToJSValueUsingPointers($function->signature, 1, $interface, "memoizedResult", $thisObject) . ";\n");
            push(@implContent, $indent . "} else if (cursor.isReplaying()) {\n");
            push(@implContent, $indent . "    MemoizedDOMResultBase* input = cursor.fetchInput<MemoizedDOMResultBase>();\n");
            push(@implContent, $indent . "    $memoizedType memoizedResult;\n");
            # FIXME: the generated code should report an error if an input cannot be fetched or converted.
            push(@implContent, $indent . "    if (input && input->convertTo<$memoizedType>(memoizedResult)) {\n");
            push(@implContent, $indent . "        result = " . NativeToJSValueUsingPointers($function->signature, 1, $interface, "memoizedResult", $thisObject) . ";\n");
            push(@implContent, $indent . "        ec = input->exceptionCode();\n") if $mayThrowLegacyException;
            push(@implContent, $indent . "    } else\n");
            push(@implContent, $indent . "        result = " . NativeToJSValueUsingPointers($function->signature, 1, $interface, $functionString, $thisObject) . ";\n");
            push(@implContent, $indent . "} else\n");
            push(@implContent, $indent . "    result = " . NativeToJSValueUsingPointers($function->signature, 1, $interface, $functionString, $thisObject) . ";\n");
            push(@implContent, "#else\n");
            push(@implContent, $indent . "result = " . NativeToJSValueUsingPointers($function->signature, 1, $interface, $functionString, $thisObject) . ";\n");
            push(@implContent, "#endif\n");
        } else {
            push(@implContent, $indent . "JSValue result = " . NativeToJSValueUsingPointers($function->signature, 1, $interface, $functionString, $thisObject) . ";\n");
        }

        push(@implContent, "\n" . $indent . "setDOMException(state, throwScope, ec);\n") if $mayThrowLegacyException;

        if ($codeGenerator->ExtendedAttributeContains($function->signature->extendedAttributes->{CallWith}, "ScriptState")) {
            push(@implContent, $indent . "RETURN_IF_EXCEPTION(throwScope, encodedJSValue());\n");
        }

        push(@implContent, $indent . "return JSValue::encode(result);\n");
    }
}

sub IsValueIterableInterface
{
    my $interface = shift;
    return 0 unless $interface->iterable;
    return 0 if length $interface->iterable->keyType;
    # FIXME: See https://webkit.org/b/159140, we should die if the next check is false.
    return 0 unless GetIndexedGetterFunction($interface);
    return 1;
}

sub IsKeyValueIterableInterface
{
    my $interface = shift;
    return 0 unless $interface->iterable;
    return 0 if IsValueIterableInterface($interface);
    return 1;
}

sub GenerateImplementationIterableFunctions
{
    my $interface = shift;

    my $interfaceName = $interface->name;
    my $className = "JS$interfaceName";
    my $visibleInterfaceName = $codeGenerator->GetVisibleInterfaceName($interface);

    AddToImplIncludes("JSDOMIterator.h");

    return unless IsKeyValueIterableInterface($interface);

    my $iteratorName = "${interfaceName}Iterator";
    my $iteratorPrototypeName = "${interfaceName}IteratorPrototype";

    my $iteratorTraitsName = "${interfaceName}IteratorTraits";
    my $iteratorTraitsType = $interface->iterable->isKeyValue ? "JSDOMIteratorType::Map" : "JSDOMIteratorType::Set";
    my $iteratorTraitsKeyType = $interface->iterable->isKeyValue ? GetIDLType($interface, $interface->iterable->idlKeyType) : "void";
    my $iteratorTraitsValueType = GetIDLType($interface, $interface->iterable->idlValueType);

    push(@implContent,  <<END);
struct ${iteratorTraitsName} {
    static constexpr JSDOMIteratorType type = ${iteratorTraitsType};
    using KeyType = ${iteratorTraitsKeyType};
    using ValueType = ${iteratorTraitsValueType};
};

using ${iteratorName} = JSDOMIterator<${className}, ${iteratorTraitsName}>;
using ${iteratorPrototypeName} = JSDOMIteratorPrototype<${className}, ${iteratorTraitsName}>;

template<>
const JSC::ClassInfo ${iteratorName}::s_info = { "${visibleInterfaceName} Iterator", &Base::s_info, 0, CREATE_METHOD_TABLE(${iteratorName}) };

template<>
const JSC::ClassInfo ${iteratorPrototypeName}::s_info = { "${visibleInterfaceName} Iterator", &Base::s_info, 0, CREATE_METHOD_TABLE(${iteratorPrototypeName}) };

END

    foreach my $function (@{$interface->iterable->functions}) {
        my $propertyName = $function->signature->name;
        my $functionName = GetFunctionName($interface, $className, $function);

        if ($propertyName eq "forEach") {
            push(@implContent,  <<END);
static inline EncodedJSValue ${functionName}Caller(ExecState* state, JS$interfaceName* thisObject, JSC::ThrowScope& throwScope)
{
    return JSValue::encode(iteratorForEach<${iteratorName}>(*state, *thisObject, throwScope));
}

END
        } else {
            my $iterationKind = "KeyValue";
            $iterationKind = "Key" if $propertyName eq "keys";
            $iterationKind = "Value" if $propertyName eq "values";
            $iterationKind = "Value" if $propertyName eq "[Symbol.Iterator]" and not $interface->iterable->isKeyValue;

            push(@implContent,  <<END);
static inline EncodedJSValue ${functionName}Caller(ExecState*, JS$interfaceName* thisObject, JSC::ThrowScope&)
{
    return JSValue::encode(iteratorCreate<${iteratorName}>(*thisObject, IterationKind::${iterationKind}));
}

END
        }

        push(@implContent,  <<END);
JSC::EncodedJSValue JSC_HOST_CALL ${functionName}(JSC::ExecState* state)
{
    return BindingCaller<$className>::callOperation<${functionName}Caller>(state, "${propertyName}");
}

END
    }
}

sub addIterableProperties()
{
    my $interface = shift;
    my $className = shift;

    if ($interface->iterable->extendedAttributes->{EnabledAtRuntime}) {
        AddToImplIncludes("RuntimeEnabledFeatures.h");
        my $enable_function = GetRuntimeEnableFunctionName($interface->iterable);
        push(@implContent, "    if (${enable_function}())\n    ");
    }

    if (IsKeyValueIterableInterface($interface)) {
        my $functionName = GetFunctionName($interface, $className, @{$interface->iterable->functions}[0]);
        push(@implContent, "    putDirect(vm, vm.propertyNames->iteratorSymbol, JSFunction::create(vm, globalObject(), 0, ASCIILiteral(\"[Symbol.Iterator]\"), $functionName), DontEnum);\n");
    } else {
        push(@implContent, "    addValueIterableMethods(*globalObject(), *this);\n");
    }
}

sub GetNativeTypeFromSignature
{
    my ($interface, $signature) = @_;

    return GetNativeType($interface, $signature->type);
}

my %nativeType = (
    "DOMString" => "String",
    "USVString" => "String",
    "Date" => "double",
    "Dictionary" => "Dictionary",
    "EventListener" => "RefPtr<EventListener>",
    "SerializedScriptValue" => "RefPtr<SerializedScriptValue>",
    "XPathNSResolver" => "RefPtr<XPathNSResolver>",
    "any" => "JSC::JSValue",
    "boolean" => "bool",
    "byte" => "int8_t",
    "double" => "double",
    "float" => "float",
    "long long" => "int64_t",
    "long" => "int32_t",
    "octet" => "uint8_t",
    "short" => "int16_t",
    "unrestricted double" => "double",
    "unrestricted float" => "float",
    "unsigned long long" => "uint64_t",
    "unsigned long" => "uint32_t",
    "unsigned short" => "uint16_t",
);

sub GetNativeVectorType
{
    my ($interface, $type) = @_;

    die "This should only be called for sequence or array types" unless $codeGenerator->IsSequenceOrFrozenArrayType($type);

    my $innerType = $codeGenerator->GetSequenceOrFrozenArrayInnerType($type);
    return "Vector<" . GetNativeVectorInnerType($innerType) . ">";
}

# http://heycam.github.io/webidl/#dfn-flattened-union-member-types
sub GetFlattenedMemberTypes
{
    my ($idlUnionType) = @_;

    my @flattenedMemberTypes = ();

    foreach my $memberType (@{$idlUnionType->subtypes}) {
        if ($memberType->isUnion) {
            push(@flattenedMemberTypes, GetFlattenedMemberTypes($memberType));
        } else {
            push(@flattenedMemberTypes, $memberType);
        }
    }

    return @flattenedMemberTypes;
}

# http://heycam.github.io/webidl/#dfn-number-of-nullable-member-types
sub GetNumberOfNullableMemberTypes
{
    my ($idlUnionType) = @_;

    my $count = 0;

    foreach my $memberType (@{$idlUnionType->subtypes}) {
        $count++ if $memberType->isNullable;
        $count += GetNumberOfNullableMemberTypes($memberType) if $memberType->isUnion;
    }

    return $count;
}

sub GetIDLUnionMemberTypes
{
    my ($interface, $idlUnionType) = @_;

    my $numberOfNullableMembers = GetNumberOfNullableMemberTypes($idlUnionType);
    assert("Union types must only have 0 or 1 nullable types.") if $numberOfNullableMembers > 1;

    my @idlUnionMemberTypes = ();

    push(@idlUnionMemberTypes, "IDLNull") if $numberOfNullableMembers == 1;

    foreach my $memberType (GetFlattenedMemberTypes($idlUnionType)) {
        push(@idlUnionMemberTypes, GetBaseIDLType($interface, $memberType));
    }

    return @idlUnionMemberTypes;
}

sub GetBaseIDLType
{
    my ($interface, $idlType) = @_;

    my %IDLTypes = (
        "any" => "IDLAny",
        "boolean" => "IDLBoolean",
        "byte" => "IDLByte",
        "octet" => "IDLOctet",
        "short" => "IDLShort",
        "unsigned short" => "IDLUnsignedShort",
        "long" => "IDLLong",
        "unsigned long" => "IDLUnsignedLong",
        "long long" => "IDLLongLong",
        "unsigned long long" => "IDLUnsignedLongLong",
        "float" => "IDLFloat",
        "unrestricted float" => "IDLUnrestrictedFloat",
        "double" => "IDLDouble",
        "unrestricted double" => "IDLUnrestrictedDouble",
        "DOMString" => "IDLDOMString",
        "ByteString" => "IDLByteString",
        "USVString" => "IDLUSVString",
        
        # Non-WebIDL extensions
        "Date" => "IDLDate",
        "BufferSource" => "IDLBufferSource",
    );

    return $IDLTypes{$idlType->name} if exists $IDLTypes{$idlType->name};
    return "IDLEnumeration<" . GetEnumerationClassName($idlType->name, $interface) . ">" if $codeGenerator->IsEnumType($idlType->name);
    return "IDLDictionary<" . GetDictionaryClassName($idlType->name, $interface) . ">" if $codeGenerator->IsDictionaryType($idlType->name);
    return "IDLSequence<" . GetIDLType($interface, @{$idlType->subtypes}[0]) . ">" if $codeGenerator->IsSequenceType($idlType->name);
    return "IDLFrozenArray<" . GetIDLType($interface, @{$idlType->subtypes}[0]) . ">" if $codeGenerator->IsFrozenArrayType($idlType->name);
    return "IDLUnion<" . join(", ", GetIDLUnionMemberTypes($interface, $idlType)) . ">" if $idlType->isUnion;
    return "IDLInterface<" . $idlType->name . ">";
}

sub GetIDLType
{
    my ($interface, $idlType) = @_;

    my $baseIDLType = GetBaseIDLType($interface, $idlType);
    return "IDLNullable<" . $baseIDLType . ">" if $idlType->isNullable;
    return $baseIDLType;
}

sub GetNativeType
{
    my ($interface, $type) = @_;

    return $nativeType{$type} if exists $nativeType{$type};

    return GetEnumerationClassName($type, $interface) if $codeGenerator->IsEnumType($type);
    return GetDictionaryClassName($type, $interface) if $codeGenerator->IsDictionaryType($type);

    my $tearOffType = $codeGenerator->GetSVGTypeNeedingTearOff($type);
    return "${tearOffType}*" if $tearOffType;

    return "RefPtr<${type}>" if $codeGenerator->IsTypedArrayType($type) and $type ne "ArrayBuffer";

    return GetNativeVectorType($interface, $type) if $codeGenerator->IsSequenceOrFrozenArrayType($type);

    return "${type}*";
}

sub ShouldPassWrapperByReference
{
    my $parameter = shift;
    my $interface = shift;

    my $nativeType = GetNativeType($interface, $parameter->type);
    return $codeGenerator->ShouldPassWrapperByReference($parameter, $interface) && (substr($nativeType, -1) eq '*' || $nativeType =~ /^RefPtr/);
}

sub GetNativeVectorInnerType
{
    my $innerType = shift;

    return $nativeType{$innerType} if exists $nativeType{$innerType};
    return "RefPtr<$innerType>";
}

sub GetNativeTypeForCallbacks
{
    my ($interface, $type) = @_;

    return "RefPtr<SerializedScriptValue>&&" if $type eq "SerializedScriptValue";
    return "const String&" if $codeGenerator->IsStringType($type);
    return GetNativeType($interface, $type);
}

sub GetNativeTypeForMemoization
{
    my ($interface, $type) = @_;

    return GetNativeType($interface, $type);
}

sub GetSVGPropertyTypes
{
    my $implType = shift;

    my $svgPropertyType;
    my $svgListPropertyType;
    my $svgNativeType;

    return ($svgPropertyType, $svgListPropertyType, $svgNativeType) if not $implType =~ /SVG/;

    $svgNativeType = $codeGenerator->GetSVGTypeNeedingTearOff($implType);
    return ($svgPropertyType, $svgListPropertyType, $svgNativeType) if not $svgNativeType;

    my $svgWrappedNativeType = $codeGenerator->GetSVGWrappedTypeNeedingTearOff($implType);
    if ($svgNativeType =~ /SVGPropertyTearOff/) {
        $svgPropertyType = $svgWrappedNativeType;
        $headerIncludes{"$svgWrappedNativeType.h"} = 1;
        $headerIncludes{"SVGAnimatedPropertyTearOff.h"} = 1;
    } elsif ($svgNativeType =~ /SVGListPropertyTearOff/ or $svgNativeType =~ /SVGStaticListPropertyTearOff/) {
        $svgListPropertyType = $svgWrappedNativeType;
        $headerIncludes{"$svgWrappedNativeType.h"} = 1;
        $headerIncludes{"SVGAnimatedListPropertyTearOff.h"} = 1;
    } elsif ($svgNativeType =~ /SVGTransformListPropertyTearOff/) {
        $svgListPropertyType = $svgWrappedNativeType;
        $headerIncludes{"$svgWrappedNativeType.h"} = 1;
        $headerIncludes{"SVGAnimatedListPropertyTearOff.h"} = 1;
        $headerIncludes{"SVGTransformListPropertyTearOff.h"} = 1;
    } elsif ($svgNativeType =~ /SVGPathSegListPropertyTearOff/) {
        $svgListPropertyType = $svgWrappedNativeType;
        $headerIncludes{"$svgWrappedNativeType.h"} = 1;
        $headerIncludes{"SVGAnimatedListPropertyTearOff.h"} = 1;
        $headerIncludes{"SVGPathSegListPropertyTearOff.h"} = 1;
    }

    return ($svgPropertyType, $svgListPropertyType, $svgNativeType);
}

sub IsNativeType
{
    my $type = shift;
    return exists $nativeType{$type};
}

sub GetIntegerConversionConfiguration
{
    my $signature = shift;

    return "EnforceRange" if $signature->extendedAttributes->{EnforceRange};
    return "Clamp" if $signature->extendedAttributes->{Clamp};
    return "NormalConversion";
}

sub JSValueToNativeIsHandledByDOMConvert
{
    my $idlType = shift;

    return 1 if $idlType->isUnion;
    return 1 if $idlType->name eq "any";
    return 1 if $idlType->name eq "boolean";
    return 1 if $idlType->name eq "Date";
    return 1 if $idlType->name eq "BufferSource";
    return 1 if $codeGenerator->IsIntegerType($idlType->name);
    return 1 if $codeGenerator->IsFloatingPointType($idlType->name);
    return 1 if $codeGenerator->IsSequenceOrFrozenArrayType($idlType->name);
    return 1 if $codeGenerator->IsDictionaryType($idlType->name);
    return 0;
}

# Returns (convertString, mayThrowException).

sub JSValueToNative
{
    my ($interface, $signature, $value, $conditional, $statePointer, $stateReference, $thisObjectReference) = @_;

    my $type = $signature->type;
    my $idlType = $signature->idlType;

    # FIXME: Remove these 3 variables when all JSValueToNative use references.
    $statePointer = "state" unless $statePointer;
    $stateReference = "*state" unless $stateReference;
    $thisObjectReference = "*castedThis" unless $thisObjectReference;

    if (JSValueToNativeIsHandledByDOMConvert($idlType)) {
        AddToImplIncludes("JSDOMConvert.h");
        AddToImplIncludesForIDLType($idlType, $conditional);

        my $IDLType = GetIDLType($interface, $idlType);

        if ($codeGenerator->IsIntegerType($type)) {
            my $conversionType = GetIntegerConversionConfiguration($signature);
            return ("convert<$IDLType>($stateReference, $value, $conversionType)", 1);
        }

        return ("convert<$IDLType>($stateReference, $value)", 1);
    }

    if ($type eq "DOMString") {
        return ("AtomicString($value.toString($statePointer)->toExistingAtomicString($statePointer))", 1) if $signature->extendedAttributes->{RequiresExistingAtomicString};

        my $treatNullAs = $signature->extendedAttributes->{TreatNullAs};
        return ("valueToStringTreatingNullAsEmptyString($statePointer, $value)", 1) if $treatNullAs && $treatNullAs eq "EmptyString";
        return ("valueToStringWithUndefinedOrNullCheck($statePointer, $value)", 1) if $signature->isNullable;
        return ("$value.toString($statePointer)->toAtomicString($statePointer)", 1) if $signature->extendedAttributes->{AtomicString};
        return ("$value.toWTFString($statePointer)", 1);
    }
    if ($type eq "USVString") {
        my $treatNullAs = $signature->extendedAttributes->{TreatNullAs};
        return ("valueToUSVStringTreatingNullAsEmptyString($statePointer, $value)", 1) if $treatNullAs && $treatNullAs eq "EmptyString";
        return ("valueToUSVStringWithUndefinedOrNullCheck($statePointer, $value)", 1) if $signature->isNullable;
        return ("valueToUSVString($statePointer, $value)", 1);
    }

    if ($type eq "SerializedScriptValue") {
        AddToImplIncludes("SerializedScriptValue.h", $conditional);
        return ("SerializedScriptValue::create($stateReference, $value)", 1);
    }

    if ($type eq "Dictionary") {
        AddToImplIncludes("Dictionary.h", $conditional);
        return ("Dictionary($statePointer, $value)", 0);
    }

    return ("to$type($value)", 1) if $codeGenerator->IsTypedArrayType($type);
    return ("parseEnumeration<" . GetEnumerationClassName($type, $interface) . ">($stateReference, $value)", 1) if $codeGenerator->IsEnumType($type);

    AddToImplIncludes("JS$type.h", $conditional);

    # FIXME: EventListener should be a callback interface.
    return "JSEventListener::create($value, $thisObjectReference, false, currentWorld($statePointer))" if $type eq "EventListener";

    my $extendedAttributes = $codeGenerator->getInterfaceExtendedAttributesFromName($type);
    return ("JS${type}::toWrapped($stateReference, $value)", 1) if $type eq "XPathNSResolver";
    return ("JS${type}::toWrapped($value)", 0);
}

sub NativeToJSValueIsHandledByDOMConvert
{
    my ($idlType) = @_;
    
    return 1 if $idlType->name eq "any";
    return 1 if $idlType->name eq "boolean";
    return 1 if $idlType->name eq "Date";
    return 1 if $codeGenerator->IsIntegerType($idlType->name);
    return 1 if $codeGenerator->IsFloatingPointType($idlType->name);
    return 1 if $codeGenerator->IsStringType($idlType->name);
    return 1 if $codeGenerator->IsEnumType($idlType->name);
    return 1 if $idlType->isUnion;
    return 1 if $codeGenerator->IsSequenceOrFrozenArrayType($idlType->name);

    return 0;
}

sub NativeToJSValueDOMConvertNeedsState
{
    my ($idlType) = @_;
    
    # FIXME: This should actually check if all the sub-objects of the union need the state.
    return 1 if $idlType->isUnion;
    return 1 if $codeGenerator->IsSequenceOrFrozenArrayType($idlType->name);
    return 1 if $codeGenerator->IsStringType($idlType->name);
    return 1 if $codeGenerator->IsEnumType($idlType->name);
    return 1 if $idlType->name eq "Date";

    return 0;
}

sub NativeToJSValueDOMConvertNeedsGlobalObject
{
    my ($idlType) = @_;
    
    # FIXME: This should actually check if all the sub-objects of the union need the global object.
    return 1 if $idlType->isUnion;
    return 1 if $codeGenerator->IsSequenceOrFrozenArrayType($idlType->name);

    return 0;
}

sub NativeToJSValueUsingReferences
{
    my ($signature, $inFunctionCall, $interface, $value, $thisValue) = @_;
    my $statePointer = "&state";
    my $stateReference = "state";
    my $wrapped = "$thisValue.wrapped()";
    my $globalObject = $thisValue ? "$thisValue.globalObject()" : "jsCast<JSDOMGlobalObject*>(state.lexicalGlobalObject())";

    return NativeToJSValue($signature, $inFunctionCall, $interface, $value, $statePointer, $stateReference, $wrapped, $globalObject);
}

# FIXME: We should remove NativeToJSValueUsingPointers and combine NativeToJSValueUsingReferences and NativeToJSValue
sub NativeToJSValueUsingPointers
{
    my ($signature, $inFunctionCall, $interface, $value, $thisValue) = @_;
    my $statePointer = "state";
    my $stateReference = "*state";
    my $wrapped = "$thisValue->wrapped()";
    my $globalObject = $thisValue ? "$thisValue->globalObject()" : "jsCast<JSDOMGlobalObject*>(state->lexicalGlobalObject())";

    return NativeToJSValue($signature, $inFunctionCall, $interface, $value, $statePointer, $stateReference, $wrapped, $globalObject);
}

sub NativeToJSValue
{
    my ($signature, $inFunctionCall, $interface, $value, $statePointer, $stateReference, $wrapped, $globalObject) = @_;

    my $conditional = $signature->extendedAttributes->{Conditional};
    my $type = $signature->type;
    my $idlType = $signature->idlType;
    my $isNullable = $signature->isNullable;
    my $mayThrowException = $signature->extendedAttributes->{GetterMayThrowException} || $signature->extendedAttributes->{MayThrowException};

    # We could instead overload a function to work with optional as well as non-optional numbers, but this
    # is slightly better because it guarantees we will fail to compile if the IDL file doesn't match the C++.
    if ($signature->extendedAttributes->{Reflect} and ($type eq "unsigned long" or $type eq "unsigned short")) {
        $value =~ s/getUnsignedIntegralAttribute/getIntegralAttribute/g;
        $value = "std::max(0, $value)";
    }

    if ($type eq "any") {
        my $returnType = $signature->extendedAttributes->{ImplementationReturnType};
        if (defined $returnType and ($returnType eq "IDBKeyPath" or $returnType eq "IDBKey")) {
            AddToImplIncludes("IDBBindingUtilities.h", $conditional);
            return "toJS($stateReference, *$globalObject, $value)";
        }
    }

    if (NativeToJSValueIsHandledByDOMConvert($idlType)) {
        AddToImplIncludes("JSDOMConvert.h");
        AddToImplIncludesForIDLType($idlType, $conditional);

        my $IDLType = GetIDLType($interface, $idlType);

        my @conversionArguments = ();
        push(@conversionArguments, "$stateReference") if NativeToJSValueDOMConvertNeedsState($idlType) || $mayThrowException;
        push(@conversionArguments, "*$globalObject") if NativeToJSValueDOMConvertNeedsGlobalObject($idlType);
        push(@conversionArguments, "throwScope") if $mayThrowException;
        push(@conversionArguments, "$value");

        return "toJS<$IDLType>(" . join(", ", @conversionArguments) . ")";
    }

    if ($type eq "SerializedScriptValue") {
        AddToImplIncludes("SerializedScriptValue.h", $conditional);
        return "$value ? $value->deserialize($stateReference, $globalObject) : jsNull()";
    }

    AddToImplIncludes("StyleProperties.h", $conditional) if $type eq "CSSStyleDeclaration";
    AddToImplIncludes("NameNodeList.h", $conditional) if $type eq "NodeList";
    AddToImplIncludes("JS$type.h", $conditional) if !$codeGenerator->IsTypedArrayType($type);

    return $value if $codeGenerator->IsSVGAnimatedType($type);

    if ($codeGenerator->IsSVGAnimatedType($interface->name) or ($interface->name eq "SVGViewSpec" and $type eq "SVGTransformList")) {
        # Convert from abstract RefPtr<ListProperty> to real type, so the right toJS() method can be invoked.
        $value = "static_cast<" . GetNativeType($interface, $type) . ">($value.get())";
    } elsif ($interface->name eq "SVGViewSpec") {
        # Convert from abstract SVGProperty to real type, so the right toJS() method can be invoked.
        $value = "static_cast<" . GetNativeType($interface, $type) . ">($value)";
    } elsif ($codeGenerator->IsSVGTypeNeedingTearOff($type) and not $interface->name =~ /List$/) {
        my $tearOffType = $codeGenerator->GetSVGTypeNeedingTearOff($type);
        if ($codeGenerator->IsSVGTypeWithWritablePropertiesNeedingTearOff($type) and !$inFunctionCall and not defined $signature->extendedAttributes->{Immutable}) {
            my $getter = $value;
            $getter =~ s/impl\.//;
            $getter =~ s/impl->//;
            $getter =~ s/\(\)//;
            my $updateMethod = "&" . $interface->name . "::update" . $codeGenerator->WK_ucfirst($getter);

            my $selfIsTearOffType = $codeGenerator->IsSVGTypeNeedingTearOff($interface->name);
            if ($selfIsTearOffType) {
                # FIXME: Why SVGMatrix specifically?
                AddToImplIncludes("SVGMatrixTearOff.h", $conditional);
                $value = "SVGMatrixTearOff::create($wrapped, $value)";
            } else {
                AddToImplIncludes("SVGStaticPropertyTearOff.h", $conditional);
                my $interfaceName = $interface->name;
                $tearOffType =~ s/SVGPropertyTearOff</SVGStaticPropertyTearOff<$interfaceName, /;
                $value = "${tearOffType}::create(impl, $value, $updateMethod)";
            }
        } elsif ($tearOffType =~ /SVGStaticListPropertyTearOff/) {
            $value = "${tearOffType}::create(impl, $value)";
        } elsif (not $tearOffType =~ /SVG(Point|PathSeg)List/) {
            $value = "${tearOffType}::create($value)";
        }
    }

    my $functionName = "toJS";
    $functionName = "toJSNewlyCreated" if $signature->extendedAttributes->{NewObject};

    my $arguments = "$statePointer, $globalObject, $value";
    $arguments = "$stateReference, *$globalObject, throwScope, $value" if $mayThrowException;

    return "$functionName($arguments)";
}

sub ceilingToPowerOf2
{
    my ($size) = @_;

    my $powerOf2 = 1;
    while ($size > $powerOf2) {
        $powerOf2 <<= 1;
    }

    return $powerOf2;
}

# Internal Helper
sub GenerateHashTableValueArray
{
    my $keys = shift;
    my $specials = shift;
    my $value1 = shift;
    my $value2 = shift;
    my $conditionals = shift;
    my $nameEntries = shift;

    my $packedSize = scalar @{$keys};
    push(@implContent, "\nstatic const HashTableValue $nameEntries\[\] =\n\{\n");

    my $hasSetter = "false";

    my $i = 0;
    foreach my $key (@{$keys}) {
        my $conditional;
        my $firstTargetType;
        my $secondTargetType = "";

        if ($conditionals) {
            $conditional = $conditionals->{$key};
        }
        if ($conditional) {
            my $conditionalString = $codeGenerator->GenerateConditionalStringFromAttributeValue($conditional);
            push(@implContent, "#if ${conditionalString}\n");
        }

        if ("@$specials[$i]" =~ m/Function/) {
            $firstTargetType = "static_cast<NativeFunction>";
        } elsif ("@$specials[$i]" =~ m/Builtin/) {
            $firstTargetType = "static_cast<BuiltinGenerator>";
        } elsif ("@$specials[$i]" =~ m/ConstantInteger/) {
            $firstTargetType = "";
        } elsif ("@$specials[$i]" =~ m/DOMJITAttribute/) {
            $firstTargetType = "static_cast<DOMJITGetterSetterGenerator>";
        } else {
            $firstTargetType = "static_cast<PropertySlot::GetValueFunc>";
            $secondTargetType = "static_cast<PutPropertySlot::PutValueFunc>";
            $hasSetter = "true";
        }
        if ("@$specials[$i]" =~ m/ConstantInteger/) {
            push(@implContent, "    { \"$key\", @$specials[$i], NoIntrinsic, { (long long)" . $firstTargetType . "(@$value1[$i]) } },\n");
        } else {
            push(@implContent, "    { \"$key\", @$specials[$i], NoIntrinsic, { (intptr_t)" . $firstTargetType . "(@$value1[$i]), (intptr_t) " . $secondTargetType . "(@$value2[$i]) } },\n");
        }
        if ($conditional) {
            push(@implContent, "#else\n") ;
            push(@implContent, "    { 0, 0, NoIntrinsic, { 0, 0 } },\n");
            push(@implContent, "#endif\n") ;
        }
        ++$i;
    }

    push(@implContent, "    { 0, 0, NoIntrinsic, { 0, 0 } }\n") if (!$packedSize);
    push(@implContent, "};\n\n");

    return $hasSetter;
}

sub GenerateHashTable
{
    my $object = shift;

    my $name = shift;
    my $size = shift;
    my $keys = shift;
    my $specials = shift;
    my $value1 = shift;
    my $value2 = shift;
    my $conditionals = shift;
    my $justGenerateValueArray = shift;

    my $nameEntries = "${name}Values";
    $nameEntries =~ s/:/_/g;
    my $nameIndex = "${name}Index";
    $nameIndex =~ s/:/_/g;

    if (($name =~ /Prototype/) or ($name =~ /Constructor/)) {
        my $type = $name;
        my $implClass;

        if ($name =~ /Prototype/) {
            $type =~ s/Prototype.*//;
            $implClass = $type; $implClass =~ s/Wrapper$//;
            push(@implContent, "/* Hash table for prototype */\n");
        } else {
            $type =~ s/Constructor.*//;
            $implClass = $type; $implClass =~ s/Constructor$//;
            push(@implContent, "/* Hash table for constructor */\n");
        }
    } else {
        push(@implContent, "/* Hash table */\n");
    }

    if ($justGenerateValueArray) {
        GenerateHashTableValueArray($keys, $specials, $value1, $value2, $conditionals, $nameEntries) if $size;
        return;
    }

    # Generate size data for compact' size hash table

    my @table = ();
    my @links = ();

    my $compactSize = ceilingToPowerOf2($size * 2);

    my $maxDepth = 0;
    my $collisions = 0;
    my $numEntries = $compactSize;

    my $i = 0;
    foreach (@{$keys}) {
        my $depth = 0;
        my $h = Hasher::GenerateHashValue($_) % $numEntries;

        while (defined($table[$h])) {
            if (defined($links[$h])) {
                $h = $links[$h];
                $depth++;
            } else {
                $collisions++;
                $links[$h] = $compactSize;
                $h = $compactSize;
                $compactSize++;
            }
        }

        $table[$h] = $i;

        $i++;
        $maxDepth = $depth if ($depth > $maxDepth);
    }

    push(@implContent, "\nstatic const struct CompactHashIndex ${nameIndex}\[$compactSize\] = {\n");
    for (my $i = 0; $i < $compactSize; $i++) {
        my $T = -1;
        if (defined($table[$i])) { $T = $table[$i]; }
        my $L = -1;
        if (defined($links[$i])) { $L = $links[$i]; }
        push(@implContent, "    { $T, $L },\n");
    }
    push(@implContent, "};\n\n");

    # Dump the hash table
    my $hasSetter = GenerateHashTableValueArray($keys, $specials, $value1, $value2, $conditionals, $nameEntries);
    my $packedSize = scalar @{$keys};

    my $compactSizeMask = $numEntries - 1;
    push(@implContent, "static const HashTable $name = { $packedSize, $compactSizeMask, $hasSetter, $nameEntries, $nameIndex };\n");
}

sub WriteData
{
    my $object = shift;
    my $interface = shift;
    my $outputDir = shift;

    my $name = $interface->name;
    my $prefix = FileNamePrefix;
    my $headerFileName = "$outputDir/$prefix$name.h";
    my $implFileName = "$outputDir/$prefix$name.cpp";
    my $depsFileName = "$outputDir/$prefix$name.dep";

    # Update a .cpp file if the contents are changed.
    my $contents = join "", @implContentHeader;

    my @includes = ();
    my %implIncludeConditions = ();
    foreach my $include (keys %implIncludes) {
        next if $headerIncludes{$include};
        next if $headerTrailingIncludes{$include};

        my $condition = $implIncludes{$include};

        my $checkType = $include;
        $checkType =~ s/\.h//;
        next if $codeGenerator->IsSVGAnimatedType($checkType);

        $include = "\"$include\"" unless $include =~ /^["<]/; # "

        if ($condition eq 1) {
            push @includes, $include;
        } else {
            push @{$implIncludeConditions{$codeGenerator->GenerateConditionalStringFromAttributeValue($condition)}}, $include;
        }
    }
    foreach my $include (sort @includes) {
        $contents .= "#include $include\n";
    }
    foreach my $condition (sort keys %implIncludeConditions) {
        $contents .= "\n#if " . $condition . "\n";
        foreach my $include (sort @{$implIncludeConditions{$condition}}) {
            $contents .= "#include $include\n";
        }
        $contents .= "#endif\n";
    }

    $contents .= join "", @implContent;
    $codeGenerator->UpdateFile($implFileName, $contents);

    @implContentHeader = ();
    @implContent = ();
    %implIncludes = ();

    # Update a .h file if the contents are changed.
    $contents = join "", @headerContentHeader;

    @includes = ();
    foreach my $include (keys %headerIncludes) {
        $include = "\"$include\"" unless $include =~ /^["<]/; # "
        push @includes, $include;
    }
    foreach my $include (sort @includes) {
        # "JSClassName.h" is already included right after config.h.
        next if $include eq "\"$prefix$name.h\"";
        $contents .= "#include $include\n";
    }

    $contents .= join "", @headerContent;

    @includes = ();
    foreach my $include (keys %headerTrailingIncludes) {
        $include = "\"$include\"" unless $include =~ /^["<]/; # "
        push @includes, $include;
    }
    foreach my $include (sort @includes) {
        $contents .= "#include $include\n";
    }
    $codeGenerator->UpdateFile($headerFileName, $contents);

    @headerContentHeader = ();
    @headerContent = ();
    %headerIncludes = ();
    %headerTrailingIncludes = ();

    if (@depsContent) {
        # Update a .dep file if the contents are changed.
        $contents = join "", @depsContent;
        $codeGenerator->UpdateFile($depsFileName, $contents);

        @depsContent = ();
    }
}

sub GeneratePrototypeDeclaration
{
    my ($outputArray, $className, $interface) = @_;

    my $prototypeClassName = "${className}Prototype";

    my %structureFlags = ();
    push(@$outputArray, "class ${prototypeClassName} : public JSC::JSNonFinalObject {\n");
    push(@$outputArray, "public:\n");
    push(@$outputArray, "    using Base = JSC::JSNonFinalObject;\n");

    push(@$outputArray, "    static ${prototypeClassName}* create(JSC::VM& vm, JSC::JSGlobalObject* globalObject, JSC::Structure* structure)\n");
    push(@$outputArray, "    {\n");
    push(@$outputArray, "        ${className}Prototype* ptr = new (NotNull, JSC::allocateCell<${className}Prototype>(vm.heap)) ${className}Prototype(vm, globalObject, structure);\n");
    push(@$outputArray, "        ptr->finishCreation(vm);\n");
    push(@$outputArray, "        return ptr;\n");
    push(@$outputArray, "    }\n\n");

    push(@$outputArray, "    DECLARE_INFO;\n");

    push(@$outputArray,
        "    static JSC::Structure* createStructure(JSC::VM& vm, JSC::JSGlobalObject* globalObject, JSC::JSValue prototype)\n" .
        "    {\n" .
        "        return JSC::Structure::create(vm, globalObject, prototype, JSC::TypeInfo(JSC::ObjectType, StructureFlags), info());\n" .
        "    }\n");

    push(@$outputArray, "\nprivate:\n");
    push(@$outputArray, "    ${prototypeClassName}(JSC::VM& vm, JSC::JSGlobalObject*, JSC::Structure* structure)\n");
    push(@$outputArray, "        : JSC::JSNonFinalObject(vm, structure)\n");
    push(@$outputArray, "    {\n");
    push(@$outputArray, "    }\n");

    if (PrototypeHasStaticPropertyTable($interface)) {
        if (IsDOMGlobalObject($interface)) {
            $structureFlags{"JSC::HasStaticPropertyTable"} = 1;
        } else {
            push(@$outputArray, "\n");
            push(@$outputArray, "    void finishCreation(JSC::VM&);\n");
        }
    }

    if ($interface->extendedAttributes->{JSCustomNamedGetterOnPrototype}) {
        push(@$outputArray, "\n");
        push(@$outputArray, "    static bool put(JSC::JSCell*, JSC::ExecState*, JSC::PropertyName, JSC::JSValue, JSC::PutPropertySlot&);\n");
        push(@$outputArray, "    bool putDelegate(JSC::ExecState*, JSC::PropertyName, JSC::JSValue, JSC::PutPropertySlot&, bool& putResult);\n");
    }

    # Custom defineOwnProperty function
    if ($interface->extendedAttributes->{JSCustomDefineOwnPropertyOnPrototype}) {
        push(@$outputArray, "\n");
        push(@$outputArray, "    static bool defineOwnProperty(JSC::JSObject*, JSC::ExecState*, JSC::PropertyName, const JSC::PropertyDescriptor&, bool shouldThrow);\n");
    }

    # structure flags
    if (%structureFlags) {
        push(@$outputArray, "public:\n");
        push(@$outputArray, "    static const unsigned StructureFlags = ");
        foreach my $structureFlag (sort (keys %structureFlags)) {
            push(@$outputArray, $structureFlag . " | ");
        }
        push(@$outputArray, "Base::StructureFlags;\n");
    }

    push(@$outputArray, "};\n\n");
}

sub GetConstructorTemplateClassName
{
    my $interface = shift;
    return "JSDOMConstructorNotConstructable" if $interface->extendedAttributes->{NamedConstructor};
    return "JSDOMConstructorNotConstructable" unless IsConstructable($interface);
    return "JSBuiltinConstructor" if IsJSBuiltinConstructor($interface);
    return "JSDOMConstructor";
}

sub GenerateConstructorDeclaration
{
    my ($outputArray, $className, $interface) = @_;

    my $interfaceName = $interface->name;
    my $constructorClassName = "${className}Constructor";
    my $templateClassName = GetConstructorTemplateClassName($interface);

    $implIncludes{"JSDOMConstructor.h"} = 1;

    push(@$outputArray, "using $constructorClassName = $templateClassName<$className>;\n");
    push(@$outputArray, "using JS${interfaceName}NamedConstructor = JSDOMNamedConstructor<$className>;\n") if $interface->extendedAttributes->{NamedConstructor};
    push(@$outputArray, "\n");
}

sub GenerateConstructorDefinitions
{
    my ($outputArray, $className, $protoClassName, $visibleInterfaceName, $interface, $generatingNamedConstructor) = @_;

    if (IsConstructable($interface)) {
        my @constructors = @{$interface->constructors};
        if (@constructors > 1) {
            foreach my $constructor (@constructors) {
                GenerateConstructorDefinition($outputArray, $className, $protoClassName, $visibleInterfaceName, $interface, $generatingNamedConstructor, $constructor);
            }
            GenerateOverloadedFunctionOrConstructor(@{$interface->constructors}[0], $interface, 1);
        } elsif (@constructors == 1) {
            GenerateConstructorDefinition($outputArray, $className, $protoClassName, $visibleInterfaceName, $interface, $generatingNamedConstructor, $constructors[0]);
        } else {
            GenerateConstructorDefinition($outputArray, $className, $protoClassName, $visibleInterfaceName, $interface, $generatingNamedConstructor);
        }
    }

    GenerateConstructorHelperMethods($outputArray, $className, $protoClassName, $visibleInterfaceName, $interface, $generatingNamedConstructor);
}

sub GenerateConstructorDefinition
{
    my ($outputArray, $className, $protoClassName, $visibleInterfaceName, $interface, $generatingNamedConstructor, $function) = @_;

    return if IsJSBuiltinConstructor($interface);

    my $interfaceName = $interface->name;
    my $constructorClassName = $generatingNamedConstructor ? "${className}NamedConstructor" : "${className}Constructor";

    if (IsConstructable($interface)) {
        if ($interface->extendedAttributes->{CustomConstructor}) {
            push(@$outputArray, "template<> JSC::EncodedJSValue JSC_HOST_CALL ${constructorClassName}::construct(JSC::ExecState* exec)\n");
            push(@$outputArray, "{\n");
            push(@$outputArray, "    ASSERT(exec);\n");
            push(@$outputArray, "    return construct${className}(*exec);\n");
            push(@$outputArray, "}\n\n");
         } elsif (!HasCustomConstructor($interface) && (!$interface->extendedAttributes->{NamedConstructor} || $generatingNamedConstructor)) {
            my $isOverloaded = $function->{overloads} && @{$function->{overloads}} > 1;
            if ($isOverloaded) {
                push(@$outputArray, "static inline EncodedJSValue construct${className}$function->{overloadIndex}(ExecState* state)\n");
            } else {
                push(@$outputArray, "template<> EncodedJSValue JSC_HOST_CALL ${constructorClassName}::construct(ExecState* state)\n");
            }

            push(@$outputArray, "{\n");
            push(@$outputArray, "    VM& vm = state->vm();\n");
            push(@$outputArray, "    auto throwScope = DECLARE_THROW_SCOPE(vm);\n");
            push(@$outputArray, "    UNUSED_PARAM(throwScope);\n");
            push(@$outputArray, "    auto* castedThis = jsCast<${constructorClassName}*>(state->callee());\n");
            push(@$outputArray, "    ASSERT(castedThis);\n");

            my @constructorArgList;

            $implIncludes{"<runtime/Error.h>"} = 1;

            GenerateArgumentsCountCheck($outputArray, $function, $interface);

            if ($function->signature->extendedAttributes->{MayThrowLegacyException} || $interface->extendedAttributes->{ConstructorMayThrowLegacyException}) {
                $implIncludes{"ExceptionCode.h"} = 1;
                push(@$outputArray, "    ExceptionCode ec = 0;\n");
            }

            # FIXME: For now, we do not support SVG constructors.
            # FIXME: Currently [Constructor(...)] does not yet support optional arguments without [Default=...]
            my ($dummy, $paramIndex) = GenerateParametersCheck($outputArray, $function, $interface, "constructorCallback", undef, undef, undef);

            push(@constructorArgList, "*state") if $codeGenerator->ExtendedAttributeContains($interface->extendedAttributes->{ConstructorCallWith}, "ScriptState");;

            if ($codeGenerator->ExtendedAttributeContains($interface->extendedAttributes->{ConstructorCallWith}, "ScriptExecutionContext")) {
                push(@constructorArgList, "*context");
                push(@$outputArray, "    ScriptExecutionContext* context = castedThis->scriptExecutionContext();\n");
                push(@$outputArray, "    if (UNLIKELY(!context))\n");
                push(@$outputArray, "        return throwConstructorScriptExecutionContextUnavailableError(*state, throwScope, \"${visibleInterfaceName}\");\n");
            }

            if ($codeGenerator->ExtendedAttributeContains($interface->extendedAttributes->{ConstructorCallWith}, "Document")) {
                $implIncludes{"Document.h"} = 1;
                push(@constructorArgList, "document");
                push(@$outputArray, "    ScriptExecutionContext* context = castedThis->scriptExecutionContext();\n");
                push(@$outputArray, "    if (UNLIKELY(!context))\n");
                push(@$outputArray, "        return throwConstructorScriptExecutionContextUnavailableError(*state, throwScope, \"${visibleInterfaceName}\");\n");
                push(@$outputArray, "    ASSERT(context->isDocument());\n");
                push(@$outputArray, "    auto& document = downcast<Document>(*context);\n");
            }

            push(@constructorArgList, "*castedThis->document()") if $generatingNamedConstructor;

            my $index = 0;
            foreach my $parameter (@{$function->parameters}) {
                last if $index eq $paramIndex;
                if (ShouldPassWrapperByReference($parameter, $interface)) {
                    push(@constructorArgList, "*" . $parameter->name);
                } else {
                    push(@constructorArgList, "WTFMove(" . $parameter->name . ")");
                }
                $index++;
            }

            push(@constructorArgList, "ec") if $interface->extendedAttributes->{ConstructorMayThrowLegacyException};

            my $constructorArg = join(", ", @constructorArgList);
            if ($generatingNamedConstructor) {
                push(@$outputArray, "    auto object = ${interfaceName}::createForJSConstructor(${constructorArg});\n");
            } else {
                push(@$outputArray, "    auto object = ${interfaceName}::create(${constructorArg});\n");
            }

            if ($interface->extendedAttributes->{ConstructorMayThrowLegacyException}) {
                push(@$outputArray, "    if (UNLIKELY(ec)) {\n");
                push(@$outputArray, "        setDOMException(state, throwScope, ec);\n");
                push(@$outputArray, "        return encodedJSValue();\n");
                push(@$outputArray, "    }\n");
            }

            push(@$outputArray, "    RETURN_IF_EXCEPTION(throwScope, encodedJSValue());\n") if $codeGenerator->ExtendedAttributeContains($interface->extendedAttributes->{ConstructorCallWith}, "ScriptState");
            if ($interface->extendedAttributes->{ConstructorMayThrowException}) {
                push(@$outputArray, "    return JSValue::encode(toJSNewlyCreated(*state, *castedThis->globalObject(), throwScope, WTFMove(object)));\n");
            } else {
                push(@$outputArray, "    return JSValue::encode(toJSNewlyCreated(state, castedThis->globalObject(), WTFMove(object)));\n");
            }
            push(@$outputArray, "}\n\n");
        }
    }
}

sub ConstructorHasProperties
{
    my $interface = shift;

    foreach my $constant (@{$interface->constants}) {
        return 1;
    }

    foreach my $attribute (@{$interface->attributes}) {
        next unless ($attribute->isStatic);
        return 1;
    }

    foreach my $function (@{$interface->functions}) {
        next unless ($function->isStatic);
        return 1;
    }

    return 0;
}

sub GenerateConstructorHelperMethods
{
    my ($outputArray, $className, $protoClassName, $visibleInterfaceName, $interface, $generatingNamedConstructor) = @_;

    my $constructorClassName = $generatingNamedConstructor ? "${className}NamedConstructor" : "${className}Constructor";
    my $leastConstructorLength = 0;
    if ($interface->extendedAttributes->{Constructor} || $interface->extendedAttributes->{CustomConstructor}) {
        my @constructors = @{$interface->constructors};
        my @customConstructors = @{$interface->customConstructors};
        $leastConstructorLength = 255;
        foreach my $constructor (@constructors, @customConstructors) {
            my $constructorLength = GetFunctionLength($constructor);
            $leastConstructorLength = $constructorLength if ($constructorLength < $leastConstructorLength);
        }
    } else {
        $leastConstructorLength = 0;
    }

    # If the interface has a parent interface which does not have [NoInterfaceObject], then use its interface object as prototype,
    # otherwise use FunctionPrototype: http://heycam.github.io/webidl/#interface-object
    push(@$outputArray, "template<> JSValue ${constructorClassName}::prototypeForStructure(JSC::VM& vm, const JSDOMGlobalObject& globalObject)\n");
    push(@$outputArray, "{\n");
    # FIXME: IDL does not allow an interface without [NoInterfaceObject] to inherit one that is marked as [NoInterfaceObject]
    # so we should be able to use our parent's interface object no matter what. However, some of our IDL files (e.g. CanvasRenderingContext2D)
    # are not valid so we need this check for now.
    if ($interface->parent && !$codeGenerator->getInterfaceExtendedAttributesFromName($interface->parent)->{NoInterfaceObject}) {
        my $parentClassName = "JS" . $interface->parent;
        push(@$outputArray, "    return ${parentClassName}::getConstructor(vm, &globalObject);\n");
    } elsif ($interface->isCallback) {
        # The internal [[Prototype]] property of an interface object for a callback interface must be the Object.prototype object.
        AddToImplIncludes("<runtime/ObjectPrototype.h>");
        push(@$outputArray, "    UNUSED_PARAM(vm);\n");
        push(@$outputArray, "    return globalObject.objectPrototype();\n");
    } else {
        AddToImplIncludes("<runtime/FunctionPrototype.h>");
        push(@$outputArray, "    UNUSED_PARAM(vm);\n");
        push(@$outputArray, "    return globalObject.functionPrototype();\n");
    }
    push(@$outputArray, "}\n\n");


    push(@$outputArray, "template<> void ${constructorClassName}::initializeProperties(VM& vm, JSDOMGlobalObject& globalObject)\n");
    push(@$outputArray, "{\n");

    # There must exist an interface prototype object for every non-callback interface defined, regardless
    # of whether the interface was declared with the [NoInterfaceObject] extended attribute.
    # https://heycam.github.io/webidl/#interface-prototype-object
    if (ShouldUseGlobalObjectPrototype($interface)) {
        push(@$outputArray, "    putDirect(vm, vm.propertyNames->prototype, globalObject.getPrototypeDirect(), DontDelete | ReadOnly | DontEnum);\n");
    } elsif ($interface->isCallback) {
        push(@$outputArray, "    UNUSED_PARAM(globalObject);\n");
    } else {
        push(@$outputArray, "    putDirect(vm, vm.propertyNames->prototype, ${className}::prototype(vm, &globalObject), DontDelete | ReadOnly | DontEnum);\n");
    }

    push(@$outputArray, "    putDirect(vm, vm.propertyNames->name, jsNontrivialString(&vm, String(ASCIILiteral(\"$visibleInterfaceName\"))), ReadOnly | DontEnum);\n");
    push(@$outputArray, "    putDirect(vm, vm.propertyNames->length, jsNumber(${leastConstructorLength}), ReadOnly | DontEnum);\n") if defined $leastConstructorLength;
    push(@$outputArray, "    reifyStaticProperties(vm, ${className}ConstructorTableValues, *this);\n") if ConstructorHasProperties($interface);

    push(@$outputArray, "}\n\n");

    if (IsJSBuiltinConstructor($interface)) {
        push(@$outputArray, "template<> FunctionExecutable* ${constructorClassName}::initializeExecutable(VM& vm)\n");
        push(@$outputArray, "{\n");
        push(@$outputArray, "    return " . GetJSBuiltinFunctionNameFromString($interface->name, "initialize" . $interface->name) . "(vm);\n");
        push(@$outputArray, "}\n");
        push(@$outputArray, "\n");
    }
    push(@$outputArray, "template<> const ClassInfo ${constructorClassName}::s_info = { \"${visibleInterfaceName}\", &Base::s_info, 0, CREATE_METHOD_TABLE($constructorClassName) };\n\n");
 
}

sub HasCustomConstructor
{
    my $interface = shift;
    return $interface->extendedAttributes->{CustomConstructor};
}

sub HasCustomGetter
{
    my $extendedAttributes = shift;
    return $extendedAttributes->{Custom} || $extendedAttributes->{CustomGetter} ;
}

sub HasCustomSetter
{
    my $extendedAttributes = shift;
    return $extendedAttributes->{Custom} || $extendedAttributes->{CustomSetter};
}

sub HasCustomMethod
{
    my $extendedAttributes = shift;
    return $extendedAttributes->{Custom};
}

sub NeedsConstructorProperty
{
    my $interface = shift;
    return !$interface->extendedAttributes->{NoInterfaceObject} || $interface->extendedAttributes->{CustomConstructor};
}

sub IsReturningPromise
{
    my $function = shift;
    return $function->signature->type && $function->signature->type eq "Promise";
}

sub IsConstructable
{
    my $interface = shift;
    return HasCustomConstructor($interface)
        || $interface->extendedAttributes->{Constructor}
        || $interface->extendedAttributes->{NamedConstructor}
        || $interface->extendedAttributes->{JSBuiltinConstructor};
}

sub HeaderNeedsPrototypeDeclaration
{
    my $interface = shift;
    return IsDOMGlobalObject($interface) || $interface->extendedAttributes->{JSCustomNamedGetterOnPrototype} || $interface->extendedAttributes->{JSCustomDefineOwnPropertyOnPrototype};
}

sub IsUnforgeable
{
    my $interface = shift;
    my $property = shift;
    return $property->signature->extendedAttributes->{Unforgeable} || $interface->extendedAttributes->{Unforgeable};
}

sub ComputeFunctionSpecial
{
    my $interface = shift;
    my $function = shift;

    my @specials = ();
    push(@specials, ("DontDelete", "ReadOnly")) if IsUnforgeable($interface, $function);
    push(@specials, "DontEnum") if $function->signature->extendedAttributes->{NotEnumerable};
    if (IsJSBuiltin($interface, $function)) {
        push(@specials, "JSC::Builtin");
    }
    else {
        push(@specials, "JSC::Function");
    }
    return (@specials > 0) ? join(" | ", @specials) : "0";
}

sub IsJSBuiltin
{
    my ($interface, $object) = @_;

    return 0 if $object->signature->extendedAttributes->{Custom};
    return 0 if $object->signature->extendedAttributes->{CustomGetter};
    return 0 if $object->signature->extendedAttributes->{CustomSetter};

    return 1 if $object->signature->extendedAttributes->{JSBuiltin};
    return 1 if $interface->extendedAttributes->{JSBuiltin};

    return 0;
}

sub IsJSBuiltinConstructor
{
    my ($interface) = @_;

    return 0 if $interface->extendedAttributes->{CustomConstructor};
    return 1 if $interface->extendedAttributes->{JSBuiltin};
    return 1 if $interface->extendedAttributes->{JSBuiltinConstructor};
    return 0;
}

sub GetJSBuiltinFunctionName
{
    my ($className, $function) = @_;
    my $scopeName = $function->signature->extendedAttributes->{ImplementedBy};
    $scopeName = substr $className, 2 unless $scopeName;
    return GetJSBuiltinFunctionNameFromString($scopeName, $function->signature->name);
}

sub GetJSBuiltinFunctionNameFromString
{
    my ($scopeName, $functionName) = @_;
    return $codeGenerator->WK_lcfirst($scopeName) . $codeGenerator->WK_ucfirst($functionName) . "CodeGenerator";
}

sub GetJSBuiltinScopeName
{
    my ($interface, $object) = @_;
    return $object->signature->extendedAttributes->{ImplementedBy} || $interface->name;
}

sub AddJSBuiltinIncludesIfNeeded()
{
    my $interface = shift;

    if ($interface->extendedAttributes->{JSBuiltin} || $interface->extendedAttributes->{JSBuiltinConstructor}) {
        AddToImplIncludes($interface->name . "Builtins.h");
        return;
    }

    foreach my $function (@{$interface->functions}) {
        AddToImplIncludes(GetJSBuiltinScopeName($interface, $function) . "Builtins.h", $function->signature->extendedAttributes->{Conditional}) if IsJSBuiltin($interface, $function);
    }

    foreach my $attribute (@{$interface->attributes}) {
        AddToImplIncludes(GetJSBuiltinScopeName($interface, $attribute) . "Builtins.h", $attribute->signature->extendedAttributes->{Conditional}) if IsJSBuiltin($interface, $attribute);
    }
}

1;
