/*
 * Copyright (C) 2018 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(LAYOUT_FORMATTING_CONTEXT)

#include "LayoutPoint.h"
#include "LayoutRect.h"
#include "LayoutUnit.h"
#include "RenderStyleConstants.h"
#include <wtf/IsoMalloc.h>

namespace WebCore {

class RenderStyle;

namespace Layout {
class BlockFormattingContext;
class FormattingContext;
class LayoutContext;
}

namespace Display {

class Box {
    WTF_MAKE_ISO_ALLOCATED(Box);
public:
    friend class Layout::BlockFormattingContext;
    friend class Layout::FormattingContext;
    friend class Layout::LayoutContext;

    class Rect {
    public:
        Rect() = default;
        
        LayoutUnit top() const;
        LayoutUnit left() const;
        LayoutPoint topLeft() const;

        LayoutUnit bottom() const;
        LayoutUnit right() const;        
        LayoutPoint bottomRight() const;

        LayoutUnit width() const;
        LayoutUnit height() const;
        LayoutSize size() const;

        void setTop(LayoutUnit);
        void setLeft(LayoutUnit);
        void setTopLeft(const LayoutPoint&);
        void setWidth(LayoutUnit);
        void setHeight(LayoutUnit);
        void setSize(const LayoutSize&);

        void shiftLeftTo(LayoutUnit);
        void shiftRightTo(LayoutUnit);
        void shiftTopTo(LayoutUnit);
        void shiftBottomTo(LayoutUnit);

        void expand(LayoutUnit, LayoutUnit);

        Rect clone() const;
        operator LayoutRect() const;

    private:
#if !ASSERT_DISABLED
        void invalidateTop() { m_hasValidTop = false; }
        void invalidateLeft() { m_hasValidLeft = false; }
        void invalidateWidth() { m_hasValidWidth = false; }
        void invalidateHeight() { m_hasValidHeight = false; }
        void invalidatePosition();

        bool hasValidPosition() const { return m_hasValidTop && m_hasValidLeft; }
        bool hasValidSize() const { return m_hasValidWidth && m_hasValidHeight; }
        bool hasValidGeometry() const { return hasValidPosition() && hasValidSize(); }
    
        void setHasValidPosition();
        void setHasValidSize();

        bool m_hasValidTop { false };
        bool m_hasValidLeft { false };
        bool m_hasValidWidth { false };
        bool m_hasValidHeight { false };
#endif
        LayoutRect m_rect;
    };

    ~Box();

    Rect rect() const { return m_rect; }

    LayoutUnit top() const { return m_rect.top(); }
    LayoutUnit left() const { return m_rect.left(); }
    LayoutUnit bottom() const { return m_rect.bottom(); }
    LayoutUnit right() const { return m_rect.right(); }

    LayoutPoint topLeft() const { return m_rect.topLeft(); }
    LayoutPoint bottomRight() const { return m_rect.bottomRight(); }

    LayoutSize size() const { return m_rect.size(); }
    LayoutUnit width() const { return m_rect.width(); }
    LayoutUnit height() const { return m_rect.height(); }

    LayoutUnit marginTop() const;
    LayoutUnit marginLeft() const;
    LayoutUnit marginBottom() const;
    LayoutUnit marginRight() const;

    LayoutUnit borderTop() const;
    LayoutUnit borderLeft() const;
    LayoutUnit borderBottom() const;
    LayoutUnit borderRight() const;

    LayoutUnit paddingTop() const;
    LayoutUnit paddingLeft() const;
    LayoutUnit paddingBottom() const;
    LayoutUnit paddingRight() const;

    Rect marginBox() const;
    Rect borderBox() const;
    Rect paddingBox() const;
    Rect contentBox() const;

private:
    Box(const RenderStyle&);

    struct Style {
        Style(const RenderStyle&);

        BoxSizing boxSizing { BoxSizing::ContentBox };
    };

    void setTopLeft(const LayoutPoint& topLeft) { m_rect.setTopLeft(topLeft); }
    void setTop(LayoutUnit top) { m_rect.setTop(top); }
    void setLeft(LayoutUnit left) { m_rect.setLeft(left); }
    void setWidth(LayoutUnit width) { m_rect.setWidth(width); }
    void setHeight(LayoutUnit height) { m_rect.setHeight(height); }
    void setSize(const LayoutSize& size) { m_rect.setSize(size); }

    struct HorizontalEdges {
        LayoutUnit left;
        LayoutUnit right;
    };

    struct VerticalEdges {
        LayoutUnit top;
        LayoutUnit bottom;
    };

    struct Edges {
        HorizontalEdges horizontal;
        VerticalEdges vertical;
    };

    void setHorizontalMargin(HorizontalEdges);
    void setVerticalMargin(VerticalEdges);
    void setBorder(Edges);
    void setPadding(Edges);

#if !ASSERT_DISABLED
    void invalidateMargin();
    void invalidateBorder() { m_hasValidBorder = false; }
    void invalidatePadding() { m_hasValidPadding = false; }

    void setHasValidVerticalMargin() { m_hasValidVerticalMargin = true; }
    void setHasValidHorizontalMargin() { m_hasValidHorizontalMargin = true; }

    void setHasValidBorder() { m_hasValidBorder = true; }
    void setHasValidPadding() { m_hasValidPadding = true; }
#endif

    const Style m_style;

    Rect m_rect;

    Edges m_margin;
    Edges m_border;
    Edges m_padding;

#if !ASSERT_DISABLED
    bool m_hasValidHorizontalMargin { false };
    bool m_hasValidVerticalMargin { false };
    bool m_hasValidBorder { false };
    bool m_hasValidPadding { false };
#endif
};

#if !ASSERT_DISABLED
inline void Box::Rect::invalidatePosition()
{
    invalidateTop();
    invalidateLeft();
}

inline void Box::Rect::setHasValidPosition()
{
    m_hasValidTop = true;
    m_hasValidLeft = true;
}

inline void Box::Rect::setHasValidSize()
{
    m_hasValidWidth = true;
    m_hasValidHeight = true;
}

inline void Box::invalidateMargin()
{
    m_hasValidHorizontalMargin = false;
    m_hasValidVerticalMargin = false;
}
#endif

inline LayoutUnit Box::Rect::top() const
{
    ASSERT(m_hasValidTop);
    return m_rect.y();
}

inline LayoutUnit Box::Rect::left() const
{
    ASSERT(m_hasValidLeft);
    return m_rect.x();
}

inline LayoutUnit Box::Rect::bottom() const
{
    ASSERT(m_hasValidTop && m_hasValidHeight);
    return m_rect.maxY();
}

inline LayoutUnit Box::Rect::right() const
{
    ASSERT(m_hasValidLeft && m_hasValidWidth);
    return m_rect.maxX();
}

inline LayoutPoint Box::Rect::topLeft() const
{
    ASSERT(hasValidPosition());
    return m_rect.minXMinYCorner();
}

inline LayoutPoint Box::Rect::bottomRight() const
{
    ASSERT(hasValidGeometry());
    return m_rect.maxXMaxYCorner();
}

inline LayoutSize Box::Rect::size() const
{
    ASSERT(hasValidSize());
    return m_rect.size();
}

inline LayoutUnit Box::Rect::width() const
{
    ASSERT(m_hasValidWidth);
    return m_rect.width();
}

inline LayoutUnit Box::Rect::height() const
{
    ASSERT(m_hasValidHeight);
    return m_rect.height();
}

inline void Box::Rect::setTopLeft(const LayoutPoint& topLeft)
{
#if !ASSERT_DISABLED
    setHasValidPosition();
#endif
    m_rect.setLocation(topLeft);
}

inline void Box::Rect::setTop(LayoutUnit top)
{
#if !ASSERT_DISABLED
    m_hasValidTop = true;
#endif
    m_rect.setY(top);
}

inline void Box::Rect::setLeft(LayoutUnit left)
{
#if !ASSERT_DISABLED
    m_hasValidLeft = true;
#endif
    m_rect.setX(left);
}

inline void Box::Rect::setWidth(LayoutUnit width)
{
#if !ASSERT_DISABLED
    m_hasValidWidth = true;
#endif
    ASSERT(m_hasValidLeft);
    m_rect.setWidth(width);
}

inline void Box::Rect::setHeight(LayoutUnit height)
{
#if !ASSERT_DISABLED
    m_hasValidHeight = true;
#endif
    ASSERT(m_hasValidTop);
    m_rect.setHeight(height);
}

inline void Box::Rect::setSize(const LayoutSize& size)
{
#if !ASSERT_DISABLED
    setHasValidSize();
#endif
    m_rect.setSize(size);
}

inline void Box::Rect::shiftLeftTo(LayoutUnit left)
{
    ASSERT(m_hasValidLeft);
    m_rect.shiftXEdgeTo(left);
}

inline void Box::Rect::shiftRightTo(LayoutUnit right)
{
    ASSERT(m_hasValidLeft && m_hasValidWidth);
    m_rect.shiftMaxXEdgeTo(right);
}

inline void Box::Rect::shiftTopTo(LayoutUnit top)
{
    ASSERT(m_hasValidTop);
    m_rect.shiftYEdgeTo(top);
}

inline void Box::Rect::shiftBottomTo(LayoutUnit bottom)
{
    ASSERT(m_hasValidTop && m_hasValidHeight);
    m_rect.shiftMaxYEdgeTo(bottom);
}

inline void Box::Rect::expand(LayoutUnit width, LayoutUnit height)
{
    ASSERT(hasValidGeometry());
    m_rect.expand(width, height);
}

inline Box::Rect Box::Rect::clone() const
{
    Rect rect;
#if !ASSERT_DISABLED
    rect.m_hasValidTop = m_hasValidTop;
    rect.m_hasValidLeft = m_hasValidLeft;
    rect.m_hasValidWidth = m_hasValidWidth;
    rect.m_hasValidHeight  = m_hasValidHeight;
#endif 
    rect.m_rect = m_rect;
    return rect;
}

inline Box::Rect::operator LayoutRect() const
{
    ASSERT(hasValidGeometry()); 
    return m_rect;
}

inline void Box::setHorizontalMargin(HorizontalEdges margin)
{
#if !ASSERT_DISABLED
    setHasValidHorizontalMargin();
#endif
    m_margin.horizontal = margin;
}

inline void Box::setVerticalMargin(VerticalEdges margin)
{
#if !ASSERT_DISABLED
    setHasValidVerticalMargin();
#endif
    m_margin.vertical = margin;
}

inline void Box::setBorder(Edges border)
{
#if !ASSERT_DISABLED
    setHasValidBorder();
#endif
    m_border = border;
}

inline void Box::setPadding(Edges padding)
{
#if !ASSERT_DISABLED
    setHasValidPadding();
#endif
    m_padding = padding;
}

inline LayoutUnit Box::marginTop() const
{
    ASSERT(m_hasValidVerticalMargin);
    return m_margin.vertical.top;
}

inline LayoutUnit Box::marginLeft() const
{
    ASSERT(m_hasValidHorizontalMargin);
    return m_margin.horizontal.left;
}

inline LayoutUnit Box::marginBottom() const
{
    ASSERT(m_hasValidVerticalMargin);
    return m_margin.vertical.bottom;
}

inline LayoutUnit Box::marginRight() const
{
    ASSERT(m_hasValidHorizontalMargin);
    return m_margin.horizontal.right;
}

inline LayoutUnit Box::paddingTop() const
{
    ASSERT(m_hasValidPadding);
    return m_padding.vertical.top;
}

inline LayoutUnit Box::paddingLeft() const
{
    ASSERT(m_hasValidPadding);
    return m_padding.horizontal.left;
}

inline LayoutUnit Box::paddingBottom() const
{
    ASSERT(m_hasValidPadding);
    return m_padding.vertical.bottom;
}

inline LayoutUnit Box::paddingRight() const
{
    ASSERT(m_hasValidPadding);
    return m_padding.horizontal.right;
}

inline LayoutUnit Box::borderTop() const
{
    ASSERT(m_hasValidBorder);
    return m_border.vertical.top;
}

inline LayoutUnit Box::borderLeft() const
{
    ASSERT(m_hasValidBorder);
    return m_border.horizontal.left;
}

inline LayoutUnit Box::borderBottom() const
{
    ASSERT(m_hasValidBorder);
    return m_border.vertical.bottom;
}

inline LayoutUnit Box::borderRight() const
{
    ASSERT(m_hasValidBorder);
    return m_border.horizontal.right;
}

}
}
#endif
