/************************************************************************/
/* File		database.cpp						*/
/*									*/
/* Purpose	This C++ program file contains the implementation for	*/
/*		the DataBase class. The DataBase class manages manages	*/
/*		an array of DataFile objects. Each DataFile object	*/
/*		stores DataRecord records in a file and the DataFile	*/
/*		object indexes the DataRecord records with a DataIndex	*/
/*		file. Each DataRecord record is an array of DataElement	*/
/*		objects.						*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, June 23, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <time.h>			// struct tm definition.
#include "database.h"			// DataBase class.

/* The following variable is the error message array that is specific	*/
/* to the DataBase class. The default UtilityModule error values are	*/
/* overwritten by the DataBase object constructor.			*/

static char *errormessages[] =
   {
      "No error",				// NoError
      "Could not create directory",		// BadDir
      "DataBase object not prepared for use",	// CanOpen
      "DataBase is not open",			// NotOpen
      "DataFile index is out of range",		// Range
      "Could not create DataBase files",	// NoMake
      "Could not open DataBase files",		// NoOpen
      "Could not close DataBase files",		// NoClose
      "Could not store DataBase record",	// Store
      "Could not remove DataBase record",	// Remove
      "Could not get DataBase record",		// Get
      "Could not get first DataBase record",	// First
      "Could not get next DataBase record",	// Next
      "Could not get last DataBase record",	// Last
      "Could not get previous DataBase record",	// Previous
   };

/************************************************************************/
/* Function	DataBase()						*/
/*									*/
/* Purpose	This is the default constructor for a DataBase object.	*/
/*		The default constructor can be used when the DataBase	*/
/*		object is to be initialized by being read in from a	*/
/*		file (such as an ObjectFile).				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The DataBase object will be empty. The DataBase object	*/
/*		can be prepared for use by reading in the object from	*/
/*		a file using the Read_From_File function (see		*/
/*		ObjectFile).						*/
/************************************************************************/

DataBase::DataBase()
   {
      itsmodulename		= "DataBase";
      itsmaxerror		= DataBaseErrors;
      itserrormessages		= errormessages;
      itscanopenflag		= false;
      itsisopenflag		= false;
      itsfilecount		= 0;
      itsfiles			= (DataFile*)0;
   }

/************************************************************************/
/* Function	DataBase(const String& directory, const int files,	*/
/*		const String* filenames, const int* elementcounts,	*/
/*		const DataElementType* recordelements, const int mode)	*/
/*									*/
/* Purpose	This is an alternate DataBase constructor. This		*/
/*		DataBase constructor will prepare a DataBase object for	*/
/*		use. The variables passed to this constructor must	*/
/*		contain the required information that describes the	*/
/*		DataBase object.					*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the path of the directory that contains (or	*/
/*		will contain) the DataBase files. If the directory does	*/
/*		not exist then it will be created. This is true even if	*/
/*		several subdirectories need to be created first. This	*/
/*		function expects the variable 'files' to contain the	*/
/*		number of data files that will be managed by the	*/
/*		DataBase object. The variable 'filenames' must point	*/
/*		to the array of names to use when creating and opening	*/
/*		the DataBase object's files. The variable		*/
/*		'elementcounts' must point to an array of integers.	*/
/*		Each entry must contain the number of DataElement	*/
/*		elements that are in each DataFile's DataRecord. The	*/
/*		variable 'recordelements' must point to a list of the	*/
/*		DataElement types. Each DataFile's DataRecord element	*/
/*		list must be included in the array. Finally, the	*/
/*		variable 'mode' must contain the operating system	*/
/*		read/write mode to use if the directory needs to be	*/
/*		created. If the variable 'mode' is not given then the	*/
/*		default value 'DataBaseDirMode' will be used.		*/
/*									*/
/* Output	If there are no errors preparing the DataBase object	*/
/*		for use then the private member variable		*/
/*		'itscanopenflag' will be set to true. If this		*/
/*		constructor encountered an error creating the directory	*/
/*		then 'itscanopenflag' will be set to false and an error	*/
/*		message will be written to stderr.			*/
/************************************************************************/

DataBase::DataBase(const String& directory, const int files,
   const String* filenames, const int* elementcounts,
   const DataElementType* recordelements, const int mode)
   {
      itsmodulename		= "DataBase";
      itsmaxerror		= DataBaseErrors;
      itserrormessages		= errormessages;
      itsfiles			= (DataFile*)0;
      Prepare_DataBase(directory, files, filenames, elementcounts,
         recordelements, mode);
   }

/************************************************************************/
/* Function	~DataBase()						*/
/*									*/
/* Purpose	This is the default destructor for a DataBase object.	*/
/*		This destructor will delete the DataFile array if it	*/
/*		has been allocated.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Ouput	None.							*/
/************************************************************************/

DataBase::~DataBase()
   {
      if (itsfiles != (DataFile*)0)
         {
            delete [] itsfiles;
	 }
   }

/************************************************************************/
/* Function	int Record_Size(void)					*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. The	*/
/*		function returns the number of bytes that are required	*/
/*		to write the object to a file.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the number of bytes that are	*/
/*		required to write the object to a file.			*/
/************************************************************************/

int DataBase::Record_Size(void)
   {
      int		size;
      register int	index;

      size		= sizeof(int) + itsdirectory.Length() + 1 +
			  sizeof(int);
      for (index = 0; index < itsfilecount; index++)
         {
	    size	+= itsfiles[index].Record_Size();
	 }
      return(size);
   }

/************************************************************************/
/* Function	status Read_From_File(File& file, const int filehandle)	*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will read a DataBase object from a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		reading the object from the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataBase object is to be read from	*/
/*									*/
/* Output	If this function is able to read in a DataBase object	*/
/*		from 'file' then this function will return OK and the	*/
/*		DataBase object will contain the contents that were	*/
/*		read in from the file. However, the DataBase object	*/
/*		cannot be used until the files have been opened. If	*/
/*		this function is not able to read in a DataBase object	*/
/*		from 'file' then this function will return ERROR.  All	*/
/*		errors by this function are handled by the ObjectFile	*/
/*		class.							*/
/************************************************************************/

status DataBase::Read_From_File(File& file, const int filehandle)
   {
      status		result		= OK;
      register int	index;

      if (itsfiles != (DataFile*)0)
         {
	    delete [] itsfiles;
	 }
      itscanopenflag	= false;
      itsisopenflag	= false;
      if (Read_String_From_ObjectFile(file, filehandle, itsdirectory)
         == ERROR)
         {
	    /* Could not read DataBase directory from file.		*/

	    file.Set_Error_Info_String(
	       "Could not read DataBase directory from file");
	    result	= ERROR;
	 }
      else if (file.Read_File(filehandle, (void*)&itsfilecount,
         sizeof(int)) == ERROR)
         {
	    /* Could not read DataBase file count from file.		*/

	    file.Set_Error_Info_String(
	       "Could not read DataBase file count from file");
	    result	= ERROR;
	 }
      for (index = 0; index < itsfilecount && result == OK; index++)
         {
	    if (itsfiles[index].Read_From_File(file, filehandle)
	       == ERROR)
	       {
	          /* Could not read DataBase object's DataFile.		*/

	          file.Set_Error_Info_String(
	             "Could not read DataBase object's DataFile");
	          result	= ERROR;
	       }
	 }
      if (result == OK)
         {
	    itscanopenflag	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_From_File(File& file, const int filehandle)*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will write a DataBase object to a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		writing the object to the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataBase object is to be written to.	*/
/*									*/
/* Output	If this function is able to write the DataBase object	*/
/*		to 'file' then this function will return OK. If this	*/
/*		function is not able to write a DataBase object to	*/
/*		'file' then this function will return ERROR. All errors	*/
/*		by this function are handled by the ObjectFile class.	*/
/************************************************************************/

status DataBase::Write_To_File(File& file, const int filehandle)
   {
      status		result		= OK;
      register int	index;

      if (itscanopenflag == false)
         {
	    /* DataBase object is not prepared for use.			*/

	    file.Set_Error_Info_String(
	       "DataBase object is not prepared for use");
	    result	= ERROR;
	 }
      if (Write_String_To_ObjectFile(file, filehandle, itsdirectory)
         == ERROR)
         {
	    /* Could not write DataBase directory to file.		*/

	    file.Set_Error_Info_String(
	       "Could not write DataBase directory to file");
	    result	= ERROR;
	 }
      else if (file.Write_File(filehandle, (void*)&itsfilecount,
         sizeof(int)) == ERROR)
         {
	    /* Could not write DataBase file count from file.		*/

	    file.Set_Error_Info_String(
	       "Could not write DataBase file count from file");
	    result	= ERROR;
	 }
      for (index = 0; index < itsfilecount && result == OK; index++)
         {
	    if (itsfiles[index].Write_To_File(file, filehandle)
	       == ERROR)
	       {
	          /* Could not write DataBase object's DataFile.	*/

	          file.Set_Error_Info_String(
	             "Could not write DataBase object's DataFile");
	          result	= ERROR;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	void Prepare_DataBase(const String& directory,		*/
/*		   const int files, const String* filenames,		*/
/*		   const int* elementcounts,				*/
/*		   const DataElementType* recordelements,		*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This function will prepare a DataBase object for use	*/
/*		The variables passed to this constructor must contain	*/
/*		the required information that describes the DataBase	*/
/*		object.							*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the path of the directory that contains (or	*/
/*		will contain) the DataBase files. If the directory does	*/
/*		not exist then it will be created. This is true even if	*/
/*		several subdirectories need to be created first. This	*/
/*		function expects the variable 'files' to contain the	*/
/*		number of data files that will be managed by the	*/
/*		DataBase object. The variable 'filenames' must point	*/
/*		to the array of names to use when creating and opening	*/
/*		the DataBase object's files. The variable		*/
/*		'elementcounts' must point to an array of integers.	*/
/*		Each entry must contain the number of DataElement	*/
/*		elements that are in each DataFile's DataRecord. The	*/
/*		variable 'recordelements' must point to a list of the	*/
/*		DataElement types. Each DataFile's DataRecord element	*/
/*		list must be included in the array. Finally, the	*/
/*		variable 'mode' must contain the operating system	*/
/*		read/write mode to use if the directory needs to be	*/
/*		created. If the variable 'mode' is not given then the	*/
/*		default value 'DataBaseDirMode' will be used.		*/
/*									*/
/* Output	If there are no errors preparing the DataBase object	*/
/*		for use then the private member variable		*/
/*		'itscanopenflag' will be set to true. If this		*/
/*		constructor encountered an error creating the directory	*/
/*		then 'itscanopenflag' will be set to false and an error	*/
/*		message will be written to stderr.			*/
/************************************************************************/

void DataBase::Prepare_DataBase(const String& directory, const int files,
   const String* filenames, const int* elementcounts,
   const DataElementType* recordelements, const int mode)
   {
      status			result		= OK;
      register int		index;
      const DataElementType*	elementpointer;

      if (itsfiles != (DataFile*)0)
         {
	    delete [] itsfiles;
	 }
      itscanopenflag		= false;
      itsisopenflag		= false;
      itsfilecount		= 0;
      itsfiles			= (DataFile*)0;
      itsdirectory		= directory;

      /* Make sure that the directory path ends with '/'.		*/

      if (itsdirectory.Data()[itsdirectory.Length() - 1] != '/')
         {
	    itsdirectory	+= "/";
	 }

      /* See if the directory already exists.				*/

      if (itsdir.Does_Directory_Exist(itsdirectory) == false)
         {
	    /* Create the directory.					*/

	    if (itsdir.Create_Recursive_Directory(itsdirectory, mode)
	       == ERROR)
	       {
	          /* Could not create database directory.		*/

		  itserrorinfo	= "Attempted to create directory ";
		  itserrorinfo	+= itsdirectory;
		  itserror	= DataBaseBadDir;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      if (result == OK)
         {
	    /* Save the number of DataBase files and allocate an array.	*/

	    itsfilecount	= files;
	    itsfiles		= new DataFile[itsfilecount];

	    /* Prepare each DataFile in the array by supplying it with	*/
	    /* its filename and its DataRecord element list.		*/

	    for (index = 0, elementpointer = recordelements;
	       index < itsfilecount; index++)
	       {
	          itsfiles[index].Prepare_DataFile(filenames[index],
		     elementcounts[index], elementpointer);
		  elementpointer		+= elementcounts[index];
	       }
	    itscanopenflag	= true;
	 }
   }

/************************************************************************/
/* Function	void Format_Record(const int file, DataRecord& record)	*/
/*									*/
/* Purpose	This function can be used to format a record for use	*/
/*		with a particular DataFile. Any data that the record	*/
/*		contains will be deleted and the record will have	*/
/*		memory allocated to hold a record that is specific to	*/
/*		the DataFile that is referenced by the variable 'file'.	*/
/*									*/
/* Input	This function expects the variable 'file' to specify	*/
/*		which DataFile format to use. The variable 'record'	*/
/*		must refernce the DataRecord object that will be	*/
/*		formatted.						*/
/*									*/
/* Output	If the DataBase object is prepared for use and if the	*/
/*		DataFile index is in range, the DataRecord will be	*/
/*		formatted for use with the specified DataFile. If the	*/
/*		DataBase object is not prepared for use or if the	*/
/*		DataFile index is out of range then an error message	*/
/*		will be written to stderr and the record will be	*/
/*		cleared.						*/
/************************************************************************/

void DataBase::Format_Record(const int file, DataRecord& record)
   {
      if (itscanopenflag == false)
         {
	    /* DataBase object is not prepared for use.			*/

	    itserrorinfo	= "Attempted to format a record";
	    itserror		= DataBaseCanOpen;
	    Report_Error();
	    record.Clear_Record();
	 }
      else if (file < 0 || file > itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to format a record";
	    itserror		= DataBaseRange;
	    Report_Error();
	    record.Clear_Record();
	 }
      else
         {
	    itsfiles[file].Format_Record(record);
	 }
   }

/************************************************************************/
/* Function	status Create_DataBase_Files(void)			*/
/*									*/
/* Purpose	This function will call each DataFile object in the	*/
/*		DataFile array to create the files that are managed by	*/
/*		the DataBase object.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to create the files that are	*/
/*		managed by the DataBase class then this function will	*/
/*		return OK. The created files will need to be opened	*/
/*		before they can be used. If this function is not able	*/
/*		to create the files that are managed by the DataBase	*/
/*		class then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Create_DataBase_Files(void)
   {
      status		result		= OK;
      register int	index;

      if (itscanopenflag == false)
         {
	    /* DataBase object is not prepared for use.			*/

	    itserrorinfo	= "Attempted to create DataBase";
	    itserror		= DataBaseCanOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      for (index = 0; index < itsfilecount && result == OK; index++)
         {
	    if (itsfiles[index].Create_Files(itsdirectory) == ERROR)
	       {
	          /* Could not create database files.			*/

		  itserrorinfo	= "Attempted to create DataBase";
		  itserror	= DataBaseNoMake;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Open_DataBase_Files(void)			*/
/*									*/
/* Purpose	This function will call each DataFile object in the	*/
/*		DataFile array to open the files that are managed by	*/
/*		the DataBase object.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to open the files that are	*/
/*		managed by the DataBase class then this function will	*/
/*		return OK. If this function is not able to open the	*/
/*		files that are managed by the DataBase class then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataBase::Open_DataBase_Files(void)
   {
      status		result		= OK;
      register int	index;

      if (itscanopenflag == false)
         {
	    /* DataBase object is not prepared for use.			*/

	    itserrorinfo	= "Attempted to open DataBase";
	    itserror		= DataBaseCanOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      for (index = 0; index < itsfilecount && result == OK; index++)
         {
	    if (itsfiles[index].Open_Files(itsdirectory) == ERROR)
	       {
	          /* Could not open database files.			*/

		  itserrorinfo	= "Attempted to open DataBase";
		  itserror	= DataBaseNoOpen;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      if (result == OK)
         {
	    itsisopenflag	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Close_DataBase_Files(void)			*/
/*									*/
/* Purpose	This function will call each DataFile object in the	*/
/*		DataFile array to close the files that are managed by	*/
/*		the DataBase object.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to close the files that are	*/
/*		managed by the DataBase class then this function will	*/
/*		return OK. If this function is not able to close the	*/
/*		files that are managed by the DataBase class then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataBase::Close_DataBase_Files(void)
   {
      status		result		= OK;
      register int	index;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to close DataBase";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      for (index = 0; index < itsfilecount && result == OK; index++)
         {
	    if (itsfiles[index].Close_Files() == ERROR)
	       {
	          /* Could not close database files.			*/

		  itserrorinfo	= "Attempted to close DataBase";
		  itserror	= DataBaseNoClose;
		  result	= ERROR;
		  Report_Error();
	       }
	 }
      if (result == OK)
         {
	    itsisopenflag	= false;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Store_Record(const int datafile,			*/
/*		   DataRecord& record)					*/
/*									*/
/* Purpose	This function will write a record to a DataBase file.	*/
/*		The first element in the record is used as the search	*/
/*		key. If the record does not already exist in the file	*/
/*		then the record will be added to the file. If the	*/
/*		record does exist in the file then the old record will	*/
/*		be replaced with the new record.			*/
/*									*/
/* Input	This function expects the variable 'datafile' to	*/
/*		specify which DataFile the record will be stored in.	*/
/*		The variable 'record' must be reference to a DataRecord	*/
/*		object that is formatted for the specified DataFile.	*/
/*									*/
/* Output	If this function is able to add the record to the file,	*/
/*		or if this function is able to replace an old record	*/
/*		with the new record, then this function will return OK.	*/
/*		If this function is not able to store the record in the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Store_Record(const int datafile, DataRecord& record)
   {
      status			result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to store record";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (datafile < 0 || datafile >= itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to store a record";
	    itserror		= DataBaseRange;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsfiles[datafile].Store_Record(record) == ERROR)
	 {
	    /* Could not store record.					*/

	    itserror		= DataBaseStore;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_Record(const int datafile,		*/
/*		   DataRecord& record)					*/
/*									*/
/* Purpose	This function will remove a record from a DataBase	*/
/*		file. The first element in the record is used as the	*/
/*		search key.						*/
/*									*/
/* Input	This function expects the variable 'datafile' to	*/
/*		specify which DataFile the record will be removed from.	*/
/*		The variable 'record' must be a reference to a		*/
/*		DataRecord object. The DataRecord object does not need	*/
/*		to be formatted for the specified DataFile, but the	*/
/*		first element of 'record' must contain the correct key.	*/
/*									*/
/* Output	If this function is able to remove the record from the	*/
/*		DataBase then this function will return OK. If this	*/
/*		function is not able to remove the record from the	*/
/*		DataBase then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Remove_Record(const int datafile, DataRecord& record)
   {
      status		result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to remove record";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (datafile < 0 || datafile >= itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to remove a record";
	    itserror		= DataBaseRange;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsfiles[datafile].Remove_Record(record) == ERROR)
	 {
	    /* Could not remove record.				*/

	    itserror		= DataBaseRemove;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Record(const int datafile,			*/
/*		   DataRecord& record)					*/
/*									*/
/* Purpose	This function will get a record from a DataBase file.	*/
/*		The first element in the record is used as the search	*/
/*		key.							*/
/*									*/
/* Input	This function expects the variable 'datafile' to	*/
/*		specify which DataFile to get the record from. The	*/
/*		variable 'record' must be a reference to a DataRecord	*/
/*		object. The DataRecord object does not need to be	*/
/*		formatted for the specified DataFile, but the first	*/
/*		element of 'record' must contain the correct key.	*/
/*									*/
/* Output	If this function is able to get the record from the	*/
/*		DataBase then this function will return OK. If this	*/
/*		function is not able to get the record from the		*/
/*		DataBase then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Get_Record(const int datafile, DataRecord& record)
   {
      status		result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to get record";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (datafile < 0 || datafile >= itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to get a record";
	    itserror		= DataBaseRange;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsfiles[datafile].Get_Record(record) == ERROR)
	 {
	    /* Could not get record.				*/

	    itserror		= DataBaseGet;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_First_Record(const int datafile,		*/
/*		   DataRecord& record)					*/
/*									*/
/* Purpose	This function will get the first record in a DataBase	*/
/*		file.							*/
/*									*/
/* Input	This function expects the variable 'datafile' to	*/
/*		specify which DataFile to get the record from. The	*/
/*		variable 'record' must be a reference to a DataRecord	*/
/*		object. The DataRecord object does not need to be	*/
/*		formatted for the specified DataFile as its data will	*/
/*		be deleted and new memory will be allocated for it.	*/
/*									*/
/* Output	If this function is able to get the record from the	*/
/*		DataBase then this function will return OK. If this	*/
/*		function is not able to get the record from the		*/
/*		DataBase then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Get_First_Record(const int datafile, DataRecord& record)
   {
      status		result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to get first record";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (datafile < 0 || datafile >= itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to get first record";
	    itserror		= DataBaseRange;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsfiles[datafile].Get_First_Record(record) == ERROR)
	 {
	    /* Could not get first record.			*/

	    itserror		= DataBaseFirst;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Next_Record(const int datafile,		*/
/*		   DataRecord& record)					*/
/*									*/
/* Purpose	This function will get the next record in a DataBase	*/
/*		file.							*/
/*									*/
/* Input	This function expects the variable 'datafile' to	*/
/*		specify which DataFile to get the record from. The	*/
/*		variable 'record' must be a reference to a DataRecord	*/
/*		object. The DataRecord object does not need to be	*/
/*		formatted for the specified DataFile as its data will	*/
/*		be deleted and new memory will be allocated for it.	*/
/*									*/
/* Output	If this function is able to get the record from the	*/
/*		DataBase then this function will return OK. If this	*/
/*		function is not able to get the record from the		*/
/*		DataBase then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Get_Next_Record(const int datafile, DataRecord& record)
   {
      status		result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to get next record";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (datafile < 0 || datafile >= itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to get next record";
	    itserror		= DataBaseRange;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsfiles[datafile].Get_Next_Record(record) == ERROR)
	 {
	    /* Could not get next record.				*/

	    itserror		= DataBaseNext;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Last_Record(const int datafile,		*/
/*		   DataRecord& record)					*/
/*									*/
/* Purpose	This function will get the last record in a DataBase	*/
/*		file.							*/
/*									*/
/* Input	This function expects the variable 'datafile' to	*/
/*		specify which DataFile to get the record from. The	*/
/*		variable 'record' must be a reference to a DataRecord	*/
/*		object. The DataRecord object does not need to be	*/
/*		formatted for the specified DataFile as its data will	*/
/*		be deleted and new memory will be allocated for it.	*/
/*									*/
/* Output	If this function is able to get the record from the	*/
/*		DataBase then this function will return OK. If this	*/
/*		function is not able to get the record from the		*/
/*		DataBase then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Get_Last_Record(const int datafile, DataRecord& record)
   {
      status		result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to get last record";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (datafile < 0 || datafile >= itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to get last record";
	    itserror		= DataBaseRange;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsfiles[datafile].Get_Last_Record(record) == ERROR)
	 {
	    /* Could not get last record.				*/

	    itserror		= DataBaseLast;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Previous_Record(const int datafile,		*/
/*		   DataRecord& record)					*/
/*									*/
/* Purpose	This function will get the previous record in a		*/
/*		DataBase file.						*/
/*									*/
/* Input	This function expects the variable 'datafile' to	*/
/*		specify which DataFile to get the record from. The	*/
/*		variable 'record' must be a reference to a DataRecord	*/
/*		object. The DataRecord object does not need to be	*/
/*		formatted for the specified DataFile as its data will	*/
/*		be deleted and new memory will be allocated for it.	*/
/*									*/
/* Output	If this function is able to get the record from the	*/
/*		DataBase then this function will return OK. If this	*/
/*		function is not able to get the record from the		*/
/*		DataBase then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataBase::Get_Previous_Record(const int datafile, DataRecord& record)
   {
      status		result		= OK;

      if (itsisopenflag == false)
         {
	    /* DataBase is not open.					*/

	    itserrorinfo	= "Attempted to get previous record";
	    itserror		= DataBaseNotOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (datafile < 0 || datafile >= itsfilecount)
         {
	    /* DataFile index is out of range.				*/

	    itserrorinfo	= "Attempted to get previous record";
	    itserror		= DataBaseRange;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsfiles[datafile].Get_Previous_Record(record) == ERROR)
	 {
	    /* Could not get previous record.			*/

	    itserror		= DataBasePrevious;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }
