/************************************************************************/
/* File		file.cpp						*/
/*									*/
/* Purpose	This C++ program file implements the File class. The	*/
/*		File class provides a C++ interface to the physical	*/
/*		file routines. These physical file routines include	*/
/*		routines for creating, removing, opening, closing,	*/
/*		reading, writing, and positioning a file.		*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00						*/
/*									*/
/* Date		Thursday, May 2, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <sys/types.h>			// System types.
#include <sys/stat.h>			// stat structure.
#include <unistd.h>			// Unix standards.
#include <fcntl.h>			// Physical file stuff.
#include <errno.h>			// OS error codes.
#include "file.h"			// File class.

/* Static	The following variable is the error message array that	*/
/*		is specific to the File class. The default		*/
/*		UtilityModule error values are overwritten by the File	*/
/*		object constructor.					*/

static char* errormessages[] =
   {
      "No error",			// FileNoError
      "Could not obtain file stats",	// FileNoStats
      "Could not creat file",		// FileNoMake
      "Could not delete file",		// FileNoDel
      "Could not open file",		// FileNoOpen
      "Could not close file",		// FileNoClose
      "Could not read from file",	// FileNoRead
      "Could not write to file",	// FileNoWrite
      "Could not position file",	// FileNoSeek
   };

/************************************************************************/
/* Function	File()							*/
/*									*/
/* Purpose	This is the default constructor for a File object. This	*/
/*		constructor simply prepares the object to report errors	*/
/*		if it needs to.						*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

File::File()
   {
      itsmodulename	= "File";
      itsmaxerror	= FileMaxError;
      itserrormessages	= errormessages;
   }

/************************************************************************/
/* Function	~File()							*/
/*									*/
/* Purpose	This is the default destructor for a File object. This	*/
/*		destructor does nothing.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

File::~File()
   {
   }

/************************************************************************/
/* Function	condition Does_File_Exist(const char* filepath)		*/
/*									*/
/* Purpose	This function can be used to determine if a file exists	*/
/*		or not. It will return true if the named file does	*/
/*		exist and false if it does not.				*/
/*									*/
/* Input	'filepath' must contain the path of the file that you	*/
/*		would like to check.					*/
/*									*/
/* Output	This function will return true if the file does exist	*/
/*		and it will return false if it does not exist or if	*/
/*		there was an OS error. If there was an error, it is not	*/
/*		saved and it is not reported.				*/
/************************************************************************/

condition File::Does_File_Exist(const char* filepath)
   {
      condition		result		= false;
      struct stat	statbuff;

      if (stat(filepath, &statbuff) == 0)
         {
	    /* We have the stats now make sure that it is a file.	*/

	    if (S_ISREG(statbuff.st_mode) == 1)
	       {
	          /* The file does exist.				*/

	          result	= true;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	condition Does_File_Exist(const String& filepath)	*/
/*									*/
/* Purpose	This function can be used to determine if a file exists	*/
/*		or not. It will return true if the named file does	*/
/*		exist and false if it does not.				*/
/*									*/
/* Input	'filepath' must contain the path of the file that you	*/
/*		would like to check.					*/
/*									*/
/* Output	This function will return true if the file does exist	*/
/*		and it will return false if it does not exist or if	*/
/*		there was an OS error. If there was an error, it is not	*/
/*		saved and it is not reported.				*/
/************************************************************************/

condition File::Does_File_Exist(const String& filepath)
   {
      return(Does_File_Exist(filepath.Data()));
   }

/************************************************************************/
/* Function	status Create_File(const char* filepath,		*/
/*		   int* filehandle, const int mode)			*/
/*									*/
/* Purpose	This function will create a file. If the file already	*/
/*		exists, it will be truncated. 'filepath' must contain	*/
/*		the path for the new file. The handle that must be used	*/
/*		to access the created file will be returned in the	*/
/*		integer pointed to by 'filehandle'. If the variable	*/
/*		'mode' is included, it must contain the mode to use	*/
/*		when creating the file. If 'mode' is not specified then	*/
/*		the value of FileDefaultMode will be used.		*/
/*									*/
/*		The following constants are defined in file.h for your	*/
/*		use:							*/
/*									*/
/*		Constant	Value	Description			*/
/*									*/
/*		FileModeOwnRWX		0700	Owner read/write/exec.	*/
/*		FileModeOwnRead		0400	Owner can read.		*/
/*		FileModeOwnWrite	0200	Owner can write.	*/
/*		FileModeOwnExec		0100	Owner can execute.	*/
/*		FileModeGrpRWX		0070	Group read/write/exec.	*/
/*		FileModeGrpRead		0040	Group can read.		*/
/*		FileModeGrpWrite	0020	Group can write.	*/
/*		FileModeGrpExec		0010	Group can execute.	*/
/*		FileModeAllRWX		0007	Others read/write/exec.	*/
/*		FileModeAllRead		0004	Others can read.	*/
/*		FileModeAllWrite	0002	Others can write.	*/
/*		FileModeAllExec		0001	Others can execute.	*/
/*									*/
/*		Combine the modes together that you would like the file	*/
/*		to be created with (e.g. FileModeOwnRWX |		*/
/*		FileModeGrpRWX).					*/
/*									*/
/*		NOTE: Depending on the 'umask' value, the file may be	*/
/*		created with a different mode than you specify. (See	*/
/*		Dir::Get_Linux_Mask()).					*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the path for the file that is to be created.	*/
/*		This functions expects the variable 'filehandle' to	*/
/*		point to an integer. The filehandle for the created	*/
/*		file will be stored at this integer location. Finally,	*/
/*		If the variable 'mode' is included, it must contain the	*/
/*		permissions to use when creating the file. If 'mode'	*/
/*		is not included then the default value of		*/
/*		FileDefaultMode will be used.				*/
/*									*/
/* Output	This function will return OK if it was successful and	*/
/*		the integer pointed to by 'filehandle' will contain	*/
/*		the filehandle to use when accessing the file. If there	*/
/*		were any errors, this function will return ERROR and	*/
/*		the integer pointed to by 'filehandle' will contain a	*/
/*		-1. You can use File::Report_Error() to report any	*/
/*		errors by this function.				*/
/************************************************************************/

status File::Create_File(const char* filepath, int* filehandle,
   const int mode)
   {
      status		result		= OK;

      *filehandle	= creat(filepath, mode);
      if (*filehandle == -1)
         {
	    /* Could not create file.					*/

	    itsoserror	= errno;
	    itserror	= FileNoMake;
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Create_File(const String& filepath,		*/
/*		   int* filehandle, const int mode)			*/
/*									*/
/* Purpose	This function will create a file. If the file already	*/
/*		exists, it will be truncated. 'filepath' must contain	*/
/*		the path for the new file. The handle that must be used	*/
/*		to access the created file will be returned in the	*/
/*		integer pointed to by 'filehandle'. If the variable	*/
/*		'mode' is included, it must contain the mode to use	*/
/*		when creating the file. If 'mode' is not specified then	*/
/*		the value of FileDefaultMode will be used.		*/
/*									*/
/*		The following constants are defined in file.h for your	*/
/*		use:							*/
/*									*/
/*		Constant	Value	Description			*/
/*									*/
/*		FileModeOwnRWX		0700	Owner read/write/exec.	*/
/*		FileModeOwnRead		0400	Owner can read.		*/
/*		FileModeOwnWrite	0200	Owner can write.	*/
/*		FileModeOwnExec		0100	Owner can execute.	*/
/*		FileModeGrpRWX		0070	Group read/write/exec.	*/
/*		FileModeGrpRead		0040	Group can read.		*/
/*		FileModeGrpWrite	0020	Group can write.	*/
/*		FileModeGrpExec		0010	Group can execute.	*/
/*		FileModeAllRWX		0007	Others read/write/exec.	*/
/*		FileModeAllRead		0004	Others can read.	*/
/*		FileModeAllWrite	0002	Others can write.	*/
/*		FileModeAllExec		0001	Others can execute.	*/
/*									*/
/*		Combine the modes together that you would like the file	*/
/*		to be created with (e.g. FileModeOwnRWX |		*/
/*		FileModeGrpRWX).					*/
/*									*/
/*		NOTE: Depending on the 'umask' value, the file may be	*/
/*		created with a different mode than you specify. (See	*/
/*		Dir::Get_Linux_Mask()).					*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the path for the file that is to be created.	*/
/*		This functions expects the variable 'filehandle' to	*/
/*		point to an integer. The filehandle for the created	*/
/*		file will be stored at this integer location. Finally,	*/
/*		If the variable 'mode' is included, it must contain the	*/
/*		permissions to use when creating the file. If 'mode'	*/
/*		is not included then the default value of		*/
/*		FileDefaultMode will be used.				*/
/*									*/
/* Output	This function will return OK if it was successful and	*/
/*		the integer pointed to by 'filehandle' will contain	*/
/*		the filehandle to use when accessing the file. If there	*/
/*		were any errors, this function will return ERROR and	*/
/*		the integer pointed to by 'filehandle' will contain a	*/
/*		-1. You can use File::Report_Error() to report any	*/
/*		errors by this function.				*/
/************************************************************************/

status File::Create_File(const String& filepath, int* filehandle,
   const int mode)
   {
      return(Create_File(filepath.Data(), filehandle, mode));
   }

/************************************************************************/
/* Function	status Remove_File(const char* filepath)		*/
/*									*/
/* Purpose	This function will remove a file. The variable		*/
/*		'filepath' must contain the path of the file to be	*/
/*		deleted.						*/
/*									*/
/* Input	The variable 'filepath' must contain the path of the	*/
/*		file that is to be removed. This can be a relative path	*/
/*		from the current directory; it can be a full path; or	*/
/*		it can be the name of a file in the current directory.	*/
/*									*/
/* Output	This function will return OK if it removed the file. It	*/
/*		will return ERROR if an OS error occurred. You can use	*/
/*		File::Report_Error() to report any errors by this	*/
/*		function.						*/
/************************************************************************/

status File::Remove_File(const char* filepath)
   {
      status		result		= OK;

      if (unlink(filepath) == -1)
         {
	    /* Could not delete file.					*/

	    itsoserror	= errno;
	    itserror	= FileNoDel;
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_File(const String& filepath)		*/
/*									*/
/* Purpose	This function will remove a file. The variable		*/
/*		'filepath' must contain the path of the file to be	*/
/*		deleted.						*/
/*									*/
/* Input	The variable 'filepath' must contain the path of the	*/
/*		file that is to be removed. This can be a relative path	*/
/*		from the current directory; it can be a full path; or	*/
/*		it can be the name of a file in the current directory.	*/
/*									*/
/* Output	This function will return OK if it removed the file. It	*/
/*		will return ERROR if an OS error occurred. You can use	*/
/*		File::Report_Error() to report any errors by this	*/
/*		function.						*/
/************************************************************************/

status File::Remove_File(const String& filepath)
   {
      return(Remove_File(filepath.Data()));
   }

/************************************************************************/
/* Function	status Open_File(const char* filepath,			*/
/*		   int* filehandle, const int flag)			*/
/*									*/
/* Purpose	This function will open a file that already exists. The	*/
/*		variable 'filepath' must contain the path for the file	*/
/*		that is to be opened. The handle for the opened file	*/
/*		will be stored in the integer that is pointed to by the	*/
/*		variable 'filehandle'. Finally, If the variable 'flag'	*/
/*		is passed to this function then it must contain the	*/
/*		read/write access to use when the file is opened. If	*/
/*		the variable 'flag' is not passed to this function then	*/
/*		the value of FileDefaultFlag is used.			*/
/*									*/
/*		The following constants are defined in file.h for you	*/
/*		to use:							*/
/*									*/
/*		Constant		Description			*/
/*									*/
/*		FileFlagRead		Open file in read only mode.	*/
/*		FileFlagWrite		Open file in write only mode.	*/
/*		FileFlagReadWrite	Open file in read/write mode.	*/
/*									*/
/* Input	This function expects the variable 'filepath' to	*/
/*		contain the path for the file that is to be opened.	*/
/*		The path can be a full path; it can be a relative path	*/
/*		from the current directory; or it can be the name of a	*/
/*		file in the current directory. The variable		*/
/*		'filehandle' must point to an integer. The handle that	*/
/*		is used to access the opened file will be stored in the	*/
/*		integer pointed to by 'filehandle'. If the variable	*/
/*		'flag' is passed to this function then it must contain	*/
/*		the read/write access to use when opening the file. If	*/
/*		the variable 'flag' is not passed to this function then	*/
/*		the value of FileDefaultFlag is used.			*/
/*									*/
/* Output	This function will return OK if it opened the file and	*/
/*		the integer pointed to by 'filehandle' will contain the	*/
/*		handle to use when accessing the opened file. This	*/
/*		function will return ERROR if there was an OS error.	*/
/*		You can use File::Report_Error() to report any errors	*/
/*		by this function.					*/
/************************************************************************/

status File::Open_File(const char* filepath, int* filehandle,
   const int flag)
   {
      status		result		= OK;

      *filehandle	= open(filepath, flag);
      if (*filehandle == -1)
         {
	    /* Could not open file.					*/

	    itsoserror	= errno;
	    itserror	= FileNoOpen;
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Open_File(const String& filepath,		*/
/*		   int* filehandle, const int flag)			*/
/*									*/
/* Purpose	This function will open a file that already exists. The	*/
/*		variable 'filepath' must contain the path for the file	*/
/*		that is to be opened. The handle for the opened file	*/
/*		will be stored in the integer that is pointed to by the	*/
/*		variable 'filehandle'. Finally, If the variable 'flag'	*/
/*		is passed to this function then it must contain the	*/
/*		read/write access to use when the file is opened. If	*/
/*		the variable 'flag' is not passed to this function then	*/
/*		the value of FileDefaultFlag is used.			*/
/*									*/
/*		The following constants are defined in file.h for you	*/
/*		to use:							*/
/*									*/
/*		Constant		Description			*/
/*									*/
/*		FileFlagRead		Open file in read only mode.	*/
/*		FileFlagWrite		Open file in write only mode.	*/
/*		FileFlagReadWrite	Open file in read/write mode.	*/
/*									*/
/* Input	This function expects the variable 'filepath' to	*/
/*		contain the path for the file that is to be opened.	*/
/*		The path can be a full path; it can be a relative path	*/
/*		from the current directory; or it can be the name of a	*/
/*		file in the current directory. The variable		*/
/*		'filehandle' must point to an integer. The handle that	*/
/*		is used to access the opened file will be stored in the	*/
/*		integer pointed to by 'filehandle'. If the variable	*/
/*		'flag' is passed to this function then it must contain	*/
/*		the read/write access to use when opening the file. If	*/
/*		the variable 'flag' is not passed to this function then	*/
/*		the value of FileDefaultFlag is used.			*/
/*									*/
/* Output	This function will return OK if it opened the file and	*/
/*		the integer pointed to by 'filehandle' will contain the	*/
/*		handle to use when accessing the opened file. This	*/
/*		function will return ERROR if there was an OS error.	*/
/*		You can use File::Report_Error() to report any errors	*/
/*		by this function.					*/
/************************************************************************/

status File::Open_File(const String& filepath, int* filehandle,
   const int flag)
   {
      return(Open_File(filepath.Data(), filehandle, flag));
   }

/************************************************************************/
/* Function	status Close_File(const int filehandle)			*/
/*									*/
/* Purpose	This function will close a file that was previously	*/
/*		opened with File::Open_File() or File::Create_File().	*/
/*		The variable 'filehandle' must contain the handle that	*/
/*		was returned when the file was opened or created.	*/
/*									*/
/* Input	This function expects the variable 'filehandle' to	*/
/*		contain the handle of the file to be closed.		*/
/*									*/
/* Output	This function will return OK if it closed the file.	*/
/*		This function will return ERROR if there were any	*/
/*		errors closing the file. You can use			*/
/*		File::Report_Error() to report any errors by this	*/
/*		function.						*/
/************************************************************************/

status File::Close_File(const int filehandle)
   {
      status		result		= OK;

      if (close(filehandle) == -1)
         {
	    /* Could not close file.					*/

	    itsoserror	= errno;
	    itserror	= FileNoClose;
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Read_File(const int filehandle, void* buffer,	*/
/*		   const int bytestoread)				*/
/*									*/
/* Purpose	This function will read in data from a file. The	*/
/*		variable 'filehandle' must contain a valid file		*/
/*		descriptor for an open file. The variable 'buffer'	*/
/*		points to the area of memory that the data is to be	*/
/*		written to. Finally, the variable 'bytestoread' should	*/
/*		contain the number of bytes that are to be read in.	*/
/*									*/
/* Input	This function expects the variable 'filehandle' to	*/
/*		contain a valid file descriptor for an open file. This	*/
/*		is the value that was returned when the file was opened	*/
/*		or created. The variable 'buffer' must point to the	*/
/*		area of memory where the data is to be saved. Finally,	*/
/*		the variable 'bytestoread' must contain the number of	*/
/*		bytes that are to be read in from the file.		*/
/*									*/
/* Output	This function will return OK if successfully read from	*/
/*		the open file. If there were any errors, this function	*/
/*		will return ERROR. You can use File::Report_Error() to	*/
/*		report any errors by this function.			*/
/************************************************************************/

status File::Read_File(const int filehandle, void* buffer,
   const int bytestoread)
   {
      status		result		= OK;

      if (read(filehandle, buffer, bytestoread) == -1)
         {
	    /* Could not read from file.				*/

	    itsoserror	= errno;
	    itserror	= FileNoRead;
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_File(const int filehandle, void* buffer,	*/
/*		   const int bytestowrite)				*/
/*									*/
/* Purpose	This function will write a block of data to a file.	*/
/*		The variable 'filehandle' must contain a valid file	*/
/*		descriptor for an open file. The variable 'buffer' must	*/
/*		point to the block of data that is to be written to the	*/
/*		file. Finally, the variable 'bytestowrite' must contain	*/
/*		the number of bytes to write to the file.		*/
/*									*/
/* Input	This function expects the variable 'filehandle' to	*/
/*		contain a valid file descriptor for an open file. This	*/
/*		is the value that was returned when the file was opened	*/
/*		or created. The variable 'buffer' must point to the	*/
/*		area of memory that is to be written to the file.	*/
/*		Finally, the variable 'bytestowrite' must contain the	*/
/*		number of bytes that are to be written to the file.	*/
/*									*/
/* Output	This function will return OK if it successfully wrote	*/
/*		to the open file. If there were any errors, this	*/
/*		function will return ERROR. You can use			*/
/*		File::Report_Error() to report any errors by this	*/
/*		function.						*/
/************************************************************************/

status File::Write_File(const int filehandle, void* buffer,
   const int bytestowrite)
   {
      status		result		= OK;

      if (write(filehandle, buffer, bytestowrite) == -1)
         {
	    /* Could not write to file.					*/

	    itsoserror	= errno;
	    itserror	= FileNoWrite;
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Position_File(const int filehandle,		*/
/*		   const unsigned long int position)			*/
/*									*/
/* Purpose	This function will position a file's pointer prior to	*/
/*		a read or write operation. The variable 'filehandle'	*/
/*		must contain a valid file descriptor for the open file.	*/
/*		The variable 'position' must contain the number of	*/
/*		bytes, from the beginning of the file, that you would	*/
/*		like to position the pointer to.			*/
/*									*/
/* Input	This function expects 'filehandle' to contain a valid	*/
/*		file descriptor for the file that is to be positioned.	*/
/*		This is the value that was returned when the file was	*/
/*		opened or created. The variable 'position' must contain	*/
/*		the number of bytes, from the beginning of the file,	*/
/*		where the file's pointer is to be positioned to.	*/
/*									*/
/* Output	This function will return OK if it successfully		*/
/*		positioned the file's pointer. If there were any	*/
/*		errors, this function will return ERROR. You can use	*/
/*		File::Report_Error() to report any errors by this	*/
/*		function.						*/
/************************************************************************/

status File::Position_File(const int filehandle,
   const FilePointer position)
   {
      status		result		= OK;

      if (lseek(filehandle, position, SEEK_SET) == -1)
         {
	    /* Could not position file.					*/

	    itsoserror	= errno;
	    itserror	= FileNoSeek;
	    result	= ERROR;
	 }
      return(result);
   }
