/************************************************************************/
/* File		string.cpp						*/
/*									*/
/* Purpose	This C++ program file implements the String class. The	*/
/*		String class manages null-terminated strings. The	*/
/*		String class is defined in string.h.			*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	0.0.0 (Prototype)					*/
/*									*/
/* Date		Sunday, April 28, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <string.h>			// Standard C string functions.
#include <ctype.h>			// Standard C char functions.
#include "string++.h"			// String class.

/************************************************************************/
/* Function	String()						*/
/*									*/
/* Purpose	This is the default constructor for a String object.	*/
/*		The default constructor creates a String object with a	*/
/*		zero length string.					*/
/*									*/
/* Input	This function does not accept any input.		*/
/*									*/
/* Output	This function initializes private object data.		*/
/************************************************************************/

String::String()
   {
      itslength		= 0;
      itsdata		= new char[1];
      *itsdata		= '\0';
   }

/************************************************************************/
/* Function	String(const char* data)				*/
/*									*/
/* Purpose	This String object constructor will initialize a String	*/
/*		object with the null-terminated string that is pointed	*/
/*		to by the variable 'data'.		 		*/
/*									*/
/* Input	This function accepts a pointer to a character array	*/
/*		containing a null-terminated string.			*/
/*									*/
/* Output	This function will intialize a String object with the	*/
/*		null-terminated string pointed to by 'data'.		*/
/************************************************************************/

String::String(const char* data)
   {
      itslength		= strlen(data);
      itsdata		= new char[itslength + 1];
      strcpy(itsdata, data);
   }

/************************************************************************/
/* Function	String(const String& object)				*/
/*									*/
/* Purpose	This is the copy constructor for a String object. The	*/
/*		copy constructor can be used to create a deep copy of	*/
/*		a String object.					*/
/*									*/
/* Input	This function accepts a pointer to a String object in	*/
/*		the variable 'object'.					*/
/*									*/
/* Output	This function will make a deep copy of the String	*/
/*		object that is referenced by the variable 'object'.	*/
/************************************************************************/

String::String(const String& object)
   {
      itslength		= object.itslength;
      itsdata		= new char[itslength + 1];
      strcpy(itsdata, object.itsdata);
   }

/************************************************************************/
/* Function	~String()						*/
/*									*/
/* Purpose	This is the default destructor for a String object. It	*/
/*		will delete the allocated memory holding the object's	*/
/*		data.							*/
/*									*/
/* Input	This function accepts no input.				*/
/*									*/
/* Output	This function operates on private object data.		*/
/************************************************************************/

String::~String()
   {
      delete itsdata;
   }

/************************************************************************/
/* Function	String& operator=(const char* data)			*/
/*									*/
/* Purpose	This String class member function overloads the equal	*/
/*		('=') operator. Overloading the equal operator allows a	*/
/*		String class object to be assigned a null-terminated	*/
/*		string. The original contents of the object will be	*/
/*		lost as the object's original data will be freed.	*/
/*									*/
/* Input	This function accepts a pointer to a character array	*/
/*		containing a null-terminated string in the variable	*/
/*		'data'.							*/
/*									*/
/* Output	This function will replace the original contents of the	*/
/*		String object with the null-terminated string pointed	*/
/*		to by the variable 'data'.				*/
/************************************************************************/

String& String::operator=(const char* data)
   {
      delete itsdata;
      itslength		= strlen(data);
      itsdata		= new char[itslength + 1];
      strcpy(itsdata, data);
      return(*this);
   }

/************************************************************************/
/* Function	String& operator=(const String& object)			*/
/*									*/
/* Purpose	This String class member function overloads the equal	*/
/*		('=') operator. Overloading the equal operator allows	*/
/*		one String class object to be assigned to another	*/
/*		String class object. The original contents of the	*/
/*		object will be lost.					*/
/*									*/
/* Input	This function accepts a reference to a String object in	*/
/*		the variable 'object'.					*/
/*									*/
/* Output	This function will replace the original contents of	*/
/*		this String object with the String object that is	*/
/*		referenced by the variable 'object'.			*/
/************************************************************************/

String& String::operator=(const String& object)
   {
      /* If the object pointer and the pointer 'object' are not equal.	*/

      if (this != &object)
         {
            /* This is not a copy to itself.				*/

            delete itsdata;
            itslength	= strlen(object.itsdata);
            itsdata	= new char[itslength + 1];
            strcpy(itsdata, object.itsdata);
         }
      return(*this);
   }

/************************************************************************/
/* Function	String& operator+=(const char* data)			*/
/*									*/
/* Purpose	This String class member function overloads the plus	*/
/*		equal ('+=') operator. Overloading the += operator	*/
/*		allows a character string pointed to by the variable	*/
/*		'data' to be added to a String object.			*/
/*									*/
/* Input	This function accepts a char array pointer containing a	*/
/*		null-terminated string in the variable 'data'.		*/
/*									*/
/* Output	This function will add a character string pointed to by	*/
/*		the variable 'data' to this String object.		*/
/************************************************************************/

String& String::operator+=(const char* data)
   {
      String		tempstring(data);

      return(operator+=(tempstring));
   }

/************************************************************************/
/* Function	String& operator+=(const String& object)		*/
/*									*/
/* Purpose	This String class member function overloads the plus	*/
/*		equal ('+=') operator. Overloading the += operator	*/
/*		allows two String class objects to be added together.	*/
/*		The result is a concatenation of the two object's	*/
/*		strings.						*/
/*									*/
/* Input	This function accepts a reference to a String object in	*/
/*		the variable 'object'.					*/
/*									*/
/* Output	This function will concatenate this String objects with	*/
/*		the String object that is referenced by the variable	*/
/*		'object'.						*/
/************************************************************************/

String& String::operator+=(const String& object)
   {
      char*		tempstring;		// Temp string pointer.

      /* See if the two objects are the same. If the two objects are	*/
      /* the same then we need to add the string to itself.		*/

      if (this == &object)
         {
            /* At this point, we need to add the string to itself. See	*/
            /* if the String object actually contains a string. If the	*/
            /* String object does not contain any data then we are	*/
            /* done.							*/

            if (itslength > 0)
               {
                  /* Add the object's string to itself and save the	*/
                  /* result as the object's new string.			*/

                  itslength	= itslength + itslength;
                  tempstring	= new char[itslength + itslength + 1];
                  strcpy(tempstring, itsdata);
                  strcat(tempstring, itsdata);
                  delete itsdata;
                  itsdata	= tempstring;
               }
         }
      else
         {
            /* The two strings are different so we need to add one	*/
            /* string to another if the string is not zero length.	*/

            if (object.itslength > 0)
               {
                  /* There is a string to add so add the two strings	*/
                  /* together.						*/

                  itslength	= itslength + object.itslength;
                  tempstring	= new char[itslength + 1];
                  strcpy(tempstring, itsdata);
                  strcat(tempstring, object.itsdata);
                  delete itsdata;
                  itsdata		= tempstring;
               }
         }
      return(*this);
   }

/************************************************************************/
/* Function	int Length(void)					*/
/*									*/
/* Purpose	This String class member function will return the	*/
/*		length of the string.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function returns the length of the string.		*/
/************************************************************************/

int String::Length(void) const
   {
      return(itslength);
   }

/************************************************************************/
/* Function	char* Data(void)					*/
/*									*/
/* Purpose	This String class member function will return a pointer	*/
/*		to the String object's character array containing the	*/
/*		null-terminated string.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a pointer to the character	*/
/*		array containing the null-terminated string.		*/
/************************************************************************/

char* String::Data(void) const
   {
      return(itsdata);
   }

int String::Search_For_Character(const char character) const
   {
      int		result;
      char*		found;

      found		= strchr(itsdata, character);
      if (found == (char*)0)
         {
	    result	= -1;
	 }
      else
         {
	    result	= found - itsdata;
	 }
      return(result);
   }

/************************************************************************/
/* Function	relation Compare(const String& object)			*/
/*									*/
/* Purpose	This String class member function will compare two	*/
/*		String class objects. This function will return		*/
/*		LESS, MORE, or EQUAL.					*/
/*									*/
/* Input	This function accepts a reference to a String class	*/
/*		object in the variable 'object'.			*/
/*									*/
/* Output	This function will return LESS, MORE, or EQUAL.		*/
/************************************************************************/

relation String::Compare(const String& object) const
   {
      int		comparison;
      relation		result;

      comparison	= strcmp(itsdata, object.itsdata);
      if (comparison < 0)
         {
	    result	= LESS;
	 }
      else if (comparison > 0)
         {
	    result	= MORE;
	 }
      else
         {
	    result	= EQUAL;
	 }
      return(result);
   }

/************************************************************************/
/* Function	relation Compare(const char* data)			*/
/*									*/
/* Purpose	This String class member function will compare a String	*/
/*		class object and a character array. This function will	*/
/*		return 	LESS, MORE, or EQUAL.				*/
/*									*/
/* Input	This function accepts a pointer to a character array in	*/
/*		the variable 'data'.					*/
/*									*/
/* Output	This function will return LESS, MORE, or EQUAL.		*/
/************************************************************************/

relation String::Compare(const char* data) const
   {
      int		comparison;
      relation		result;

      comparison	= strcmp(itsdata, data);
      if (comparison < 0)
         {
	    result	= LESS;
	 }
      else if (comparison > 0)
         {
	    result	= MORE;
	 }
      else
         {
	    result	= EQUAL;
	 }
      return(result);
   }

/************************************************************************/
/* Function	relation Case_Compare(const String& object)		*/
/*									*/
/* Purpose	This String class member function will compare two	*/
/*		String class objects, ignoring case. This function will	*/
/*		return LESS, MORE, or EQUAL.				*/
/*									*/
/* Input	This function accepts a reference to a String class	*/
/*		object in the variable 'object'.			*/
/*									*/
/* Output	This function will return LESS, MORE, or EQUAL.		*/
/************************************************************************/

relation String::Case_Compare(const String& object) const
   {
      int		comparison;
      relation		result;

      comparison		= strcasecmp(itsdata, object.itsdata);
      if (comparison < 0)
         {
	    result	= LESS;
	 }
      else if (comparison > 0)
         {
	    result	= MORE;
	 }
      else
         {
	    result	= EQUAL;
	 }
      return(result);
   }

/************************************************************************/
/* Function	relation Case_Compare(const char* data)			*/
/*									*/
/* Purpose	This String class member function will compare a String	*/
/*		class object and a character array, ignoring case. This	*/
/*		function will return LESS, MORE, or EQUAL.		*/
/*									*/
/* Input	This function accepts a pointer to a character array in	*/
/*		the variable 'data'.					*/
/*									*/
/* Output	This function will return LESS, MORE, or EQUAL.		*/
/************************************************************************/

relation String::Case_Compare(const char* data) const
   {
      int		comparison;
      relation		result;

      comparison	= strcasecmp(itsdata, data);
      if (comparison < 0)
         {
	    result	= LESS;
	 }
      else if (comparison > 0)
         {
	    result	= MORE;
	 }
      else
         {
	    result	= EQUAL;
	 }
      return(result);
   }

/************************************************************************/
/* Function	void Convert_To_Upper_Case(void)			*/
/*									*/
/* Purpose	This String class member function will convert all of	*/
/*		the characters in the object's string to upper case.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The object's data will be upper case.			*/
/************************************************************************/

void String::Convert_To_Upper_Case(void)
   {
      register int	index;
      char*		charpointer;

      for (index = 0; index < itslength; index++)
         {
	    charpointer		= &itsdata[index];
	    *charpointer	= toupper(*charpointer);
	 }
   }

/************************************************************************/
/* Function	void Convert_To_Lower_Case(void)			*/
/*									*/
/* Purpose	This String class member function will convert all of	*/
/*		the characters in the object's string to lower case.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The object's data will all be lower case.		*/
/************************************************************************/

void String::Convert_To_Lower_Case(void)
   {
      register int	index;
      char*		charpointer;

      for (index = 0; index < itslength; index++)
         {
	    charpointer		= &itsdata[index];
	    *charpointer	= tolower(*charpointer);
	 }
   }
