/************************************************************************/
/* File		webpublishutility.cpp					*/
/*									*/
/* Purpose	This C++ program file contains private WebPublish	*/
/*		member functions. The private WebPublish member		*/
/*		functions that are contained in this file include the	*/
/*		functions that support the WebPublish class.		*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	0.1.0 (Prototype)					*/
/*									*/
/* Date		Sunday, July 7, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include "ctype.h"			// toupper.
#include "webpublish.h"			// WebPublish class

static char*	copy_message[] =
   {
      "WebPublish -- Version 0.1.0 -- A website publishing utility for "
      "GNU/Linux.",
      "Copyright (C) 2002 by Padre Software [http://PadreSoftware.com].",
      "Released to the public domain by Padre Software under "
      "the GNU license.",
      "ftplib Release 3.1 6/xx/98, copyright 1996, 1997, 1998 "
      "Thomas Pfau.",                                                 
      "",
      0
   };
   
static char*	help_message[] =
   {
      "Help:",
      "WebPublish transfers a local copy of a website to a remote FTP",
      "server using an account. An account contains the information that",
      "WebPublish uses to transfer a website to an FTP server.",
      "",
      "Usage: webpublish COMMAND [[DATA]...] [[FLAG]...] [[FILE]...]",
      "",
      "COMMAND is always required on the command line (except when the",
      "--help, and/or --version, FLAG is included). Only one COMMAND can be",
      "included on each command line. The COMMAND tells WebPublish what",
      "to do. The valid COMMANDs are:",
      "",
      "COMMAND                  Description",
      "",
      "--publish                Publish files and directories to server.",
      "--synchronize            Remove files and directories from server.",
      "--add                    Add a new database record.",
      "--change                 Change an existing database record.",
      "--remove                 Remove an existing database record.",
      "--list                   List database records.",
      "",
      "DATA command line options specify the information that WebPublish",
      "needs to perform the requested COMMAND. Most COMMANDs require",
      "more than one DATA option to be included on the command line.",
      "Each DATA option requires an argument. The DATA options are:",
      "",
      "-a or --account ACCT     ACCT specifies the name of the account to",
      "                         operate on.",
      "-w or --website DIR      DIR specifies the website (the local",
      "                         directory) to operate on.",
      "-s or --server FTP       FTP specifies the name of the FTP server",
      "                         to connect with.",
      "-u or --user USER        USER specifies the user name to use when",
      "                         logging into the FTP server.",
      "-p or --password PASS    PASS specifies the password to use when",
      "                         logging into the FTP server.",
      "-d or --directory DIR    DIR specifies a directory path.",
      "-e or --extension EXT    EXT specifies a file extension.",
      "-P or --path PATH        PATH specifies a file or directory path.",
      "-D or --depth NUM        NUM specifies the number of directory",
      "                         levels to traverse.",
      "-F or -file FILE         FILE specifies the database file to",
      "                         operate on. The value of FILE can be",
      "                         either \"account\", \"shell\", \"synch\", ",
      "                         \"publish\", or \"mode\" (case is not",
      "                         compared).",
      "",
      "The WebPublish FLAGs modify the behavior of WebPublish. Multiple",
      "FLAGs are allowed on the same command line. The valid FLAGs are:",
      "",
      "-h or --help             Write this help message to stdout.",
      "-v or --version          Write a version string to stdout.",
      "-V or --verbose          Write descriptive messages to stdout.",
      "-f or --force            Force files to be transferred.",
      "--prompt                 Prompt the user before removing anything",
      "                         from the server.",
      "",
      "Specific FILEs can be published by listing each file on the",
      "command line. The file path can be a full path or a relative",
      "path. Wildcards can be used with a full path.",
      "",
      "The following example will create a new account called \"example\".",
      "The account will publish the website that is located at",
      "\"/home/mydir/example\" to the ftp server \"ftp.example.com\".",
      "",
      "webpublish --add -F account -a example -w /home/mydir/example",
      "-s ftp.example.com -u username -p password",
      "",
      "The following example will publish the account \"example\".",
      "",
      "webpublish --publish -a example -V",
      "",
      "WebPublish home webpage: http://www.gnu.org/software/webpublish",
      "Report bugs to <bug-webpublish@gnu.org>.",
      "",
      0
   };
			
static String		filenames[]	=
			   {
			      "accounts",
			      "shells",
			      "xsynch",
			      "xpublish",
			      "modes"
			   };
static int		sizes[]		= { 6, 4, 3, 3, 2 };
static DataElementType	elements[]	=
   {
      // accounts

      DataElementString,	// Accnt name.
      DataElementString,	// Website path.
      DataElementString,	// Server domain.
      DataElementString,	// User name.
      DataElementString,	// Password.
      DataElementString,	// Directory.

      // shells

      DataElementString,	// Accnt name + extension
      DataElementString,	// Accnt name
      DataElementString,	// Extension
      DataElementString,	// Shell program.

      // xsynch

      DataElementString,	// Accnt name + x synch file or dir.
      DataElementString,	// Accnt name
      DataElementString,	// Local directory path

      // xpublish

      DataElementString,	// Accnt name + x publish file or dir.
      DataElementString,	// Accnt name
      DataElementString,	// Local directory path

      // modes

      DataElementString,	// Extension
      DataElementNumber,	// Mode
   };

/************************************************************************/
/* Function	initialize_webpublish_directory(void)			*/
/*									*/
/* Purpose	This function is responsible for initializing the	*/
/*		WebPublish object's working directory. If the working	*/
/*		directory does not exist in the user's home directory	*/
/*		then this function will create it.			*/
/*									*/
/* Input	This function uses environment variables to perform its	*/
/*		task.							*/
/*									*/
/* Output	This function will return 'itserror'. If 'itserror'	*/
/*		equals WEBPUBERR_NOERROR then the function was		*/
/*		successful.						*/
/************************************************************************/

status WebPublish::initialize_webpublish_database(void)
   {
      status		result		= OK;
      Dir		dir;
      condition		testresult;

      if (dir.Get_Home_Directory(itshomedirectory) == ERROR)
         {
            /* Could not retrieve the user's home directory.		*/

            itserror	= WebPublishNoHome;
	    result	= ERROR;
	    Report_Error();
         }
      else
         {
            /* Use home directory to initialize itsworkdirectory.	*/

	    if (itshomedirectory.Data()[itshomedirectory.Length()
	       - 1] != '/')
	       {
	          itshomedirectory	+= "/";
	       }
	    itsworkdirectory	= itshomedirectory;
	    itsworkdirectory	+= WebPublishDir;
	    itstempfile		= itsworkdirectory;
	    itstempfile		+= WebPublishTempFile;
	    testresult	= dir.Does_Directory_Exist(itsworkdirectory);

	    /* Initialize the DataBase object. This will create the	*/
	    /* directory if it does not exist.				*/

	    itsdatabase.Prepare_DataBase(itsworkdirectory,
	       WebPublishFiles, filenames, sizes, elements,
	       WebPublishDirMode);

            /* Did the directory exist?					*/

	    if (testresult == false)
	       {
	          /* The directory did not exist so create the DataBase	*/
		  /* files in the directory that was created when the	*/
		  /* DataBase object was prepared for use.		*/

	          if (itsdatabase.Create_DataBase_Files() == ERROR)
		     {
		        /* Could not create WebPublish database.	*/

			itserrorinfo	= "Attempted to create "
					  "database files in ";
			itserrorinfo	+= itsworkdirectory;
			itserror	= WebPublishDataMake;
			result		= ERROR;
			Report_Error();
		     }
	       }
            if (result == OK)
               {
	          /* The working directory exists so open the database	*/
		  /* files and we are done.				*/

                  if (itsdatabase.Open_DataBase_Files() == ERROR)
                     {
                        /* Could not open WebPublish database files.	*/

			itserrorinfo	= "Attempted to open database "
					  " files in ";
			itserrorinfo	+= itsworkdirectory;
			itserror	= WebPublishDataOpen;
			result		= ERROR;
			Report_Error();
                     }
		  else
		     {
		        itsdbopenflag	= true;
		     }
               }
         }
      return(result);
   }

char WebPublish::response(void)
   {
      char		answer;

      do
         {
            std::cin >> answer;
	 }
      while (answer != 'y' && answer != 'Y' && answer != 'n' &&
         answer != 'N' && answer != 'a' && answer != 'A');
      return(toupper(answer));
   }

/************************************************************************/
/* Function	status does_record_exist(const int file,		*/
/*		   const String& key, condition& value)			*/
/*									*/
/* Purpose	This function can be used to determine if a record	*/
/*		already exists in a database file. If the record does	*/
/*		exist in the file then the variable 'value' will be	*/
/*		set to true. 'value' will be set to false if the record	*/
/*		is not in the database file.				*/
/*									*/
/* Input	This function expects the variable 'file' to contain	*/
/*		index for the DataFile that is to be searched. The	*/
/*		variable 'key' must contain the key to search for in	*/
/*		the database file.					*/
/*									*/
/* Output	If this function did not encounter any errors then this	*/
/*		function will return OK and the variable 'value' will	*/
/*		be set to true if the record exists in the database	*/
/*		file. The variable 'value' will be set to false if the	*/
/*		record does not exist in the database file. If this	*/
/*		function encounters an error then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status WebPublish::does_record_exist(const int file, const String& key,
   condition& value)
   {
      status		result		= OK;
      DataRecord	record;
      String		recordkey;

      value		= false;
      itsdatabase.Format_Record(file, record);
      record.Set_Data(0, key);
      if (itsdatabase.Get_Record(file, record) == ERROR)
         {
	    /* Could not search for record.				*/

	    itserrorinfo	= "Attempted to see if record exists";
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    record.Get_Key(recordkey);
	    if (recordkey.Length() != 0)
	       {
	          /* Record does exist in database file.		*/

	          value		= true;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	print_webpublish_version(void)				*/
/*									*/
/* Purpose	This function is responsible for writing the WebPublish	*/
/*		version message to stdout.				*/
/*									*/
/* Input	This function uses static data to write the version	*/
/*		message to stdout.					*/
/*									*/
/* Output	This function will write a version message to stdout.	*/
/************************************************************************/

void WebPublish::print_webpublish_version(void) const
   {
      std::cout << "Version " << WebPublishVersion << ".\n\n";
   }

/************************************************************************/
/* Function	print_webpublish_help(void)				*/
/*									*/
/* Purpose	This function is responsible for writing the WebPublish	*/
/*		help message to stdout.					*/
/*									*/
/* Input	This function uses static data to write the help	*/
/*		message to stdout.					*/
/*									*/
/* Output	This function will write a help message to stdout.	*/
/************************************************************************/

void WebPublish::print_webpublish_help(void) const
   {
      register int	index;

      for (index = 0; help_message[index] != 0; index++)
         {
            std::cout << help_message[index] << std::endl;
         }

   }

/************************************************************************/
/* Function	print_webpublish_copyright(void)			*/
/*									*/
/* Purpose	This function is responsible for writing the WebPublish	*/
/*		copyright message to stdout.				*/
/*									*/
/* Input	This function uses static data to write the copyright	*/
/*		message to stdout.					*/
/*									*/
/* Output	This function will write a copyright message to stdout.	*/
/************************************************************************/

void WebPublish::print_webpublish_copyright(void) const
   {
      register int	index;

      for (index = 0; copy_message[index] != 0; index++)
         {
            std::cout << copy_message[index] << std::endl;
         }
   }
