//! This crate generates Cranelift-specific assembly code for x64 instructions; see the `README.md`
//! for more information.

pub mod dsl;
mod generate;
pub mod instructions;

use std::path::{Path, PathBuf};
use std::process::Command;

/// Generate the assembler `file` containing the core assembler logic; each of
/// the DSL-defined instructions is emitted into a Rust `enum Inst`.
pub fn generate_rust_assembler<P: AsRef<Path>>(file: P) -> PathBuf {
    let mut fmt = generate::Formatter::new("// Generated by `generate::rust_assembler`", file!(), line!());
    generate(file, &mut fmt, generate::rust_assembler, true)
}

/// Generate a macro containing builder functions for the assembler's ISLE
/// constructors; this uses the `build` module emitted in
/// [`generate_rust_assembler`].
pub fn generate_isle_macro<P: AsRef<Path>>(file: P) -> PathBuf {
    let mut fmt = generate::Formatter::new("// Generated by `generate::isle_macro`", file!(), line!());
    generate(file, &mut fmt, generate::isle_macro, true)
}

/// Generate the ISLE definitions; this provides ISLE glue to access the builder
/// functions from [`generate_isle_macro`].
pub fn generate_isle_definitions<P: AsRef<Path>>(file: P) -> PathBuf {
    let mut fmt = generate::Formatter::new(";; Generated by `generate::isle_definitions`", file!(), line!());
    generate(file, &mut fmt, generate::isle_definitions, false)
}

/// Helper for emitting generated lines into a formatted file.
///
/// # Panics
///
/// This function panics if we cannot update the file.
fn generate<P: AsRef<Path>>(
    file: P,
    fmt: &mut generate::Formatter,
    generator: fn(&mut generate::Formatter, &[dsl::Inst]),
    format_rust: bool,
) -> PathBuf {
    let file = file.as_ref();
    eprintln!("Generating {}", file.display());
    generator(fmt, &instructions::list());
    fmt.write(file).unwrap();
    if format_rust {
        rustfmt(file);
    }
    file.to_path_buf()
}

/// Use the installed `rustfmt` binary to format the generated code; if it
/// fails, skip formatting with a warning.
fn rustfmt(file: &Path) {
    if let Ok(status) = Command::new("rustfmt").arg(file).status() {
        if !status.success() {
            eprintln!("`rustfmt` exited with a non-zero status; skipping formatting of generated files");
        }
    } else {
        eprintln!("`rustfmt` not found; skipping formatting of generated files");
    }
}
