/* doscan - Denial Of Service Capable Auditing of Networks       -*- C++ -*-
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SCAN_UDP_H
#define SCAN_UDP_H

#include "event_queue.h"
#include "results.h"
#include "scan_trigger.h"
#include "subnets.h"

#include <cerrno>
#include <map>
#include <netinet/in.h>
#include <string>
#include <sys/types.h>
#include <sys/socket.h>

class scan_udp_single {
  scan_udp_single(const scan_udp_single&);

public:
  struct handler {
    virtual ~handler();

    // clone() creates a new, independent handler.  Invoked as a
    // factory method.

    virtual handler* clone() const = 0;

    virtual void create_query(ipv4_t, std::string& data) = 0;
    virtual void reply_received(ipv4_t host, unsigned short remote_port,
                                const std::string& data) = 0;
    virtual void timeout(ipv4_t host, ticks_t ticks) = 0;
  };

private:
  event_queue& queue;
  int listen_fd;

  const handler& handler_template;

  class send_trigger : public scan_trigger::handler {
    scan_udp_single& scanner;
  public:
    send_trigger(scan_udp_single&);
    virtual void connect(event_queue&, scan_trigger&, ipv4_t);
    virtual void all_connected();
  };

  send_trigger trigger_handler;
  scan_trigger trigger;

  class listener : public event_queue::fd_handler {
    scan_udp_single& scanner;

  public:
    listener(scan_udp_single&);
    virtual bool on_activity(activity);
    virtual bool on_timeout(ticks_t);
  };

  class sender : public event_queue::handler {
    friend class listener;

    scan_udp_single::handler* the_handler;
    scan_udp_single& scanner;
    ipv4_t host;
    std::string data;

    // retries counts the remaining number of retransmits.
    // Special values are:
    //    0: Return false on next timeout (deallocation) -- timeout
    //   -1: Same, but no timeout (manually triggered termination)

    int retries;

    sockaddr_in address;

    bool do_send();

  public:
    sender(scan_udp_single&, ipv4_t);
    virtual ~sender();

    void stop_sending();

    virtual bool on_timeout(ticks_t);
  };

  // retries_count specifies the number of retries to send.

  unsigned retries;

  // retries_timeout specifies the number of milliseconds before a
  // retry packet is sent.

  unsigned timeout;

  typedef std::map<ipv4_t, sender*> senders_t;
  senders_t senders;

  listener* the_listener;

public:
  scan_udp_single(event_queue&, subnets&, handler&, unsigned maximum,
                  unsigned burst_delay, unsigned burst_size,
                  unsigned retries, unsigned timeout);
};

#endif // SCAN_UDP_H

// arch-tag: 8116f97e-af95-4820-8f96-b7828b1a9341
