------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2006 Dmitriy Anisimkov.                                   --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-containers.adb,v 1.17 2008/05/05 09:10:57 vagul Exp $

with Ada.Strings.Fixed;

with OCI.Thick.Number_Functions;
with OCI.Thick.Date;

package body OCI.Thick.Containers is

   use type SWord;
   use type Ub4;
   use type OCIHandle;

   ------------
   -- Append --
   ------------

   procedure Append (Holder : in out Data_Holder; Item : in String) is
      use C;
   begin
      Ada.Strings.Unbounded.Append (Holder.Str, Item);
   end Append;

   procedure Append
     (Container : in out Data_Container;
      Item      : in     String;
      Position  : in     Positive;
      Iteration : in     Positive := 1)
   is
      use type Ada.Containers.Count_Type;

      procedure Modify (V : in out Positions.Vector);

      ------------
      -- Modify --
      ------------

      procedure Modify (V : in out Positions.Vector) is
         procedure Modify (D : in out Data_Holder);

         procedure Modify (D : in out Data_Holder) is
         begin
            Append (D, Item);
         end Modify;

      begin
         V.Update_Element (Position, Modify'Access);
      end Modify;

   begin
      Container.Data.Update_Element (Iteration, Modify'Access);
   end Append;

   -------------
   -- C_Slice --
   -------------

   procedure C_Slice
     (Holder : in     Data_Holder;
      Low    : in     Positive;
      Item   :    out C.char_array;
      Last   :    out C.size_t;
      Done   :    out Boolean)
   is
      use type Lib.C.size_t;
      use Ada.Strings.Unbounded;
      High : Positive := Low + Item'Length - 1;
   begin
      Done := High >= Length (Holder.Str);

      if Done then
         High := Length (Holder.Str);
         Last := Lib.C.size_t (High - Low) + Item'First;
      else
         Last := Item'Last;
      end if;

      Item (Item'First .. Last)
        := C.To_C
             (Slice (Holder.Str, Low => Low, High => High),
              Append_Nul => False);
   end C_Slice;

   -----------
   -- Clear --
   -----------

   procedure Clear (Container : in out Data_Container) is
   begin
      Container.Data.Clear;
   end Clear;

   procedure Clear
     (Container : in out Container_Type;
      Position  : in     Positive;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Null_Data,
         Position  => Position,
         Iteration => Iteration);
   end Clear;

   ---------
   -- Get --
   ---------

   procedure Get
     (Container : in     Container_Type;
      Item      :    out Data_Holder;
      Position  : in     Positive;
      Iteration : in     Positive := 1)
   is
      procedure Query (Item : in Positions.Vector);

      procedure Query (Item : in Positions.Vector) is
      begin
         if Position <= Integer (Item.Length) then
            Get.Item := Positions.Element (Item, Position);
         else
            Get.Item := Null_Data;
         end if;
      end Query;

   begin
      Container.Data.Query_Element (Iteration, Query'Access);
   end Get;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Integer
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Position  => Position,
         Iteration => Iteration);

      return Result.Int;
   end Get;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Long_Float
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Position  => Position,
         Iteration => Iteration);

      return Result.Flt;
   end Get;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return String
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Position  => Position,
         Iteration => Iteration);

      return Value (Result);
   end Get;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return OCINumber
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Position  => Position,
         Iteration => Iteration);

      return Result.Numb;
   end Get;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Ada.Calendar.Time
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Position  => Position,
         Iteration => Iteration);

      return Result.Dat;
   end Get;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Lobs.Locator
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Position  => Position,
         Iteration => Iteration);

      return Result.Loc;
   end Get;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Data_Holder
   is
      Result : Data_Holder;

   begin
      Get
        (Container => Container,
         Item      => Result,
         Position  => Position,
         Iteration => Iteration);

      return Result;
   end Get;

   -----------
   -- Image --
   -----------

   function Image
     (Item : in Data_Holder; Null_Image : in String := "") return String
   is
      use OCI.Thick.Date;

      function Reduced_Float_Image return String;

      function Reduced_Float_Image return String is
         use Ada.Strings;

         Img    : constant String := Long_Float'Image (Item.Flt);
         Last_0 : constant Natural := Fixed.Index (Img, "0E", Backward);
      begin
         if Last_0 = 0 then
            return Img;
         end if;

         for J in reverse Img'First + 1 .. Last_0 - 1 loop
            if Img (J) /= '0' then
               return Img (Img'First .. J - Boolean'Pos (Img (J) = '.'))
                      & Img (Last_0 + 1 .. Img'Last);
            end if;
         end loop;

         raise Program_Error;
      end Reduced_Float_Image;

   begin
      case Item.Kind is
      when Type_Null       => return Null_Image;
      when Type_String     => return Value (Item);
      when Type_Number     => return Number_Functions.To_String (Item.Numb);
      when Type_Integer    => return Item.Int'Img;
      when Type_Long_Float => return Reduced_Float_Image;
      when Type_Date       => return To_String (To_OCI (Item.Dat));
      when Type_Lob        => return "(LOB)";
      end case;
   end Image;

   -------------
   -- Is_Null --
   -------------

   function Is_Null (Item : in Data_Holder) return Boolean is
      use Ada.Strings.Unbounded;
   begin
      return Item.Kind = Type_Null
         or else (Item.Kind = Type_String
                  and then Item.Str = Null_Unbounded_String);
   end Is_Null;

   function Is_Null
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Boolean
   is
      Result : Boolean := True;

      procedure Query (Item : in Positions.Vector);

      -----------
      -- Query --
      -----------

      procedure Query (Item : in Positions.Vector) is
      begin
         if Position <= Integer (Item.Length) then
            Result := Is_Null (Item.Element (Position));
         end if;
      end Query;

   begin
      Container.Data.Query_Element (Iteration, Query'Access);

      return Result;
   end Is_Null;

   ------------
   -- Length --
   ------------

   function Length (Container : in Container_Type) return Natural is
   begin
      return Natural (Container.Data.Length);
   end Length;

   ---------
   -- Set --
   ---------

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Data_Holder;
      Position  : in     Positive;
      Iteration : in     Positive := 1)
   is
      use type Ada.Containers.Count_Type;

      procedure Modify (Item : in out Positions.Vector);

      ------------
      -- Modify --
      ------------

      procedure Modify (Item : in out Positions.Vector) is
      begin
         if Natural (Item.Length) < Position then
            Item.Append
              (Null_Data, Ada.Containers.Count_Type (Position) - Item.Length);
         end if;

         Item.Replace_Element (Position, Set.Item);
      end Modify;

   begin
      if Container.Length < Iteration then
         Container.Data.Append
           (Positions.Empty_Vector,
            Ada.Containers.Count_Type (Iteration - Container.Length));
      end if;

      Container.Data.Update_Element (Iteration, Modify'Access);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Integer;
      Position  : in     Positive;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Integer, Int => Item),
         Position  => Position,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Long_Float;
      Position  : in     Positive;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Long_Float, Flt => Item),
         Position  => Position,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     String;
      Position  : in     Positive;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'
                        (Kind => Type_String,
                         Str  =>
                           Ada.Strings.Unbounded.To_Unbounded_String (Item)),
         Position  => Position,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     OCINumber;
      Position  : in     Positive;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Number, Numb => Item),
         Position  => Position,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Ada.Calendar.Time;
      Position  : in     Positive;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Date, Dat => Item),
         Position  => Position,
         Iteration => Iteration);
   end Set;

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Lobs.Locator;
      Position  : in     Positive;
      Iteration : in     Positive := 1) is
   begin
      Set
        (Container => Container,
         Item      => Data_Holder'(Kind => Type_Lob, Loc => Item),
         Position  => Position,
         Iteration => Iteration);
   end Set;

   -------------
   -- To_Data --
   -------------

   function To_Data (Item : Integer) return Data_Holder is
   begin
      return (Kind => Type_Integer, Int => Item);
   end To_Data;

   function To_Data (Item : String) return Data_Holder is
   begin
      return (Kind => Type_String,
              Str  => Ada.Strings.Unbounded.To_Unbounded_String (Item));
   end To_Data;

   function To_Data (Item : C.char_array) return Data_Holder is
      use type C.char;
   begin
      return To_Data (C.To_Ada (Item, Trim_Nul => Item (Item'Last) = C.nul));
   end To_Data;

   function To_Data (Item : OCINumber) return Data_Holder is
   begin
      return (Kind => Type_Number, Numb => Item);
   end To_Data;

   function To_Data (Item : Long_Float) return Data_Holder is
   begin
      return (Kind => Type_Long_Float, Flt => Item);
   end To_Data;

   function To_Data (Item : Ada.Calendar.Time) return Data_Holder is
   begin
      return (Kind => Type_Date, Dat => Item);
   end To_Data;

   function To_Data (Item : Lobs.Locator) return Data_Holder is
   begin
      return (Kind => Type_Lob, Loc => Item);
   end To_Data;

   -----------
   -- Value --
   -----------

   function Value
     (Item : in Data_Holder; Default : in Integer) return Integer is
   begin
      if Item.Kind = Type_Null then
         return Default;
      else
         return Item.Int;
      end if;
   end Value;

   function Value (Item : in Data_Holder; Default : in String) return String is
   begin
      if Item.Kind = Type_Null then
         return Default;
      else
         return Ada.Strings.Unbounded.To_String (Item.Str);
      end if;
   end Value;

   function Value
     (Item : in Data_Holder; Default : in Long_Float) return Long_Float is
   begin
      if Item.Kind = Type_Null then
         return Default;
      else
         return Item.Flt;
      end if;
   end Value;

   function Value
     (Item : in Data_Holder; Default : in OCINumber) return OCINumber is
   begin
      if Item.Kind = Type_Null then
         return Default;
      else
         return Item.Numb;
      end if;
   end Value;

   function Value
     (Item : in Data_Holder; Default : in Ada.Calendar.Time)
      return Ada.Calendar.Time is
   begin
      if Item.Kind = Type_Null then
         return Default;
      else
         return Item.Dat;
      end if;
   end Value;

   function Value (Item : in Data_Holder) return Integer is
   begin
      return Item.Int;
   end Value;

   function Value (Item : in Data_Holder) return String is
   begin
      return Value (Item, "");
   end Value;

   function Value (Item : in Data_Holder) return Long_Float is
   begin
      return Item.Flt;
   end Value;

   function Value (Item : in Data_Holder) return OCINumber is
   begin
      return Item.Numb;
   end Value;

   function Value (Item : in Data_Holder) return Ada.Calendar.Time is
   begin
      return Item.Dat;
   end Value;

   function Value (Item : in Data_Holder) return Lobs.Locator is
   begin
      return Item.Loc;
   end Value;

end OCI.Thick.Containers;
